;;; iso-639.el --- ISO 639                   -*- lexical-binding: t; -*-

;; Copyright (C) 2024  Jean Libète

;; Author: Jean Libète <tomenzgg@mail.mayfirst.org>
;; URL: https://codeberg.org/tomenzgg/emacs-iso-639
;; Package-Requires: ((emacs "27.1"))
;; Package-Version: 1.1.0
;; Package-Revision: d55dbbea5291
;; Keywords: tools, multilingual, language, iso-639

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU Affero General Public License as published
;; by the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU Affero General Public License
;; along with this program.  If not, see <http://www.gnu.org/licenses/>.

;;; Commentary:

;; A library to handle ISO 639 language codes; attempts to cover 1–3.

;;; Code:
;; seq-find
;; seq-subseq
(require 'seq)

(defconst iso-639---languages
          ;; 3     1    2     ret sco typ mac family          nat english
          '(("aaa" nil  nil   nil I   L   () "Atlantic–Congo" nil "Ghotuo")
            ("aab" nil  nil   nil I   L   () "Atlantic–Congo" nil "Alumu-Tesu")
            ("aac" nil  nil   nil I   L   () "Suki-Gogodala" nil "Ari")
            ("aad" nil  nil   nil I   L   () "Sepik" nil "Amal")
            ("aae" nil  nil   nil I   L   () "Indo-European" nil "Albanian (Arbëreshë Dialect)")
            ("aaf" nil  nil   nil I   L   () "Dravidian" nil "Aranadan")
            ("aag" nil  nil   nil I   L   () "Torricelli" nil "Ambrak")
            ("aah" nil  nil   nil I   L   () "Torricelli" nil "Abu’ Arapesh")
            ("aai" nil  nil   nil I   L   () "Austronesian" nil "Arifama-Miniafia")
            ("aak" nil  nil   nil I   L   () "Angan" nil "Ankave")
            ("aal" nil  nil   nil I   L   () "Afro-Asiatic" "afaë" "Afade")
            ("aam" nil  nil   t   I   L   () "spurious language" nil "Aramanik")
            ("aan" nil  nil   nil I   L   () "Tupian" nil "Anambé")
            ("aao" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic (Algerian Saharan)")
            ("aap" nil  nil   nil I   L   () "Cariban" nil "Arára, Pará")
            ("aaq" nil  nil   nil I   E   () "Algic" nil "Abnaki (Eastern)")
            ("aar" "aa" "aar" nil I   L   () "Afro-Asiatic" "Afaraf" "Afar")
            ("aas" nil  nil   nil I   L   () "Afro-Asiatic" nil "Aasáx")
            ("aat" nil  nil   nil I   L   () "Indo-European" nil "Albanian (Arvanitika)")
            ("aau" nil  nil   nil I   L   () "Sepik" nil "Abau")
            ("aaw" nil  nil   nil I   L   () "Austronesian" nil "Solong")
            ("aax" nil  nil   nil I   L   () "Trans–New Guinea" nil "Mandobo Atas")
            ("aay" nil  nil   t   I   L   () "spurious language" nil "Aariya")
            ("aaz" nil  nil   nil I   L   () "Austronesian" nil "Amarasi")
            ("aba" nil  nil   nil I   L   () "Atlantic–Congo" nil "Abé")
            ("abb" nil  nil   nil I   L   () "Atlantic–Congo" nil "Bankon")
            ("abc" nil  nil   nil I   L   () "Austronesian" nil "Ayta, Ambala")
            ("abd" nil  nil   nil I   L   () "Austronesian" nil "Agta, Camarines Norte")
            ("abe" nil  nil   nil I   L   () "Algic" "Wôbanakiôdwawôgan" "Abnaki, Western")
            ("abf" nil  nil   nil I   L   () "Austronesian" nil "Abai Sungai")
            ("abg" nil  nil   nil I   L   () "Trans–New Guinea" nil "Abaga")
            ("abh" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic (Tajiki)")
            ("abi" nil  nil   nil I   L   () "Atlantic–Congo" nil "Abidji")
            ("abj" nil  nil   nil I   E   () "Great Andamanese" nil "Aka-Bea")
            ("abk" "ab" "abk" nil I   L   () "Northwest Caucasian" "Аҧсуа" "Abkhazian")
            ("abl" nil  nil   nil I   L   () "Austronesian" nil "Abung")
            ("abm" nil  nil   nil I   L   () "Atlantic–Congo" nil "Abanyom")
            ("abn" nil  nil   nil I   L   () "Atlantic–Congo" nil "Abua")
            ("abo" nil  nil   nil I   L   () "Atlantic–Congo" nil "Abon")
            ("abp" nil  nil   nil I   L   () "Austronesian" nil "Ayta, Abenlen")
            ("abq" nil  nil   nil I   L   () "Northwest Caucasian" "абаза" "Abaza")
            ("abr" nil  nil   nil I   L   () "Atlantic–Congo" nil "Abron")
            ("abs" nil  nil   nil I   L   () "Austronesian" nil "Malay, Ambonese")
            ("abt" nil  nil   nil I   L   () "Ndu" nil "Ambulas")
            ("abu" nil  nil   nil I   L   () "Atlantic–Congo" "ɔbule ɔyʋɛ" "Abure")
            ("abv" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic (Baharna)")
            ("abw" nil  nil   nil I   L   () "Trans–New Guinea" nil "Pal")
            ("abx" nil  nil   nil I   L   () "Austronesian" nil "Inabaknon")
            ("aby" nil  nil   nil I   L   () "Yareban" nil "Aneme Wake")
            ("abz" nil  nil   nil I   L   () "Timor-Alor-Pantar" nil "Abui")
            ("aca" nil  nil   nil I   L   () "Arawakan" nil "Achagua")
            ("acb" nil  nil   nil I   L   () "spurious language" nil "Áncá")
            ("acc" nil  nil   t   I   L   () "spurious language" nil "Achí, Cubulco")
            ("acd" nil  nil   nil I   L   () "Atlantic–Congo" nil "Gikyode")
            ("ace" nil  "ace" nil I   L   () "Austronesian" "Aceh" "Achinese")
            ("acf" nil  nil   nil I   L   () "French Creole" "kwéyòl" "Saint Lucian Creole French")
            ("ach" nil  "ach" nil I   L   () "Nilotic" nil "Acoli")
            ("aci" nil  nil   nil I   E   () "Great Andamanese" nil "Aka-Cari")
            ("ack" nil  nil   nil I   E   () "Great Andamanese" nil "Aka-Kora")
            ("acl" nil  nil   nil I   E   () "Great Andamanese" nil "Akar-Bale")
            ("acm" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic (Mesopotamian)")
            ("acn" nil  nil   nil I   L   () "Sino-Tibetan" "Mönghsa" "Achang")
            ("acp" nil  nil   nil I   L   () "Atlantic–Congo" nil "Acipa, Eastern")
            ("acq" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic, Ta'izzi-Adeni Spoken")
            ("acr" nil  nil   nil I   L   () "Mayan" nil "Achí, Rabinal")
            ("acs" nil  nil   nil I   E   () "Macro-Jê" nil "Acroá")
            ("act" nil  nil   nil I   L   () "Indo-European" "Achterhooks" "Achterhooks")
            ("acu" nil  nil   nil I   L   () "Jivaroan" nil "Achuar-Shiwiar")
            ("acv" nil  nil   nil I   L   () "Palaihnihan" "Ajúmmááwí" "Achumawi")
            ("acw" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic (Hijazi)")
            ("acx" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic (Omani)")
            ("acy" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic (Cypriot)")
            ("acz" nil  nil   nil I   L   () "Talodi" nil "Acheron")
            ("ada" nil  "ada" nil I   L   () "Atlantic–Congo" nil "Adangme")
            ("adb" nil  nil   nil I   L   () "spurious language" nil "Adabe")
            ("add" nil  nil   nil I   L   () "Atlantic–Congo" nil "Dzodinka")
            ("ade" nil  nil   nil I   L   () "Atlantic–Congo" nil "Adele")
            ("adf" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic (Dhofari)")
            ("adg" nil  nil   nil I   L   () "Pama–Nyungan" nil "Andegerebinha")
            ("adh" nil  nil   nil I   L   () "Nilotic" nil "Adhola")
            ("adi" nil  nil   nil I   L   () "Sino-Tibetan" nil "Adi")
            ("adj" nil  nil   nil I   L   () "Atlantic–Congo" "mɔjukru" "Adioukrou")
            ("adl" nil  nil   nil I   L   () "Sino-Tibetan" nil "Adi (Galo)")
            ("adn" nil  nil   nil I   L   () "Timor-Alor-Pantar" nil "Adang")
            ("ado" nil  nil   nil I   L   () "Lower Sepik-Ramu" nil "Abu")
            ("adp" nil  nil   t   I   L   () "spurious language" nil "Adap")
            ("adq" nil  nil   nil I   L   () "Atlantic–Congo" nil "Adangbe")
            ("adr" nil  nil   nil I   L   () "Austronesian" nil "Adonara")
            ("ads" nil  nil   nil I   L   () "West African gestural area" nil "Adamorobe Sign Language")
            ("adt" nil  nil   nil I   L   () "Pama–Nyungan" "Yura Ngawarla" "Adynyamathanha")
            ("adu" nil  nil   nil I   L   () "spurious language" nil "Aduge")
            ("adw" nil  nil   nil I   L   () "Tupian" nil "Amundava")
            ("adx" nil  nil   nil I   L   () "Sino-Tibetan" nil "Tibetan, Amdo")
            ("ady" nil  "ady" nil I   L   () "Northwest Caucasian" "адыгэбзэ" "Adyghe; Adygei")
            ("adz" nil  nil   nil I   L   () "Austronesian" nil "Adzera")
            ("aea" nil  nil   nil I   E   () "Pama–Nyungan" nil "Areba")
            ("aeb" nil  nil   nil I   L   () "Afro-Asiatic" "تونسي" "Arabic, Tunisian Spoken")
            ("aec" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic, Saidi Spoken")
            ("aed" nil  nil   nil I   L   () "Sign Language" nil "Argentine Sign Language")
            ("aee" nil  nil   nil I   L   () "Indo-European" nil "Pashayi, Northeast")
            ("aek" nil  nil   nil I   L   () "Austronesian" nil "Haeke")
            ("ael" nil  nil   nil I   L   () "Atlantic–Congo" nil "Ambele")
            ("aem" nil  nil   nil I   L   () "Austroasiatic" nil "Arem")
            ("aen" nil  nil   nil I   L   () "Sign Language" nil "Armenian Sign Language")
            ("aeq" nil  nil   nil I   L   () "Indo-Aryan" nil "Aer")
            ("aer" nil  nil   nil I   L   () "Pama–Nyungan" nil "Arrernte, Eastern")
            ("aes" nil  nil   nil I   E   () "isolate" nil "Alsea")
            ("aeu" nil  nil   nil I   L   () "Sino-Tibetan" nil "Akeu")
            ("aew" nil  nil   nil I   L   () "Keram" nil "Ambakich")
            ("aex" nil  nil   t   I   L   () "spurious language" nil "Amerax")
            ("aey" nil  nil   nil I   L   () "Trans–New Guinea" nil "Amele")
            ("aez" nil  nil   nil I   L   () "Trans–New Guinea" nil "Aeka")
            (nil   nil  "afa" nil C   nil () "Afro-Asiatic" nil "Afro-Asiatic languages")
            ("afb" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic, Gulf Spoken")
            ("afd" nil  nil   nil I   L   () "Arafundi" nil "Andai")
            ("afe" nil  nil   nil I   L   () "Atlantic–Congo" nil "Putukwam")
            ("afg" nil  nil   nil I   L   () "Sign Language" nil "Afghan Sign Language")
            ("afh" nil  "afh" nil I   C   () "constructed" nil "Afrihili")
            ("afi" nil  nil   nil I   L   () "Lower Sepik-Ramu" nil "Akrukay")
            ("afk" nil  nil   nil I   L   () "Arafundi" nil "Nanubae")
            ("afn" nil  nil   nil I   L   () "Ijoid" nil "Defaka")
            ("afo" nil  nil   nil I   L   () "Atlantic–Congo" nil "Eloyi")
            ("afp" nil  nil   nil I   L   () "Arafundi" nil "Tapei")
            ("afr" "af" "afr" nil I   L   () "Indo-European" "Afrikaans" "Afrikaans")
            ("afs" nil  nil   nil I   L   () "English Creole" nil "Afro-Seminole Creole")
            ("aft" nil  nil   nil I   L   () "Nyimang" nil "Afitti")
            ("afu" nil  nil   nil I   L   () "Atlantic-Congo" nil "Awutu")
            ("afz" nil  nil   nil I   L   () "Lakes Plain" nil "Obokuitai")
            ("aga" nil  nil   nil I   E   () "unattested" nil "Aguano")
            ("agb" nil  nil   nil I   L   () "Atlantic–Congo" nil "Legbo")
            ("agc" nil  nil   nil I   L   () "Atlantic–Congo" nil "Agatu")
            ("agd" nil  nil   nil I   L   () "Trans–New Guinea" nil "Agarabi")
            ("age" nil  nil   nil I   L   () "Trans–New Guinea" nil "Angal")
            ("agf" nil  nil   nil I   L   () "Austronesian" nil "Arguni")
            ("agg" nil  nil   nil I   L   () "Senagi" nil "Angor")
            ("agh" nil  nil   nil I   L   () "Atlantic–Congo" nil "Ngelima")
            ("agi" nil  nil   nil I   L   () "unattested" nil "Agariya")
            ("agj" nil  nil   nil I   L   () "Afro-Asiatic" nil "Argobba")
            ("agk" nil  nil   nil I   L   () "Austronesian" nil "Agta, Isarog")
            ("agl" nil  nil   nil I   L   () "East Strickland" nil "Fembe")
            ("agm" nil  nil   nil I   L   () "Angan" nil "Angaatiha")
            ("agn" nil  nil   nil I   L   () "Austronesian" nil "Agutaynen")
            ("ago" nil  nil   nil I   L   () "Angan" nil "Tainae")
            ("agp" nil  nil   t   I   L   () "Austronesian" nil "Paranan")
            ("agq" nil  nil   nil I   L   () "Atlantic–Congo" "aghím" "Aghem")
            ("agr" nil  nil   nil I   L   () "Jivaroan" "awajun" "Aguaruna")
            ("ags" nil  nil   nil I   L   () "Atlantic–Congo" nil "Esimbi")
            ("agt" nil  nil   nil I   L   () "Austronesian" nil "Agta, Central Cagayan")
            ("agu" nil  nil   nil I   L   () "Mayan" "awakateko" "Aguacateco")
            ("agv" nil  nil   nil I   L   () "Austronesian" nil "Agta, Remontado")
            ("agw" nil  nil   nil I   L   () "Austronesian" nil "Kahua")
            ("agx" nil  nil   nil I   L   () "Northeast Caucasian" "агъул" "Aghul")
            ("agy" nil  nil   nil I   L   () "Austronesian" nil "Alta, Southern")
            ("agz" nil  nil   nil I   L   () "Austronesian" nil "Agta, Mt. Iriga")
            ("aha" nil  nil   nil I   L   () "Atlantic–Congo" nil "Ahanta")
            ("ahb" nil  nil   nil I   L   () "Austronesian" nil "Axamb")
            ("ahe" nil  nil   t   I   L   () "spurious language" nil "Ahe")
            ("ahg" nil  nil   nil I   L   () "Afro-Asiatic" nil "Qimant")
            ("ahh" nil  nil   nil I   L   () "Trans–New Guinea" nil "Aghu")
            ("ahi" nil  nil   nil I   L   () "Kru" nil "Aizi, Tiagbamrin")
            ("ahk" nil  nil   nil I   L   () "Sino-Tibetan" nil "Akha")
            ("ahl" nil  nil   nil I   L   () "Atlantic–Congo" nil "Igo")
            ("ahm" nil  nil   nil I   L   () "Kru" nil "Aizi, Mobumrin")
            ("ahn" nil  nil   nil I   L   () "Atlantic–Congo" nil "Àhàn")
            ("aho" nil  nil   nil I   E   () "Tai–Kadai" nil "Ahom")
            ("ahp" nil  nil   nil I   L   () "Atlantic–Congo" nil "Aizi, Aproumu")
            ("ahr" nil  nil   nil I   L   () "spurious language" nil "Ahirani")
            ("ahs" nil  nil   nil I   L   () "Atlantic–Congo" nil "Ashe")
            ("aht" nil  nil   nil I   L   () "Na-Dené" nil "Ahtena")
            ("aia" nil  nil   nil I   L   () "Austronesian" nil "Arosi")
            ("aib" nil  nil   nil I   L   () "Turkic" nil "Ainu (China)")
            ("aic" nil  nil   nil I   L   () "Border" nil "Ainbai")
            ("aid" nil  nil   nil I   E   () "Pama–Nyungan" nil "Alngith")
            ("aie" nil  nil   nil I   L   () "Austronesian" nil "Amara")
            ("aif" nil  nil   nil I   L   () "Torricelli" nil "Agi")
            ("aig" nil  nil   nil I   L   () "English Creole" nil "Antigua and Barbuda Creole English")
            ("aih" nil  nil   nil I   L   () "Tai–Kadai" nil "Ai-Cham")
            ("aii" nil  nil   nil I   L   () "Afro-Asiatic" "ܣܘܪܝܝܐ ܣܘܪܝܬ,ܐܬܘܪܝܐ ܣܘܪܝܝܐ" "Assyrian Neo-Aramaic")
            ("aij" nil  nil   nil I   L   () "Afro-Asiatic" nil "Lishanid Noshan")
            ("aik" nil  nil   nil I   L   () "Atlantic–Congo" nil "Ake")
            ("ail" nil  nil   nil I   L   () "Bosavi" nil "Aimele")
            ("aim" nil  nil   nil I   L   () "Sino-Tibetan" nil "Aimol")
            ("ain" nil  "ain" nil I   L   () "isolate" "アイヌ イタク(イタック)" "Ainu (Japan)")
            ("aio" nil  nil   nil I   L   () "Tai–Kadai" nil "Aiton")
            ("aip" nil  nil   nil I   L   () "Trans–New Guinea" nil "Burumakok")
            ("aiq" nil  nil   nil I   L   () "Indo-European" nil "Aimaq")
            ("air" nil  nil   nil I   L   () "Tor-Kwerba" nil "Airoran")
            ("ais" nil  nil   t   I   L   () "Austronesian" nil "Amis, Nataoran")
            ("ait" nil  nil   nil I   E   () "Tupian" nil "Arikem")
            ("aiw" nil  nil   nil I   L   () "South Omotic" nil "Aari")
            ("aix" nil  nil   nil I   L   () "Austronesian" nil "Aigon")
            ("aiy" nil  nil   nil I   L   () "Atlantic–Congo" nil "Ali")
            ("aiz" nil  nil   t   I   L   () "Afro-Asiatic" nil "Aari")
            ("aja" nil  nil   nil I   L   () "Kresh-Aja" nil "Aja (Sudan)")
            ("ajg" nil  nil   nil I   L   () "Atlantic–Congo" nil "Aja (Benin)")
            ("aji" nil  nil   nil I   L   () "Austronesian" nil "Ajië")
            ("ajn" nil  nil   nil I   L   () "Worroran" nil "Andajin")
            ("ajp" nil  nil   t   I   L   () "Afro-Asiatic" nil "Arabic (South Levantine)")
            ("ajs" nil  nil   nil I   L   () "village sign language" nil "Algerian Jewish Sign Language")
            ("ajt" nil  nil   t   I   L   () "Afro-Asiatic" nil "Arabic (Judeo-Tunisian)")
            ("aju" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic (Judeo-Moroccan)")
            ("ajw" nil  nil   nil I   E   () "Afro-Asiatic" nil "Ajawa")
            ("ajz" nil  nil   nil I   L   () "Sino-Tibetan" nil "Amri")
            ("aka" "ak" "aka" nil M   L   ("fat" "twi") "Atlantic–Congo" "Akan" "Akan")
            ("akb" nil  nil   nil I   L   () "Austronesian" nil "Batak Angkola")
            ("akc" nil  nil   nil I   L   () "isolate" nil "Mpur")
            ("akd" nil  nil   nil I   L   () "Atlantic–Congo" nil "Ukpet-Ehom")
            ("ake" nil  nil   nil I   L   () "Cariban" nil "Akawaio")
            ("akf" nil  nil   nil I   L   () "Niger–Congo" nil "Akpa")
            ("akg" nil  nil   nil I   L   () "Austronesian" nil "Anakalangu")
            ("akh" nil  nil   nil I   L   () "Trans–New Guinea" nil "Angal Heneng")
            ("aki" nil  nil   nil I   L   () "Ramu" nil "Aiome")
            ("akj" nil  nil   nil I   E   () "Great Andamanese" nil "Aka-Jeru")
            ("akk" nil  "akk" nil I   A   () "Afro-Asiatic" "akkadû" "Akkadian")
            ("akl" nil  nil   nil I   L   () "Austronesian" "Inakeanon" "Aklanon")
            ("akm" nil  nil   nil I   E   () "Great Andamanese" nil "Aka-Bo")
            ("akn" nil  nil   t   I   L   () "spurious language" nil "Amikoana")
            ("ako" nil  nil   nil I   L   () "Cariban" nil "Akurio")
            ("akp" nil  nil   nil I   L   () "Niger–Congo" nil "Siwu")
            ("akq" nil  nil   nil I   L   () "Sepik" nil "Ak")
            ("akr" nil  nil   nil I   L   () "Austronesian" nil "Araki")
            ("aks" nil  nil   nil I   L   () "Niger–Congo" nil "Akaselem")
            ("akt" nil  nil   nil I   L   () "Austronesian" nil "Akolet")
            ("aku" nil  nil   nil I   L   () "Niger–Congo" "aakuem" "Akum")
            ("akv" nil  nil   nil I   L   () "Northeast Caucasian" nil "Akhvakh")
            ("akw" nil  nil   nil I   L   () "Niger–Congo" nil "Akwa")
            ("akx" nil  nil   nil I   E   () "Great Andamanese" nil "Aka-Kede")
            ("aky" nil  nil   nil I   E   () "Great Andamanese" nil "Aka-Kol")
            ("akz" nil  nil   nil I   L   () "Muskogean" "Albaamo innaaɬiilka" "Alabama")
            ("ala" nil  nil   nil I   L   () "Niger–Congo" nil "Alago")
            ("alc" nil  nil   nil I   L   () "Alacalufan" "alacalufe" "Qawasqar")
            ("ald" nil  nil   nil I   L   () "Niger–Congo" nil "Alladian")
            ("ale" nil  "ale" nil I   L   () "Eskimo-Aleut" "Unangax tunuu" "Aleut")
            ("alf" nil  nil   nil I   L   () "Niger–Congo" nil "Alege")
            (nil   nil  "alg" nil C   nil () "Algic" nil "Algonquian languages")
            ("alh" nil  nil   nil I   L   () "Arnhem" nil "Alawa")
            ("ali" nil  nil   nil I   L   () "Trans–New Guinea" nil "Amaimon")
            ("alj" nil  nil   nil I   L   () "Austronesian" nil "Alangan")
            ("alk" nil  nil   nil I   L   () "Austroasiatic" nil "Alak")
            ("all" nil  nil   nil I   L   () "Dravidian" nil "Allar")
            ("alm" nil  nil   nil I   L   () "Austronesian" nil "Amblong")
            ("aln" nil  nil   nil I   L   () "Indo-European" "gegnisht" "Albanian (Gheg)")
            ("alo" nil  nil   nil I   L   () "Austronesian" nil "Larike-Wakasihu")
            ("alp" nil  nil   nil I   L   () "Austronesian" nil "Alune")
            ("alq" nil  nil   nil I   L   () "Algic" "Anishnaabemowin (Omaamiwininimowin)" "Algonquin")
            ("alr" nil  nil   nil I   L   () "Chukotko-Kamchatkan" nil "Alutor")
            ("als" nil  nil   nil I   L   () "Indo-European" "toskërishte" "Albanian (Tosk)")
            ("alt" nil  "alt" nil I   L   () "Turkic" "Алтай тили" "Altai (Southern)")
            ("alu" nil  nil   nil I   L   () "Austronesian" nil "'Are'are")
            ("alw" nil  nil   nil I   L   () "Afro-Asiatic" nil "Alaba")
            ("alx" nil  nil   nil I   L   () "Torricelli" nil "Alatil")
            ("aly" nil  nil   nil I   L   () "Pama–Nyungan" nil "Alyawarr")
            ("alz" nil  nil   nil I   L   () "Nilo-Saharan" nil "Alur")
            ("ama" nil  nil   nil I   E   () "Tupian" nil "Amanayé")
            ("amb" nil  nil   nil I   L   () "Niger–Congo" nil "Ambo")
            ("amc" nil  nil   nil I   L   () "Pano-Tacanan" nil "Amahuaca")
            ("amd" nil  nil   t   I   L   () "spurious language" nil "Amapá Creole")
            ("ame" nil  nil   nil I   L   () "Arawakan" "Yanešač" "Yanesha'")
            ("amf" nil  nil   nil I   L   () "Afro-Asiatic" nil "Hamer-Banna")
            ("amg" nil  nil   nil I   L   () "Iwaidjan" nil "Amarag")
            ("amh" "am" "amh" nil I   L   () "Afro-Asiatic" "አማርኛ" "Amharic")
            ("ami" nil  nil   nil I   L   () "Austronesian" nil "Amis")
            ("amj" nil  nil   nil I   L   () "Nilo-Saharan" nil "Amdang")
            ("amk" nil  nil   nil I   L   () "Austronesian" nil "Ambai")
            ("aml" nil  nil   nil I   L   () "Austroasiatic" nil "War")
            ("amm" nil  nil   nil I   L   () "Left May" nil "Ama (Papua New Guinea)")
            ("amn" nil  nil   nil I   L   () "Border" nil "Amanab")
            ("amo" nil  nil   nil I   L   () "Niger–Congo" nil "Amo")
            ("amp" nil  nil   nil I   L   () "Sepik" nil "Alamblak")
            ("amq" nil  nil   nil I   L   () "Austronesian" nil "Amahai")
            ("amr" nil  nil   nil I   L   () "Otomakoan" nil "Amarakaeri")
            ("ams" nil  nil   nil I   L   () "Japonic" nil "Amami-Oshima, Southern")
            ("amt" nil  nil   nil I   L   () "Amto-Musan" nil "Amto")
            ("amu" nil  nil   nil I   L   () "Oto-Manguean" nil "Amuzgo, Guerrero")
            ("amv" nil  nil   nil I   L   () "Austronesian" nil "Ambelau")
            ("amw" nil  nil   nil I   L   () "Afro-Asiatic" "ܐܪܡܝܬ, آرامي" "Western Neo-Aramaic")
            ("amx" nil  nil   nil I   L   () "Pama–Nyungan" nil "Anmatyerre")
            ("amy" nil  nil   nil I   L   () "Northern Daly" nil "Ami")
            ("amz" nil  nil   nil I   E   () "Pama–Nyungan" nil "Atampaya")
            ("ana" nil  nil   nil I   E   () "Barbacoan" nil "Andaqui")
            ("anb" nil  nil   nil I   E   () "Zaparoan" nil "Andoa")
            ("anc" nil  nil   nil I   L   () "Afro-Asiatic" nil "Ngas")
            ("and" nil  nil   nil I   L   () "Austronesian" nil "Ansus")
            ("ane" nil  nil   nil I   L   () "Austronesian" nil "Xârâcùù")
            ("anf" nil  nil   nil I   L   () "Niger–Congo" nil "Animere")
            ("ang" nil  "ang" nil I   H   () "Indo-European" "Englisc" "Anglo-Saxon or Old English")
            ("anh" nil  nil   nil I   L   () "Trans–New Guinea" nil "Nend")
            ("ani" nil  nil   nil I   L   () "Northeast Caucasian" nil "Andi")
            ("anj" nil  nil   nil I   L   () "Ramu" nil "Anor")
            ("ank" nil  nil   nil I   L   () "Afro-Asiatic" nil "Goemai")
            ("anl" nil  nil   nil I   L   () "Sino-Tibetan" nil "Anu")
            ("anm" nil  nil   nil I   L   () "Sino-Tibetan" nil "Anal")
            ("ann" nil  nil   nil I   L   () "Niger–Congo" nil "Obolo")
            ("ano" nil  nil   nil I   L   () "isolate" nil "Andoque")
            ("anp" nil  "anp" nil I   L   () "Indo-European" nil "Angika")
            ("anq" nil  nil   nil I   L   () "Ongan" nil "Jarawa (India)")
            ("anr" nil  nil   nil I   L   () "Indo-European" nil "Andh")
            ("ans" nil  nil   nil I   E   () "Chocoan" nil "Anserma")
            ("ant" nil  nil   nil I   L   () "Pama–Nyungan" nil "Antakarinya")
            ("anu" nil  nil   nil I   L   () "Nilo-Saharan" nil "Anuak")
            ("anv" nil  nil   nil I   L   () "Niger–Congo" nil "Denya")
            ("anw" nil  nil   nil I   L   () "Niger–Congo" nil "Anaang")
            ("anx" nil  nil   nil I   L   () "Austronesian" nil "Andra-Hus")
            ("any" nil  nil   nil I   L   () "Niger–Congo" nil "Anyin")
            ("anz" nil  nil   nil I   L   () "Yele-West New Britain" nil "Anem")
            ("aoa" nil  nil   nil I   L   () "Portuguese Creole" nil "Angolar")
            ("aob" nil  nil   nil I   L   () "Trans–New Guinea" nil "Abom")
            ("aoc" nil  nil   nil I   L   () "Cariban" nil "Pemon")
            ("aod" nil  nil   nil I   L   () "Ramu" nil "Andarum")
            ("aoe" nil  nil   nil I   L   () "Trans–New Guinea" nil "Angal Enen")
            ("aof" nil  nil   nil I   L   () "Torricelli" nil "Bragat")
            ("aog" nil  nil   nil I   L   () "Ramu" nil "Angoram")
            ("aoh" nil  nil   t   I   E   () "spurious language" nil "Arma")
            ("aoi" nil  nil   nil I   L   () "Arnhem" nil "Anindilyakwa")
            ("aoj" nil  nil   nil I   L   () "Torricelli" nil "Mufian")
            ("aok" nil  nil   nil I   L   () "Austronesian" nil "Arhö")
            ("aol" nil  nil   nil I   L   () "Austronesian" nil "Alor")
            ("aom" nil  nil   nil I   L   () "Trans–New Guinea" nil "Ömie")
            ("aon" nil  nil   nil I   L   () "Torricelli" nil "Arapesh, Bumbita")
            ("aor" nil  nil   nil I   E   () "Austronesian" nil "Aore")
            ("aos" nil  nil   nil I   L   () "Border" nil "Taikat")
            ("aot" nil  nil   nil I   L   () "Sino-Tibetan" nil "A'tong")
            ("aou" nil  nil   nil I   L   () "Tai–Kadai" nil "Gelao")
            ("aox" nil  nil   nil I   L   () "Arawakan" nil "Atorada")
            ("aoz" nil  nil   nil I   L   () "Austronesian" nil "Uab Meto")
            (nil   nil  "apa" nil C   nil () "Apache" nil "Apache languages")
            ("apb" nil  nil   nil I   L   () "Austronesian" nil "Sa'a")
            ("apc" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic (North Levantine)")
            ("apd" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic (Sudanese)")
            ("ape" nil  nil   nil I   L   () "Torricelli" nil "Bukiyip")
            ("apf" nil  nil   nil I   L   () "Austronesian" nil "Pahanan Agta")
            ("apg" nil  nil   nil I   L   () "Austronesian" nil "Ampanang")
            ("aph" nil  nil   nil I   L   () "Sino-Tibetan" nil "Athpariya")
            ("api" nil  nil   nil I   L   () "Tupian" nil "Apiacá")
            ("apj" nil  nil   nil I   L   () "Apache" "Abáachi mizaa" "Apache (Jicarilla)")
            ("apk" nil  nil   nil I   L   () "Apache" nil "Apache (Kiowa)")
            ("apl" nil  nil   nil I   L   () "Apache" nil "Apache (Lipan)")
            ("apm" nil  nil   nil I   L   () "Apache" nil "Apache (Mescalero-Chiricahua)")
            ("apn" nil  nil   nil I   L   () "Macro-Jê" nil "Apinayé")
            ("apo" nil  nil   nil I   L   () "Austronesian" nil "Apalik")
            ("app" nil  nil   nil I   L   () "Austronesian" nil "Apma")
            ("apq" nil  nil   nil I   L   () "Great Andamanese" nil "A-Pucikwar")
            ("apr" nil  nil   nil I   L   () "Austronesian" nil "Arop-Lukep")
            ("aps" nil  nil   nil I   L   () "Austronesian" nil "Arop-Sissano")
            ("apt" nil  nil   nil I   L   () "Sino-Tibetan" nil "Apatani")
            ("apu" nil  nil   nil I   L   () "Arawakan" nil "Apurinã")
            ("apv" nil  nil   nil I   E   () "Nambikwaran" nil "Alapmunte")
            ("apw" nil  nil   nil I   L   () "Apache" "Ndéé biyáti'" "Apache (Western)")
            ("apx" nil  nil   nil I   L   () "Austronesian" nil "Aputai")
            ("apy" nil  nil   nil I   L   () "Cariban" nil "Apalaí")
            ("apz" nil  nil   nil I   L   () "Trans–New Guinea" nil "Safeyoka")
            ("aqc" nil  nil   nil I   L   () "Northeast Caucasian" nil "Archi")
            ("aqd" nil  nil   nil I   L   () "Niger–Congo" nil "Ampari Dogon")
            ("aqg" nil  nil   nil I   L   () "Niger–Congo" nil "Arigidi")
            ("aqk" nil  nil   nil I   L   () "Niger–Congo" nil "Aninka")
            ("aqm" nil  nil   nil I   L   () "Trans–New Guinea" nil "Atohwaim")
            ("aqn" nil  nil   nil I   L   () "Austronesian" nil "Alta (Northern)")
            ("aqp" nil  nil   nil I   E   () "isolate" nil "Atakapa")
            ("aqr" nil  nil   nil I   L   () "Austronesian" nil "Arhâ")
            ("aqt" nil  nil   nil I   L   () "Mascoian" nil "Angaité")
            ("aqz" nil  nil   nil I   L   () "Tupian" nil "Akuntsu")
            ("ara" "ar" "ara" nil M   L   ("aao" "abh" "abv"
                                           "acm" "acq" "acw"
                                           "acx" "acy" "adf"
                                           "aeb" "aec" "afb"
                                           "apc" "apd" "arb"
                                           "arq" "ars" "ary"
                                           "arz" "auz" "avl"
                                           "ayh" "ayl" "ayn"
                                           "ayp" "pga" "shu"
                                           "ssh" "ajp" "bbz") "Afro-Asiatic" "العربية" "Arabic")
            ("arb" nil  nil   nil I   L   () "Afro-Asiatic" "لعربية" "Arabic (standard)")
            ("arc" nil  "arc" nil I   A   () "Afro-Asiatic" "ܐܪܡܝܐ" "Aramaic")
            ("ard" nil  nil   nil I   E   () "Pama–Nyungan" nil "Arabana")
            ("are" nil  nil   nil I   L   () "Pama–Nyungan" nil "Arrarnta, Western")
            ("arf" nil  nil   t   I   L   () "Arafundi" nil "Arafundi")
            ("arg" "an" "arg" nil I   L   () "Indo-European" "aragonés" "Aragonese")
            ("arh" nil  nil   nil I   L   () "Chibchan" nil "Arhuaco")
            ("ari" nil  nil   nil I   L   () "Caddoan" "sáhniš" "Arikara")
            ("arj" nil  nil   nil I   E   () "Tucanoan" nil "Arapaso")
            ("ark" nil  nil   nil I   L   () "Macro-Jê" nil "Arikapú")
            ("arl" nil  nil   nil I   L   () "Zaparoan" nil "Arabela")
            ("arn" nil  "arn" nil I   L   () "Araucanian" "Mapudungun" "Araucanian")
            ("aro" nil  nil   nil I   L   () "Pano-Tacanan" nil "Araona")
            ("arp" nil  "arp" nil I   L   () "Algic" "Hinono'eitiit" "Arapaho")
            ("arq" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic (Algerian)")
            ("arr" nil  nil   nil I   L   () "Tupian" nil "Karo (Brazil)")
            ("ars" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic (Najdi)")
            (nil   nil  "art" nil C   nil () nil nil "Artificial languages")
            ("aru" nil  nil   nil I   E   () "Arawan" nil "Arua")
            ("arv" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arbore")
            ("arw" nil  "arw" nil I   L   () "Maipurean" nil "Arawak")
            ("arx" nil  nil   nil I   L   () "Tupian" nil "Aruá")
            ("ary" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic (Moroccan)")
            ("arz" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic (Egyptian)")
            ("asa" nil  nil   nil I   L   () "Niger–Congo" nil "Asu (Tanzania)")
            ("asb" nil  nil   nil I   L   () "Siouan" "Nakʰóda" "Assiniboine")
            ("asc" nil  nil   nil I   L   () "Trans–New Guinea" nil "Asmat, Casuarina Coast")
            ("asd" nil  nil   t   I   L   () "Trans–New Guinea" nil "Asas")
            ("ase" nil  nil   nil I   L   () "French Sign" nil "American Sign Language")
            ("asf" nil  nil   nil I   L   () "BANZSL" nil "Australian Sign Language")
            ("asg" nil  nil   nil I   L   () "Niger–Congo" nil "Cishingini")
            ("ash" nil  nil   nil I   E   () "Tequiraca–Canichana" nil "Abishira")
            ("asi" nil  nil   nil I   L   () "Trans–New Guinea" nil "Buruwai")
            ("asj" nil  nil   nil I   L   () "Niger–Congo" nil "Nsari")
            ("ask" nil  nil   nil I   L   () "Indo-European" nil "Ashkun")
            ("asl" nil  nil   nil I   L   () "Austronesian" nil "Asilulu")
            ("asm" "as" "asm" nil I   L   () "Indo-European" "অসমীয়া" "Assamese")
            ("asn" nil  nil   nil I   L   () "Tupian" nil "Asuriní, Xingú")
            ("aso" nil  nil   nil I   L   () "Trans–New Guinea" nil "Dano")
            ("asp" nil  nil   nil I   L   () "French Sign" nil "Algerian Sign Language")
            ("asq" nil  nil   nil I   L   () "French Sign" nil "Austrian Sign Language")
            ("asr" nil  nil   nil I   L   () "Austroasiatic" nil "Asuri")
            ("ass" nil  nil   nil I   L   () "Niger–Congo" nil "Ipulo")
            ("ast" nil  "ast" nil I   L   () "Indo-European" "asturianu" "Asturian")
            ("asu" nil  nil   nil I   L   () "Tupian" nil "Asuriní")
            ("asv" nil  nil   nil I   L   () "Nilo-Saharan" nil "Asoa")
            ("asw" nil  nil   nil I   L   () "Australian Aboriginal Sign" nil "Australian Aborigines Sign Language")
            ("asx" nil  nil   nil I   L   () "Trans–New Guinea" nil "Muratayak")
            ("asy" nil  nil   nil I   L   () "Trans–New Guinea" nil "Asmat (Yaosakor)")
            ("asz" nil  nil   nil I   L   () "Austronesian" nil "As")
            ("ata" nil  nil   nil I   L   () "Yele-West New Britain" nil "Pele-Ata")
            ("atb" nil  nil   nil I   L   () "Sino-Tibetan" nil "Zaiwa")
            ("atc" nil  nil   nil I   E   () "Pano-Tacanan" nil "Atsahuaca")
            ("atd" nil  nil   nil I   L   () "Austronesian" nil "Manobo, Ata")
            ("ate" nil  nil   nil I   L   () "Trans–New Guinea" nil "Atemble")
            ("atf" nil  nil   t   I   L   () "spurious language" nil "Atuence")
            ("atg" nil  nil   nil I   L   () "Niger–Congo" nil "Ivbie North-Okpela-Arhe")
            (nil   nil  "ath" nil C   nil () nil nil "Athapascan languages")
            ("ati" nil  nil   nil I   L   () "Niger–Congo" nil "Attié")
            ("atj" nil  nil   nil I   L   () "Algic" "Atikamekw" "Atikamekw")
            ("atk" nil  nil   nil I   L   () "Austronesian" nil "Ati")
            ("atl" nil  nil   nil I   L   () "Austronesian" nil "Agta (Mt. Iraya)")
            ("atm" nil  nil   nil I   L   () "Yele-West New Britain" nil "Ata")
            ("atn" nil  nil   nil I   L   () "Indo-European" nil "Ashtiani")
            ("ato" nil  nil   nil I   L   () "Niger–Congo" nil "Atong")
            ("atp" nil  nil   nil I   L   () "Austronesian" nil "Atta (Pudtol)")
            ("atq" nil  nil   nil I   L   () "Austronesian" nil "Aralle-Tabulahan")
            ("atr" nil  nil   nil I   L   () "Cariban" nil "Atruahí")
            ("ats" nil  nil   nil I   L   () "Algic" nil "Gros Ventre")
            ("att" nil  nil   nil I   L   () "Austronesian" nil "Atta (Pamplona)")
            ("atu" nil  nil   nil I   L   () "Nilo-Saharan" nil "Reel")
            ("atv" nil  nil   nil I   L   () "Turkic" nil "Altai (Northern)")
            ("atw" nil  nil   nil I   L   () "Hokan" nil "Atsugewi")
            ("atx" nil  nil   nil I   L   () "Arutani–Sape" nil "Arutani")
            ("aty" nil  nil   nil I   L   () "Austronesian" nil "Aneityum")
            ("atz" nil  nil   nil I   L   () "Austronesian" nil "Arta")
            ("aua" nil  nil   nil I   L   () "Austronesian" nil "Asumboa")
            ("aub" nil  nil   nil I   L   () "Sino-Tibetan" nil "Alugu")
            ("auc" nil  nil   nil I   L   () "isolate" "Huao Terero" "Waorani")
            ("aud" nil  nil   nil I   L   () "Austronesian" nil "Anuta")
            ("aue" nil  nil   t   I   L   () "Kx'a" nil "ǂKx’auǁ’ein")
            ("aug" nil  nil   nil I   L   () "Niger–Congo" nil "Aguna")
            ("auh" nil  nil   nil I   L   () "Niger–Congo" nil "Aushi")
            ("aui" nil  nil   nil I   L   () "Austronesian" nil "Anuki")
            ("auj" nil  nil   nil I   L   () "Afro-Asiatic" nil "Awjilah")
            ("auk" nil  nil   nil I   L   () "Torricelli" nil "Heyo")
            ("aul" nil  nil   nil I   L   () "Austronesian" nil "Aulua")
            ("aum" nil  nil   nil I   L   () "Niger–Congo" nil "Asu (Nigeria)")
            ("aun" nil  nil   nil I   L   () "Torricelli" nil "One, Molmo")
            ("auo" nil  nil   nil I   E   () "Afro-Asiatic" nil "Auyokawa")
            ("aup" nil  nil   nil I   L   () "Trans–New Guinea" nil "Makayam")
            ("auq" nil  nil   nil I   L   () "Austronesian" nil "Anus")
            ("aur" nil  nil   nil I   L   () "Torricelli" nil "Aruek")
            (nil   nil  "aus" nil C   nil () nil nil "Australian languages")
            ("aut" nil  nil   nil I   L   () "Austronesian" nil "Austral")
            ("auu" nil  nil   nil I   L   () "Trans–New Guinea" nil "Auye")
            ("auv" nil  nil   t   I   L   () "Indo-European" nil "Auvergnat")
            ("auw" nil  nil   nil I   L   () "Border" nil "Awyi")
            ("aux" nil  nil   nil I   E   () "Tupian" nil "Aurá")
            ("auy" nil  nil   nil I   L   () "Trans–New Guinea" nil "Awiyaana")
            ("auz" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic (Uzbeki)")
            ("ava" "av" "ava" nil I   L   () "Northeast Caucasian" "авар" "Avaric")
            ("avb" nil  nil   nil I   L   () "Austronesian" nil "Avau")
            ("avd" nil  nil   nil I   L   () "Indo-European" nil "Alviri-Vidari")
            ("ave" "ae" "ave" nil I   A   () "Indo-European" "avesta" "Avestan")
            ("avi" nil  nil   nil I   L   () "Niger–Congo" nil "Avikam")
            ("avk" nil  nil   nil I   C   () "constructed" "Kotava" "Kotava")
            ("avl" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic (Eastern Egyptian Bedawi)")
            ("avm" nil  nil   nil I   E   () "Pama–Nyungan" nil "Angkamuthi")
            ("avn" nil  nil   nil I   L   () "Niger–Congo" nil "Avatime")
            ("avo" nil  nil   nil I   E   () "Arawakan" nil "Agavotaguerra")
            ("avs" nil  nil   nil I   E   () "Zaparoan" nil "Aushiri")
            ("avt" nil  nil   nil I   L   () "Torricelli" nil "Au")
            ("avu" nil  nil   nil I   L   () "Nilo-Saharan" nil "Avokaya")
            ("avv" nil  nil   nil I   L   () "Tupian" nil "Avá-Canoeiro")
            ("awa" nil  "awa" nil I   L   () "Indo-European" "आवधी" "Awadhi")
            ("awb" nil  nil   nil I   L   () "Trans–New Guinea" nil "Awa")
            ("awc" nil  nil   nil I   L   () "Niger–Congo" nil "Acipa (Western)")
            ("awe" nil  nil   nil I   L   () "Tupian" nil "Awetí")
            ("awg" nil  nil   nil I   E   () "Pama–Nyungan" nil "Anguthimri")
            ("awh" nil  nil   nil I   L   () "Trans–New Guinea" nil "Awbono")
            ("awi" nil  nil   nil I   L   () "Trans–New Guinea" nil "Aekyom")
            ("awk" nil  nil   nil I   E   () "Pama–Nyungan" nil "Awabakal")
            ("awm" nil  nil   nil I   L   () "Trans–New Guinea" nil "Arawum")
            ("awn" nil  nil   nil I   L   () "Afro-Asiatic" nil "Awngi")
            ("awo" nil  nil   nil I   L   () "Niger–Congo" nil "Awak")
            ("awr" nil  nil   nil I   L   () "Lakes Plain" nil "Awera")
            ("aws" nil  nil   nil I   L   () "Trans–New Guinea" nil "Awyu (South)")
            ("awt" nil  nil   nil I   L   () "Tupian" nil "Araweté")
            ("awu" nil  nil   nil I   L   () "Trans–New Guinea" nil "Awyu (Central)")
            ("awv" nil  nil   nil I   L   () "Trans–New Guinea" nil "Awyu (Jair)")
            ("aww" nil  nil   nil I   L   () "Trans–New Guinea" nil "Awun")
            ("awx" nil  nil   nil I   L   () "Trans–New Guinea" nil "Awara")
            ("awy" nil  nil   nil I   L   () "Trans–New Guinea" nil "Awyu (Edera)")
            ("axb" nil  nil   nil I   E   () "Guaicuruan" nil "Abipon")
            ("axe" nil  nil   nil I   E   () "Pama–Nyungan" nil "Ayerrerenge")
            ("axg" nil  nil   nil I   E   () "unclassified" nil "Arára (Mato Grosso)")
            ("axk" nil  nil   nil I   L   () "Niger–Congo" nil "Yaka (Central African Republic)")
            ("axl" nil  nil   nil I   E   () "Pama–Nyungan" nil "Lower Southern Aranda")
            ("axm" nil  nil   nil I   H   () "Indo-European" nil "Armenian, Middle")
            ("axx" nil  nil   nil I   L   () "Austronesian" nil "Xaragure")
            ("aya" nil  nil   nil I   L   () "Ramu" nil "Awar")
            ("ayb" nil  nil   nil I   L   () "Niger–Congo" nil "Gbe, Ayizo")
            ("ayc" nil  nil   nil I   L   () "Aymaran" nil "Aymara, Southern")
            ("ayd" nil  nil   nil I   E   () "Pama–Nyungan" nil "Ayabadhu")
            ("aye" nil  nil   nil I   L   () "Niger–Congo" nil "Ayere")
            ("ayg" nil  nil   nil I   L   () "Niger–Congo" nil "Ginyanga")
            ("ayh" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic (Hadrami)")
            ("ayi" nil  nil   nil I   L   () "Niger–Congo" nil "Yigha")
            ("ayk" nil  nil   nil I   L   () "Niger–Congo" nil "Akuku")
            ("ayl" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic (Libyan)")
            ("aym" "ay" "aym" nil M   L   ("ayc" "ayr") "Aymaran" "aymar" "Aymara")
            ("ayn" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic (Sanaani)")
            ("ayo" nil  nil   nil I   L   () "Zamucoan" nil "Ayoreo")
            ("ayp" nil  nil   nil I   L   () "Afro-Asiatic" nil "Arabic (North Mesopotamian)")
            ("ayq" nil  nil   nil I   L   () "Sepik" nil "Ayi (Papua New Guinea)")
            ("ayr" nil  nil   nil I   L   () "Aymaran" nil "Aymara (Central)")
            ("ays" nil  nil   nil I   L   () "Austronesian" nil "Ayta (Sorsogon)")
            ("ayt" nil  nil   nil I   L   () "Austronesian" nil "Ayta (Bataan)")
            ("ayu" nil  nil   nil I   L   () "Niger–Congo" nil "Ayu")
            ("ayx" nil  nil   t   I   L   () "Sino-Tibetan" nil "Ayi (China)")
            ("ayy" nil  nil   t   I   E   () "spurious language" nil "Ayta, Tayabas")
            ("ayz" nil  nil   nil I   L   () "West Papuan" nil "Mai Brat")
            ("aza" nil  nil   nil I   L   () "Sino-Tibetan" nil "Azha")
            ("azb" nil  nil   nil I   L   () "Turkic" nil "Azerbaijani, South")
            ("azd" nil  nil   nil I   L   () "Uto-Aztecan" nil "Eastern Durango Nahuatl")
            ("aze" "az" "aze" nil M   L   ("azb" "azj") "Turkic" "Azərbaycan" "Azerbaijani")
            ("azg" nil  nil   nil I   L   () "Oto-Manguean" nil "Amuzgo (San Pedro Amuzgos)")
            ("azj" nil  nil   nil I   L   () "Azeri" nil "Azerbaijani (North)")
            ("azm" nil  nil   nil I   L   () "Oto-Manguean" nil "Amuzgo (Ipalapa)")
            ("azn" nil  nil   nil I   L   () "Uto-Aztecan" nil "Western Durango Nahuatl")
            ("azo" nil  nil   nil I   L   () "Niger–Congo" nil "Awing")
            ("azr" nil  nil   t   I   L   () "Austronesian" nil "Adzera")
            ("azt" nil  nil   nil I   L   () "Austronesian" nil "Atta (Faire)")
            ("azz" nil  nil   nil I   L   () "Uto-Aztecan" nil "Nahuatl (Highland Puebla)")
            ;; B
            ("baa" nil  nil   nil I   L   () "Austronesian" nil "Babatana")
            ("bab" nil  nil   nil I   L   () "Niger-Congo" nil "Bainouk-Gunyuño")
            ("bac" nil  nil   nil I   L   () "Austronesian" nil "Badui")
            (nil   nil  "bad" nil C   nil () nil nil "Banda languages")
            ("bae" nil  nil   nil I   E   () "Arawakan" nil "Baré")
            ("baf" nil  nil   nil I   L   () "Niger-Congo" nil "Nubaca")
            ("bag" nil  nil   nil I   L   () "Niger-Congo" nil "Tuki")
            ("bah" nil  nil   nil I   L   () "English Creole" nil "Bahamas Creole English")
            (nil   nil  "bai" nil C   nil () nil nil "Bamileke languages")
            ("baj" nil  nil   nil I   L   () "Austronesian" nil "Barakai")
            ("bak" "ba" "bak" nil I   L   () "Turkic" "башҡорт" "Bashkir")
            ("bal" nil  "bal" nil M   L   ("bcc" "bgn" "bgp") "Indo-European" "بلوچی" "Baluchi")
            ("bam" "bm" "bam" nil I   L   () "Niger–Congo" "bamanankan" "Bambara")
            ("ban" nil  "ban" nil I   L   () "Austronesian" "Basa Bali" "Balinese")
            ("bao" nil  nil   nil I   L   () "Tucanoan" nil "Waimaha")
            ("bap" nil  nil   nil I   L   () "Sino-Tibetan" nil "Bantawa")
            ("bar" nil  nil   nil I   L   () "Indo-European" nil "Bavarian")
            ("bas" nil  "bas" nil I   L   () "Niger-Congo" "ɓasaá" "Basa (Cameroon)")
            (nil   nil  "bat" nil C   nil () nil nil "Baltic languages")
            ("bau" nil  nil   nil I   L   () "Niger-Congo" nil "Bada (Nigeria)")
            ("bav" nil  nil   nil I   L   () "Niger-Congo" nil "Vengo")
            ("baw" nil  nil   nil I   L   () "Niger-Congo" nil "Bambili-Bambui")
            ("bax" nil  nil   nil I   L   () "Niger-Congo" nil "Bamun")
            ("bay" nil  nil   nil I   L   () "Austronesian" nil "Batuley")
            ("baz" nil  nil   t   I   L   () "Niger-Congo" nil "Tunen")
            ("bba" nil  nil   nil I   L   () "Niger-Congo" nil "Baatonum")
            ("bbb" nil  nil   nil I   L   () "Trans-New Guinea" nil "Barai")
            ("bbc" nil  nil   nil I   L   () "Austronesian" nil "Batak Toba")
            ("bbd" nil  nil   nil I   L   () "Trans-New Guinea" nil "Bau")
            ("bbe" nil  nil   nil I   L   () "Niger-Congo" nil "Bangba")
            ("bbf" nil  nil   nil I   L   () "Fas" nil "Baibai")
            ("bbg" nil  nil   nil I   L   () "Niger-Congo" nil "Barama")
            ("bbh" nil  nil   nil I   L   () "Austroasiatic" nil "Bugan")
            ("bbi" nil  nil   nil I   L   () "Niger-Congo" nil "Barombi")
            ("bbj" nil  nil   nil I   L   () "Niger-Congo" nil "Ghomálá'")
            ("bbk" nil  nil   nil I   L   () "Niger-Congo" nil "Babanki")
            ("bbl" nil  nil   nil I   L   () "Northeast Caucasian" nil "Bats")
            ("bbm" nil  nil   nil I   L   () "Niger-Congo" nil "Babango")
            ("bbn" nil  nil   nil I   L   () "Austronesian" nil "Uneapa")
            ("bbo" nil  nil   nil I   L   () "Niger-Congo" nil "Bobo Madaré, Northern")
            ("bbp" nil  nil   nil I   L   () "Niger-Congo" nil "Banda, West Central")
            ("bbq" nil  nil   nil I   L   () "Niger-Congo" nil "Bamali")
            ("bbr" nil  nil   nil I   L   () "Trans-New Guinea" nil "Girawa")
            ("bbs" nil  nil   nil I   L   () "Niger-Congo" nil "Bakpinka")
            ("bbt" nil  nil   nil I   L   () "Afro-Asiatic" nil "Mburku")
            ("bbu" nil  nil   nil I   L   () "Niger-Congo" nil "Kulung (Nigeria)")
            ("bbv" nil  nil   nil I   L   () "Austronesian" nil "Karnai")
            ("bbw" nil  nil   nil I   L   () "Niger-Congo" nil "Baba")
            ("bbx" nil  nil   nil I   L   () "Niger-Congo" nil "Bubia")
            ("bby" nil  nil   nil I   L   () "Niger-Congo" nil "Befang")
            ("bbz" nil  nil   t   I   L   () "Creole" nil "Arabic, Babalia Creole")
            ("bca" nil  nil   nil I   L   () "Sino-Tibetan" nil "Bai, Central")
            ("bcb" nil  nil   nil I   L   () "Niger-Congo" nil "Bainouk-Samik")
            ("bcc" nil  nil   nil I   L   () "Indo-European" "بلوچی" "Balochi, Southern")
            ("bcd" nil  nil   nil I   L   () "Austronesian" nil "Babar, North")
            ("bce" nil  nil   nil I   L   () "Niger-Congo" nil "Bamenyam")
            ("bcf" nil  nil   nil I   L   () "Kiwaian" nil "Bamu")
            ("bcg" nil  nil   nil I   L   () "Niger-Congo" nil "Baga Binari")
            ("bch" nil  nil   nil I   L   () "Austronesian" nil "Bariai")
            ("bci" nil  nil   nil I   L   () "Niger-Congo" nil "Baoulé")
            ("bcj" nil  nil   nil I   L   () "Nyulnyulan" nil "Bardi")
            ("bck" nil  nil   nil I   L   () "Bunuban" nil "Bunaba")
            ("bcl" nil  nil   nil I   L   () "Austronesian" nil "Bicolano, Central")
            ("bcm" nil  nil   nil I   L   () "Austronesian" nil "Bannoni")
            ("bcn" nil  nil   nil I   L   () "Niger-Congo" nil "Bali (Nigeria)")
            ("bco" nil  nil   nil I   L   () "Trans-New Guinea" nil "Kaluli")
            ("bcp" nil  nil   nil I   L   () "Niger-Congo" nil "Bali (Democratic Republic of Congo)")
            ("bcq" nil  nil   nil I   L   () "Afro-Asiatic" nil "Bench")
            ("bcr" nil  nil   nil I   L   () "Na-Dené" "Witsuwit'en" "Babine")
            ("bcs" nil  nil   nil I   L   () "Niger-Congo" nil "Kohumono")
            ("bct" nil  nil   nil I   L   () "Nilo-Saharan" nil "Bendi")
            ("bcu" nil  nil   nil I   L   () "Austronesian" nil "Awad Bing")
            ("bcv" nil  nil   nil I   L   () "Niger-Congo" nil "Shoo-Minda-Nye")
            ("bcw" nil  nil   nil I   L   () "Afro-Asiatic" nil "Bana")
            ("bcx" nil  nil   t   nil nil () "Austronesian" nil "Pamona")
            ("bcy" nil  nil   nil I   L   () "Afro-Asiatic" nil "Bacama")
            ("bcz" nil  nil   nil I   L   () "Niger-Congo" nil "Bainouk-Gunyaamolo")
            ("bda" nil  nil   nil I   L   () "Niger-Congo" nil "Bayot")
            ("bdb" nil  nil   nil I   L   () "Austronesian" nil "Basap")
            ("bdc" nil  nil   nil I   L   () "Chocoan" nil "Emberá-Baudó")
            ("bdd" nil  nil   nil I   L   () "Austronesian" nil "Bunama")
            ("bde" nil  nil   nil I   L   () "Afro-Asiatic" nil "Bade")
            ("bdf" nil  nil   nil I   L   () "Trans-New Guinea" nil "Biage")
            ("bdg" nil  nil   nil I   L   () "Austronesian" nil "Bonggi")
            ("bdh" nil  nil   nil I   L   () "Nilo-Saharan" nil "Baka (Sudan)")
            ("bdi" nil  nil   nil I   L   () "Nilo-Saharan" nil "Burun")
            ("bdj" nil  nil   nil I   L   () "Niger-Congo" "Bairt⤧ngvrt⤧zix" "Bai")
            ("bdk" nil  nil   nil I   L   () "Northeast Caucasian" nil "Budukh")
            ("bdl" nil  nil   nil I   L   () "Austronesian" nil "Bajau, Indonesian")
            ("bdm" nil  nil   nil I   L   () "Afro-Asiatic" nil "Buduma")
            ("bdn" nil  nil   nil I   L   () "Afro-Asiatic" nil "Baldemu")
            ("bdo" nil  nil   nil I   L   () "Nilo-Saharan" nil "Bernde")
            ("bdp" nil  nil   nil I   L   () "Niger-Congo" nil "Bende")
            ("bdq" nil  nil   nil I   L   () "Austroasiatic" nil "Bahnar")
            ("bdr" nil  nil   nil I   L   () "Austronesian" nil "Bajau, West Coast")
            ("bds" nil  nil   nil I   L   () "Afro-Asiatic" nil "Burunge")
            ("bdt" nil  nil   nil I   L   () "Niger-Congo" nil "Bokoto")
            ("bdu" nil  nil   nil I   L   () "Niger-Congo" nil "Oroko")
            ("bdv" nil  nil   nil I   L   () "Indo-European" nil "Bodo Parja")
            ("bdw" nil  nil   nil I   L   () "Trans-New Guinea" nil "Baham")
            ("bdx" nil  nil   nil I   L   () "Austronesian" nil "Budong-Budong")
            ("bdy" nil  nil   nil I   L   () "Pama-Nyungan" nil "Bandjalang")
            ("bdz" nil  nil   nil I   L   () "Indo-European" nil "Badeshi")
            ("bea" nil  nil   nil I   L   () "Na-Dené" "Dunne-za" "Beaver")
            ("beb" nil  nil   nil I   L   () "Niger-Congo" nil "Bebele")
            ("bec" nil  nil   nil I   L   () "Niger-Congo" nil "Iceve-Maci")
            ("bed" nil  nil   nil I   L   () "Austronesian" nil "Bedoanas")
            ("bee" nil  nil   nil I   L   () "Sino-Tibetan" nil "Byangsi")
            ("bef" nil  nil   nil I   L   () "Trans-New Guinea" nil "Benabena")
            ("beg" nil  nil   nil I   L   () "Austronesian" nil "Belait")
            ("beh" nil  nil   nil I   L   () "Niger-Congo" nil "Biali")
            ("bei" nil  nil   nil I   L   () "Austronesian" nil "Bekati'")
            ("bej" nil  "bej" nil I   L   () "Afro-Asiatic" "بداوية" "Beja")
            ("bek" nil  nil   nil I   L   () "Austronesian" nil "Bebeli")
            ("bel" "be" "bel" nil I   L   () "Indo-European" "беларуская" "Belarusian")
            ("bem" nil  "bem" nil I   L   () "Niger-Congo" "ichibemba" "Bemba (Zambia)")
            ("ben" "bn" "ben" nil I   L   () "Indo-European" "বাংলা" "Bengali")
            ("beo" nil  nil   nil I   L   () "Niger-Congo" nil "Beami")
            ("bep" nil  nil   nil I   L   () "Trans-New Guinea" nil "Besoa")
            ("beq" nil  nil   nil I   L   () "Austronesian" nil "Beembe")
            (nil   nil  "ber" nil C   nil () nil nil "Berber languages")
            ("bes" nil  nil   nil I   L   () "Niger-Congo" nil "Besme")
            ("bet" nil  nil   nil I   L   () "Niger-Congo" nil "Béte, Guiberoua")
            ("beu" nil  nil   nil I   L   () "Timor-Alar-Pantar" nil "Blagar")
            ("bev" nil  nil   nil I   L   () "Niger-Congo" nil "Bété, Daloa")
            ("bew" nil  nil   nil I   L   () "Malay-based Creole" "Bahasa Betawi" "Betawi")
            ("bex" nil  nil   nil I   L   () "Nilo-Saharan" nil "Jur Modo")
            ("bey" nil  nil   nil I   L   () "Torricelli" nil "Beli (Papua New Guinea)")
            ("bez" nil  nil   nil I   L   () "Niger-Congo" nil "Bena (Tanzania)")
            ("bfa" nil  nil   nil I   L   () "Nilo-Saharan" nil "Bari")
            ("bfb" nil  nil   nil I   L   () "Indo-European" nil "Bareli, Pauri")
            ("bfc" nil  nil   nil I   L   () "Sino-Tibetan" nil "Bai, Northern")
            ("bfd" nil  nil   nil I   L   () "Niger-Congo" nil "Bafut")
            ("bfe" nil  nil   nil I   L   () "Foja Range" nil "Betaf")
            ("bff" nil  nil   nil I   L   () "Niger-Congo" nil "Bofi")
            ("bfg" nil  nil   nil I   L   () "Austronesian" nil "Kayan, Busang")
            ("bfh" nil  nil   nil I   L   () "Yam" nil "Blafe")
            ("bfi" nil  nil   nil I   L   () "BANZSL" nil "British Sign Language")
            ("bfj" nil  nil   nil I   L   () "Niger-Congo" nil "Bafanji")
            ("bfk" nil  nil   nil I   L   () "village sign language" nil "Ban Khor Sign Language")
            ("bfl" nil  nil   nil I   L   () "Niger-Congo" nil "Banda-Ndélé")
            ("bfm" nil  nil   nil I   L   () "Niger-Congo" nil "Mmen")
            ("bfn" nil  nil   nil I   L   () "Trans-New Guinea" nil "Bunak")
            ("bfo" nil  nil   nil I   L   () "Niger-Congo" nil "Birifor, Malba")
            ("bfp" nil  nil   nil I   L   () "Niger-Congo" nil "Beba")
            ("bfq" nil  nil   nil I   L   () "Dravidian" "ಬಡಗ" "Badaga")
            ("bfr" nil  nil   nil I   L   () "Indo-European" nil "Bazigar")
            ("bfs" nil  nil   nil I   L   () "Sino-Tibetan" nil "Bai, Southern")
            ("bft" nil  nil   nil I   L   () "Sino-Tibetan" "بلتی" "Balti")
            ("bfu" nil  nil   nil I   L   () "Sino-Tibetan" nil "Gahri")
            ("bfw" nil  nil   nil I   L   () "Austroasiatic" nil "Bondo")
            ("bfx" nil  nil   nil I   L   () "Austronesian" nil "Bantayanon")
            ("bfy" nil  nil   nil I   L   () "Indo-European" nil "Bagheli")
            ("bfz" nil  nil   nil I   L   () "Indo-European" nil "Pahari, Mahasu")
            ("bga" nil  nil   nil I   L   () "Niger-Congo" nil "Gwamhi-Wuri")
            ("bgb" nil  nil   nil I   L   () "Austronesian" nil "Bobongko")
            ("bgc" nil  nil   nil I   L   () "Indo-European" nil "Haryanvi")
            ("bgd" nil  nil   nil I   L   () "Indo-European" nil "Bareli, Rathwi")
            ("bge" nil  nil   nil I   L   () "Indo-European" nil "Bauria")
            ("bgf" nil  nil   nil I   L   () "Niger-Congo" nil "Bangandu")
            ("bgg" nil  nil   nil I   L   () "Sino-Tibetan" nil "Bugun")
            ("bgh" nil  nil   t   nil nil () "Austroasiatic" nil "Bogan")
            ("bgi" nil  nil   nil I   L   () "Austronesian" nil "Giangan")
            ("bgj" nil  nil   nil I   L   () "Niger-Congo" nil "Bangolan")
            ("bgk" nil  nil   nil I   L   () "Austroasiatic" nil "Bit")
            ("bgl" nil  nil   nil I   L   () "Austroasiatic" nil "Bo (Laos)")
            ("bgm" nil  nil   t   I   L   () "Niger-Congo" nil "Baga Mboteni")
            ("bgn" nil  nil   nil I   L   () "Indo-European" nil "Balochi, Western")
            ("bgo" nil  nil   nil I   L   () "Niger-Congo" nil "Baga Koga")
            ("bgp" nil  nil   nil I   L   () "Indo-European" nil "Balochi, Eastern")
            ("bgq" nil  nil   nil I   L   () "Indo-European" nil "Bagri")
            ("bgr" nil  nil   nil I   L   () "Sino-Tgibetan" nil "Chin, Bawm")
            ("bgs" nil  nil   nil I   L   () "Austronesian" nil "Tagabawa")
            ("bgt" nil  nil   nil I   L   () "Austronesian" nil "Bughotu")
            ("bgu" nil  nil   nil I   L   () "Niger-Congo" nil "Mbongno")
            ("bgv" nil  nil   nil I   L   () "Trans-New Guinea" nil "Warkay-Bipim")
            ("bgw" nil  nil   nil I   L   () "Indo-European" nil "Bhatri")
            ("bgx" nil  nil   nil I   L   () "Turkic" nil "Balkan Gagauz Turkish")
            ("bgy" nil  nil   nil I   L   () "Austronesian" nil "Benggoi")
            ("bgz" nil  nil   nil I   L   () "Austronesian" nil "Banggai")
            ("bha" nil  nil   nil I   L   () "Indo-European" nil "Bharia")
            ("bhb" nil  nil   nil I   L   () "Indo-European" "भीली" "Bhili")
            ("bhc" nil  nil   nil I   L   () "Austronesian" nil "Biga")
            ("bhd" nil  nil   nil I   L   () "Indo-European" nil "Bhadrawahi")
            ("bhe" nil  nil   nil I   L   () "Indo-European" nil "Bhaya")
            ("bhf" nil  nil   nil I   L   () "Senu River" nil "Odiai")
            ("bhg" nil  nil   nil I   L   () "Trans-New Guinea" nil "Binandere")
            ("bhh" nil  nil   nil I   L   () "Indo-European" nil "Bukharic")
            ("bhi" nil  nil   nil I   L   () "Indo-European" nil "Bhilali")
            ("bhj" nil  nil   nil I   L   () "Sino-Tibetan" nil "Bahing")
            ("bhk" nil  nil   t   I   L   () "Austronesian" nil "Bicolano, Albay")
            ("bhl" nil  nil   nil I   L   () "Trans-New Guinea" nil "Bimin")
            ("bhm" nil  nil   nil I   L   () "Afro-Asiatic" nil "Bathari")
            ("bhn" nil  nil   nil I   L   () "Afro-Asiatic" nil "Bohtan Neo-Aramaic")
            ("bho" nil  "bho" nil I   L   () "Indo-European" "भोजपुरी" "Bhojpuri")
            ("bhp" nil  nil   nil I   L   () "Austronesian" nil "Bima")
            ("bhq" nil  nil   nil I   L   () "Austronesian" nil "Tukang Besi South")
            ("bhr" nil  nil   nil I   L   () "Austronesian" nil "Malagasy, Bara")
            ("bhs" nil  nil   nil I   L   () "Afro-Asiatic" nil "Buwal")
            ("bht" nil  nil   nil I   L   () "Indo-European" nil "Bhattiyali")
            ("bhu" nil  nil   nil I   L   () "Indo-European" nil "Bhunjia")
            ("bhv" nil  nil   nil I   L   () "Austronesian" nil "Bahau")
            ("bhw" nil  nil   nil I   L   () "Austronesian" nil "Biak")
            ("bhx" nil  nil   nil I   L   () "spurious" nil "Bhalay")
            ("bhy" nil  nil   nil I   L   () "Niger-Congo" nil "Bhele")
            ("bhz" nil  nil   nil I   L   () "Austronesian" nil "Bada (Indonesia)")
            ("bia" nil  nil   nil I   L   () "Pama-Nyungan" nil "Badimaya")
            ("bib" nil  nil   nil I   L   () "Niger-Congo" nil "Bissa")
            ("bic" nil  nil   t   I   L   () "Engan" nil "Bikaru")
            ("bid" nil  nil   nil I   L   () "Afro-Asiatic" nil "Bidiyo")
            ("bie" nil  nil   nil I   L   () "Madang" nil "Bepour")
            ("bif" nil  nil   nil I   L   () "Niger-Congo" nil "Biafada")
            ("big" nil  nil   nil I   L   () "Trans-New Guinea" nil "Biangai")
            (nil   nil  "bih" nil C   nil () nil nil "Bihari languages")
            ("bii" nil  nil   t   nil nil () "Sino-Tibetan" nil "Bisu")
            ("bij" nil  nil   t   I   L   () "Niger-Congo" nil "Vaghat-Ya-Bijim-Legeri")
            ("bik" nil  "bik" nil M   L   ("bcl" "bln" "bto"
                                           "cts" "fbl" "lbl"
                                           "rbl" "ubl" "bhk") "Austronesian" "Bicol" "Bikol")
            ("bil" nil  nil   nil I   L   () "Niger-Congo" nil "Bile")
            ("bim" nil  nil   nil I   L   () "Niger-Congo" nil "Bimoba")
            ("bin" nil  "bin" nil I   L   () "Niger-Congo" "Èdó" "Bini")
            ("bio" nil  nil   nil I   L   () "Senu River" nil "Nai")
            ("bip" nil  nil   nil I   L   () "Niger-Congo" nil "Bila")
            ("biq" nil  nil   nil I   L   () "Austronesian" nil "Bipi")
            ("bir" nil  nil   nil I   L   () "Engan" nil "Bisorio")
            ("bis" "bi" "bis" nil I   L   () "English Creole" "Bislama" "Bislama")
            ("bit" nil  nil   nil I   L   () "Sepik" nil "Berinomo")
            ("biu" nil  nil   nil I   L   () "Sino-Tibetan" nil "Biete")
            ("biv" nil  nil   nil I   L   () "Niger-Congo" nil "Birifor, Southern")
            ("biw" nil  nil   nil I   L   () "Niger-Congo" nil "Kol (Cameroon)")
            ("bix" nil  nil   nil I   L   () "Austroasiatic" nil "Bijori")
            ("biy" nil  nil   nil I   L   () "Austroasiatic" nil "Birhor")
            ("biz" nil  nil   nil I   L   () "Niger-Congo" nil "Baloi")
            ("bja" nil  nil   nil I   L   () "Niger-Congo" nil "Budza")
            ("bjb" nil  nil   nil I   E   () "Pama-Nyungan" nil "Banggarla")
            ("bjc" nil  nil   nil I   L   () "Trans-New Guinea" nil "Bariji")
            ("bjd" nil  nil   t   I   L   () "Pama-Nyungan" nil "Bandjigali")
            ("bje" nil  nil   nil I   L   () "Hmong-Mien" nil "Biao-Jiao Mien")
            ("bjf" nil  nil   nil I   L   () "Afro-Asiatic" nil "Barzani Jewish Neo-Aramaic")
            ("bjg" nil  nil   nil I   L   () "Niger-Congo" nil "Bidyogo")
            ("bjh" nil  nil   nil I   L   () "Sepik" nil "Bahinemo")
            ("bji" nil  nil   nil I   L   () "Afro-Asiatic" nil "Burji")
            ("bjj" nil  nil   nil I   L   () "Indo-European" nil "Kanauji")
            ("bjk" nil  nil   nil I   L   () "Austronesian" nil "Barok")
            ("bjl" nil  nil   nil I   L   () "Austronesian" nil "Bulu (Papua New Guinea)")
            ("bjm" nil  nil   nil I   L   () "Indo-European" nil "Bajelani")
            ("bjn" nil  nil   nil I   L   () "Austronesian" nil "Banjar")
            ("bjo" nil  nil   nil I   L   () "Niger-Congo" nil "Banda, Mid-Southern")
            ("bjp" nil  nil   nil I   L   () "Austronesian" nil "Fanamaket")
            ("bjq" nil  nil   t   I   L   () "Austronesian" nil "Malagasy, Southern Betsimisaraka")
            ("bjr" nil  nil   nil I   L   () "Trans-New Guinea" nil "Binumarien")
            ("bjs" nil  nil   nil I   L   () "English Creole" nil "Bajan")
            ("bjt" nil  nil   nil I   L   () "Niger-Congo" nil "Balanta-Ganja")
            ("bju" nil  nil   nil I   L   () "Niger-Congo" nil "Busuu")
            ("bjv" nil  nil   nil I   L   () "Nilo-Saharan" nil "Bedjond")
            ("bjw" nil  nil   nil I   L   () "Niger-Congo" nil "Bakwé")
            ("bjx" nil  nil   nil I   L   () "Austronesian" nil "Itneg, Banao")
            ("bjy" nil  nil   nil I   E   () "Pama-Nyungan" nil "Bayali")
            ("bjz" nil  nil   nil I   L   () "Trans-New Guinea" nil "Baruga")
            ("bka" nil  nil   nil I   L   () "Niger-Congo" nil "Kyak")
            ("bkb" nil  nil   t   I   L   () "Austronesian" nil "Finallig")
            ("bkc" nil  nil   nil I   L   () "Niger-Congo" nil "Baka (Cameroon)")
            ("bkd" nil  nil   nil I   L   () "Austronesian" nil "Binukid")
            ("bke" nil  nil   t   nil nil () "Austronesian" nil "Bengkulu")
            ("bkf" nil  nil   nil I   L   () "Niger-Congo" nil "Beeke")
            ("bkg" nil  nil   nil I   L   () "Niger-Congo" nil "Buraka")
            ("bkh" nil  nil   nil I   L   () "Niger-Congo" nil "Bakoko")
            ("bki" nil  nil   nil I   L   () "Austronesian" nil "Baki")
            ("bkj" nil  nil   nil I   L   () "Niger-Congo" nil "Pande")
            ("bkk" nil  nil   nil I   L   () "Indo-European" nil "Brokskat")
            ("bkl" nil  nil   nil I   L   () "Foja Range" nil "Berik")
            ("bkm" nil  nil   nil I   L   () "Niger-Congo" nil "Kom (Cameroon)")
            ("bkn" nil  nil   nil I   L   () "Austronesian" nil "Bukitan")
            ("bko" nil  nil   nil I   L   () "Niger-Congo" nil "Kwa'")
            ("bkp" nil  nil   nil I   L   () "Niger-Congo" nil "Boko (Democratic Republic of Congo)")
            ("bkq" nil  nil   nil I   L   () "Cariban" nil "Bakairí")
            ("bkr" nil  nil   nil I   L   () "Austronesian" nil "Bakumpai")
            ("bks" nil  nil   nil I   L   () "Austronesian" nil "Sorsogon, Masbate")
            ("bkt" nil  nil   nil I   L   () "Niger-Congo" nil "Boloki")
            ("bku" nil  nil   nil I   L   () "Austronesian" nil "Buhid")
            ("bkv" nil  nil   nil I   L   () "Niger-Congo" nil "Bekwarra")
            ("bkw" nil  nil   nil I   L   () "Niger-Congo" nil "Bekwil")
            ("bkx" nil  nil   nil I   L   () "Austronesian" nil "Baikeno")
            ("bky" nil  nil   nil I   L   () "Niger-Congo" nil "Bokyi")
            ("bkz" nil  nil   nil I   L   () "Austronesian" nil "Bungku")
            ("bla" nil  "bla" nil I   L   () "Algic" nil "Blackfoot")
            ("blb" nil  nil   nil I   L   () "Central Solomon" nil "Bilua")
            ("blc" nil  nil   nil I   L   () "Salishan" nil "Bella Coola")
            ("bld" nil  nil   nil I   L   () "Austronesian" nil "Bolango")
            ("ble" nil  nil   nil I   L   () "Niger-Congo" nil "Balanta-Kentohe")
            ("blf" nil  nil   nil I   L   () "Austronesian" nil "Buol")
            ("blg" nil  nil   t   I   L   () "Austronesian" nil "Balau")
            ("blh" nil  nil   nil I   L   () "Niger-Congo" nil "Kuwaa")
            ("bli" nil  nil   nil I   L   () "Niger-Congo" nil "Bolia")
            ("blj" nil  nil   nil I   L   () "Austronesian" nil "Bolongan")
            ("blk" nil  nil   nil I   L   () "Sino-Tibetan" nil "Karen, Pa'o")
            ("bll" nil  nil   nil I   E   () "Siouan" nil "Biloxi")
            ("blm" nil  nil   nil I   L   () "Nilo-Saharan" nil "Beli (Sudan)")
            ("bln" nil  nil   nil I   L   () "Austronesian" nil "Bicolano, Southern Catanduanes")
            ("blo" nil  nil   nil I   L   () "Niger-Congo" nil "Anii")
            ("blp" nil  nil   nil I   L   () "Austronesian" nil "Blablanga")
            ("blq" nil  nil   nil I   L   () "Austronesian" nil "Baluan-Pam")
            ("blr" nil  nil   nil I   L   () "Austroasiatic" nil "Blang")
            ("bls" nil  nil   nil I   L   () "Austronesian" nil "Balaesang")
            ("blt" nil  nil   nil I   L   () "Kra-Dai" nil "Tai Dam")
            ("blu" nil  nil   t   nil nil () "Hmong-Mien" nil "Hmong Njua")
            ("blv" nil  nil   nil I   L   () "Niger-Congo" nil "Bolo")
            ("blw" nil  nil   nil I   L   () "Austronesian" nil "Balangao")
            ("blx" nil  nil   nil I   L   () "Austronesian" nil "Ayta, Mag-Indi")
            ("bly" nil  nil   nil I   L   () "Niger-Congo" nil "Notre")
            ("blz" nil  nil   nil I   L   () "Austronesian" nil "Balantak")
            ("bma" nil  nil   nil I   L   () "Niger-Congo" nil "Lame")
            ("bmb" nil  nil   nil I   L   () "Niger-Congo" nil "Bembe")
            ("bmc" nil  nil   nil I   L   () "Austronesian" nil "Biem")
            ("bmd" nil  nil   nil I   L   () "Niger-Congo" nil "Baga Manduri")
            ("bme" nil  nil   nil I   L   () "Niger-Congo" nil "Limassa")
            ("bmf" nil  nil   nil I   L   () "Niger-Congo" nil "Bom")
            ("bmg" nil  nil   nil I   L   () "Niger-Congo" nil "Bamwe")
            ("bmh" nil  nil   nil I   L   () "Trans-New Guinea" nil "Kein")
            ("bmi" nil  nil   nil I   L   () "Nilo-Saharan" nil "Bagirmi")
            ("bmj" nil  nil   nil I   L   () "Indo-European" nil "Bote-Majhi")
            ("bmk" nil  nil   nil I   L   () "Austronesian" nil "Ghayavi")
            ("bml" nil  nil   nil I   L   () "Niger-Congo" nil "Bomboli")
            ("bmm" nil  nil   nil I   L   () "Austronesian" nil "Malagasy, Northern Betsimisaraka")
            ("bmn" nil  nil   nil I   E   () "Austronesian" nil "Bina (Papua New Guinea)")
            ("bmo" nil  nil   nil I   L   () "Niger-Congo" nil "Bambalang")
            ("bmp" nil  nil   nil I   L   () "Trans-New Guinea" nil "Bulgebi")
            ("bmq" nil  nil   nil I   L   () "Niger-Congo" nil "Bomu")
            ("bmr" nil  nil   nil I   L   () "Boran" nil "Muinane")
            ("bms" nil  nil   nil I   L   () "Nilo-Saharan" nil "Kanuri, Bilma")
            ("bmt" nil  nil   nil I   L   () "Hmong-Mien" nil "Biao Mon")
            ("bmu" nil  nil   nil I   L   () "Trans-New Guinea" nil "Burum-Mindik")
            ("bmv" nil  nil   nil I   L   () "Niger-Congo" nil "Bum")
            ("bmw" nil  nil   nil I   L   () "Niger-Congo" nil "Bomwali")
            ("bmx" nil  nil   nil I   L   () "Trans-New Guinea" nil "Baimak")
            ("bmy" nil  nil   t   I   L   () "spurious" nil "Bemba (Democratic Republic of Congo)")
            ("bmz" nil  nil   nil I   L   () "Trans-New Guinea" nil "Baramu")
            ("bna" nil  nil   nil I   L   () "Austronesian" nil "Bonerate")
            ("bnb" nil  nil   nil I   L   () "Austronesian" nil "Bookan")
            ("bnc" nil  nil   nil M   L   ("ebk" "lbk"
                                           "obk" "rbk" "vbk") "Austronesian" nil "Bontoc, Central")
            ("bnd" nil  nil   nil I   L   () "Austronesian" nil "Banda (Indonesia)")
            ("bne" nil  nil   nil I   L   () "Austronesian" nil "Bintauna")
            ("bnf" nil  nil   nil I   L   () "Austronesian" nil "Masiwang")
            ("bng" nil  nil   nil I   L   () "Niger-Congo" nil "Benga")
            ("bnh" nil  nil   t   I   L   () "Arawan" nil "Banawá")
            ("bni" nil  nil   nil I   L   () "Niger-Congo" nil "Bangi")
            ("bnj" nil  nil   nil I   L   () "Austronesian" nil "Tawbuid, Eastern")
            ("bnk" nil  nil   nil I   L   () "Austronesian" nil "Bierebo")
            ("bnl" nil  nil   nil I   L   () "Afro-Asiatic" nil "Boon")
            ("bnm" nil  nil   nil I   L   () "Niger-Congo" nil "Batanga")
            ("bnn" nil  nil   nil I   L   () "Austronesian" nil "Bunun")
            ("bno" nil  nil   nil I   L   () "Austronesian" nil "Bantoanon")
            ("bnp" nil  nil   nil I   L   () "Austronesian" nil "Bola")
            ("bnq" nil  nil   nil I   L   () "Austronesian" nil "Bantik")
            ("bnr" nil  nil   nil I   L   () "Austronesian" nil "Butmas-Tur")
            ("bns" nil  nil   nil I   L   () "Indo-European" nil "Bundeli")
            (nil   nil  "bnt" nil C   nil () nil nil "Bantu languages")
            ("bnu" nil  nil   nil I   L   () "Austronesian" nil "Bentong")
            ("bnv" nil  nil   nil I   L   () "Foja Range" nil "Beneraf, Bonerif, Edwas")
            ("bnw" nil  nil   nil I   L   () "Sepik" nil "Bisis")
            ("bnx" nil  nil   nil I   L   () "Niger-Congo" nil "Bangubangu")
            ("bny" nil  nil   nil I   L   () "Austronesian" nil "Bintulu")
            ("bnz" nil  nil   nil I   L   () "Niger-Congo" nil "Beezen")
            ("boa" nil  nil   nil I   L   () "Boran" nil "Bora")
            ("bob" nil  nil   nil I   L   () "Afro-Asiatic" nil "Boni")
            ("boc" nil  nil   t   I   L   () "Austronesian" nil "Bakung Kenyah")
            ("bod" "bo" "tib" nil I   L   () "Sino-Tibetan" "བོད་ཡིག" "Tibetan")
            ("boe" nil  nil   nil I   L   () "Niger-Congo" nil "Mundabli")
            ("bof" nil  nil   nil I   L   () "Niger-Congo" nil "Bolon")
            ("bog" nil  nil   nil I   L   () "Deaf-community sign language" nil "Bamako Sign Language")
            ("boh" nil  nil   nil I   L   () "Niger-Congo" nil "Boma")
            ("boi" nil  nil   nil I   E   () "Chumashan" nil "Barbareño")
            ("boj" nil  nil   nil I   L   () "Trans-New Guinea" nil "Anjam")
            ("bok" nil  nil   nil I   L   () "Niger-Congo" nil "Bonjo")
            ("bol" nil  nil   nil I   L   () "Afro-Asiatic" nil "Bole")
            ("bom" nil  nil   nil I   L   () "Niger-Congo" nil "Berom")
            ("bon" nil  nil   nil I   L   () "Trans-Fly" nil "Bine")
            ("boo" nil  nil   nil I   L   () "Niger-Congo" nil "Bozo, Tièma Cièwè")
            ("bop" nil  nil   nil I   L   () "Trans-New Guinea" nil "Bonkiman")
            ("boq" nil  nil   nil I   L   () "Trans-New Guinea" nil "Bogaya")
            ("bor" nil  nil   nil I   L   () "Macro-Jê" nil "Borôro")
            ("bos" "bs" "bos" nil I   L   () "Indo-European" "bosanski" "Bosnian")
            ("bot" nil  nil   nil I   L   () "Nilo-Saharan" nil "Bongo")
            ("bou" nil  nil   nil I   L   () "Niger-Congo" nil "Bondei")
            ("bov" nil  nil   nil I   L   () "Niger-Congo" nil "Tuwuli")
            ("bow" nil  nil   nil I   E   () "Trans-Fly" nil "Rema")
            ("box" nil  nil   nil I   L   () "Niger-Congo" nil "Buamu")
            ("boy" nil  nil   nil I   L   () "Niger-Congo" nil "Bodo (Central African Republic)")
            ("boz" nil  nil   nil I   L   () "Niger-Congo" nil "Bozo, Tiéyaxo")
            ("bpa" nil  nil   nil I   L   () nil nil "Dakaka")
            ("bpb" nil  nil   t   I   E   () "spurious language" nil "Barbacoas")
            ("bpc" nil  nil   nil I   L   () "Niger–Congo" nil "Mbuk")
            ("bpd" nil  nil   nil I   L   () nil nil "Banda-Banda")
            ("bpe" nil  nil   nil I   L   () "Skou" nil "Bauni")
            ("bpg" nil  nil   nil I   L   () nil nil "Bonggo")
            ("bph" nil  nil   nil I   L   () nil nil "Botlikh")
            ("bpi" nil  nil   nil I   L   () nil nil "Bagupi")
            ("bpj" nil  nil   nil I   L   () nil nil "Binji")
            ("bpk" nil  nil   nil I   L   () nil nil "Orowe")
            ("bpl" nil  nil   nil I   L   () nil nil "Broome Pearling Lugger Pidgin")
            ("bpm" nil  nil   nil I   L   () nil nil "Biyom")
            ("bpn" nil  nil   nil I   L   () nil nil "Dzao Min")
            ("bpo" nil  nil   nil I   L   () nil nil "Anasi")
            ("bpp" nil  nil   nil I   L   () nil nil "Kaure")
            ("bpq" nil  nil   nil I   L   () nil nil "Malay, Banda")
            ("bpr" nil  nil   nil I   L   () nil nil "Blaan, Koronadal")
            ("bps" nil  nil   nil I   L   () nil nil "Blaan, Sarangani")
            ("bpt" nil  nil   nil I   E   () nil nil "Barrow Point")
            ("bpu" nil  nil   nil I   L   () nil nil "Bongu")
            ("bpv" nil  nil   nil I   L   () nil nil "Marind, Bian")
            ("bpw" nil  nil   nil I   L   () nil nil "Bo (Papua New Guinea)")
            ("bpx" nil  nil   nil I   L   () nil nil "Bareli, Palya")
            ("bpy" nil  nil   nil I   L   () "Indo-European" "বিষ্ণুপ্রিয়া মণিপুরী" "Bishnupriya Manipuri")
            ("bpz" nil  nil   nil I   L   () nil nil "Bilba")
            ("bqa" nil  nil   nil I   L   () nil nil "Tchumbuli")
            ("bqb" nil  nil   nil I   L   () nil nil "Bagusa")
            ("bqc" nil  nil   nil I   L   () nil nil "Boko (Benin)")
            ("bqd" nil  nil   nil I   L   () nil nil "Bung")
            ("bqe" nil  nil   t   nil nil () nil nil "Navarro-Labourdin Basque")
            ("bqf" nil  nil   nil I   E   () nil nil "Baga Kaloum")
            ("bqg" nil  nil   nil I   L   () nil nil "Bago-Kusuntu")
            ("bqh" nil  nil   nil I   L   () nil nil "Baima")
            ("bqi" nil  nil   nil I   L   () nil nil "Bakhtiari")
            ("bqj" nil  nil   nil I   L   () nil nil "Bandial")
            ("bqk" nil  nil   nil I   L   () nil nil "Banda-Mbrès")
            ("bql" nil  nil   nil I   L   () nil nil "Bilakura")
            ("bqm" nil  nil   nil I   L   () nil nil "Wumboko")
            ("bqn" nil  nil   nil I   L   () nil nil "Bulgarian Sign Language")
            ("bqo" nil  nil   nil I   L   () nil nil "Balo")
            ("bqp" nil  nil   nil I   L   () nil nil "Busa")
            ("bqq" nil  nil   nil I   L   () nil nil "Biritai")
            ("bqr" nil  nil   nil I   L   () nil nil "Burusu")
            ("bqs" nil  nil   nil I   L   () nil nil "Bosngun")
            ("bqt" nil  nil   nil I   L   () nil nil "Bamukumbit")
            ("bqu" nil  nil   nil I   L   () nil nil "Boguru")
            ("bqv" nil  nil   nil I   L   () nil nil "Begbere-Ejar")
            ("bqw" nil  nil   nil I   L   () nil nil "Buru (Nigeria)")
            ("bqx" nil  nil   nil I   L   () nil nil "Baangi")
            ("bqy" nil  nil   nil I   L   () nil nil "Bali Sign Language")
            ("bqz" nil  nil   nil I   L   () nil nil "Bakaka")
            ("bra" nil  "bra" nil I   L   () nil nil "Braj")
            ("brb" nil  nil   nil I   L   () nil nil "Lave")
            ("brc" nil  nil   nil I   E   () nil nil "Berbice Creole Dutch")
            ("brd" nil  nil   nil I   L   () nil nil "Baraamu")
            ("bre" "br" "bre" nil I   L   () "Indo-European" "brezhoneg" "Breton")
            ("brf" nil  nil   nil I   L   () nil nil "Bera")
            ("brg" nil  nil   nil I   L   () nil nil "Baure")
            ("brh" nil  nil   nil I   L   () nil nil "Brahui")
            ("bri" nil  nil   nil I   L   () nil nil "Mokpwe")
            ("brj" nil  nil   nil I   L   () nil nil "Bieria")
            ("brk" nil  nil   nil I   E   () nil nil "Birked")
            ("brl" nil  nil   nil I   L   () nil nil "Birwa")
            ("brm" nil  nil   nil I   L   () nil nil "Barambu")
            ("brn" nil  nil   nil I   L   () nil nil "Boruca")
            ("bro" nil  nil   nil I   L   () nil nil "Brokkat")
            ("brp" nil  nil   nil I   L   () nil nil "Barapasi")
            ("brq" nil  nil   nil I   L   () nil nil "Breri")
            ("brr" nil  nil   nil I   L   () nil nil "Birao")
            ("brs" nil  nil   nil I   L   () nil nil "Baras")
            ("brt" nil  nil   nil I   L   () nil nil "Bitare")
            ("bru" nil  nil   nil I   L   () nil nil "Bru, Eastern")
            ("brv" nil  nil   nil I   L   () nil nil "Bru, Western")
            ("brw" nil  nil   nil I   L   () nil nil "Bellari")
            ("brx" nil  nil   nil I   L   () nil nil "Bodo (India)")
            ("bry" nil  nil   nil I   L   () nil nil "Burui")
            ("brz" nil  nil   nil I   L   () nil nil "Bilbil")
            ("bsa" nil  nil   nil I   L   () nil nil "Abinomn")
            ("bsb" nil  nil   nil I   L   () nil nil "Bisaya, Brunei")
            ("bsc" nil  nil   nil I   L   () nil nil "Bassari")
            ("bsd" nil  nil   t   nil nil () nil nil "Sarawak Bisaya")
            ("bse" nil  nil   nil I   L   () nil nil "Wushi")
            ("bsf" nil  nil   nil I   L   () nil nil "Bauchi")
            ("bsg" nil  nil   nil I   L   () nil nil "Bashkardi")
            ("bsh" nil  nil   nil I   L   () nil nil "Kati")
            ("bsi" nil  nil   nil I   L   () nil nil "Bassossi")
            ("bsj" nil  nil   nil I   L   () nil nil "Bangwinji")
            ("bsk" nil  nil   nil I   L   () nil nil "Burushaski")
            ("bsl" nil  nil   nil I   E   () nil nil "Basa-Gumna")
            ("bsm" nil  nil   nil I   L   () nil nil "Busami")
            ("bsn" nil  nil   nil I   L   () nil nil "Barasana")
            ("bso" nil  nil   nil I   L   () nil nil "Buso")
            ("bsp" nil  nil   nil I   L   () nil nil "Baga Sitemu")
            ("bsq" nil  nil   nil I   L   () nil nil "Bassa")
            ("bsr" nil  nil   nil I   L   () nil nil "Bassa-Kontagora")
            ("bss" nil  nil   nil I   L   () nil nil "Akoose")
            ("bst" nil  nil   nil I   L   () nil nil "Basketo")
            ("bsu" nil  nil   nil I   L   () nil nil "Bahonsuai")
            ("bsv" nil  nil   nil I   E   () nil nil "Baga Sobané")
            ("bsw" nil  nil   nil I   L   () nil nil "Baiso")
            ("bsx" nil  nil   nil I   L   () nil nil "Yangkam")
            ("bsy" nil  nil   nil I   L   () nil nil "Bisaya, Sabah")
            ("bsz" nil  nil   t   nil nil () nil nil "Souletin Basque")
            ("bta" nil  nil   nil I   L   () nil nil "Bata")
            ("btb" nil  nil   t   I   L   () nil nil "Beti (Cameroon)")
            ("btc" nil  nil   nil I   L   () nil nil "Bati (Cameroon)")
            ("btd" nil  nil   nil I   L   () nil nil "Batak Dairi")
            ("bte" nil  nil   nil I   E   () nil nil "Gamo-Ningi")
            ("btf" nil  nil   nil I   L   () nil nil "Birgit")
            ("btg" nil  nil   nil I   L   () nil nil "Bété, Gagnoa")
            ("bth" nil  nil   nil I   L   () nil nil "Biatah")
            ("bti" nil  nil   nil I   L   () nil nil "Burate")
            ("btj" nil  nil   nil I   L   () nil nil "Malay, Bacanese")
            (nil   nil  "btk" nil C   nil () nil nil "Batak languages")
            ("btl" nil  nil   t   I   L   () nil nil "Bhatola")
            ("btm" nil  nil   nil I   L   () nil nil "Batak Mandailing")
            ("btn" nil  nil   nil I   L   () nil nil "Ratagnon")
            ("bto" nil  nil   nil I   L   () nil nil "Bicolano, Iriga")
            ("btp" nil  nil   nil I   L   () nil nil "Budibud")
            ("btq" nil  nil   nil I   L   () nil nil "Batek")
            ("btr" nil  nil   nil I   L   () nil nil "Baetora")
            ("bts" nil  nil   nil I   L   () nil nil "Batak Simalungun")
            ("btt" nil  nil   nil I   L   () nil nil "Bete-Bendi")
            ("btu" nil  nil   nil I   L   () nil nil "Batu")
            ("btv" nil  nil   nil I   L   () nil nil "Bateri")
            ("btw" nil  nil   nil I   L   () nil nil "Butuanon")
            ("btx" nil  nil   nil I   L   () nil nil "Batak Karo")
            ("bty" nil  nil   nil I   L   () nil nil "Bobot")
            ("btz" nil  nil   nil I   L   () nil nil "Batak Alas-Kluet")
            ("bua" nil  "bua" nil M   L   ("bxm" "bxr" "bxu") nil "буряад" "Buriat")
            ("bub" nil  nil   nil I   L   () nil nil "Bua")
            ("buc" nil  nil   nil I   L   () nil nil "Bushi")
            ("bud" nil  nil   nil I   L   () nil nil "Ntcham")
            ("bue" nil  nil   nil I   E   () nil nil "Beothuk")
            ("buf" nil  nil   nil I   L   () nil nil "Bushoong")
            ("bug" nil  "bug" nil I   L   () nil "ᨅᨔ ᨕᨘᨁᨗ" "Buginese")
            ("buh" nil  nil   nil I   L   () nil nil "Bunu, Younuo")
            ("bui" nil  nil   nil I   L   () nil nil "Bongili")
            ("buj" nil  nil   nil I   L   () nil nil "Basa-Gurmana")
            ("buk" nil  nil   nil I   L   () nil nil "Bugawac")
            ("bul" "bg" "bul" nil I   L   () "Indo-European" "български" "Bulgarian")
            ("bum" nil  nil   nil I   L   () nil nil "Bulu (Cameroon)")
            ("bun" nil  nil   nil I   L   () nil nil "Sherbro")
            ("buo" nil  nil   nil I   L   () nil nil "Terei")
            ("bup" nil  nil   nil I   L   () nil nil "Busoa")
            ("buq" nil  nil   nil I   L   () nil nil "Brem")
            ("bus" nil  nil   nil I   L   () nil nil "Bokobaru")
            ("but" nil  nil   nil I   L   () nil nil "Bungain")
            ("buu" nil  nil   nil I   L   () nil nil "Budu")
            ("buv" nil  nil   nil I   L   () nil nil "Bun")
            ("buw" nil  nil   nil I   L   () nil nil "Bubi")
            ("bux" nil  nil   nil I   L   () nil nil "Boghom")
            ("buy" nil  nil   nil I   L   () nil nil "Bullom So")
            ("buz" nil  nil   nil I   L   () nil nil "Bukwen")
            ("bva" nil  nil   nil I   L   () nil nil "Barein")
            ("bvb" nil  nil   nil I   L   () nil nil "Bube")
            ("bvc" nil  nil   nil I   L   () nil nil "Baelelea")
            ("bvd" nil  nil   nil I   L   () nil nil "Baeggu")
            ("bve" nil  nil   nil I   L   () nil nil "Malay, Berau")
            ("bvf" nil  nil   nil I   L   () nil nil "Boor")
            ("bvg" nil  nil   nil I   L   () nil nil "Bonkeng")
            ("bvh" nil  nil   nil I   L   () nil nil "Bure")
            ("bvi" nil  nil   nil I   L   () nil nil "Belanda Viri")
            ("bvj" nil  nil   nil I   L   () nil nil "Baan")
            ("bvk" nil  nil   nil I   L   () nil nil "Bukat")
            ("bvl" nil  nil   nil I   L   () nil nil "Bolivian Sign Language")
            ("bvm" nil  nil   nil I   L   () nil nil "Bamunka")
            ("bvn" nil  nil   nil I   L   () nil nil "Buna")
            ("bvo" nil  nil   nil I   L   () nil nil "Bolgo")
            ("bvp" nil  nil   nil I   L   () nil nil "Bumang")
            ("bvq" nil  nil   nil I   L   () nil nil "Birri")
            ("bvr" nil  nil   nil I   L   () nil nil "Burarra")
            ("bvs" nil  nil   t   nil nil () nil nil "Belgian Sign Language")
            ("bvt" nil  nil   nil I   L   () nil nil "Bati (Indonesia)")
            ("bvu" nil  nil   nil I   L   () nil nil "Malay, Bukit")
            ("bvv" nil  nil   nil I   E   () nil nil "Baniva")
            ("bvw" nil  nil   nil I   L   () nil nil "Boga")
            ("bvx" nil  nil   nil I   L   () nil nil "Dibole")
            ("bvy" nil  nil   nil I   L   () nil nil "Baybayanon")
            ("bvz" nil  nil   nil I   L   () nil nil "Bauzi")
            ("bwa" nil  nil   nil I   L   () nil nil "Bwatoo")
            ("bwb" nil  nil   nil I   L   () nil nil "Namosi-Naitasiri-Serua")
            ("bwc" nil  nil   nil I   L   () nil nil "Bwile")
            ("bwd" nil  nil   nil I   L   () nil nil "Bwaidoka")
            ("bwe" nil  nil   nil I   L   () nil nil "Karen, Bwe")
            ("bwf" nil  nil   nil I   L   () nil nil "Boselewa")
            ("bwg" nil  nil   nil I   L   () nil nil "Barwe")
            ("bwh" nil  nil   nil I   L   () nil nil "Bishuo")
            ("bwi" nil  nil   nil I   L   () nil nil "Baniwa")
            ("bwj" nil  nil   nil I   L   () nil nil "Bwamu, Láá Láá")
            ("bwk" nil  nil   nil I   L   () nil nil "Bauwaki")
            ("bwl" nil  nil   nil I   L   () nil nil "Bwela")
            ("bwm" nil  nil   nil I   L   () nil nil "Biwat")
            ("bwn" nil  nil   nil I   L   () nil nil "Bunu, Wunai")
            ("bwo" nil  nil   nil I   L   () nil nil "Boro")
            ("bwp" nil  nil   nil I   L   () nil nil "Mandobo Bawah")
            ("bwq" nil  nil   nil I   L   () nil nil "Bobo Madaré, Southern")
            ("bwr" nil  nil   nil I   L   () nil nil "Bura-Pabir")
            ("bws" nil  nil   nil I   L   () nil nil "Bomboma")
            ("bwt" nil  nil   nil I   L   () nil nil "Bafaw-Balong")
            ("bwu" nil  nil   nil I   L   () nil nil "Buli (Ghana)")
            ("bwv" nil  nil   t   nil nil () nil nil "Bahau River Kenyah")
            ("bww" nil  nil   nil I   L   () nil nil "Bwa")
            ("bwx" nil  nil   nil I   L   () nil nil "Bunu, Bu-Nao")
            ("bwy" nil  nil   nil I   L   () nil nil "Bwamu, Cwi")
            ("bwz" nil  nil   nil I   L   () nil nil "Bwisi")
            ("bxa" nil  nil   nil I   L   () nil nil "Bauro")
            ("bxb" nil  nil   nil I   L   () nil nil "Belanda Bor")
            ("bxc" nil  nil   nil I   L   () nil nil "Molengue")
            ("bxd" nil  nil   nil I   L   () nil nil "Pela")
            ("bxe" nil  nil   nil I   L   () nil nil "Birale")
            ("bxf" nil  nil   nil I   L   () nil nil "Bilur")
            ("bxg" nil  nil   nil I   L   () nil nil "Bangala")
            ("bxh" nil  nil   nil I   L   () nil nil "Buhutu")
            ("bxi" nil  nil   nil I   E   () nil nil "Pirlatapa")
            ("bxj" nil  nil   nil I   L   () nil nil "Bayungu")
            ("bxk" nil  nil   nil I   L   () nil nil "Bukusu")
            ("bxl" nil  nil   nil I   L   () nil nil "Jalkunan")
            ("bxm" nil  nil   nil I   L   () nil nil "Buriat, Mongolia")
            ("bxn" nil  nil   nil I   L   () nil nil "Burduna")
            ("bxo" nil  nil   nil I   L   () nil nil "Barikanchi")
            ("bxp" nil  nil   nil I   L   () nil nil "Bebil")
            ("bxq" nil  nil   nil I   L   () nil nil "Beele")
            ("bxr" nil  nil   nil I   L   () nil nil "Buriat, Russia")
            ("bxs" nil  nil   nil I   L   () nil nil "Busam")
            ("bxt" nil  nil   t   nil nil () nil nil "Buxinhua")
            ("bxu" nil  nil   nil I   L   () nil nil "Buriat, China")
            ("bxv" nil  nil   nil I   L   () nil nil "Berakou")
            ("bxw" nil  nil   nil I   L   () nil nil "Bankagooma")
            ("bxx" nil  nil   t   I   L   () nil nil "Borna")
            ("bxz" nil  nil   nil I   L   () nil nil "Binahari")
            ("bya" nil  nil   nil I   L   () nil nil "Batak")
            ("byb" nil  nil   nil I   L   () nil nil "Bikya")
            ("byc" nil  nil   nil I   L   () nil nil "Ubaghara")
            ("byd" nil  nil   nil I   L   () nil nil "Benyadu'")
            ("bye" nil  nil   nil I   L   () nil nil "Pouye")
            ("byf" nil  nil   nil I   L   () nil nil "Bete")
            ("byg" nil  nil   nil I   E   () nil nil "Baygo")
            ("byh" nil  nil   nil I   L   () nil nil "Bujhyal")
            ("byi" nil  nil   nil I   L   () nil nil "Buyu")
            ("byj" nil  nil   nil I   L   () nil nil "Bina (Nigeria)")
            ("byk" nil  nil   nil I   L   () nil nil "Biao")
            ("byl" nil  nil   nil I   L   () nil nil "Bayono")
            ("bym" nil  nil   nil I   L   () nil nil "Bidyara")
            ("byn" nil  "byn" nil I   L   () nil nil "Bilin")
            ("byo" nil  nil   nil I   L   () nil nil "Biyo")
            ("byp" nil  nil   nil I   L   () nil nil "Bumaji")
            ("byq" nil  nil   nil I   E   () nil nil "Basay")
            ("byr" nil  nil   nil I   L   () nil nil "Baruya")
            ("bys" nil  nil   nil I   L   () nil nil "Burak")
            ("byt" nil  nil   nil I   E   () nil nil "Berti")
            ("byu" nil  nil   t   nil nil () nil nil "Buyang")
            ("byv" nil  nil   nil I   L   () nil nil "Medumba")
            ("byw" nil  nil   nil I   L   () nil nil "Belhariya")
            ("byx" nil  nil   nil I   L   () nil nil "Qaqet")
            ("byy" nil  nil   t   I   L   () nil nil "Buya")
            ("byz" nil  nil   nil I   L   () nil nil "Banaro")
            ("bza" nil  nil   nil I   L   () nil nil "Bandi")
            ("bzb" nil  nil   nil I   L   () nil nil "Andio")
            ("bzc" nil  nil   nil I   L   () nil nil "Southern Betsimisaraka Malagasy")
            ("bzd" nil  nil   nil I   L   () nil nil "Bribri")
            ("bze" nil  nil   nil I   L   () nil nil "Bozo, Jenaama")
            ("bzf" nil  nil   nil I   L   () nil nil "Boikin")
            ("bzg" nil  nil   nil I   L   () nil nil "Babuza")
            ("bzh" nil  nil   nil I   L   () nil nil "Buang, Mapos")
            ("bzi" nil  nil   nil I   L   () nil nil "Bisu")
            ("bzj" nil  nil   nil I   L   () nil nil "Belize Kriol English")
            ("bzk" nil  nil   nil I   L   () nil nil "Nicaragua Creole English")
            ("bzl" nil  nil   nil I   L   () nil nil "Boano (Sulawesi)")
            ("bzm" nil  nil   nil I   L   () nil nil "Bolondo")
            ("bzn" nil  nil   nil I   L   () nil nil "Boano (Maluku)")
            ("bzo" nil  nil   nil I   L   () nil nil "Bozaba")
            ("bzp" nil  nil   nil I   L   () nil nil "Kemberano")
            ("bzq" nil  nil   nil I   L   () nil nil "Buli (Indonesia)")
            ("bzr" nil  nil   nil I   E   () nil nil "Biri")
            ("bzs" nil  nil   nil I   L   () nil nil "Brazilian Sign Language")
            ("bzt" nil  nil   nil I   C   () nil nil "Brithenig")
            ("bzu" nil  nil   nil I   L   () nil nil "Burmeso")
            ("bzv" nil  nil   nil I   L   () nil nil "Bebe")
            ("bzw" nil  nil   nil I   L   () nil nil "Basa (Nigeria)")
            ("bzx" nil  nil   nil I   L   () nil nil "Bozo, Hainyaxo")
            ("bzy" nil  nil   nil I   L   () nil nil "Obanliku")
            ("bzz" nil  nil   nil I   L   () nil nil "Evant")
            ;; C
            ("caa" nil  nil   nil I   L   () nil "čorti'" "Chortí")
            ("cab" nil  nil   nil I   L   () nil nil "Garifuna")
            ("cac" nil  nil   nil I   L   () nil nil "Chuj (San Sebastián Coatán)")
            ("cad" nil  "cad" nil I   L   () nil "Hasí:nay" "Caddo")
            ("cae" nil  nil   nil I   L   () nil nil "Lehar")
            ("caf" nil  nil   nil I   L   () nil "ᑕᗸᒡ" "Carrier, Southern")
            ("cag" nil  nil   nil I   L   () nil nil "Nivaclé")
            ("cah" nil  nil   nil I   L   () nil nil "Cahuarano")
            (nil   nil  "cai" nil C   nil () nil nil "Central American Indian languages")
            ("caj" nil  nil   nil I   E   () nil nil "Chané")
            ("cak" nil  nil   nil I   L   () nil nil "Cakchiquel, Central")
            ("cal" nil  nil   nil I   L   () nil nil "Carolinian")
            ("cam" nil  nil   nil I   L   () nil nil "Cemuhî")
            ("can" nil  nil   nil I   L   () nil nil "Chambri")
            ("cao" nil  nil   nil I   L   () nil nil "Chácobo")
            ("cap" nil  nil   nil I   L   () nil nil "Chipaya")
            ("caq" nil  nil   nil I   L   () nil nil "Nicobarese, Car")
            ("car" nil  "car" nil I   L   () nil nil "Carib")
            ("cas" nil  nil   nil I   L   () nil nil "Tsimané")
            ("cat" "ca" "cat" nil I   L   () "Indo-European" "català" "Catalan")
            (nil   nil  "cau" nil C   nil () nil nil "Caucasian languages")
            ("cav" nil  nil   nil I   L   () nil nil "Cavineña")
            ("caw" nil  nil   nil I   L   () nil nil "Callawalla")
            ("cax" nil  nil   nil I   L   () nil nil "Chiquitano")
            ("cay" nil  nil   nil I   L   () nil "Goyogo̱hó:nǫ’" "Cayuga")
            ("caz" nil  nil   nil I   E   () nil nil "Canichana")
            ("cbb" nil  nil   nil I   L   () nil nil "Cabiyarí")
            ("cbc" nil  nil   nil I   L   () nil nil "Carapana")
            ("cbd" nil  nil   nil I   L   () nil nil "Carijona")
            ("cbe" nil  nil   t   nil nil () "spurious language" nil "Chipiajes")
            ("cbg" nil  nil   nil I   L   () nil nil "Chimila")
            ("cbh" nil  nil   t   nil nil () "spurious language" nil "Cagua")
            ("cbi" nil  nil   nil I   L   () nil nil "Chachi")
            ("cbj" nil  nil   nil I   L   () nil nil "Ede Cabe")
            ("cbk" nil  nil   nil I   L   () nil nil "Chavacano")
            ("cbl" nil  nil   nil I   L   () nil nil "Chin, Bualkhaw")
            ("cbm" nil  nil   t   I   L   () nil nil "Cakchiquel, Yepocapa Southwestern")
            ("cbn" nil  nil   nil I   L   () nil nil "Nyahkur")
            ("cbo" nil  nil   nil I   L   () nil nil "Izora")
            ("cbq" nil  nil   nil I   L   () "Niger–Congo" nil "Cuba, Tsucuba")
            ("cbr" nil  nil   nil I   L   () nil nil "Cashibo-Cacataibo")
            ("cbs" nil  nil   nil I   L   () nil nil "Cashinahua")
            ("cbt" nil  nil   nil I   L   () nil nil "Chayahuita")
            ("cbu" nil  nil   nil I   L   () nil nil "Candoshi-Shapra")
            ("cbv" nil  nil   nil I   L   () nil nil "Cacua")
            ("cbw" nil  nil   nil I   L   () nil nil "Kinabalian")
            ("cby" nil  nil   nil I   L   () nil nil "Carabayo")
            ("cca" nil  nil   t   I   E   () "spurious language" nil "Cauca")
            ("ccc" nil  nil   nil I   L   () nil nil "Chamicuro")
            ("ccd" nil  nil   nil I   L   () nil nil "Cafundo Creole")
            ("cce" nil  nil   nil I   L   () nil nil "Chopi")
            ("ccg" nil  nil   nil I   L   () nil nil "Samba Daka")
            ("cch" nil  nil   nil I   L   () nil nil "Atsam")
            ("ccj" nil  nil   nil I   L   () nil nil "Kasanga")
            ("ccl" nil  nil   nil I   L   () nil nil "Cutchi-Swahili")
            ("ccm" nil  nil   nil I   L   () nil nil "Malaccan Creole Malay")
            ("cco" nil  nil   nil I   L   () nil nil "Chinantec, Comaltepec")
            ("ccp" nil  nil   nil I   L   () nil nil "Chakma")
            ("ccq" nil  nil   t   I   L   () nil nil "Chaungtha")
            ("ccr" nil  nil   nil I   E   () nil nil "Cacaopera")
            ("ccx" nil  nil   t   nil nil () nil nil "Northern Zhuang")
            ("ccy" nil  nil   t   nil nil () nil nil "Southern Zhuang")
            ("cda" nil  nil   nil I   L   () nil nil "Choni")
            ("cde" nil  nil   nil I   L   () nil nil "Chenchu")
            ("cdf" nil  nil   nil I   L   () nil nil "Chiru")
            ("cdg" nil  nil   t   I   L   () nil nil "Chamari")
            ("cdh" nil  nil   nil I   L   () nil nil "Chambeali")
            ("cdi" nil  nil   nil I   L   () nil nil "Chodri")
            ("cdj" nil  nil   nil I   L   () nil nil "Churahi")
            ("cdm" nil  nil   nil I   L   () nil nil "Chepang")
            ("cdn" nil  nil   nil I   L   () nil nil "Chaudangsi")
            ("cdo" nil  nil   nil I   L   () "Chinese" "閩東語" "Min Dong Chinese")
            ("cdr" nil  nil   nil I   L   () nil nil "Cinda-Regi-Tiyal")
            ("cds" nil  nil   nil I   L   () nil nil "Chadian Sign Language")
            ("cdy" nil  nil   nil I   L   () nil nil "Chadong")
            ("cdz" nil  nil   nil I   L   () nil nil "Koda")
            ("cea" nil  nil   nil I   E   () nil nil "Chehalis, Lower")
            ("ceb" nil  "ceb" nil I   L   () nil "S(in)ugboanon" "Cebuano")
            ("ceg" nil  nil   nil I   L   () nil nil "Chamacoco")
            ("cek" nil  nil   nil I   L   () nil nil "Eastern Khumi Chin")
            (nil   nil  "cel" nil C   nil () nil nil "Celtic languages")
            ("cen" nil  nil   nil I   L   () nil nil "Cen")
            ("ces" "cs" "cze" nil I   L   () "Indo-European" "čeština" "Czech")
            ("cet" nil  nil   nil I   L   () nil nil "Centúúm")
            ("cey" nil  nil   nil I   L   () "Sino-Tibetan" nil "Ekai Chin")
            ("cfa" nil  nil   nil I   L   () nil nil "Dijim-Bwilim")
            ("cfd" nil  nil   nil I   L   () nil nil "Cara")
            ("cfg" nil  nil   nil I   L   () nil nil "Como Karim")
            ("cfm" nil  nil   nil I   L   () nil nil "Falam Chin")
            ("cga" nil  nil   nil I   L   () nil nil "Changriwa")
            ("cgc" nil  nil   nil I   L   () nil nil "Kagayanen")
            ("cgg" nil  nil   nil I   L   () nil nil "Chiga")
            ("cgk" nil  nil   nil I   L   () nil nil "Chocangacakha")
            ("cha" "ch" "cha" nil I   L   () "Austronesian" "Chamoru" "Chamorro")
            ("chb" nil  "chb" nil I   E   () nil nil "Chibcha")
            ("chc" nil  nil   nil I   E   () nil "Iyeye" "Catawba")
            ("chd" nil  nil   nil I   L   () nil nil "Chontal, Highland Oaxaca")
            ("che" "ce" "che" nil I   L   () "Northeast Caucasian" "нохчийн" "Chechen")
            ("chf" nil  nil   nil I   L   () nil nil "Chontal, Tabasco")
            ("chg" nil  "chg" nil I   E   () nil "جغتای" "Chagatai")
            ("chh" nil  nil   nil I   E   () nil nil "Chinook")
            ("chj" nil  nil   nil I   L   () nil nil "Chinantec, Ojitlán")
            ("chk" nil  "chk" nil I   L   () nil nil "Chuukese")
            ("chl" nil  nil   nil I   L   () nil nil "Cahuilla")
            ("chm" nil  "chm" nil M   L   ("mhr" "mrj") nil "марий" "Mari (Russia)")
            ("chn" nil  "chn" nil I   L   () nil "chinuk wawa" "Chinook jargon")
            ("cho" nil  "cho" nil I   L   () nil "Chahta" "Choctaw")
            ("chp" nil  "chp" nil I   L   () nil "ᑌᓀᓲᒢᕄᓀ (Dëne Sųłiné)" "Chipewyan")
            ("chq" nil  nil   nil I   L   () nil nil "Chinantec, Quiotepec")
            ("chr" nil  "chr" nil I   L   () nil "ᏣᎳᎩ" "Cherokee")
            ("chs" nil  nil   t   I   E   () nil nil "Chumash")
            ("cht" nil  nil   nil I   E   () nil nil "Cholón")
            ("chu" "cu" "chu" nil I   A   () "Indo-European" "ѩзыкъ словѣньскъ" "Old Church Slavonic")
            ("chv" "cv" "chv" nil I   L   () "Turkic" "Чӑваш" "Chuvash")
            ("chw" nil  nil   nil I   L   () nil nil "Chuwabu")
            ("chx" nil  nil   nil I   L   () nil nil "Chantyal")
            ("chy" nil  "chy" nil I   L   () nil "Tsêhést" "Cheyenne")
            ("chz" nil  nil   nil I   L   () nil nil "Chinantec, Ozumacín")
            ("cia" nil  nil   nil I   L   () nil nil "Cia-Cia")
            ("cib" nil  nil   nil I   L   () nil nil "Gbe, Ci")
            ("cic" nil  nil   nil I   L   () nil "Chikasha" "Chickasaw")
            ("cid" nil  nil   nil I   E   () nil nil "Chimariko")
            ("cie" nil  nil   nil I   L   () nil nil "Cineni")
            ("cih" nil  nil   nil I   L   () nil nil "Chinali")
            ("cik" nil  nil   nil I   L   () nil nil "Kinnauri, Chitkuli")
            ("cim" nil  nil   nil I   L   () nil "Zimbrisch" "Cimbrian")
            ("cin" nil  nil   nil I   L   () nil nil "Cinta Larga")
            ("cip" nil  nil   nil I   L   () nil nil "Chiapanec")
            ("cir" nil  nil   nil I   L   () nil nil "Tiri")
            ("cit" nil  nil   t   I   L   () "Indo-European" "চাঁটগাঁইয়া" "Chittagonian")
            ("ciw" nil  nil   nil I   L   () nil "ᐊᓂᐦᔑᓈᐯᒧᐧᐃᓐ / ᐅᒋᐧᐯᒧᐧᐃᓐ (Anishinaabemowin / Ojibwemowin)" "Chippewa")
            ("ciy" nil  nil   nil I   L   () nil nil "Chaima")
            ("cja" nil  nil   nil I   L   () nil nil "Cham, Western")
            ("cje" nil  nil   nil I   L   () nil nil "CHARu")
            ("cjh" nil  nil   nil I   E   () nil nil "Chehalis, Upper")
            ("cji" nil  nil   nil I   L   () nil nil "Chamalal")
            ("cjk" nil  nil   nil I   L   () nil nil "Chokwe")
            ("cjm" nil  nil   nil I   L   () nil nil "Cham, Eastern")
            ("cjn" nil  nil   nil I   L   () nil nil "Chenapian")
            ("cjo" nil  nil   nil I   L   () nil nil "Ashéninka Pajonal")
            ("cjp" nil  nil   nil I   L   () nil nil "Cabécar")
            ("cjr" nil  nil   t   I   E   () nil nil "Chorotega")
            ("cjs" nil  nil   nil I   L   () nil "Шор" "Shor")
            ("cjv" nil  nil   nil I   L   () nil nil "Chuave")
            ("cjy" nil  nil   nil I   L   () "Chinese" "晋语" "Jinyu Chinese")
            ("cka" nil  nil   t   I   L   () nil nil "Chin, Khumi Awa")
            ("ckb" nil  nil   nil I   L   () "Indo-European" "کوردی" "Kurdish, Central")
            ("ckc" nil  nil   t   I   L   () "Cakchiquel" nil "Cakchiquel, Northern")
            ("ckd" nil  nil   t   I   L   () "Cakchiquel" nil "Cakchiquel, South Central")
            ("cke" nil  nil   t   I   L   () "Cakchiquel" nil "Cakchiquel, Eastern")
            ("ckf" nil  nil   t   I   L   () "Cakchiquel" nil "Cakchiquel, Southern")
            ("ckh" nil  nil   nil I   L   () nil nil "Chak")
            ("cki" nil  nil   t   I   L   () "Cakchiquel" nil "Cakchiquel, Santa María De Jesús")
            ("ckj" nil  nil   t   I   L   () "Cakchiquel" nil "Cakchiquel, Santo Domingo Xenacoj")
            ("ckk" nil  nil   t   I   L   () "Cakchiquel" nil "Cakchiquel, Acatenango Southwestern")
            ("ckl" nil  nil   nil I   L   () nil nil "Cibak")
            ("ckm" nil  nil   nil I   L   () "Indo-European" nil "Chakavian")
            ("ckn" nil  nil   nil I   L   () nil nil "Kaang Chin")
            ("cko" nil  nil   nil I   L   () nil nil "Anufo")
            ("ckq" nil  nil   nil I   L   () nil nil "Kajakse")
            ("ckr" nil  nil   nil I   L   () nil nil "Kairak")
            ("cks" nil  nil   nil I   L   () nil nil "Tayo")
            ("ckt" nil  nil   nil I   L   () nil "чаучу" "Chukot")
            ("cku" nil  nil   nil I   L   () nil nil "Koasati")
            ("ckv" nil  nil   nil I   L   () nil nil "Kavalan")
            ("ckw" nil  nil   t   I   L   () nil nil "Cakchiquel, Western")
            ("ckx" nil  nil   nil I   L   () nil nil "Caka")
            ("cky" nil  nil   nil I   L   () nil nil "Cakfem-Mushere")
            ("ckz" nil  nil   nil I   L   () nil nil "Cakchiquel-Quiché Mixed Language")
            ("cla" nil  nil   nil I   L   () nil nil "Ron")
            ("clc" nil  nil   nil I   L   () nil "Tšinlhqot⤙in, Tsilhqot’in" "Chilcotin")
            ("cld" nil  nil   nil I   L   () nil "ܟܠܕܝܐ" "Chaldean Neo-Aramaic")
            ("cle" nil  nil   nil I   L   () nil nil "Chinantec, Lealao")
            ("clh" nil  nil   nil I   L   () nil nil "Chilisso")
            ("cli" nil  nil   nil I   L   () nil nil "Chakali")
            ("clj" nil  nil   nil I   L   () nil nil "Laitu Chin")
            ("clk" nil  nil   nil I   L   () nil nil "Idu-Mishmi")
            ("cll" nil  nil   nil I   L   () nil nil "Chala")
            ("clm" nil  nil   nil I   L   () "Salishan" "nəxʷsƛʼayʼəmʼúcən" "Clallam, Klallam")
            ("clo" nil  nil   nil I   L   () nil nil "Chontal, Lowland Oaxaca")
            ("clt" nil  nil   nil I   L   () nil nil "Lautu Chin")
            ("clu" nil  nil   nil I   L   () nil nil "Caluyanun")
            ("clw" nil  nil   nil I   L   () nil nil "Chulym")
            ("cly" nil  nil   nil I   L   () nil nil "Chatino, Eastern Highland")
            ("cma" nil  nil   nil I   L   () nil nil "Maa")
            (nil   nil  "cmc" nil C   nil () nil nil "Chamic languages")
            ("cme" nil  nil   nil I   L   () nil nil "Cerma")
            ("cmg" nil  nil   nil I   H   () nil nil "Mongolian, Classical")
            ("cmi" nil  nil   nil I   L   () nil nil "Emberá-Chamí")
            ("cmk" nil  nil   t   I   E   () nil nil "Chimakum")
            ("cml" nil  nil   nil I   L   () nil nil "Campalagian")
            ("cmm" nil  nil   nil I   E   () nil nil "Michigamea")
            ("cmn" nil  nil   nil I   L   () "Chinese" "官話; 北方話" "Mandarin Chinese")
            ("cmo" nil  nil   nil I   L   () nil nil "Mnong, Central")
            ("cmr" nil  nil   nil I   L   () nil nil "Chin, Mro")
            ("cms" nil  nil   nil I   A   () nil nil "Messapic")
            ("cmt" nil  nil   nil I   L   () nil nil "Camtho")
            ("cna" nil  nil   nil I   L   () nil nil "Changthang")
            ("cnb" nil  nil   nil I   L   () nil nil "Chin, Chinbon")
            ("cnc" nil  nil   nil I   L   () nil nil "Côông")
            ("cng" nil  nil   nil I   L   () nil nil "Qiang, Northern")
            ("cnh" nil  nil   nil I   L   () nil nil "Chin, Lai")
            ("cni" nil  nil   nil I   L   () nil nil "Asháninka")
            ("cnk" nil  nil   nil I   L   () nil nil "Chin, Khumi")
            ("cnl" nil  nil   nil I   L   () nil nil "Chinantec, Lalana")
            ("cnm" nil  nil   t   I   L   () nil nil "Chuj, Ixtatán")
            ("cno" nil  nil   nil I   L   () nil nil "Con")
            ("cnp" nil  nil   nil I   L   () "Sino-Tibetan" nil "Northern Ping Chinese, Northern Pinghua")
            ("cnq" nil  nil   nil I   L   () "Niger–Congo" nil "Chung")
            ("cnr" nil  "cnr" nil I   L   () nil "crnogorski / црногорски" "Montenegrin")
            ("cns" nil  nil   nil I   L   () nil nil "Asmat, Central")
            ("cnt" nil  nil   nil I   L   () nil nil "Chinantec, Tepetotutla")
            ("cnu" nil  nil   nil I   L   () nil nil "Chenoua")
            ("cnw" nil  nil   nil I   L   () nil nil "Chin, Ngawn")
            ("cnx" nil  nil   nil I   H   () nil nil "Cornish, Middle")
            ("coa" nil  nil   nil I   L   () nil nil "Malay, Cocos Islands")
            ("cob" nil  nil   nil I   E   () nil nil "Chicomuceltec")
            ("coc" nil  nil   nil I   L   () nil "Kwikapa" "Cocopa")
            ("cod" nil  nil   nil I   L   () nil nil "Cocama-Cocamilla")
            ("coe" nil  nil   nil I   L   () nil nil "Koreguaje")
            ("cof" nil  nil   nil I   L   () nil nil "Colorado")
            ("cog" nil  nil   nil I   L   () nil nil "Chong")
            ("coh" nil  nil   nil I   L   () nil nil "Chonyi")
            ("coj" nil  nil   nil I   E   () nil "Tipai" "Cochimi")
            ("cok" nil  nil   nil I   L   () nil nil "Cora, Santa Teresa")
            ("col" nil  nil   nil I   L   () nil nil "Columbia-Wenatchi")
            ("com" nil  nil   nil I   L   () nil "nʉmʉ tekwapʉ̱" "Comanche")
            ("con" nil  nil   nil I   L   () nil nil "Cofán")
            ("coo" nil  nil   nil I   L   () nil "Saɬuɬtxʷ" "Comox")
            ("cop" nil  "cop" nil I   E   () nil "ⲙⲉⲧⲛ̀ⲣⲉⲙⲛ̀ⲭⲏⲙⲓ" "Coptic")
            ("coq" nil  nil   nil I   E   () nil nil "Coquille")
            ("cor" "kw" "cor" nil I   L   () "Indo-European" "Kernewek" "Cornish")
            ("cos" "co" "cos" nil I   L   () "Indo-European" "corsu" "Corsican")
            ("cot" nil  nil   nil I   L   () nil nil "Caquinte")
            ("cou" nil  nil   nil I   L   () nil nil "Wamey")
            ("cov" nil  nil   nil I   L   () nil nil "Cao Miao")
            ("cow" nil  nil   nil I   E   () nil nil "Cowlitz")
            ("cox" nil  nil   nil I   L   () nil nil "Nanti")
            ("coy" nil  nil   t   I   E   () nil nil "Coyaima")
            ("coz" nil  nil   nil I   L   () nil nil "Chochotec")
            ("cpa" nil  nil   nil I   L   () nil nil "Chinantec, Palantla")
            ("cpb" nil  nil   nil I   L   () nil nil "Ashéninka, Ucayali-Yurúa")
            ("cpc" nil  nil   nil I   L   () nil nil "Ajyíninka Apurucayali")
            (nil   nil  "cpe" nil C   nil () nil nil "Creoles and pidgins, English based")
            (nil   nil  "cpf" nil C   nil () nil nil "Creoles and pidgins, French-based")
            ("cpg" nil  nil   nil I   E   () nil nil "Greek (Cappadocian)")
            ("cpi" nil  nil   nil I   L   () nil nil "Chinese Pidgin English")
            ("cpn" nil  nil   nil I   L   () nil nil "Cherepon")
            ("cpo" nil  nil   nil I   L   () nil nil "Kpeego")
            (nil   nil  "cpp" nil C   nil () nil nil "Creoles and pidgins, Portuguese-based")
            ("cps" nil  nil   nil I   L   () nil nil "Capiznon")
            ("cpu" nil  nil   nil I   L   () nil nil "Ashéninka, Pichis")
            ("cpx" nil  nil   nil I   L   () "Chinese" nil "Pu-Xian Chinese")
            ("cpy" nil  nil   nil I   L   () nil nil "Ashéninka, South Ucayali")
            ("cqd" nil  nil   nil I   L   () nil nil "Chuanqiandian Cluster Miao")
            ("cqu" nil  nil   t   I   L   () nil nil "Quechua, Chilean")
            ("cra" nil  nil   nil I   L   () nil nil "Chara")
            ("crb" nil  nil   nil I   E   () nil nil "Carib, Island")
            ("crc" nil  nil   nil I   L   () nil nil "Lonwolwol")
            ("crd" nil  nil   nil I   L   () nil nil "Coeur d'Alene")
            ("cre" "cr" "cre" nil M   L   ("crj" "crk" "crl"
                                           "crm" "csw" "cwd") "Cree" "ᓀᐦᐃᔭᐤ (Nehiyāw)" "Cree")
            ("crf" nil  nil   nil I   E   () nil nil "Caramanta")
            ("crg" nil  nil   nil I   L   () nil nil "Michif")
            ("crh" nil  "crh" nil I   L   () nil "Къырым Татар" "Crimean Tatar")
            ("cri" nil  nil   nil I   L   () nil nil "Sãotomense")
            ("crj" nil  nil   nil I   L   () "Cree" nil "East Cree (Southern)")
            ("crk" nil  nil   nil I   L   () "Cree" nil "Cree (Plains)")
            ("crl" nil  nil   nil I   L   () "Cree" nil "East Cree (Northern)")
            ("crm" nil  nil   nil I   L   () "Cree" nil "Cree (Moose)")
            ("crn" nil  nil   nil I   L   () nil nil "Cora, El Nayar")
            ("cro" nil  nil   nil I   L   () nil nil "Crow")
            (nil   nil  "crp" nil C   nil () nil nil "Creoles and pidgins")
            ("crq" nil  nil   nil I   L   () nil nil "Chorote, Iyo'wujwa")
            ("crr" nil  nil   nil I   E   () nil nil "Carolina Algonquian")
            ("crs" nil  nil   nil I   L   () nil nil "Seselwa Creole French")
            ("crt" nil  nil   nil I   L   () nil nil "Chorote, Iyojwa'ja")
            ("cru" nil  nil   t   nil nil () nil nil "Carútana")
            ("crv" nil  nil   nil I   L   () nil nil "Chaura")
            ("crw" nil  nil   nil I   L   () nil nil "Chrau")
            ("crx" nil  nil   nil I   L   () nil "ᑕᗸᒡ" "Carrier")
            ("cry" nil  nil   nil I   L   () nil nil "Cori")
            ("crz" nil  nil   nil I   E   () nil nil "Cruzeño")
            ("csa" nil  nil   nil I   L   () nil nil "Chinantec, Chiltepec")
            ("csb" nil  "csb" nil I   L   () nil "kaszëbsczi" "Kashubian")
            ("csc" nil  nil   nil I   L   () nil nil "Catalan Sign Language")
            ("csd" nil  nil   nil I   L   () nil nil "Chiangmai Sign Language")
            ("cse" nil  nil   nil I   L   () nil nil "Czech Sign Language")
            ("csf" nil  nil   nil I   L   () nil nil "Cuba Sign Language")
            ("csg" nil  nil   nil I   L   () nil nil "Chilean Sign Language")
            ("csh" nil  nil   nil I   L   () nil nil "Chin, Asho")
            ("csi" nil  nil   nil I   E   () nil nil "Miwok, Coast")
            ("csj" nil  nil   nil I   L   () nil nil "Songlai Chin")
            ("csk" nil  nil   nil I   L   () nil nil "Jola-Kasa")
            ("csl" nil  nil   nil I   L   () nil nil "Chinese Sign Language")
            ("csm" nil  nil   nil I   L   () nil nil "Miwok, Central Sierra")
            ("csn" nil  nil   nil I   L   () nil nil "Colombian Sign Language")
            ("cso" nil  nil   nil I   L   () nil nil "Chinantec, Sochiapan")
            ("csp" nil  nil   nil I   L   () "Sino-Tibetan" nil "Southern Ping Chinese, Southern Pinghua")
            ("csq" nil  nil   nil I   L   () nil nil "Croatia Sign Language")
            ("csr" nil  nil   nil I   L   () nil nil "Costa Rican Sign Language")
            ("css" nil  nil   nil I   E   () nil nil "Ohlone (Southern)")
            ("cst" nil  nil   nil I   L   () nil nil "Ohlone (Northern)")
            ("csv" nil  nil   nil I   L   () nil nil "Sumtu Chin")
            ("csw" nil  nil   nil I   L   () "Cree" nil "Cree (Swampy)")
            ("csx" nil  nil   nil I   L   () nil nil "Cambodian Sign Language")
            ("csy" nil  nil   nil I   L   () nil nil "Chin, Siyin")
            ("csz" nil  nil   nil I   L   () nil nil "Coos")
            ("cta" nil  nil   nil I   L   () nil nil "Chatino, Tataltepec")
            ("ctc" nil  nil   nil I   E   () nil nil "Chetco")
            ("ctd" nil  nil   nil I   L   () nil nil "Chin, Tedim")
            ("cte" nil  nil   nil I   L   () nil nil "Chinantec, Tepinapa")
            ("ctg" nil  nil   nil I   L   () nil nil "Chittagonian")
            ("cth" nil  nil   nil I   L   () nil nil "Thaiphum Chin")
            ("cti" nil  nil   t   I   L   () nil nil "Chol, Tila")
            ("ctl" nil  nil   nil I   L   () nil nil "Chinantec, Tlacoatzintepec")
            ("ctm" nil  nil   nil I   E   () nil nil "Chitimacha")
            ("ctn" nil  nil   nil I   L   () nil nil "Chhintange")
            ("cto" nil  nil   nil I   L   () nil nil "Emberá-Catío")
            ("ctp" nil  nil   nil I   L   () nil nil "Chatino, Western Highland")
            ("cts" nil  nil   nil I   L   () nil nil "Bicolano, Northern Catanduanes")
            ("ctt" nil  nil   nil I   L   () nil nil "Wayanad Chetti")
            ("ctu" nil  nil   nil I   L   () nil nil "Chol, Tumbalá")
            ("cty" nil  nil   nil I   L   () "Dravidian" nil "Moundadan Chetty")
            ("ctz" nil  nil   nil I   L   () nil nil "Chatino, Zacatepec")
            ("cua" nil  nil   nil I   L   () nil nil "Cua")
            ("cub" nil  nil   nil I   L   () nil nil "Cubeo")
            ("cuc" nil  nil   nil I   L   () nil nil "Chinantec, Usila")
            ("cug" nil  nil   t   I   L   () nil nil "Cung")
            ("cuh" nil  nil   nil I   L   () nil nil "Chuka")
            ("cui" nil  nil   nil I   L   () nil nil "Cuiba")
            ("cuj" nil  nil   nil I   L   () nil nil "Mashco Piro")
            ("cuk" nil  nil   nil I   L   () nil nil "Kuna, San Blas")
            ("cul" nil  nil   nil I   L   () nil nil "Culina")
            ("cum" nil  nil   t   nil nil () "spurious language" nil "Cumeral")
            ("cun" nil  nil   t   I   L   () nil nil "Quiché, Cunén")
            ("cuo" nil  nil   nil I   E   () nil nil "Cumanagoto")
            ("cup" nil  nil   nil I   E   () nil "Kuupangaxwichem" "Cupeño")
            ("cuq" nil  nil   nil I   L   () nil nil "Cun")
            ("cur" nil  nil   nil I   L   () nil nil "Chhulung")
            (nil   nil  "cus" nil C   nil () nil nil "Cushitic languages")
            ("cut" nil  nil   nil I   L   () nil nil "Cuicatec, Teutila")
            ("cuu" nil  nil   nil I   L   () nil nil "Tai Ya")
            ("cuv" nil  nil   nil I   L   () nil nil "Cuvok")
            ("cuw" nil  nil   nil I   L   () nil nil "Chukwa")
            ("cux" nil  nil   nil I   L   () nil nil "Cuicatec, Tepeuxila")
            ("cuy" nil  nil   nil I   L   () "isolate" nil "Cuitlatec")
            ("cvg" nil  nil   nil I   L   () nil nil "Chug")
            ("cvn" nil  nil   nil I   L   () nil nil "Chinantec, Valle Nacional")
            ("cwa" nil  nil   nil I   L   () nil nil "Kabwa")
            ("cwb" nil  nil   nil I   L   () nil nil "Maindo")
            ("cwd" nil  nil   nil I   L   () "Cree" nil "Cree (Woods)")
            ("cwe" nil  nil   nil I   L   () nil nil "Kwere")
            ("cwg" nil  nil   nil I   L   () nil nil "Chewong")
            ("cwt" nil  nil   nil I   L   () nil nil "Kuwaataay")
            ("cxh" nil  nil   nil I   L   () "Afro-Asiatic" nil "Cha'ari")
            ("cya" nil  nil   nil I   L   () nil nil "Chatino, Nopala")
            ("cyb" nil  nil   nil I   E   () nil nil "Cayubaba")
            ("cym" "cy" "wel" nil I   L   () "Indo-European" "Cymraeg" "Welsh")
            ("cyo" nil  nil   nil I   L   () nil nil "Cuyonon")
            ("czh" nil  nil   nil I   L   () "Chinese" "徽州话" "Huizhou Chinese")
            ("czk" nil  nil   nil I   E   () nil nil "Knaanic")
            ("czn" nil  nil   nil I   L   () nil nil "Chatino, Zenzontepec")
            ("czo" nil  nil   nil I   L   () "Chinese" nil "Min Zhong Chinese")
            ("czt" nil  nil   nil I   L   () nil nil "Chin, Zotung")
            ;; D
            ("daa" nil  nil   nil I   L   () nil nil "Dangaléat")
            ("dac" nil  nil   nil I   L   () nil nil "Dambi")
            ("dad" nil  nil   nil I   L   () nil nil "Marik")
            ("dae" nil  nil   nil I   L   () nil nil "Duupa")
            ("daf" nil  nil   t   I   L   () nil nil "Dan")
            ("dag" nil  nil   nil I   L   () nil nil "Dagbani")
            ("dah" nil  nil   nil I   L   () nil nil "Gwahatike")
            ("dai" nil  nil   nil I   L   () nil nil "Day")
            ("daj" nil  nil   nil I   L   () nil nil "Daju, Dar Fur")
            ("dak" nil  "dak" nil I   L   () nil "Lakhota" "Dakota")
            ("dal" nil  nil   nil I   L   () nil nil "Dahalo")
            ("dam" nil  nil   nil I   L   () nil nil "Damakawa")
            ("dan" "da" "dan" nil I   L   () "Indo-European" "dansk" "Danish")
            ("dao" nil  nil   nil I   L   () nil nil "Chin, Daai")
            ("dap" nil  nil   t   I   L   () nil nil "Nisi")
            ("daq" nil  nil   nil I   L   () nil nil "Maria, Dandami")
            ("dar" nil  "dar" nil I   L   () nil "дарган" "Dargwa")
            ("das" nil  nil   nil I   L   () nil nil "Daho-Doo")
            ("dat" nil  nil   t   nil nil () nil nil "Darang Deng")
            ("dau" nil  nil   nil I   L   () nil nil "Daju, Dar Sila")
            ("dav" nil  nil   nil I   L   () nil nil "Taita")
            ("daw" nil  nil   nil I   L   () nil nil "Davawenyo")
            ("dax" nil  nil   nil I   L   () nil nil "Dayi")
            (nil   nil  "day" nil C   nil () nil nil "Land Dayak languages")
            ("daz" nil  nil   nil I   L   () nil nil "Dao")
            ("dba" nil  nil   nil I   L   () nil nil "Dogon, Bangeri Me")
            ("dbb" nil  nil   nil I   L   () nil nil "Deno")
            ("dbd" nil  nil   nil I   L   () nil nil "Dadiya")
            ("dbe" nil  nil   nil I   L   () nil nil "Dabe")
            ("dbf" nil  nil   nil I   L   () nil nil "Edopi")
            ("dbg" nil  nil   nil I   L   () nil nil "Dogon, Dogul Dom")
            ("dbi" nil  nil   nil I   L   () nil nil "Doka")
            ("dbj" nil  nil   nil I   L   () nil nil "Ida'an")
            ("dbl" nil  nil   nil I   L   () nil nil "Dyirbal")
            ("dbm" nil  nil   nil I   L   () nil nil "Duguri")
            ("dbn" nil  nil   nil I   L   () nil nil "Duriankere")
            ("dbo" nil  nil   nil I   L   () nil nil "Dulbu")
            ("dbp" nil  nil   nil I   L   () nil nil "Duwai")
            ("dbq" nil  nil   nil I   L   () nil nil "Daba")
            ("dbr" nil  nil   nil I   L   () nil nil "Dabarre")
            ("dbt" nil  nil   nil I   L   () nil nil "Ben Tey Dogon")
            ("dbu" nil  nil   nil I   L   () nil nil "Dogon, Bondum Dom")
            ("dbv" nil  nil   nil I   L   () nil nil "Dungu")
            ("dbw" nil  nil   nil I   L   () nil nil "Bankan Tey Dogon")
            ("dby" nil  nil   nil I   L   () nil nil "Dibiyaso")
            ("dcc" nil  nil   nil I   L   () nil nil "Deccan")
            ("dcr" nil  nil   nil I   E   () nil nil "Negerhollands")
            ("dda" nil  nil   nil I   E   () nil nil "Dadi Dadi")
            ("ddd" nil  nil   nil I   L   () nil nil "Dongotono")
            ("dde" nil  nil   nil I   L   () nil nil "Doondo")
            ("ddg" nil  nil   nil I   L   () nil nil "Fataluku")
            ("ddi" nil  nil   nil I   L   () nil nil "Diodio")
            ("ddj" nil  nil   nil I   L   () nil nil "Jaru")
            ("ddn" nil  nil   nil I   L   () nil nil "Dendi (Benin)")
            ("ddo" nil  nil   nil I   L   () nil "цез" "Tsez")
            ("ddr" nil  nil   nil I   E   () nil nil "Dhudhuroa")
            ("dds" nil  nil   nil I   L   () nil nil "Dogon, Donno So")
            ("ddw" nil  nil   nil I   L   () nil nil "Dawera-Daweloor")
            ("dec" nil  nil   nil I   L   () nil nil "Dagik")
            ("ded" nil  nil   nil I   L   () nil nil "Dedua")
            ("dee" nil  nil   nil I   L   () nil nil "Dewoin")
            ("def" nil  nil   nil I   L   () nil nil "Dezfuli")
            ("deg" nil  nil   nil I   L   () nil nil "Degema")
            ("deh" nil  nil   nil I   L   () nil nil "Dehwari")
            ("dei" nil  nil   nil I   L   () nil nil "Demisa")
            ("dek" nil  nil   nil I   L   () nil nil "Dek")
            ("del" nil  "del" nil M   L   ("umu" "unm") nil "Lënape" "Delaware")
            ("dem" nil  nil   nil I   L   () nil nil "Dem")
            ("den" nil  "den" nil M   L   ("scs" "xsl") nil nil "Slave (Athapascan)")
            ("dep" nil  nil   nil I   E   () nil nil "Delaware, Pidgin")
            ("deq" nil  nil   nil I   L   () nil nil "Dendi (Central African Republic)")
            ("der" nil  nil   nil I   L   () nil nil "Deori")
            ("des" nil  nil   nil I   L   () nil nil "Desano")
            ("deu" "de" "ger" nil I   L   () "Indo-European" "Deutsch" "German")
            ("dev" nil  nil   nil I   L   () nil nil "Domung")
            ("dez" nil  nil   nil I   L   () nil nil "Dengese")
            ("dga" nil  nil   nil I   L   () nil nil "Dagaare, Southern")
            ("dgb" nil  nil   nil I   L   () nil nil "Bunoge Dogon")
            ("dgc" nil  nil   nil I   L   () nil nil "Agta, Casiguran Dumagat")
            ("dgd" nil  nil   nil I   L   () nil nil "Dagaari Dioula")
            ("dge" nil  nil   nil I   L   () nil nil "Degenan")
            ("dgg" nil  nil   nil I   L   () nil nil "Doga")
            ("dgh" nil  nil   nil I   L   () nil nil "Dghwede")
            ("dgi" nil  nil   nil I   L   () nil nil "Dagara, Northern")
            ("dgk" nil  nil   nil I   L   () nil nil "Dagba")
            ("dgl" nil  nil   nil I   L   () nil nil "Andaandi")
            ("dgn" nil  nil   nil I   E   () nil nil "Dagoman")
            ("dgo" nil  nil   nil I   L   () nil nil "Dogri (specific)")
            ("dgr" nil  "dgr" nil I   L   () nil "Tłįchǫ" "Dogrib")
            ("dgs" nil  nil   nil I   L   () nil nil "Dogoso")
            ("dgt" nil  nil   nil I   E   () nil nil "Ndrag'ngith")
            ("dgu" nil  nil   t   I   L   () nil nil "Degaru")
            ("dgw" nil  nil   nil I   E   () nil nil "Daungwurrung")
            ("dgx" nil  nil   nil I   L   () nil nil "Doghoro")
            ("dgz" nil  nil   nil I   L   () nil nil "Daga")
            ("dha" nil  nil   t   nil nil () nil nil "Dhanwar (spurious)")
            ("dhd" nil  nil   nil I   L   () nil nil "Dhundari")
            ("dhg" nil  nil   nil I   L   () nil nil "Dhangu")
            ("dhi" nil  nil   nil I   L   () nil nil "Dhimal")
            ("dhl" nil  nil   nil I   L   () nil nil "Dhalandji")
            ("dhm" nil  nil   nil I   L   () nil nil "Zemba")
            ("dhn" nil  nil   nil I   L   () nil nil "Dhanki")
            ("dho" nil  nil   nil I   L   () nil nil "Dhodia")
            ("dhr" nil  nil   nil I   L   () nil nil "Dhargari")
            ("dhs" nil  nil   nil I   L   () nil nil "Dhaiso")
            ("dhu" nil  nil   nil I   E   () nil nil "Dhurga")
            ("dhv" nil  nil   nil I   L   () nil "drehu" "Dehu")
            ("dhw" nil  nil   nil I   L   () nil nil "Dhanwar (Nepal)")
            ("dhx" nil  nil   nil I   L   () nil nil "Dhungaloo")
            ("dia" nil  nil   nil I   L   () nil nil "Dia")
            ("dib" nil  nil   nil I   L   () nil nil "Dinka, South Central")
            ("dic" nil  nil   nil I   L   () nil nil "Dida, Lakota")
            ("did" nil  nil   nil I   L   () nil nil "Didinga")
            ("dif" nil  nil   nil I   E   () nil nil "Dieri")
            ("dig" nil  nil   nil I   L   () nil nil "Digo")
            ("dih" nil  nil   nil I   L   () nil "K'miai" "Kumiai")
            ("dii" nil  nil   nil I   L   () nil nil "Dimbong")
            ("dij" nil  nil   nil I   L   () nil nil "Dai")
            ("dik" nil  nil   nil I   L   () nil nil "Dinka, Southwestern")
            ("dil" nil  nil   nil I   L   () nil nil "Dilling")
            ("dim" nil  nil   nil I   L   () nil nil "Dime")
            ("din" nil  "din" nil M   L   ("dib" "dik"
                                           "dip" "diw" "dks") nil "Thuɔŋjäŋ" "Dinka")
            ("dio" nil  nil   nil I   L   () nil nil "Dibo")
            ("dip" nil  nil   nil I   L   () nil nil "Dinka, Northeastern")
            ("diq" nil  nil   nil I   L   () nil nil "Dimli (individual language)")
            ("dir" nil  nil   nil I   L   () nil nil "Dirim")
            ("dis" nil  nil   nil I   L   () nil nil "Dimasa")
            ("dit" nil  nil   t   I   E   () nil nil "Dirari")
            ("diu" nil  nil   nil I   L   () nil nil "Diriku")
            ("div" "dv" "div" nil I   L   () "Indo-European" "ދިވެހިބަސ" "Divehi")
            ("diw" nil  nil   nil I   L   () nil nil "Dinka, Northwestern")
            ("dix" nil  nil   nil I   L   () nil nil "Dixon Reef")
            ("diy" nil  nil   nil I   L   () nil nil "Diuwe")
            ("diz" nil  nil   nil I   L   () nil nil "Ding")
            ("dja" nil  nil   nil I   E   () nil nil "Djadjawurrung")
            ("djb" nil  nil   nil I   L   () nil nil "Djinba")
            ("djc" nil  nil   nil I   L   () nil nil "Daju, Dar Daju")
            ("djd" nil  nil   nil I   L   () nil nil "Djamindjung")
            ("dje" nil  nil   nil I   L   () nil "zarmaciine" "Zarma")
            ("djf" nil  nil   nil I   E   () nil nil "Djangun")
            ("dji" nil  nil   nil I   L   () nil nil "Djinang")
            ("djj" nil  nil   nil I   L   () nil nil "Djeebbana")
            ("djk" nil  nil   nil I   L   () nil nil "Aukan")
            ("djl" nil  nil   t   I   E   () nil nil "Djiwarli")
            ("djm" nil  nil   nil I   L   () nil nil "Dogon, Jamsay")
            ("djn" nil  nil   nil I   L   () nil nil "Djauan")
            ("djo" nil  nil   nil I   L   () nil nil "Djongkang")
            ("djr" nil  nil   nil I   L   () nil nil "Djambarrpuyngu")
            ("dju" nil  nil   nil I   L   () nil nil "Kapriman")
            ("djw" nil  nil   nil I   E   () nil nil "Djawi")
            ("dka" nil  nil   nil I   L   () nil nil "Dakpakha")
            ("dkg" nil  nil   nil I   L   () "Niger–Congo" nil "Kadung")
            ("dkk" nil  nil   nil I   L   () nil nil "Dakka")
            ("dkl" nil  nil   t   I   L   () nil nil "Dogon, Kolum So")
            ("dkr" nil  nil   nil I   L   () nil nil "Kuijau")
            ("dks" nil  nil   nil I   L   () nil nil "Dinka, Southeastern")
            ("dkx" nil  nil   nil I   L   () nil nil "Mazagway")
            ("dlg" nil  nil   nil I   L   () nil "Дулҕан" "Dolgan")
            ("dlk" nil  nil   nil I   L   () nil nil "Dahalik")
            ("dlm" nil  nil   nil I   E   () nil nil "Dalmatian")
            ("dln" nil  nil   nil I   L   () nil nil "Darlong")
            ("dma" nil  nil   nil I   L   () nil nil "Duma")
            ("dmb" nil  nil   nil I   L   () nil nil "Mombo Dogon")
            ("dmc" nil  nil   nil I   L   () nil nil "Dimir")
            ("dmd" nil  nil   nil I   E   () nil nil "Madhi Madhi")
            ("dme" nil  nil   nil I   L   () nil nil "Dugwor")
            ("dmf" nil  nil   nil I   E   () nil nil "Medefaidrin")
            ("dmg" nil  nil   nil I   L   () nil nil "Kinabatangan, Upper")
            ("dmk" nil  nil   nil I   L   () nil nil "Domaaki")
            ("dml" nil  nil   nil I   L   () nil nil "Dameli")
            ("dmm" nil  nil   nil I   L   () nil nil "Dama")
            ("dmo" nil  nil   nil I   L   () nil nil "Kemezung")
            ("dmr" nil  nil   nil I   L   () nil nil "Damar, East")
            ("dms" nil  nil   nil I   L   () nil nil "Dampelas")
            ("dmu" nil  nil   nil I   L   () nil nil "Dubu")
            ("dmv" nil  nil   nil I   L   () nil nil "Dumpas")
            ("dmw" nil  nil   nil I   L   () nil nil "Mudburra")
            ("dmx" nil  nil   nil I   L   () nil nil "Dema")
            ("dmy" nil  nil   nil I   L   () nil nil "Demta")
            ("dna" nil  nil   nil I   L   () nil nil "Dani, Upper Grand Valley")
            ("dnd" nil  nil   nil I   L   () nil nil "Daonda")
            ("dne" nil  nil   nil I   L   () nil nil "Ndendeule")
            ("dng" nil  nil   nil I   L   () "Sino-Tibetan" "Хуэйзў йүян (Huejzw jyian)" "Dungan")
            ("dni" nil  nil   nil I   L   () nil nil "Dani, Lower Grand Valley")
            ("dnj" nil  nil   nil I   L   () nil nil "Dan")
            ("dnk" nil  nil   nil I   L   () nil nil "Dengka")
            ("dnn" nil  nil   nil I   L   () nil nil "Dzùùngoo")
            ("dno" nil  nil   nil I   L   () "Nilo-Saharan" nil "Ndrulo, Northern Lendu")
            ("dnr" nil  nil   nil I   L   () nil nil "Danaru")
            ("dnt" nil  nil   nil I   L   () nil nil "Dani, Mid Grand Valley")
            ("dnu" nil  nil   nil I   L   () nil nil "Danau")
            ("dnv" nil  nil   nil I   L   () nil nil "Danu")
            ("dnw" nil  nil   nil I   L   () nil nil "Dani, Western")
            ("dny" nil  nil   nil I   L   () nil nil "Dení")
            ("doa" nil  nil   nil I   L   () nil nil "Dom")
            ("dob" nil  nil   nil I   L   () nil nil "Dobu")
            ("doc" nil  nil   nil I   L   () nil nil "Dong, Northern")
            ("doe" nil  nil   nil I   L   () nil nil "Doe")
            ("dof" nil  nil   nil I   L   () nil nil "Domu")
            ("doh" nil  nil   nil I   L   () nil "leec Gaeml" "Dong")
            ("doi" nil  "doi" nil M   L   ("dgo" "xnr") nil "डोगरी" "Dogri (generic)")
            ("dok" nil  nil   nil I   L   () nil nil "Dondo")
            ("dol" nil  nil   nil I   L   () nil nil "Doso")
            ("don" nil  nil   nil I   L   () nil nil "Toura (Papua New Guinea)")
            ("doo" nil  nil   nil I   L   () nil nil "Dongo")
            ("dop" nil  nil   nil I   L   () nil nil "Lukpa")
            ("doq" nil  nil   nil I   L   () nil nil "Dominican Sign Language")
            ("dor" nil  nil   nil I   L   () nil nil "Dori'o")
            ("dos" nil  nil   nil I   L   () nil nil "Dogosé")
            ("dot" nil  nil   nil I   L   () nil nil "Dass")
            ("dov" nil  nil   nil I   L   () nil nil "Dombe")
            ("dow" nil  nil   nil I   L   () nil nil "Doyayo")
            ("dox" nil  nil   nil I   L   () nil nil "Bussa")
            ("doy" nil  nil   nil I   L   () nil nil "Dompo")
            ("doz" nil  nil   nil I   L   () nil nil "Dorze")
            ("dpp" nil  nil   nil I   L   () nil nil "Papar")
            (nil   nil  "dra" nil C   nil () nil nil "Dravidian languages")
            ("drb" nil  nil   nil I   L   () nil nil "Dair")
            ("drc" nil  nil   nil I   L   () nil nil "Minderico")
            ("drd" nil  nil   nil I   L   () nil nil "Darmiya")
            ("dre" nil  nil   nil I   L   () nil nil "Dolpo")
            ("drg" nil  nil   nil I   L   () nil nil "Rungus")
            ("drh" nil  nil   t   I   L   () nil nil "Darkhat")
            ("dri" nil  nil   nil I   L   () nil nil "C'lela")
            ("drl" nil  nil   nil I   L   () nil nil "Darling")
            ("drn" nil  nil   nil I   L   () nil nil "Damar, West")
            ("dro" nil  nil   nil I   L   () nil nil "Daro-Matu")
            ("drq" nil  nil   nil I   E   () nil nil "Dura")
            ("drr" nil  nil   t   I   E   () nil nil "Dororo")
            ("drs" nil  nil   nil I   L   () nil nil "Gedeo")
            ("drt" nil  nil   nil I   L   () nil nil "Drents")
            ("dru" nil  nil   nil I   L   () nil nil "Rukai")
            ("drw" nil  nil   t   I   L   () nil nil "Darwazi")
            ("dry" nil  nil   nil I   L   () nil nil "Darai")
            ("dsb" nil  "dsb" nil I   L   () nil "dolnoserbski" "Sorbian, Lower")
            ("dse" nil  nil   nil I   L   () nil nil "Dutch Sign Language")
            ("dsh" nil  nil   nil I   L   () nil nil "Daasanach")
            ("dsi" nil  nil   nil I   L   () nil nil "Disa")
            ("dsk" nil  nil   nil I   L   () "Afro-Asiatic" nil "Dokshi")
            ("dsl" nil  nil   nil I   L   () nil nil "Danish Sign Language")
            ("dsn" nil  nil   nil I   E   () nil nil "Dusner")
            ("dso" nil  nil   nil I   L   () nil nil "Oriya, Desiya")
            ("dsq" nil  nil   nil I   L   () nil nil "Dawsahak")
            ("dsz" nil  nil   nil I   L   () nil nil "Mardin Sign Language")
            ("dta" nil  nil   nil I   L   () nil nil "Daur")
            ("dtb" nil  nil   nil I   L   () nil nil "Kadazan, Labuk-Kinabatangan")
            ("dtd" nil  nil   nil I   L   () nil nil "Ditidaht")
            ("dth" nil  nil   nil I   E   () nil nil "Adithinngithigh")
            ("dti" nil  nil   nil I   L   () nil nil "Dogon, Ana Tinga")
            ("dtk" nil  nil   nil I   L   () nil nil "Dogon, Tene Kan")
            ("dtm" nil  nil   nil I   L   () nil nil "Dogon, Tomo Kan")
            ("dtn" nil  nil   nil I   L   () nil nil "Daats'iin")
            ("dto" nil  nil   nil I   L   () nil nil "Tommo So Dogon")
            ("dtp" nil  nil   nil I   L   () nil nil "Dusun, Central")
            ("dtr" nil  nil   nil I   L   () nil nil "Lotud")
            ("dts" nil  nil   nil I   L   () nil nil "Dogon, Toro So")
            ("dtt" nil  nil   nil I   L   () nil nil "Dogon, Toro Tegu")
            ("dtu" nil  nil   nil I   L   () nil nil "Dogon, Tebul Ure")
            ("dty" nil  nil   nil I   L   () nil nil "Dotyali")
            ("dua" nil  "dua" nil I   L   () nil "Duala" "Duala")
            ("dub" nil  nil   nil I   L   () nil nil "Dubli")
            ("duc" nil  nil   nil I   L   () nil nil "Duna")
            ("dud" nil  nil   t   I   L   () nil nil "Hun-Saare")
            ("due" nil  nil   nil I   L   () nil nil "Agta, Umiray Dumaget")
            ("duf" nil  nil   nil I   L   () nil nil "Dumbea")
            ("dug" nil  nil   nil I   L   () nil nil "Duruma")
            ("duh" nil  nil   nil I   L   () nil nil "Dungra Bhil")
            ("dui" nil  nil   nil I   L   () nil nil "Dumun")
            ("duj" nil  nil   t   I   L   () nil nil "Dhuwal")
            ("duk" nil  nil   nil I   L   () nil nil "Duduela")
            ("dul" nil  nil   nil I   L   () nil nil "Agta, Alabat Island")
            ("dum" nil  "dum" nil I   H   () nil nil "Dutch, Middle (ca.1050-1350)")
            ("dun" nil  nil   nil I   L   () nil nil "Dusun Deyah")
            ("duo" nil  nil   nil I   L   () nil nil "Agta, Dupaninan")
            ("dup" nil  nil   nil I   L   () nil nil "Duano'")
            ("duq" nil  nil   nil I   L   () nil nil "Dusun Malang")
            ("dur" nil  nil   nil I   L   () nil nil "Dii")
            ("dus" nil  nil   nil I   L   () nil nil "Dumi")
            ("duu" nil  nil   nil I   L   () nil nil "Drung")
            ("duv" nil  nil   nil I   L   () nil nil "Duvle")
            ("duw" nil  nil   nil I   L   () nil nil "Dusun Witu")
            ("dux" nil  nil   nil I   L   () nil nil "Duungooma")
            ("duy" nil  nil   nil I   E   () nil nil "Agta, Dicamay")
            ("duz" nil  nil   nil I   E   () nil nil "Duli")
            ("dva" nil  nil   nil I   L   () nil nil "Duau")
            ("dwa" nil  nil   nil I   L   () nil nil "Diri")
            ("dwk" nil  nil   nil I   L   () nil nil "Dawik Kui")
            ("dwl" nil  nil   t   I   L   () nil nil "Dogon, Walo Kumbe")
            ("dwr" nil  nil   nil I   L   () nil nil "Dawro")
            ("dws" nil  nil   nil I   C   () nil nil "Dutton World Speedwords")
            ("dwu" nil  nil   nil I   L   () "Pama–Nyungan" nil "Dhuwal")
            ("dww" nil  nil   nil I   L   () "Austronesian" nil "Dawawa")
            ("dwy" nil  nil   nil I   L   () "Pama–Nyungan" nil "Dhuwaya")
            ("dwz" nil  nil   nil I   L   () "Indo-European" nil "Dewas Rai")
            ("dya" nil  nil   nil I   L   () nil nil "Dyan")
            ("dyb" nil  nil   nil I   E   () nil nil "Dyaberdyaber")
            ("dyd" nil  nil   nil I   E   () nil nil "Dyugun")
            ("dyg" nil  nil   nil I   E   () nil nil "Agta, Villa Viciosa")
            ("dyi" nil  nil   nil I   L   () nil nil "Senoufo, Djimini")
            ("dyk" nil  nil   t   nil nil () nil nil "Land Dayak")
            ("dym" nil  nil   nil I   L   () nil nil "Dogon, Yanda Dom")
            ("dyn" nil  nil   nil I   L   () nil nil "Dyangadi")
            ("dyo" nil  nil   nil I   L   () nil nil "Jola-Fonyi")
            ("dyr" nil  nil   nil I   L   () "Afro-Asiatic" nil "Dyarim")
            ("dyu" nil  "dyu" nil I   L   () nil "Julakan" "Dyula")
            ("dyy" nil  nil   nil I   L   () nil nil "Dyaabugay")
            ("dza" nil  nil   nil I   L   () nil nil "Duguza")
            ("dzd" nil  nil   nil I   L   () nil nil "Daza")
            ("dze" nil  nil   nil I   E   () nil nil "Djiwarli")
            ("dzg" nil  nil   nil I   L   () nil nil "Dazaga")
            ("dzl" nil  nil   nil I   L   () nil nil "Dzalakha")
            ("dzn" nil  nil   nil I   L   () nil nil "Dzando")
            ("dzo" "dz" "dzo" nil I   L   () "Sino-Tibetan" "རྫོང་ཁ" "Dzongkha")
            ;; E
            ("eaa" nil  nil   nil I   E   () nil nil "Karenggapa")
            ("ebc" nil  nil   nil I   L   () nil nil "Beginci")
            ("ebg" nil  nil   nil I   L   () nil nil "Ebughu")
            ("ebk" nil  nil   nil I   L   () nil nil "Eastern Bontok")
            ("ebo" nil  nil   nil I   L   () nil nil "Teke-Ebo")
            ("ebr" nil  nil   nil I   L   () nil nil "Ebrié")
            ("ebu" nil  nil   nil I   L   () nil nil "Embu")
            ("ecr" nil  nil   nil I   A   () nil nil "Eteocretan")
            ("ecs" nil  nil   nil I   L   () nil nil "Ecuadorian Sign Language")
            ("ecy" nil  nil   nil I   A   () nil nil "Eteocypriot")
            ("eee" nil  nil   nil I   L   () nil "E" "E")
            ("efa" nil  nil   nil I   L   () nil nil "Efai")
            ("efe" nil  nil   nil I   L   () nil nil "Efe")
            ("efi" nil  "efi" nil I   L   () nil nil "Efik")
            ("ega" nil  nil   nil I   L   () nil nil "Ega")
            ("egl" nil  nil   nil I   L   () nil "emigliàṅ" "Emilian")
            ("egm" nil  nil   nil I   L   () "Niger–Congo" nil "Benamanga")
            ("ego" nil  nil   nil I   L   () nil nil "Eggon")
            ("egy" nil  "egy" nil I   A   () nil nil "Egyptian")
            ("ehs" nil  nil   nil I   L   () nil nil "Miyakubo Sign Language")
            ("ehu" nil  nil   nil I   L   () nil nil "Ehueun")
            ("eip" nil  nil   nil I   L   () nil nil "Eipomek")
            ("eit" nil  nil   nil I   L   () nil nil "Eitiep")
            ("eiv" nil  nil   nil I   L   () nil nil "Askopan")
            ("eja" nil  nil   nil I   L   () nil nil "Ejamat")
            ("eka" nil  "eka" nil I   L   () nil nil "Ekajuk")
            ("ekc" nil  nil   t   I   E   () "spurious language" nil "Eastern Karnic")
            ("eke" nil  nil   nil I   L   () nil nil "Ekit")
            ("ekg" nil  nil   nil I   L   () nil nil "Ekari")
            ("eki" nil  nil   nil I   L   () nil nil "Eki")
            ("ekk" nil  nil   nil I   L   () nil nil "Standard Estonian")
            ("ekl" nil  nil   nil I   L   () nil nil "Kol")
            ("ekm" nil  nil   nil I   L   () nil nil "Elip")
            ("eko" nil  nil   nil I   L   () nil nil "Koti")
            ("ekp" nil  nil   nil I   L   () nil nil "Ekpeye")
            ("ekr" nil  nil   nil I   L   () nil nil "Yace")
            ("eky" nil  nil   nil I   L   () nil nil "Kayah, Eastern")
            ("ele" nil  nil   nil I   L   () nil nil "Elepi")
            ("elh" nil  nil   nil I   L   () nil nil "El Hugeirat")
            ("eli" nil  nil   nil I   E   () nil nil "Nding")
            ("elk" nil  nil   nil I   L   () nil nil "Elkei")
            ("ell" "el" "gre" nil I   L   () "Indo-European" "Ελληνικά" "(New) Greek")
            ("elm" nil  nil   nil I   L   () nil nil "Eleme")
            ("elo" nil  nil   nil I   L   () nil nil "El Molo")
            ("elp" nil  nil   t   I   L   () nil nil "Elpaputih")
            ("elu" nil  nil   nil I   L   () nil nil "Elu")
            ("elx" nil  "elx" nil I   A   () nil nil "Elamite")
            ("ema" nil  nil   nil I   L   () nil nil "Emai-Iuleha-Ora")
            ("emb" nil  nil   nil I   L   () nil nil "Embaloh")
            ("eme" nil  nil   nil I   L   () nil nil "Emerillon")
            ("emg" nil  nil   nil I   L   () nil nil "Meohang, Eastern")
            ("emi" nil  nil   nil I   L   () nil nil "Mussau-Emira")
            ("emk" nil  nil   nil I   L   () nil nil "Maninkakan, Eastern")
            ("eml" nil  nil   t   I   L   () nil "emigliàn e rumagnòl" "Emiliano-Romagnolo")
            ("emm" nil  nil   nil I   E   () nil nil "Mamulique")
            ("emn" nil  nil   nil I   L   () nil nil "Eman")
            ("emo" nil  nil   t   nil nil () "spurious language" nil "Emok")
            ("emp" nil  nil   nil I   L   () nil nil "Emberá, Northern")
            ("emq" nil  nil   nil I   L   () "Sino-Tibetan" nil "Minyag, Eastern")
            ("ems" nil  nil   nil I   L   () nil nil "Yupik, Pacific Gulf")
            ("emu" nil  nil   nil I   L   () nil nil "Muria, Eastern")
            ("emw" nil  nil   nil I   L   () nil nil "Emplawas")
            ("emx" nil  nil   nil I   L   () nil nil "Erromintxela")
            ("emy" nil  nil   nil I   A   () nil nil "Mayan, Epigraphic")
            ("emz" nil  nil   nil I   L   () "Niger–Congo" nil "Mbessa")
            ("ena" nil  nil   nil I   L   () nil nil "Apali")
            ("enb" nil  nil   nil I   L   () nil nil "Endo")
            ("enc" nil  nil   nil I   L   () nil nil "En")
            ("end" nil  nil   nil I   L   () nil nil "Ende")
            ("enf" nil  nil   nil I   L   () nil nil "Enets, Forest")
            ("eng" "en" "eng" nil I   L   () "Indo-European" "English" "English")
            ("enh" nil  nil   nil I   L   () nil nil "Enets, Tundra")
            ("eni" nil  nil   t   nil nil () nil nil "Enim")
            ("enl" nil  nil   nil I   L   () "Mascoian" nil "Enlhet")
            ("enm" nil  "enm" nil I   H   () nil "English" "English, Middle (1100-1500)")
            ("enn" nil  nil   nil I   L   () nil nil "Engenni")
            ("eno" nil  nil   nil I   L   () nil nil "Enggano")
            ("enq" nil  nil   nil I   L   () nil nil "Enga")
            ("enr" nil  nil   nil I   L   () nil nil "Emumu")
            ("enu" nil  nil   nil I   L   () nil nil "Enu")
            ("env" nil  nil   nil I   L   () nil nil "Enwan (Edu State)")
            ("enw" nil  nil   nil I   L   () nil nil "Enwan (Akwa Ibom State)")
            ("enx" nil  nil   nil I   L   () "Mascoian" nil "Enxet")
            ("eot" nil  nil   nil I   L   () nil nil "Beti (Côte d'Ivoire)")
            ("epi" nil  nil   nil I   L   () nil nil "Epie")
            ("epo" "eo" "epo" nil I   C   () "constructed" "Esperanto" "Esperanto")
            ("era" nil  nil   nil I   L   () nil nil "Eravallan")
            ("erg" nil  nil   nil I   L   () nil nil "Sie")
            ("erh" nil  nil   nil I   L   () nil nil "Eruwa")
            ("eri" nil  nil   nil I   L   () nil nil "Ogea")
            ("erk" nil  nil   nil I   L   () nil nil "Efate, South")
            ("ero" nil  nil   nil I   L   () nil nil "Horpa")
            ("err" nil  nil   nil I   E   () nil nil "Erre")
            ("ers" nil  nil   nil I   L   () nil nil "Ersu")
            ("ert" nil  nil   nil I   L   () nil nil "Eritai")
            ("erw" nil  nil   nil I   L   () nil nil "Erokwanas")
            ("ese" nil  nil   nil I   L   () nil nil "Ese Ejja")
            ("esg" nil  nil   nil I   L   () "Dravidian" nil "Aheri Gondi")
            ("esh" nil  nil   nil I   L   () nil nil "Eshtehardi")
            ("esi" nil  nil   nil I   L   () nil nil "Inupiatun, North Alaskan")
            ("esk" nil  nil   nil I   L   () nil nil "Inupiatun, Northwest Alaska")
            ("esl" nil  nil   nil I   L   () nil nil "Egypt Sign Language")
            ("esm" nil  nil   nil I   E   () nil nil "Esuma")
            ("esn" nil  nil   nil I   L   () nil nil "Salvadoran Sign Language")
            ("eso" nil  nil   nil I   L   () nil nil "Estonian Sign Language")
            ("esq" nil  nil   nil I   E   () nil nil "Esselen")
            ("ess" nil  nil   nil I   L   () nil nil "Yupik, Central Siberian")
            ("est" "et" "est" nil M   L   ("ekk" "vro") "Uralic" "eesti" "Estonian")
            ("esu" nil  nil   nil I   L   () nil nil "Yupik, Central")
            ("esy" nil  nil   nil I   L   () nil nil "Eskayan")
            ("etb" nil  nil   nil I   L   () nil nil "Etebi")
            ("etc" nil  nil   nil I   E   () nil nil "Etchemin")
            ("eth" nil  nil   nil I   L   () nil nil "Ethiopian Sign Language")
            ("etn" nil  nil   nil I   L   () nil nil "Eton (Vanuatu)")
            ("eto" nil  nil   nil I   L   () nil nil "Eton (Cameroon)")
            ("etr" nil  nil   nil I   L   () nil nil "Edolo")
            ("ets" nil  nil   nil I   L   () nil nil "Yekhee")
            ("ett" nil  nil   nil I   A   () nil nil "Etruscan")
            ("etu" nil  nil   nil I   L   () nil nil "Ejagham")
            ("etx" nil  nil   nil I   L   () nil nil "Eten")
            ("etz" nil  nil   nil I   L   () nil nil "Semimi")
            ("eud" nil  nil   nil I   L   () "Uto-Aztecan" nil "Eudeve")
            ("eur" nil  nil   t   I   C   () "constructed" "Europanto" "Europanto")
            ("eus" "eu" "baq" nil I   L   () "isolate" "euskara" "Basque")
            ("eve" nil  nil   nil I   L   () nil "эвэды" "Even")
            ("evh" nil  nil   nil I   L   () nil nil "Uvbie")
            ("evn" nil  nil   nil I   L   () nil "орочон" "Evenki")
            ("ewe" "ee" "ewe" nil I   L   () "Niger–Congo" "Eʋegbe" "Ewe")
            ("ewo" nil  "ewo" nil I   L   () nil nil "Ewondo")
            ("ext" nil  nil   nil I   L   () nil "estremeñu" "Extremaduran")
            ("eya" nil  nil   nil I   E   () nil nil "Eyak")
            ("eyo" nil  nil   nil I   L   () nil nil "Keiyo")
            ("eza" nil  nil   nil I   L   () nil nil "Ezaa")
            ("eze" nil  nil   nil I   L   () nil nil "Uzekwe")
            ;; F
            ("faa" nil  nil   nil I   L   () nil nil "Fasu")
            ("fab" nil  nil   nil I   L   () nil nil "Fa d'Ambu")
            ("fad" nil  nil   nil I   L   () nil nil "Wagi")
            ("faf" nil  nil   nil I   L   () nil nil "Fagani")
            ("fag" nil  nil   nil I   L   () nil nil "Finongan")
            ("fah" nil  nil   nil I   L   () nil nil "Fali Of Baissa")
            ("fai" nil  nil   nil I   L   () nil nil "Faiwol")
            ("faj" nil  nil   nil I   L   () nil nil "Faita")
            ("fak" nil  nil   nil I   L   () nil nil "Fang (Cameroon)")
            ("fal" nil  nil   nil I   L   () nil nil "Fali, South")
            ("fam" nil  nil   nil I   L   () nil nil "Fam")
            ("fan" nil  "fan" nil I   L   () nil "Fang" "Fang (Equatorial Guinea)")
            ("fao" "fo" "fao" nil I   L   () "Indo-European" "føroyskt" "Faroese")
            ("fap" nil  nil   nil I   L   () nil nil "Palor")
            ("far" nil  nil   nil I   L   () nil nil "Fataleka")
            ("fas" "fa" "per" nil M   L   ("pes" "prs") "Indo-European" "فارسی" "Persian")
            ("fat" nil  "fat" nil I   L   () nil nil "Fanti")
            ("fau" nil  nil   nil I   L   () nil nil "Fayu")
            ("fax" nil  nil   nil I   L   () nil "Fala" "Fala")
            ("fay" nil  nil   nil I   L   () nil nil "Fars, Southwestern")
            ("faz" nil  nil   nil I   L   () nil nil "Fars, Northwestern")
            ("fbl" nil  nil   nil I   L   () nil nil "West Albay Bikol")
            ("fcs" nil  nil   nil I   L   () nil nil "Quebec Sign Language")
            ("fer" nil  nil   nil I   L   () nil nil "Feroge")
            ("ffi" nil  nil   nil I   L   () nil nil "Foia Foia")
            ("ffm" nil  nil   nil I   L   () "Niger–Congo" nil "Fulfulde, Maasina")
            ("fgr" nil  nil   nil I   L   () nil nil "Fongoro")
            ("fia" nil  nil   nil I   L   () nil nil "Nobiin")
            ("fie" nil  nil   nil I   L   () nil nil "Fyer")
            ("fif" nil  nil   nil I   L   () "Afro-Asiatic" nil "Faifi")
            ("fij" "fj" "fij" nil I   L   () "Austronesian" "na vosa Vakaviti" "Fijian")
            ("fil" nil  "fil" nil I   L   () "Austronesian" "Filipino" "Filipino")
            ("fin" "fi" "fin" nil I   L   () "Uralic" "suomi" "Finnish")
            ("fip" nil  nil   nil I   L   () nil nil "Fipa")
            ("fir" nil  nil   nil I   L   () nil nil "Firan")
            ("fit" nil  nil   nil I   L   () "Uralic" "meänkieli" "Finnish (Tornedalen)")
            (nil   nil  "fiu" nil C   nil () nil nil "Finno-Ugrian languages")
            ("fiw" nil  nil   nil I   L   () nil nil "Fiwaga")
            ("fiz" nil  nil   t   nil nil () nil nil "Izere")
            ("fkk" nil  nil   nil I   L   () nil nil "Kirya-Konzəl")
            ("fkv" nil  nil   nil I   L   () "Uralic" nil "Finnish (Kven)")
            ("fla" nil  nil   nil I   L   () nil nil "Kalispel-Pend d'Oreille")
            ("flh" nil  nil   nil I   L   () nil nil "Foau")
            ("fli" nil  nil   nil I   L   () nil nil "Fali")
            ("fll" nil  nil   nil I   L   () nil nil "Fali, North")
            ("flm" nil  nil   t   nil nil () nil nil "Falam Chin")
            ("fln" nil  nil   nil I   E   () nil nil "Flinders Island")
            ("flr" nil  nil   nil I   L   () nil nil "Fuliiru")
            ("fly" nil  nil   nil I   L   () nil nil "Tsotsitaal")
            ("fmp" nil  nil   nil I   L   () nil nil "Fe'fe'")
            ("fmu" nil  nil   nil I   L   () nil nil "Muria, Far Western")
            ("fnb" nil  nil   nil I   L   () "Austronesian" nil "Fanbak")
            ("fng" nil  nil   nil I   L   () nil nil "Fanagalo")
            ("fni" nil  nil   nil I   L   () nil nil "Fania")
            ("fod" nil  nil   nil I   L   () nil nil "Foodo")
            ("foi" nil  nil   nil I   L   () nil nil "Foi")
            ("fom" nil  nil   nil I   L   () nil nil "Foma")
            ("fon" nil  "fon" nil I   L   () nil "Fɔngbè" "Fon")
            ("for" nil  nil   nil I   L   () nil nil "Fore")
            ("fos" nil  nil   nil I   E   () "Austronesian" nil "Siraya")
            ("fpe" nil  nil   nil I   L   () nil nil "Fernando Po Creole English")
            ("fqs" nil  nil   nil I   L   () nil nil "Fas")
            ("fra" "fr" "fre" nil I   L   () "Indo-European" "français" "French")
            ("frc" nil  nil   nil I   L   () "Indo-European" nil "French, Cajun")
            ("frd" nil  nil   nil I   L   () nil nil "Fordata")
            ("fri" nil  nil   t   nil nil () nil nil "Western Frisian")
            ("frk" nil  nil   nil I   H   () "Indo-European" "Fränkisch" "Frankish")
            ("frm" nil  "frm" nil I   H   () "Indo-European" nil "French, Middle (ca.1400-1600)")
            ("fro" nil  "fro" nil I   H   () "Indo-European" nil "French, Old (842-Ca.1400)")
            ("frp" nil  nil   nil I   L   () "Indo-European" "francoprovensal" "Franco-Provençal")
            ("frq" nil  nil   nil I   L   () nil nil "Forak")
            ("frr" nil  "frr" nil I   L   () "Indo-European" "Nordfriisk" "Frisian, Northern")
            ("frs" nil  "frs" nil I   L   () "Indo-European" nil "Frisian, Eastern")
            ("frt" nil  nil   nil I   L   () nil nil "Fortsenal")
            ("fry" "fy" "fry" nil I   L   () "Indo-European" "Frysk" "Frisian, Western")
            ("fse" nil  nil   nil I   L   () nil nil "Finnish Sign Language")
            ("fsl" nil  nil   nil I   L   () nil nil "French Sign Language")
            ("fss" nil  nil   nil I   L   () nil nil "Finnish-Swedish Sign Language")
            ("fub" nil  nil   nil I   L   () "Niger–Congo" nil "Fulfulde (Adamawa)")
            ("fuc" nil  nil   nil I   L   () nil "Pulaar" "Pulaar")
            ("fud" nil  nil   nil I   L   () nil nil "Futuna, East")
            ("fue" nil  nil   nil I   L   () "Niger–Congo" nil "Fulfulde, Borgu")
            ("fuf" nil  nil   nil I   L   () nil nil "Pular")
            ("fuh" nil  nil   nil I   L   () "Niger–Congo" nil "Fulfulde, Western Niger")
            ("fui" nil  nil   nil I   L   () "Niger–Congo" nil "Fulfulde, Bagirmi")
            ("fuj" nil  nil   nil I   L   () nil nil "Ko")
            ("ful" "ff" "ful" nil M   L   ("bdt"
                                           "gbp" "gbq" "gmm"
                                           "gso" "gya" "mdo") "Niger–Congo" "Fulfulde" "Fulah")
            ("fum" nil  nil   nil I   L   () nil nil "Fum")
            ("fun" nil  nil   nil I   L   () nil nil "Fulniô")
            ("fuq" nil  nil   nil I   L   () "Niger–Congo" nil "Fulfulde, Central-Eastern Niger")
            ("fur" nil  "fur" nil I   L   () "Indo-European" "furlan" "Friulian")
            ("fut" nil  nil   nil I   L   () nil nil "Futuna-Aniwa")
            ("fuu" nil  nil   nil I   L   () nil nil "Furu")
            ("fuv" nil  nil   nil I   L   () "Niger–Congo" nil "Fulfulde (Nigerian)")
            ("fuy" nil  nil   nil I   L   () nil nil "Fuyug")
            ("fvr" nil  nil   nil I   L   () nil "fòòr" "Fur")
            ("fwa" nil  nil   nil I   L   () nil nil "Fwâi")
            ("fwe" nil  nil   nil I   L   () nil nil "Fwe")
            ;; G
            ("gaa" nil  "gaa" nil I   L   () nil "Gã" "Ga")
            ("gab" nil  nil   nil I   L   () nil nil "Gabri")
            ("gac" nil  nil   nil I   L   () nil nil "Mixed Great Andamanese")
            ("gad" nil  nil   nil I   L   () nil nil "Gaddang")
            ("gae" nil  nil   nil I   L   () nil nil "Guarequena")
            ("gaf" nil  nil   nil I   L   () nil nil "Gende")
            ("gag" nil  nil   nil I   L   () "Turkic" "Gagauz dili" "Gagauz")
            ("gah" nil  nil   nil I   L   () nil nil "Alekano")
            ("gai" nil  nil   nil I   L   () nil nil "Borei")
            ("gaj" nil  nil   nil I   L   () nil nil "Gadsup")
            ("gak" nil  nil   nil I   L   () nil nil "Gamkonora")
            ("gal" nil  nil   nil I   L   () nil nil "Galoli")
            ("gam" nil  nil   nil I   L   () nil nil "Kandawo")
            ("gan" nil  nil   nil I   L   () "Chinese" "赣语" "Gan")
            ("gao" nil  nil   nil I   L   () nil nil "Gants")
            ("gap" nil  nil   nil I   L   () nil nil "Gal")
            ("gaq" nil  nil   nil I   L   () nil nil "Gata'")
            ("gar" nil  nil   nil I   L   () nil nil "Galeya")
            ("gas" nil  nil   nil I   L   () nil nil "Garasia, Adiwasi")
            ("gat" nil  nil   nil I   L   () nil nil "Kenati")
            ("gau" nil  nil   nil I   L   () nil nil "Gadaba, Mudhili")
            ("gav" nil  nil   t   I   L   () nil nil "Gabutamon")
            ("gaw" nil  nil   nil I   L   () nil nil "Nobonob")
            ("gax" nil  nil   nil I   L   () nil nil "Oromo, Borana-Arsi-Guji")
            ("gay" nil  "gay" nil I   L   () nil nil "Gayo")
            ("gaz" nil  nil   nil I   L   () nil nil "Oromo, West Central")
            ("gba" nil  "gba" nil M   L   () nil nil "Gbaya (Central African Republic)")
            ("gbb" nil  nil   nil I   L   () nil nil "Kaytetye")
            ("gbc" nil  nil   t   I   L   () nil nil "Garawa")
            ("gbd" nil  nil   nil I   L   () nil nil "Karadjeri")
            ("gbe" nil  nil   nil I   L   () nil nil "Niksek")
            ("gbf" nil  nil   nil I   L   () nil nil "Gaikundi")
            ("gbg" nil  nil   nil I   L   () nil nil "Gbanziri")
            ("gbh" nil  nil   nil I   L   () nil nil "Gbe, Defi")
            ("gbi" nil  nil   nil I   L   () nil nil "Galela")
            ("gbj" nil  nil   nil I   L   () nil nil "Gadaba, Bodo")
            ("gbk" nil  nil   nil I   L   () nil nil "Gaddi")
            ("gbl" nil  nil   nil I   L   () nil nil "Gamit")
            ("gbm" nil  nil   nil I   L   () nil nil "Garhwali")
            ("gbn" nil  nil   nil I   L   () nil nil "Mo'da")
            ("gbo" nil  nil   nil I   L   () nil nil "Grebo, Northern")
            ("gbp" nil  nil   nil I   L   () nil nil "Gbaya-Bossangoa")
            ("gbq" nil  nil   nil I   L   () nil nil "Gbaya-Bozoum")
            ("gbr" nil  nil   nil I   L   () nil nil "Gbagyi")
            ("gbs" nil  nil   nil I   L   () nil nil "Gbe, Gbesi")
            ("gbu" nil  nil   nil I   L   () nil nil "Gagadu")
            ("gbv" nil  nil   nil I   L   () nil nil "Gbanu")
            ("gbw" nil  nil   nil I   L   () nil nil "Gabi-Gabi")
            ("gbx" nil  nil   nil I   L   () nil nil "Gbe, Eastern Xwla")
            ("gby" nil  nil   nil I   L   () nil nil "Gbari")
            ("gbz" nil  nil   nil I   L   () nil "دَرِي" "Dari, Zoroastrian")
            ("gcc" nil  nil   nil I   L   () nil nil "Mali")
            ("gcd" nil  nil   nil I   E   () nil nil "Ganggalida")
            ("gce" nil  nil   nil I   E   () nil nil "Galice")
            ("gcf" nil  nil   nil I   L   () nil nil "Guadeloupean Creole French")
            ("gcl" nil  nil   nil I   L   () nil nil "Grenadian Creole English")
            ("gcn" nil  nil   nil I   L   () nil nil "Gaina")
            ("gcr" nil  nil   nil I   L   () nil nil "Guianese Creole French")
            ("gct" nil  nil   nil I   L   () nil nil "German, Colonia Tovar")
            ("gda" nil  nil   nil I   L   () nil nil "Lohar, Gade")
            ("gdb" nil  nil   nil I   L   () nil nil "Gadaba, Pottangi Ollar")
            ("gdc" nil  nil   nil I   E   () nil nil "Gugu Badhun")
            ("gdd" nil  nil   nil I   L   () nil nil "Gedaged")
            ("gde" nil  nil   nil I   L   () nil nil "Gude")
            ("gdf" nil  nil   nil I   L   () nil nil "Guduf-Gava")
            ("gdg" nil  nil   nil I   L   () nil nil "Ga'dang")
            ("gdh" nil  nil   nil I   L   () nil nil "Gadjerawang")
            ("gdi" nil  nil   nil I   L   () nil nil "Gundi")
            ("gdj" nil  nil   nil I   L   () nil nil "Gurdjar")
            ("gdk" nil  nil   nil I   L   () nil nil "Gadang")
            ("gdl" nil  nil   nil I   L   () nil nil "Dirasha")
            ("gdm" nil  nil   nil I   L   () nil "yəw láà:l" "Laal")
            ("gdn" nil  nil   nil I   L   () nil nil "Umanakaina")
            ("gdo" nil  nil   nil I   L   () nil nil "Ghodoberi")
            ("gdq" nil  nil   nil I   L   () nil nil "Mehri")
            ("gdr" nil  nil   nil I   L   () nil nil "Wipi")
            ("gds" nil  nil   nil I   L   () nil nil "Ghandruk Sign Language")
            ("gdt" nil  nil   nil I   E   () nil nil "Kungardutyi")
            ("gdu" nil  nil   nil I   L   () nil nil "Gudu")
            ("gdx" nil  nil   nil I   L   () nil nil "Godwari")
            ("gea" nil  nil   nil I   L   () nil nil "Geruma")
            ("geb" nil  nil   nil I   L   () nil nil "Kire")
            ("gec" nil  nil   nil I   L   () nil nil "Grebo, Gboloo")
            ("ged" nil  nil   nil I   L   () nil nil "Gade")
            ("gef" nil  nil   nil I   L   () nil nil "Gerai")
            ("geg" nil  nil   nil I   L   () nil nil "Gengle")
            ("geh" nil  nil   nil I   L   () nil "Hutterisch" "German, Hutterite")
            ("gei" nil  nil   nil I   L   () nil nil "Gebe")
            ("gej" nil  nil   nil I   L   () nil nil "Gen")
            ("gek" nil  nil   nil I   L   () nil nil "Yiwom")
            ("gel" nil  nil   nil I   L   () nil nil "Kag-Fer-Jiir-Koor-Ror-Us-Zuksun")
            (nil   nil  "gem" nil C   nil () nil nil "Germanic languages")
            ("gen" nil  nil   t   nil nil () nil nil "Geman Deng")
            ("geq" nil  nil   nil I   L   () nil nil "Geme")
            ("ges" nil  nil   nil I   L   () nil nil "Geser-Gorom")
            ("gev" nil  nil   nil I   L   () "Niger–Congo" nil "Eviya")
            ("gew" nil  nil   nil I   L   () nil nil "Gera")
            ("gex" nil  nil   nil I   L   () nil nil "Garre")
            ("gey" nil  nil   nil I   L   () nil nil "Enya")
            ("gez" nil  "gez" nil I   A   () nil "ግዕዝ" "Geez")
            ("gfk" nil  nil   nil I   L   () nil nil "Patpatar")
            ("gft" nil  nil   nil I   E   () nil nil "Gafat")
            ("gfx" nil  nil   t   I   L   () nil nil "Mangetti Dune !Xung")
            ("gga" nil  nil   nil I   L   () nil nil "Gao")
            ("ggb" nil  nil   nil I   L   () nil nil "Gbii")
            ("ggd" nil  nil   nil I   E   () nil nil "Gugadj")
            ("gge" nil  nil   nil I   L   () nil nil "Guragone")
            ("ggg" nil  nil   nil I   L   () nil nil "Gurgula")
            ("ggh" nil  nil   t   nil nil () nil nil "Garreh-Ajuran")
            ("ggk" nil  nil   nil I   E   () nil nil "Kungarakany")
            ("ggl" nil  nil   nil I   L   () nil nil "Ganglau")
            ("ggm" nil  nil   t   I   E   () nil nil "Gugu Mini")
            ("ggn" nil  nil   t   I   L   () nil nil "Gurung, Eastern")
            ("ggo" nil  nil   t   I   L   () nil nil "Gondi, Southern")
            ("ggr" nil  nil   t   I   E   () nil nil "Aghu Tharnggalu")
            ("ggt" nil  nil   nil I   L   () nil nil "Gitua")
            ("ggu" nil  nil   nil I   L   () nil nil "Gagu")
            ("ggw" nil  nil   nil I   L   () nil nil "Gogodala")
            ("gha" nil  nil   nil I   L   () nil nil "Ghadamès")
            ("ghc" nil  nil   nil I   H   () nil nil "Gaelic, Hiberno-Scottish")
            ("ghe" nil  nil   nil I   L   () nil nil "Ghale, Southern")
            ("ghh" nil  nil   nil I   L   () nil nil "Ghale, Northern")
            ("ghk" nil  nil   nil I   L   () nil nil "Karen, Geko")
            ("ghl" nil  nil   nil I   L   () nil nil "Ghulfan")
            ("ghn" nil  nil   nil I   L   () nil nil "Ghanongga")
            ("gho" nil  nil   nil I   E   () nil nil "Ghomara")
            ("ghr" nil  nil   nil I   L   () nil nil "Ghera")
            ("ghs" nil  nil   nil I   L   () nil nil "Guhu-Samane")
            ("ght" nil  nil   nil I   L   () nil nil "Ghale, Kutang")
            ("gia" nil  nil   nil I   L   () nil nil "Kitja")
            ("gib" nil  nil   nil I   L   () nil nil "Gibanawa")
            ("gic" nil  nil   nil I   L   () nil nil "Gail")
            ("gid" nil  nil   nil I   L   () nil nil "Gidar")
            ("gie" nil  nil   nil I   L   () "Niger–Congo" nil "Gaɓogbo, Guébie")
            ("gig" nil  nil   nil I   L   () nil nil "Goaria")
            ("gih" nil  nil   nil I   L   () nil nil "Githabul")
            ("gii" nil  nil   nil I   L   () "Afro-Asiatic" nil "Girirra")
            ("gil" nil  "gil" nil I   L   () nil "taetae ni Kiribati" "Gilbertese")
            ("gim" nil  nil   nil I   L   () nil nil "Gimi (Eastern Highlands)")
            ("gin" nil  nil   nil I   L   () nil nil "Hinukh")
            ("gio" nil  nil   t   I   L   () nil nil "Gelao")
            ("gip" nil  nil   nil I   L   () nil nil "Gimi (West New Britain)")
            ("giq" nil  nil   nil I   L   () nil nil "Gelao, Green")
            ("gir" nil  nil   nil I   L   () nil nil "Gelao, Red")
            ("gis" nil  nil   nil I   L   () nil nil "Giziga, North")
            ("git" nil  nil   nil I   L   () nil "Gitx̱sanimx̱" "Gitxsan")
            ("giu" nil  nil   nil I   L   () nil nil "Mulao")
            ("giw" nil  nil   nil I   L   () nil nil "Gelao, White")
            ("gix" nil  nil   nil I   L   () nil nil "Gilima")
            ("giy" nil  nil   nil I   L   () nil nil "Giyug")
            ("giz" nil  nil   nil I   L   () nil nil "Giziga, South")
            ("gji" nil  nil   t   I   L   () nil nil "Geji")
            ("gjk" nil  nil   nil I   L   () nil nil "Koli, Kachi")
            ("gjm" nil  nil   nil I   E   () nil nil "Gunditjmara")
            ("gjn" nil  nil   nil I   L   () nil nil "Gonja")
            ("gjr" nil  nil   nil I   L   () nil nil "Gurindji Kriol")
            ("gju" nil  nil   nil I   L   () nil nil "Gujari")
            ("gka" nil  nil   nil I   L   () nil nil "Guya")
            ("gkd" nil  nil   nil I   L   () "Trans–New Guinea" nil "Magɨ (Madang Province)")
            ("gke" nil  nil   nil I   L   () nil nil "Ndai")
            ("gkn" nil  nil   nil I   L   () nil nil "Gokana")
            ("gko" nil  nil   nil I   E   () nil nil "Kok-Nar")
            ("gkp" nil  nil   nil I   L   () nil nil "Kpelle, Guinea")
            ("gku" nil  nil   nil I   E   () "Tuu" nil "ǂUngkue")
            ("gla" "gd" "gla" nil I   L   () "Indo-European" "Gàidhlig" "Gaelic (Scots)")
            ("glb" nil  nil   nil I   L   () "Afro-Asiatic" nil "Belning")
            ("glc" nil  nil   nil I   L   () nil nil "Bon Gula")
            ("gld" nil  nil   nil I   L   () nil "нанай" "Nanai")
            ("gle" "ga" "gle" nil I   L   () "Indo-European" "Gaeilge" "Irish")
            ("glg" "gl" "glg" nil I   L   () "Indo-European" "galego" "Galician")
            ("glh" nil  nil   nil I   L   () nil nil "Pashayi, Northwest")
            ("gli" nil  nil   t   I   E   () nil nil "Guliguli")
            ("glj" nil  nil   nil I   L   () nil nil "Gula Iro")
            ("glk" nil  nil   nil I   L   () nil nil "Gilaki")
            ("gll" nil  nil   nil I   E   () nil nil "Garlali")
            ("glo" nil  nil   nil I   L   () nil nil "Galambu")
            ("glr" nil  nil   nil I   L   () nil nil "Glaro-Twabo")
            ("glu" nil  nil   nil I   L   () nil nil "Gula (Chad)")
            ("glv" "gv" "glv" nil I   L   () "Indo-European" "Gaelg" "Manx")
            ("glw" nil  nil   nil I   L   () nil nil "Glavda")
            ("gly" nil  nil   nil I   E   () nil nil "Gule")
            ("gma" nil  nil   nil I   E   () nil nil "Gambera")
            ("gmb" nil  nil   nil I   L   () nil nil "Gula'alaa")
            ("gmd" nil  nil   nil I   L   () nil nil "Mághdì")
            ("gmg" nil  nil   nil I   L   () "Trans–New Guinea" nil "Magɨyi")
            ("gmh" nil  "gmh" nil I   H   () nil nil "German, Middle High (ca.1050-1500)")
            ("gml" nil  nil   nil I   H   () nil nil "Middle Low German")
            ("gmm" nil  nil   nil I   L   () nil nil "Gbaya-Mbodomo")
            ("gmn" nil  nil   nil I   L   () nil nil "Gimnime")
            ("gmo" nil  nil   t   I   L   () nil nil "Gamo-Gofa-Dawro")
            ("gmr" nil  nil   nil I   L   () nil nil "Mirning, Mirniny")
            ("gmu" nil  nil   nil I   L   () nil nil "Gumalu")
            ("gmv" nil  nil   nil I   L   () nil nil "Gamo")
            ("gmx" nil  nil   nil I   L   () nil nil "Magoma")
            ("gmy" nil  nil   nil I   A   () nil nil "Greek, Mycenaean")
            ("gmz" nil  nil   nil I   L   () nil nil "Mgbolizhia")
            ("gna" nil  nil   nil I   L   () nil nil "Kaansa")
            ("gnb" nil  nil   nil I   L   () nil nil "Gangte")
            ("gnc" nil  nil   nil I   E   () nil nil "Guanche")
            ("gnd" nil  nil   nil I   L   () nil nil "Zulgo-Gemzek")
            ("gne" nil  nil   nil I   L   () nil nil "Ganang")
            ("gng" nil  nil   nil I   L   () nil nil "Ngangam")
            ("gnh" nil  nil   nil I   L   () nil nil "Lere")
            ("gni" nil  nil   nil I   L   () nil nil "Gooniyandi")
            ("gnj" nil  nil   nil I   L   () "Austronesian" nil "Ngen")
            ("gnk" nil  nil   nil I   L   () nil nil "//Gana")
            ("gnl" nil  nil   nil I   E   () nil nil "Gangulu")
            ("gnm" nil  nil   nil I   L   () nil nil "Ginuman")
            ("gnn" nil  nil   nil I   L   () nil nil "Gumatj")
            ("gno" nil  nil   nil I   L   () nil nil "Gondi, Northern")
            ("gnq" nil  nil   nil I   L   () nil nil "Gana")
            ("gnr" nil  nil   nil I   E   () nil nil "Gureng Gureng")
            ("gnt" nil  nil   nil I   L   () nil nil "Guntai")
            ("gnu" nil  nil   nil I   L   () nil nil "Gnau")
            ("gnw" nil  nil   nil I   L   () nil nil "Guaraní, Western Bolivian")
            ("gnz" nil  nil   nil I   L   () nil nil "Ganzi")
            ("goa" nil  nil   nil I   L   () nil nil "Guro")
            ("gob" nil  nil   nil I   L   () nil nil "Playero")
            ("goc" nil  nil   nil I   L   () nil nil "Gorakor")
            ("god" nil  nil   nil I   L   () nil nil "Godié")
            ("goe" nil  nil   nil I   L   () nil nil "Gongduk")
            ("gof" nil  nil   nil I   L   () nil nil "Gofa")
            ("gog" nil  nil   nil I   L   () nil nil "Gogo")
            ("goh" nil  "goh" nil I   H   () nil nil "German, Old High (ca.750-1050)")
            ("goi" nil  nil   nil I   L   () nil nil "Gobasi")
            ("goj" nil  nil   nil I   L   () nil nil "Gowlan")
            ("gok" nil  nil   nil I   L   () nil nil "Gowli")
            ("gol" nil  nil   nil I   L   () nil nil "Gola")
            ("gom" nil  nil   nil I   L   () nil "ಕೊಂಕಣಿ" "Konkani, Goan")
            ("gon" nil  "gon" nil M   L   ("esg"
                                           "gno" "wsg" "ggo") nil nil "Gondi")
            ("goo" nil  nil   nil I   L   () nil nil "Gone Dau")
            ("gop" nil  nil   nil I   L   () nil nil "Yeretuar")
            ("goq" nil  nil   nil I   L   () nil nil "Gorap")
            ("gor" nil  "gor" nil I   L   () nil nil "Gorontalo")
            ("gos" nil  nil   nil I   L   () nil nil "Gronings")
            ("got" nil  "got" nil I   A   () nil nil "Gothic")
            ("gou" nil  nil   nil I   L   () nil nil "Gavar")
            ("gov" nil  nil   nil I   L   () "Niger–Congo" nil "Goo")
            ("gow" nil  nil   nil I   L   () nil nil "Gorowa")
            ("gox" nil  nil   nil I   L   () nil nil "Gobu")
            ("goy" nil  nil   nil I   L   () nil nil "Goundo")
            ("goz" nil  nil   nil I   L   () nil nil "Gozarkhani")
            ("gpa" nil  nil   nil I   L   () nil nil "Gupa-Abawa")
            ("gpe" nil  nil   nil I   L   () nil nil "Ghanaian Pidgin English")
            ("gpn" nil  nil   nil I   L   () nil nil "Taiap")
            ("gqa" nil  nil   nil I   L   () nil nil "Ga'anda")
            ("gqi" nil  nil   nil I   L   () nil nil "Guiqiong")
            ("gqn" nil  nil   nil I   E   () nil nil "Guana (Brazil)")
            ("gqr" nil  nil   nil I   L   () nil nil "Gor")
            ("gqu" nil  nil   nil I   L   () nil nil "Qau")
            ("gra" nil  nil   nil I   L   () nil nil "Garasia, Rajput")
            ("grb" nil  "grb" nil M   L   ("gbo" "gec"
                                           "grj" "grv" "gry") nil nil "Grebo")
            ("grc" nil  "grc" nil I   H   () nil "ἑλληνικά" "Greek, Ancient (to 1453)")
            ("grd" nil  nil   nil I   L   () nil nil "Guruntum-Mbaaru")
            ("grg" nil  nil   nil I   L   () nil nil "Madi")
            ("grh" nil  nil   nil I   L   () nil nil "Gbiri-Niragu")
            ("gri" nil  nil   nil I   L   () nil nil "Ghari")
            ("grj" nil  nil   nil I   L   () nil nil "Grebo, Southern")
            ("grm" nil  nil   nil I   L   () nil nil "Kota Marudu Talantang")
            ("grn" "gn" "grn" nil M   L   ("gnw" "gug"
                                           "gui" "gun" "nhd") "Tupian" "Avañe'ẽ" "Guarani")
            ("gro" nil  nil   nil I   L   () nil nil "Groma")
            ("grq" nil  nil   nil I   L   () nil nil "Gorovu")
            ("grr" nil  nil   nil I   L   () nil nil "Taznatit")
            ("grs" nil  nil   nil I   L   () nil nil "Gresi")
            ("grt" nil  nil   nil I   L   () nil nil "Garo")
            ("gru" nil  nil   nil I   L   () nil nil "Kistane")
            ("grv" nil  nil   nil I   L   () nil nil "Grebo, Central")
            ("grw" nil  nil   nil I   L   () nil nil "Gweda")
            ("grx" nil  nil   nil I   L   () nil nil "Guriaso")
            ("gry" nil  nil   nil I   L   () nil nil "Grebo, Barclayville")
            ("grz" nil  nil   nil I   L   () nil nil "Guramalum")
            ("gsc" nil  nil   t   nil nil () nil nil "Gascon")
            ("gse" nil  nil   nil I   L   () nil nil "Ghanaian Sign Language")
            ("gsg" nil  nil   nil I   L   () nil nil "German Sign Language")
            ("gsl" nil  nil   nil I   L   () nil nil "Gusilay")
            ("gsm" nil  nil   nil I   L   () nil nil "Guatemalan Sign Language")
            ("gsn" nil  nil   nil I   L   () nil nil "Gusan")
            ("gso" nil  nil   nil I   L   () nil nil "Southwest Gbaya")
            ("gsp" nil  nil   nil I   L   () nil nil "Wasembo")
            ("gss" nil  nil   nil I   L   () nil nil "Greek Sign Language")
            ("gsw" nil  "gsw" nil I   L   () "Indo-European" "Alemannisch (specifically: Schwyzerdütsch, Elsassisch)" "Alemannic (specifically: Swiss German, Alsatian)")
            ("gta" nil  nil   nil I   L   () nil nil "Guató")
            ("gti" nil  nil   t   I   L   () nil nil "Gbati-ri")
            ("gtu" nil  nil   nil I   E   () nil nil "Aghu-Tharnggala")
            ("gua" nil  nil   nil I   L   () nil nil "Shiki")
            ("gub" nil  nil   nil I   L   () nil nil "Guajajára")
            ("guc" nil  nil   nil I   L   () nil nil "Wayuu")
            ("gud" nil  nil   nil I   L   () nil nil "Dida, Yocoboué")
            ("gue" nil  nil   nil I   L   () nil nil "Gurinji")
            ("guf" nil  nil   nil I   L   () nil nil "Gupapuyngu")
            ("gug" nil  nil   nil I   L   () nil nil "Guaraní, Paraguayan")
            ("guh" nil  nil   nil I   L   () nil nil "Guahibo")
            ("gui" nil  nil   nil I   L   () nil nil "Guaraní, Eastern Bolivian")
            ("guj" "gu" "guj" nil I   L   () "Indo-European" "ગુજરાતી" "Gujarati")
            ("guk" nil  nil   nil I   L   () nil nil "Gumuz")
            ("gul" nil  nil   nil I   L   () nil nil "Sea Island Creole English")
            ("gum" nil  nil   nil I   L   () nil nil "Guambiano")
            ("gun" nil  nil   nil I   L   () nil nil "Guaraní, Mbyá")
            ("guo" nil  nil   nil I   L   () nil nil "Guayabero")
            ("gup" nil  nil   nil I   L   () nil nil "Gunwinggu")
            ("guq" nil  nil   nil I   L   () nil nil "Aché")
            ("gur" nil  nil   nil I   L   () nil nil "Farefare")
            ("gus" nil  nil   nil I   L   () nil nil "Guinean Sign Language")
            ("gut" nil  nil   nil I   L   () nil nil "Maléku Jaíka")
            ("guu" nil  nil   nil I   L   () nil nil "Yanomamö")
            ("guv" nil  nil   t   I   E   () nil nil "Gey")
            ("guw" nil  nil   nil I   L   () nil nil "Gun")
            ("gux" nil  nil   nil I   L   () nil nil "Gourmanchéma")
            ("guz" nil  nil   nil I   L   () nil nil "Gusii")
            ("gva" nil  nil   nil I   L   () nil nil "Guana (Paraguay)")
            ("gvc" nil  nil   nil I   L   () nil nil "Guanano")
            ("gve" nil  nil   nil I   L   () nil nil "Duwet")
            ("gvf" nil  nil   nil I   L   () nil nil "Golin")
            ("gvj" nil  nil   nil I   L   () nil nil "Guajá")
            ("gvl" nil  nil   nil I   L   () nil nil "Gulay")
            ("gvm" nil  nil   nil I   L   () nil nil "Gurmana")
            ("gvn" nil  nil   nil I   L   () nil nil "Kuku-Yalanji")
            ("gvo" nil  nil   nil I   L   () nil nil "Gavião Do Jiparaná")
            ("gvp" nil  nil   nil I   L   () nil nil "Gavião, Pará")
            ("gvr" nil  nil   nil I   L   () nil nil "Gurung, Western")
            ("gvs" nil  nil   nil I   L   () nil nil "Gumawana")
            ("gvy" nil  nil   nil I   E   () nil nil "Guyani")
            ("gwa" nil  nil   nil I   L   () nil nil "Mbato")
            ("gwb" nil  nil   nil I   L   () nil nil "Gwa")
            ("gwc" nil  nil   nil I   L   () nil nil "Kalami")
            ("gwd" nil  nil   nil I   L   () nil nil "Gawwada")
            ("gwe" nil  nil   nil I   L   () nil nil "Gweno")
            ("gwf" nil  nil   nil I   L   () nil nil "Gowro")
            ("gwg" nil  nil   nil I   L   () nil nil "Moo")
            ("gwi" nil  "gwi" nil I   L   () nil "Gwich´in" "Gwich´in")
            ("gwj" nil  nil   nil I   L   () nil nil "/Gwi")
            ("gwm" nil  nil   nil I   E   () nil nil "Awngthim")
            ("gwn" nil  nil   nil I   L   () nil nil "Gwandara")
            ("gwr" nil  nil   nil I   L   () nil nil "Gwere")
            ("gwt" nil  nil   nil I   L   () nil nil "Gawar-Bati")
            ("gwu" nil  nil   nil I   E   () nil nil "Guwamu")
            ("gww" nil  nil   nil I   L   () nil nil "Kwini")
            ("gwx" nil  nil   nil I   L   () nil nil "Gua")
            ("gxx" nil  nil   nil I   L   () nil nil "Wè Southern")
            ("gya" nil  nil   nil I   L   () nil nil "Gbaya, Northwest")
            ("gyb" nil  nil   nil I   L   () nil nil "Garus")
            ("gyd" nil  nil   nil I   L   () nil nil "Kayardild")
            ("gye" nil  nil   nil I   L   () nil nil "Gyem")
            ("gyf" nil  nil   nil I   E   () nil nil "Gungabula")
            ("gyg" nil  nil   nil I   L   () nil nil "Gbayi")
            ("gyi" nil  nil   nil I   L   () nil nil "Gyele")
            ("gyl" nil  nil   nil I   L   () nil nil "Gayil")
            ("gym" nil  nil   nil I   L   () nil nil "Ngäbere")
            ("gyn" nil  nil   nil I   L   () nil nil "Guyanese Creole English")
            ("gyo" nil  nil   nil I   L   () "Sino-Tibetan" nil "Gyalsumdo")
            ("gyr" nil  nil   nil I   L   () nil nil "Guarayu")
            ("gyy" nil  nil   nil I   E   () nil nil "Gunya")
            ("gyz" nil  nil   nil I   L   () "Afro-Asiatic" nil "Geji")
            ("gza" nil  nil   nil I   L   () nil nil "Ganza")
            ("gzi" nil  nil   nil I   L   () nil nil "Gazi")
            ("gzn" nil  nil   nil I   L   () nil nil "Gane")
            ;; H
            ("haa" nil  nil   nil I   L   () nil "Hän" "Han")
            ("hab" nil  nil   nil I   L   () nil nil "Hanoi Sign Language")
            ("hac" nil  nil   nil I   L   () nil nil "Gurani")
            ("had" nil  nil   nil I   L   () nil nil "Hatam")
            ("hae" nil  nil   nil I   L   () nil nil "Oromo, Eastern")
            ("haf" nil  nil   nil I   L   () nil nil "Haiphong Sign Language")
            ("hag" nil  nil   nil I   L   () nil nil "Hanga")
            ("hah" nil  nil   nil I   L   () nil nil "Hahon")
            ("hai" nil  "hai" nil M   L   ("hax" "hdn") nil "X̲aat Kíl" "Haida")
            ("haj" nil  nil   nil I   L   () nil nil "Hajong")
            ("hak" nil  nil   nil I   L   () "Chinese" nil "Hakka Chinese")
            ("hal" nil  nil   nil I   L   () nil nil "Halang")
            ("ham" nil  nil   nil I   L   () nil nil "Hewa")
            ("han" nil  nil   nil I   L   () nil nil "Hangaza")
            ("hao" nil  nil   nil I   L   () nil nil "Hakö")
            ("hap" nil  nil   nil I   L   () nil nil "Hupla")
            ("haq" nil  nil   nil I   L   () nil nil "Ha")
            ("har" nil  nil   nil I   L   () nil nil "Harari")
            ("has" nil  nil   nil I   L   () nil "X̄a'’islak̓ala" "Haisla")
            ("hat" "ht" "hat" nil I   L   () "French Creole" "kreyòl ayisyen" "Haitian; Haitian Creole")
            ("hau" "ha" "hau" nil I   L   () "Afro-Asiatic" "حَوْسَ" "Hausa")
            ("hav" nil  nil   nil I   L   () nil nil "Havu")
            ("haw" nil  "haw" nil I   L   () nil "ʻōlelo Hawaiʻi" "Hawaiian")
            ("hax" nil  nil   nil I   L   () nil nil "Haida, Southern")
            ("hay" nil  nil   nil I   L   () nil nil "Haya")
            ("haz" nil  nil   nil I   L   () nil nil "Hazaragi")
            ("hba" nil  nil   nil I   L   () nil nil "Hamba")
            ("hbb" nil  nil   nil I   L   () nil nil "Huba")
            ("hbn" nil  nil   nil I   L   () nil nil "Heiban")
            ("hbo" nil  nil   nil I   H   () nil nil "Hebrew, Ancient")
            ("hbs" nil  nil   nil M   L   ("bos"
                                           "cnr" "hrv" "srp") "Indo-European" "српскохрватски / hrvatskosrpski" "Serbo-Croatian")
            ("hbu" nil  nil   nil I   L   () nil nil "Habu")
            ("hca" nil  nil   nil I   L   () nil nil "Andaman Creole Hindi")
            ("hch" nil  nil   nil I   L   () nil nil "Huichol")
            ("hdn" nil  nil   nil I   L   () nil nil "Haida, Northern")
            ("hds" nil  nil   nil I   L   () nil nil "Honduras Sign Language")
            ("hdy" nil  nil   nil I   L   () nil nil "Hadiyya")
            ("hea" nil  nil   nil I   L   () nil nil "Hmong, Northern Qiandong")
            ("heb" "he" "heb" nil I   L   () "Afro-Asiatic" "עִבְרִית" "Hebrew")
            ("hed" nil  nil   nil I   L   () nil nil "Herdé")
            ("heg" nil  nil   nil I   L   () nil nil "Helong")
            ("heh" nil  nil   nil I   L   () nil nil "Hehe")
            ("hei" nil  nil   nil I   L   () nil "hailhzaqvla" "Heiltsuk")
            ("hem" nil  nil   nil I   L   () nil nil "Hemba")
            ("her" "hz" "her" nil I   L   () "Niger–Congo" "Otjiherero" "Herero")
            ("hgm" nil  nil   nil I   L   () nil nil "Hai//om")
            ("hgw" nil  nil   nil I   L   () nil nil "Haigwai")
            ("hhi" nil  nil   nil I   L   () nil nil "Hoia Hoia")
            ("hhr" nil  nil   nil I   L   () nil nil "Kerak")
            ("hhy" nil  nil   nil I   L   () nil nil "Hoyahoya")
            ("hia" nil  nil   nil I   L   () nil nil "Lamang")
            ("hib" nil  nil   nil I   E   () nil nil "Hibito")
            ("hid" nil  nil   nil I   L   () nil "hiraacá" "Hidatsa")
            ("hif" nil  nil   nil I   L   () nil nil "Fiji Hindi")
            ("hig" nil  nil   nil I   L   () nil nil "Kamwe")
            ("hih" nil  nil   nil I   L   () nil nil "Pamosu")
            ("hii" nil  nil   nil I   L   () nil nil "Hinduri")
            ("hij" nil  nil   nil I   L   () nil nil "Hijuk")
            ("hik" nil  nil   nil I   L   () nil nil "Seit-Kaitetu")
            ("hil" nil  "hil" nil I   L   () nil "Ilonggo" "Hiligaynon")
            (nil   nil  "him" nil C   nil () nil nil "Himachali languages; Pahari languages")
            ("hin" "hi" "hin" nil I   L   () "Indo-European" "हिन्दी" "Hindi")
            ("hio" nil  nil   nil I   L   () nil nil "Tsoa")
            ("hir" nil  nil   nil I   L   () nil nil "Himarimã")
            ("hit" nil  "hit" nil I   A   () nil nil "Hittite")
            ("hiw" nil  nil   nil I   L   () nil nil "Hiw")
            ("hix" nil  nil   nil I   L   () nil nil "Hixkaryána")
            ("hji" nil  nil   nil I   L   () nil nil "Haji")
            ("hka" nil  nil   nil I   L   () nil nil "Kahe")
            ("hke" nil  nil   nil I   L   () nil nil "Hunde")
            ("hkh" nil  nil   nil I   L   () "Indo-European" nil "Khah")
            ("hkk" nil  nil   nil I   L   () nil nil "Hunjara-Kaina Ke")
            ("hkn" nil  nil   nil I   L   () "Austroasiatic" nil "Mel-Khaonh")
            ("hks" nil  nil   nil I   L   () nil nil "Hong Kong Sign Language")
            ("hla" nil  nil   nil I   L   () nil nil "Halia")
            ("hlb" nil  nil   nil I   L   () nil nil "Halbi")
            ("hld" nil  nil   nil I   L   () nil nil "Halang Doan")
            ("hle" nil  nil   nil I   L   () nil nil "Hlersu")
            ("hlt" nil  nil   nil I   L   () nil nil "Nga La")
            ("hlu" nil  nil   nil I   A   () nil nil "Luwian, Hieroglyphic")
            ("hma" nil  nil   nil I   L   () nil nil "Hmong, Southern Mashan")
            ("hmb" nil  nil   nil I   L   () nil nil "Songhay, Humburi Senni")
            ("hmc" nil  nil   nil I   L   () "Hmong" nil "Hmong, Central Huishui")
            ("hmd" nil  nil   nil I   L   () "Hmong" nil "Hmong, Northeastern Dian")
            ("hme" nil  nil   nil I   L   () "Hmong" nil "Hmong, Eastern Huishui")
            ("hmf" nil  nil   nil I   L   () "Hmong" nil "Hmong Don")
            ("hmg" nil  nil   nil I   L   () "Hmong" nil "Hmong, Southwestern Guiyang")
            ("hmh" nil  nil   nil I   L   () "Hmong" nil "Hmong, Southwestern Huishui")
            ("hmi" nil  nil   nil I   L   () "Hmong" nil "Hmong, Northern Huishui")
            ("hmj" nil  nil   nil I   L   () "Hmong" nil "Hmong, Chonganjiang")
            ("hmk" nil  nil   nil I   A   () nil nil "Maek")
            ("hml" nil  nil   nil I   L   () "Hmong" nil "Hmong, Luopohe")
            ("hmm" nil  nil   nil I   L   () nil nil "Hmong, Central Mashan")
            ("hmn" nil  "hmn" nil M   L   ("cqd" "hea"
                                           "hma" "hmc" "hmd"
                                           "hme" "hmg" "hmh"
                                           "hmi" "hmj" "hml"
                                           "hmm" "hmp" "hmq"
                                           "hms" "hmw" "hmy"
                                           "hmz" "hnj" "hrm"
                                           "huj" "mmr" "muq"
                                           "mww" "sfm" "blu") "Hmong" "Hmoob" "Hmong")
            ("hmo" "ho" "hmo" nil I   L   () "Austronesian" nil "Hiri Motu")
            ("hmp" nil  nil   nil I   L   () "Hmong" nil "Hmong, Northern Mashan")
            ("hmq" nil  nil   nil I   L   () "Hmong" nil "Hmong, Eastern Qiandong")
            ("hmr" nil  nil   nil I   L   () nil nil "Hmar")
            ("hms" nil  nil   nil I   L   () "Hmong" nil "Hmong, Southern Qiandong")
            ("hmt" nil  nil   nil I   L   () nil nil "Hamtai")
            ("hmu" nil  nil   nil I   L   () "Hmong" nil "Hamap")
            ("hmv" nil  nil   nil I   L   () "Hmong" nil "Hmong Dô")
            ("hmw" nil  nil   nil I   L   () "Hmong" nil "Hmong, Western Mashan")
            ("hmy" nil  nil   nil I   L   () "Hmong" nil "Hmong, Southern Guiyang")
            ("hmz" nil  nil   nil I   L   () "Hmong" nil "Hmong Shua")
            ("hna" nil  nil   nil I   L   () nil nil "Mina (Cameroon)")
            ("hnd" nil  nil   nil I   L   () nil nil "Hindko, Southern")
            ("hne" nil  nil   nil I   L   () nil nil "Chhattisgarhi")
            ("hng" nil  nil   nil I   L   () nil nil "Hungu")
            ("hnh" nil  nil   nil I   L   () nil nil "Ani")
            ("hni" nil  nil   nil I   L   () nil "Haqniqdoq" "Hani")
            ("hnj" nil  nil   nil I   L   () nil nil "Hmong Njua")
            ("hnn" nil  nil   nil I   L   () nil nil "Hanunoo")
            ("hno" nil  nil   nil I   L   () nil nil "Hindko, Northern")
            ("hns" nil  nil   nil I   L   () nil nil "Hindustani, Caribbean")
            ("hnu" nil  nil   nil I   L   () nil nil "Hung")
            ("hoa" nil  nil   nil I   L   () nil nil "Hoava")
            ("hob" nil  nil   nil I   L   () nil nil "Mari (Madang Province)")
            ("hoc" nil  nil   nil I   L   () nil nil "Ho")
            ("hod" nil  nil   nil I   E   () nil nil "Holma")
            ("hoe" nil  nil   nil I   L   () nil nil "Horom")
            ("hoh" nil  nil   nil I   L   () nil nil "Hobyót")
            ("hoi" nil  nil   nil I   L   () nil nil "Holikachuk")
            ("hoj" nil  nil   nil I   L   () nil nil "Harauti")
            ("hol" nil  nil   nil I   L   () nil nil "Holu")
            ("hom" nil  nil   nil I   E   () nil nil "Homa")
            ("hoo" nil  nil   nil I   L   () nil nil "Holoholo")
            ("hop" nil  nil   nil I   L   () nil "Hopilàvayi" "Hopi")
            ("hor" nil  nil   nil I   E   () nil nil "Horo")
            ("hos" nil  nil   nil I   L   () nil nil "Ho Chi Minh City Sign Language")
            ("hot" nil  nil   nil I   L   () nil nil "Hote")
            ("hov" nil  nil   nil I   L   () nil nil "Hovongan")
            ("how" nil  nil   nil I   L   () nil nil "Honi")
            ("hoy" nil  nil   nil I   L   () nil nil "Holiya")
            ("hoz" nil  nil   nil I   L   () nil nil "Hozo")
            ("hpo" nil  nil   nil I   E   () nil nil "Hpon")
            ("hps" nil  nil   nil I   L   () nil nil "Hawai'i Pidgin Sign Language")
            ("hra" nil  nil   nil I   L   () nil nil "Hrangkhol")
            ("hrc" nil  nil   nil I   L   () nil nil "Niwer Mil")
            ("hre" nil  nil   nil I   L   () nil nil "Hre")
            ("hrk" nil  nil   nil I   L   () nil nil "Haruku")
            ("hrm" nil  nil   nil I   L   () nil nil "Horned Miao")
            ("hro" nil  nil   nil I   L   () nil nil "Haroi")
            ("hrp" nil  nil   nil I   E   () nil nil "Nhirrpi")
            ("hrr" nil  nil   t   I   L   () nil nil "Horuru")
            ("hrt" nil  nil   nil I   L   () nil nil "Hértevin")
            ("hru" nil  nil   nil I   L   () nil nil "Hruso")
            ("hrv" "hr" "hrv" nil I   L   () "Indo-European" "hrvatski" "Croatian")
            ("hrw" nil  nil   nil I   L   () nil nil "Warwar Feni")
            ("hrx" nil  nil   nil I   L   () nil nil "Hunsrik")
            ("hrz" nil  nil   nil I   L   () nil nil "Harzani")
            ("hsb" nil  "hsb" nil I   L   () nil "hornjoserbsce" "Sorbian, Upper")
            ("hsf" nil  nil   t   I   L   () nil nil "Huastec, Southeastern")
            ("hsh" nil  nil   nil I   L   () nil nil "Hungarian Sign Language")
            ("hsl" nil  nil   nil I   L   () nil nil "Hausa Sign Language")
            ("hsn" nil  nil   nil I   L   () "Chinese" nil "Xiang Chinese")
            ("hss" nil  nil   nil I   L   () nil nil "Harsusi")
            ("hti" nil  nil   nil I   E   () nil nil "Hoti")
            ("hto" nil  nil   nil I   L   () nil nil "Huitoto, Minica")
            ("hts" nil  nil   nil I   L   () nil nil "Hadza")
            ("htu" nil  nil   nil I   L   () nil nil "Hitu")
            ("htx" nil  nil   nil I   A   () nil nil "Hittite, Middle")
            ("hub" nil  nil   nil I   L   () nil nil "Huambisa")
            ("huc" nil  nil   nil I   L   () nil nil "=/Hua")
            ("hud" nil  nil   nil I   L   () nil nil "Huaulu")
            ("hue" nil  nil   nil I   L   () nil nil "Huave, San Francisco Del Mar")
            ("huf" nil  nil   nil I   L   () nil nil "Humene")
            ("hug" nil  nil   nil I   L   () nil nil "Huachipaeri")
            ("huh" nil  nil   nil I   L   () nil nil "Huilliche")
            ("hui" nil  nil   nil I   L   () nil nil "Huli")
            ("huj" nil  nil   nil I   L   () nil nil "Hmong, Northern Guiyang")
            ("huk" nil  nil   nil I   E   () nil nil "Hulung")
            ("hul" nil  nil   nil I   L   () nil nil "Hula")
            ("hum" nil  nil   nil I   L   () nil nil "Hungana")
            ("hun" "hu" "hun" nil I   L   () "Uralic" "magyar" "Hungarian")
            ("huo" nil  nil   nil I   L   () nil nil "Hu")
            ("hup" nil  "hup" nil I   L   () nil nil "Hupa")
            ("huq" nil  nil   nil I   L   () nil nil "Tsat")
            ("hur" nil  nil   nil I   L   () nil "Hǝn̓q̓ǝmin̓ǝm̓" "Halkomelem")
            ("hus" nil  nil   nil I   L   () nil nil "Huastec, Veracruz")
            ("hut" nil  nil   nil I   L   () nil nil "Humla")
            ("huu" nil  nil   nil I   L   () nil nil "Huitoto, Murui")
            ("huv" nil  nil   nil I   L   () nil nil "Huave, San Mateo Del Mar")
            ("huw" nil  nil   nil I   E   () nil nil "Hukumina")
            ("hux" nil  nil   nil I   L   () nil nil "Huitoto, Nüpode")
            ("huy" nil  nil   nil I   L   () nil nil "Hulaulá")
            ("huz" nil  nil   nil I   L   () nil nil "Hunzib")
            ("hva" nil  nil   t   I   L   () nil nil "Huastec, San Luís Potosí")
            ("hvc" nil  nil   nil I   L   () nil nil "Haitian Vodoun Culture Language")
            ("hve" nil  nil   nil I   L   () nil nil "Huave, San Dionisio Del Mar")
            ("hvk" nil  nil   nil I   L   () nil nil "Haveke")
            ("hvn" nil  nil   nil I   L   () nil nil "Sabu")
            ("hvv" nil  nil   nil I   L   () nil nil "Huave, Santa María Del Mar")
            ("hwa" nil  nil   nil I   L   () nil nil "Wané")
            ("hwc" nil  nil   nil I   L   () nil nil "Hawai'i Creole English")
            ("hwo" nil  nil   nil I   L   () nil nil "Hwana")
            ("hya" nil  nil   nil I   L   () nil nil "Hya")
            ("hye" "hy" "arm" nil I   L   () "Indo-European" "Հայերեն" "Armenian")
            ("hyw" nil  nil   nil I   L   () "Indo-European" nil "Western Armenian")
            ;; I
            ("iai" nil  nil   nil I   L   () nil nil "Iaai")
            ("ian" nil  nil   nil I   L   () nil nil "Iatmul")
            ("iap" nil  nil   t   I   L   () nil nil "Iapama")
            ("iar" nil  nil   nil I   L   () nil nil "Purari")
            ("iba" nil  "iba" nil I   L   () nil nil "Iban")
            ("ibb" nil  nil   nil I   L   () nil nil "Ibibio")
            ("ibd" nil  nil   nil I   L   () nil nil "Iwaidja")
            ("ibe" nil  nil   nil I   L   () nil nil "Akpes")
            ("ibg" nil  nil   nil I   L   () nil nil "Ibanag")
            ("ibh" nil  nil   nil I   L   () "Austronesian" nil "Bih")
            ("ibi" nil  nil   t   I   L   () nil nil "Ibilo")
            ("ibl" nil  nil   nil I   L   () nil nil "Ibaloi")
            ("ibm" nil  nil   nil I   L   () nil nil "Agoi")
            ("ibn" nil  nil   nil I   L   () nil nil "Ibino")
            ("ibo" "ig" "ibo" nil I   L   () "Niger–Congo" "Igbo" "Igbo")
            ("ibr" nil  nil   nil I   L   () nil nil "Ibuoro")
            ("ibu" nil  nil   nil I   L   () nil nil "Ibu")
            ("iby" nil  nil   nil I   L   () nil nil "Ibani")
            ("ica" nil  nil   nil I   L   () nil nil "Ede Ica")
            ("ich" nil  nil   nil I   L   () nil nil "Etkywan")
            ("icl" nil  nil   nil I   L   () nil nil "Icelandic Sign Language")
            ("icr" nil  nil   nil I   L   () nil nil "Islander Creole English")
            ("ida" nil  nil   nil I   L   () nil nil "Idakho-Isukha-Tiriki")
            ("idb" nil  nil   nil I   L   () nil nil "Indo-Portuguese")
            ("idc" nil  nil   nil I   L   () nil nil "Idon")
            ("idd" nil  nil   nil I   L   () nil nil "Ede Idaca")
            ("ide" nil  nil   nil I   L   () nil nil "Idere")
            ("idi" nil  nil   nil I   L   () nil nil "Idi")
            ("ido" "io" "ido" nil I   C   () "constructed" "Ido" "Ido")
            ("idr" nil  nil   nil I   L   () nil nil "Indri")
            ("ids" nil  nil   nil I   L   () nil nil "Idesa")
            ("idt" nil  nil   nil I   L   () nil nil "Idaté")
            ("idu" nil  nil   nil I   L   () nil nil "Idoma")
            ("ifa" nil  nil   nil I   L   () nil nil "Ifugao, Amganad")
            ("ifb" nil  nil   nil I   L   () nil nil "Ifugao, Batad")
            ("ife" nil  nil   nil I   L   () nil nil "Ifè")
            ("iff" nil  nil   nil I   E   () nil nil "Ifo")
            ("ifk" nil  nil   nil I   L   () nil nil "Ifugao, Tuwali")
            ("ifm" nil  nil   nil I   L   () nil nil "Teke-Fuumu")
            ("ifu" nil  nil   nil I   L   () nil nil "Ifugao, Mayoyao")
            ("ify" nil  nil   nil I   L   () nil nil "Kallahan, Keley-I")
            ("igb" nil  nil   nil I   L   () nil nil "Ebira")
            ("ige" nil  nil   nil I   L   () nil nil "Igede")
            ("igg" nil  nil   nil I   L   () nil nil "Igana")
            ("igl" nil  nil   nil I   L   () nil nil "Igala")
            ("igm" nil  nil   nil I   L   () nil nil "Kanggape")
            ("ign" nil  nil   nil I   L   () nil nil "Ignaciano")
            ("igo" nil  nil   nil I   L   () nil nil "Isebe")
            ("igs" nil  nil   nil I   C   () "constructed" nil "Interglossa")
            ("igw" nil  nil   nil I   L   () nil nil "Igwe")
            ("ihb" nil  nil   nil I   L   () nil nil "Iha Based Pidgin")
            ("ihi" nil  nil   nil I   L   () nil nil "Ihievbe")
            ("ihp" nil  nil   nil I   L   () nil nil "Iha")
            ("ihw" nil  nil   nil I   E   () nil nil "Bidhawal")
            ("iii" "ii" "iii" nil I   L   () "Sino-Tibetan" "ꆇꉙ" "Yi, Sichuan")
            ("iin" nil  nil   nil I   E   () nil nil "Thiin")
            ("ijc" nil  nil   nil I   L   () nil nil "Izon")
            ("ije" nil  nil   nil I   L   () nil nil "Biseni")
            ("ijj" nil  nil   nil I   L   () nil nil "Ede Ije")
            ("ijn" nil  nil   nil I   L   () nil nil "Kalabari")
            (nil   nil  "ijo" nil C   nil () nil nil "Ijo languages")
            ("ijs" nil  nil   nil I   L   () nil nil "Ijo, Southeast")
            ("ike" nil  nil   nil I   L   () "Eskimo–Aleut" nil "Inuktitut (Eastern Canadian)")
            ("ikh" nil  nil   nil I   L   () "Niger–Congo" nil "Ikhin-Arokho")
            ("iki" nil  nil   nil I   L   () nil nil "Iko")
            ("ikk" nil  nil   nil I   L   () nil nil "Ika")
            ("ikl" nil  nil   nil I   L   () nil nil "Ikulu")
            ("iko" nil  nil   nil I   L   () nil nil "Olulumo-Ikom")
            ("ikp" nil  nil   nil I   L   () nil nil "Ikpeshi")
            ("ikr" nil  nil   nil I   E   () nil nil "Ikaranggal")
            ("iks" nil  nil   nil I   L   () nil nil "Inuit Sign Language")
            ("ikt" nil  nil   nil I   L   () "Eskimo–Aleut" nil "Inuktitut (Western Canadian)")
            ("iku" "iu" "iku" nil M   L   ("ike" "ikt") "Eskimo–Aleut" "ᐃᓄᒃᑎᑐᑦ" "Inuktitut")
            ("ikv" nil  nil   nil I   L   () nil nil "Iku-Gora-Ankwa")
            ("ikw" nil  nil   nil I   L   () nil nil "Ikwere")
            ("ikx" nil  nil   nil I   L   () nil nil "Ik")
            ("ikz" nil  nil   nil I   L   () nil nil "Ikizu")
            ("ila" nil  nil   nil I   L   () nil nil "Ile Ape")
            ("ilb" nil  nil   nil I   L   () nil nil "Ila")
            ("ile" "ie" "ile" nil I   C   () "constructed" "Interlingue" "Interlingue")
            ("ilg" nil  nil   nil I   E   () nil nil "Garig-Ilgar")
            ("ili" nil  nil   nil I   L   () nil nil "Ili Turki")
            ("ilk" nil  nil   nil I   L   () nil nil "Ilongot")
            ("ill" nil  nil   t   I   L   () nil nil "Iranun")
            ("ilm" nil  nil   nil I   L   () nil nil "Iranun (Malaysia)")
            ("ilo" nil  "ilo" nil I   L   () nil "ilokano" "Iloko")
            ("ilp" nil  nil   nil I   L   () nil nil "Iranun (Philippines)")
            ("ils" nil  nil   nil I   L   () nil nil "International Sign")
            ("ilu" nil  nil   nil I   L   () nil nil "Ili'uun")
            ("ilv" nil  nil   nil I   L   () nil nil "Ilue")
            ("ilw" nil  nil   t   I   L   () nil nil "Talur")
            ("ima" nil  nil   nil I   L   () nil nil "Mala Malasar")
            ("ime" nil  nil   t   I   L   () nil nil "Imeraguen")
            ("imi" nil  nil   nil I   L   () nil nil "Anamgura")
            ("iml" nil  nil   nil I   E   () nil nil "Miluk")
            ("imn" nil  nil   nil I   L   () nil nil "Imonda")
            ("imo" nil  nil   nil I   L   () nil nil "Imbongu")
            ("imr" nil  nil   nil I   L   () nil nil "Imroing")
            ("ims" nil  nil   nil I   A   () nil nil "Marsian")
            ("imt" nil  nil   nil I   L   () "Nilo-Saharan" nil "Imotong")
            ("imy" nil  nil   nil I   A   () nil nil "Milyan")
            ("ina" "ia" "ina" nil I   C   () "constructed" "interlingua" "Interlingua")
            ("inb" nil  nil   nil I   L   () nil nil "Inga")
            (nil   nil  "inc" nil C   nil () nil nil "Indo-Aryan languages")
            ("ind" "id" "ind" nil I   L   () "Austronesian" "bahasa Indonesia" "Indonesian")
            (nil   nil  "ine" nil C   nil () nil nil "Indo-European languages")
            ("ing" nil  nil   nil I   L   () nil nil "Degexit'an")
            ("inh" nil  "inh" nil I   L   () nil "гӀалгӀай" "Ingush")
            ("inj" nil  nil   nil I   L   () nil nil "Inga, Jungle")
            ("inl" nil  nil   nil I   L   () nil nil "Indonesian Sign Language")
            ("inm" nil  nil   nil I   A   () nil nil "Minaean")
            ("inn" nil  nil   nil I   L   () nil nil "Isinai")
            ("ino" nil  nil   nil I   L   () nil nil "Inoke-Yate")
            ("inp" nil  nil   nil I   L   () nil nil "Iñapari")
            ("ins" nil  nil   nil I   L   () nil nil "Indian Sign Language")
            ("int" nil  nil   nil I   L   () nil nil "Intha")
            ("inz" nil  nil   nil I   E   () nil nil "Ineseño")
            ("ior" nil  nil   nil I   L   () nil nil "Inor")
            ("iou" nil  nil   nil I   L   () nil nil "Tuma-Irumu")
            ("iow" nil  nil   nil I   E   () nil nil "Iowa-Oto")
            ("ipi" nil  nil   nil I   L   () nil nil "Ipili")
            ("ipk" "ik" "ipk" nil M   L   ("esi" "esk") "Eskimo–Aleut" "Iñupiaq" "Inupiaq")
            ("ipo" nil  nil   nil I   L   () nil nil "Ipiko")
            ("iqu" nil  nil   nil I   L   () nil nil "Iquito")
            ("iqw" nil  nil   nil I   L   () nil nil "Ikwo")
            (nil   nil  "ira" nil C   nil () nil nil "Iranian languages")
            ("ire" nil  nil   nil I   L   () nil nil "Iresim")
            ("irh" nil  nil   nil I   L   () nil nil "Irarutu")
            ("iri" nil  nil   nil I   L   () nil nil "Irigwe")
            ("irk" nil  nil   nil I   L   () nil nil "Iraqw")
            ("irn" nil  nil   nil I   L   () nil nil "Irántxe")
            (nil   nil  "iro" nil C   nil () nil nil "Iroquoian languages")
            ("irr" nil  nil   nil I   L   () nil nil "Ir")
            ("iru" nil  nil   nil I   L   () nil nil "Irula")
            ("irx" nil  nil   nil I   L   () nil nil "Kamberau")
            ("iry" nil  nil   nil I   L   () nil nil "Iraya")
            ("isa" nil  nil   nil I   L   () nil nil "Isabi")
            ("isc" nil  nil   nil I   L   () nil nil "Isconahua")
            ("isd" nil  nil   nil I   L   () nil nil "Isnag")
            ("ise" nil  nil   nil I   L   () nil nil "Italian Sign Language")
            ("isg" nil  nil   nil I   L   () nil nil "Irish Sign Language")
            ("ish" nil  nil   nil I   L   () nil nil "Esan")
            ("isi" nil  nil   nil I   L   () nil nil "Nkem-Nkum")
            ("isk" nil  nil   nil I   L   () nil nil "Ishkashimi")
            ("isl" "is" "ice" nil I   L   () "Indo-European" "íslenska" "Icelandic")
            ("ism" nil  nil   nil I   L   () nil nil "Masimasi")
            ("isn" nil  nil   nil I   L   () nil nil "Isanzu")
            ("iso" nil  nil   nil I   L   () nil nil "Isoko")
            ("isr" nil  nil   nil I   L   () nil nil "Israeli Sign Language")
            ("ist" nil  nil   nil I   L   () nil nil "Istriot")
            ("isu" nil  nil   nil I   L   () nil nil "Isu (Menchum Division)")
            ("ita" "it" "ita" nil I   L   () "Indo-European" "italiano" "Italian")
            ("itb" nil  nil   nil I   L   () nil nil "Itneg, Binongan")
            ("itd" nil  nil   nil I   L   () "Austronesian" nil "Southern Tidung")
            ("ite" nil  nil   nil I   E   () nil nil "Itene")
            ("iti" nil  nil   nil I   L   () nil nil "Itneg, Inlaod")
            ("itk" nil  nil   nil I   L   () nil nil "Judeo-Italian")
            ("itl" nil  nil   nil I   L   () nil "Итэнмэн" "Itelmen")
            ("itm" nil  nil   nil I   L   () nil nil "Itu Mbon Uzo")
            ("ito" nil  nil   nil I   L   () nil nil "Itonama")
            ("itr" nil  nil   nil I   L   () nil nil "Iteri")
            ("its" nil  nil   nil I   L   () nil nil "Isekiri")
            ("itt" nil  nil   nil I   L   () nil nil "Itneg, Maeng")
            ("itu" nil  nil   t   nil nil () nil nil "Itutang")
            ("itv" nil  nil   nil I   L   () nil nil "Itawit")
            ("itw" nil  nil   nil I   L   () nil nil "Ito")
            ("itx" nil  nil   nil I   L   () nil nil "Itik")
            ("ity" nil  nil   nil I   L   () nil nil "Itneg, Moyadan")
            ("itz" nil  nil   nil I   L   () nil nil "Itzá")
            ("ium" nil  nil   nil I   L   () nil nil "Iu Mien")
            ("ivb" nil  nil   nil I   L   () nil nil "Ibatan")
            ("ivv" nil  nil   nil I   L   () nil nil "Ivatan")
            ("iwk" nil  nil   nil I   L   () nil nil "I-Wak")
            ("iwm" nil  nil   nil I   L   () nil nil "Iwam")
            ("iwo" nil  nil   nil I   L   () nil nil "Iwur")
            ("iws" nil  nil   nil I   L   () nil nil "Iwam, Sepik")
            ("ixc" nil  nil   nil I   L   () nil nil "Ixcatec")
            ("ixi" nil  nil   t   I   L   () nil nil "Ixil, Nebaj")
            ("ixj" nil  nil   t   I   L   () nil nil "Ixil, Chajul")
            ("ixl" nil  nil   nil I   L   () nil nil "Ixil, San Juan Cotzal")
            ("iya" nil  nil   nil I   L   () nil nil "Iyayu")
            ("iyo" nil  nil   nil I   L   () nil nil "Mesaka")
            ("iyx" nil  nil   nil I   L   () nil nil "Yaka (Congo)")
            ("izh" nil  nil   nil I   L   () nil "ižoran keeli" "Ingrian")
            ("izi" nil  nil   t   I   L   () nil nil "Izi-Ezaa-Ikwo-Mgbo")
            ("izm" nil  nil   nil I   L   () "Niger–Congo" nil "Kizamani")
            ("izr" nil  nil   nil I   L   () nil nil "Izere")
            ("izz" nil  nil   nil I   L   () nil nil "Izii")
            ;; J
            ("jaa" nil  nil   nil I   L   () nil nil "Jamamadí")
            ("jab" nil  nil   nil I   L   () nil nil "Hyam")
            ("jac" nil  nil   nil I   L   () nil nil "Jacalteco, Eastern")
            ("jad" nil  nil   nil I   L   () nil nil "Jahanka")
            ("jae" nil  nil   nil I   L   () nil nil "Yabem")
            ("jaf" nil  nil   nil I   L   () nil nil "Jara")
            ("jah" nil  nil   nil I   L   () nil nil "Jah Hut")
            ("jai" nil  nil   t   I   L   () nil nil "Jacalteco, Western")
            ("jaj" nil  nil   nil I   L   () nil nil "Zazao")
            ("jak" nil  nil   nil I   L   () nil nil "Jakun")
            ("jal" nil  nil   nil I   L   () nil nil "Yalahatan")
            ("jam" nil  nil   nil I   L   () nil "Jumiekan Patwa" "Jamaican Creole English")
            ("jan" nil  nil   nil I   E   () nil nil "Jandai")
            ("jao" nil  nil   nil I   L   () nil nil "Yanyuwa")
            ("jap" nil  nil   t   nil nil () nil nil "Jaruára")
            ("jaq" nil  nil   nil I   L   () nil nil "Yaqay")
            ("jar" nil  nil   t   I   L   () nil nil "Jarawa (Nigeria)")
            ("jas" nil  nil   nil I   L   () nil nil "Javanese, New Caledonian")
            ("jat" nil  nil   nil I   L   () nil nil "Jakati")
            ("jau" nil  nil   nil I   L   () nil nil "Yaur")
            ("jav" "jv" "jav" nil I   L   () "Austronesian" "basa Jawa" "Javanese")
            ("jax" nil  nil   nil I   L   () nil nil "Malay, Jambi")
            ("jay" nil  nil   nil I   L   () nil nil "Jarnango")
            ("jaz" nil  nil   nil I   L   () nil nil "Jawe")
            ("jbe" nil  nil   nil I   L   () nil nil "Judeo-Berber")
            ("jbi" nil  nil   nil I   E   () nil nil "Badjiri")
            ("jbj" nil  nil   nil I   L   () nil nil "Arandai")
            ("jbk" nil  nil   nil I   L   () nil nil "Barikewa")
            ("jbm" nil  nil   nil I   L   () "Niger–Congo" nil "Bijim")
            ("jbn" nil  nil   nil I   L   () nil nil "Nafusi")
            ("jbo" nil  "jbo" nil I   C   () nil "la .lojban." "Lojban")
            ("jbr" nil  nil   nil I   L   () nil nil "Jofotek-Bromnya")
            ("jbt" nil  nil   nil I   L   () nil nil "Jabutí")
            ("jbu" nil  nil   nil I   L   () nil nil "Jukun Takum")
            ("jbw" nil  nil   nil I   E   () nil nil "Yawijibaya")
            ("jcs" nil  nil   nil I   L   () nil nil "Jamaican Country Sign Language")
            ("jct" nil  nil   nil I   L   () nil "Кърымчах" "Judeo-Crimean Tatar")
            ("jda" nil  nil   nil I   L   () nil nil "Jad")
            ("jdg" nil  nil   nil I   L   () nil nil "Jadgali")
            ("jdt" nil  nil   nil I   L   () nil nil "Judeo-Tat")
            ("jeb" nil  nil   nil I   L   () nil nil "Jebero")
            ("jee" nil  nil   nil I   L   () nil nil "Jerung")
            ("jeg" nil  nil   t   I   L   () nil nil "Jeng")
            ("jeh" nil  nil   nil I   L   () nil nil "Jeh")
            ("jei" nil  nil   nil I   L   () nil nil "Yei")
            ("jek" nil  nil   nil I   L   () nil nil "Jeri Kuo")
            ("jel" nil  nil   nil I   L   () nil nil "Yelmek")
            ("jen" nil  nil   nil I   L   () nil nil "Dza")
            ("jer" nil  nil   nil I   L   () nil nil "Jere")
            ("jet" nil  nil   nil I   L   () nil nil "Manem")
            ("jeu" nil  nil   nil I   L   () nil nil "Jonkor Bourmataguil")
            ("jgb" nil  nil   nil I   E   () nil nil "Ngbee")
            ("jge" nil  nil   nil I   L   () nil "קיברולי" "Judeo-Georgian")
            ("jgk" nil  nil   nil I   L   () nil nil "Gwak")
            ("jgo" nil  nil   nil I   L   () nil nil "Ngomba")
            ("jhi" nil  nil   nil I   L   () nil nil "Jehai")
            ("jhs" nil  nil   nil I   L   () nil nil "Jhankot Sign Language")
            ("jia" nil  nil   nil I   L   () nil nil "Jina")
            ("jib" nil  nil   nil I   L   () nil nil "Jibu")
            ("jic" nil  nil   nil I   L   () nil nil "Tol")
            ("jid" nil  nil   nil I   L   () nil nil "Bu")
            ("jie" nil  nil   nil I   L   () nil nil "Jilbe")
            ("jig" nil  nil   nil I   L   () nil nil "Djingili")
            ("jih" nil  nil   nil I   L   () nil nil "Shangzhai")
            ("jii" nil  nil   nil I   L   () nil nil "Jiiddu")
            ("jil" nil  nil   nil I   L   () nil nil "Jilim")
            ("jim" nil  nil   nil I   L   () nil nil "Jimi (Cameroon)")
            ("jio" nil  nil   nil I   L   () nil nil "Jiamao")
            ("jiq" nil  nil   nil I   L   () nil nil "Guanyinqiao")
            ("jit" nil  nil   nil I   L   () nil nil "Jita")
            ("jiu" nil  nil   nil I   L   () nil nil "Jinuo, Youle")
            ("jiv" nil  nil   nil I   L   () nil nil "Shuar")
            ("jiy" nil  nil   nil I   L   () nil nil "Jinuo, Buyuan")
            ("jje" nil  nil   nil I   L   () "Koreanic" nil "Jejueo")
            ("jjr" nil  nil   nil I   L   () nil nil "Bankal")
            ("jka" nil  nil   nil I   L   () "Trans–New Guinea" nil "Kaera")
            ("jkm" nil  nil   nil I   L   () nil nil "Mobwa Karen")
            ("jko" nil  nil   nil I   L   () nil nil "Kubo")
            ("jkp" nil  nil   nil I   L   () nil nil "Paku Karen")
            ("jkr" nil  nil   nil I   L   () nil nil "Koro (India)")
            ("jks" nil  nil   nil I   L   () nil nil "Amami Koniya Sign Language")
            ("jku" nil  nil   nil I   L   () nil nil "Labir")
            ("jle" nil  nil   nil I   L   () nil nil "Ngile")
            ("jls" nil  nil   nil I   L   () nil nil "Jamaican Sign Language")
            ("jma" nil  nil   nil I   L   () nil nil "Dima")
            ("jmb" nil  nil   nil I   L   () nil nil "Zumbun")
            ("jmc" nil  nil   nil I   L   () nil nil "Machame")
            ("jmd" nil  nil   nil I   L   () nil nil "Yamdena")
            ("jmi" nil  nil   nil I   L   () nil nil "Jimi (Nigeria)")
            ("jml" nil  nil   nil I   L   () nil nil "Jumli")
            ("jmn" nil  nil   nil I   L   () nil nil "Makuri Naga")
            ("jmr" nil  nil   nil I   L   () nil nil "Kamara")
            ("jms" nil  nil   nil I   L   () nil nil "Mashi (Nigeria)")
            ("jmw" nil  nil   nil I   L   () nil nil "Mouwase")
            ("jmx" nil  nil   nil I   L   () nil nil "Mixtec (Western Juxtlahuaca)")
            ("jna" nil  nil   nil I   L   () nil nil "Jangshung")
            ("jnd" nil  nil   nil I   L   () nil nil "Jandavra")
            ("jng" nil  nil   nil I   E   () nil nil "Yangman")
            ("jni" nil  nil   nil I   L   () nil nil "Janji")
            ("jnj" nil  nil   nil I   L   () nil nil "Yemsa")
            ("jnl" nil  nil   nil I   L   () nil nil "Rawat")
            ("jns" nil  nil   nil I   L   () nil nil "Jaunsari")
            ("job" nil  nil   nil I   L   () nil nil "Joba")
            ("jod" nil  nil   nil I   L   () nil nil "Wojenaka")
            ("jog" nil  nil   nil I   L   () "Indo-European" nil "Jogi")
            ("jor" nil  nil   nil I   E   () nil nil "Jorá")
            ("jos" nil  nil   nil I   L   () nil nil "Jordanian Sign Language")
            ("jow" nil  nil   nil I   L   () nil nil "Jowulu")
            ("jpa" nil  nil   nil I   H   () nil nil "Jewish Palestinian Aramaic")
            ("jpn" "ja" "jpn" nil I   L   () "Japonic" "日本語" "Japanese")
            ("jpr" nil  "jpr" nil I   L   () nil nil "Judeo-Persian")
            ("jqr" nil  nil   nil I   L   () nil nil "Jaqaru")
            ("jra" nil  nil   nil I   L   () nil nil "Jarai")
            ("jrb" nil  "jrb" nil M   L   ("aju" "jye"
                                           "yhd" "yud" "ajt") nil nil "Judeo-Arabic")
            ("jrr" nil  nil   nil I   L   () nil nil "Jiru")
            ("jrt" nil  nil   nil I   L   () nil nil "Jorto")
            ("jru" nil  nil   nil I   L   () nil nil "Japrería")
            ("jsl" nil  nil   nil I   L   () nil nil "Japanese Sign Language")
            ("jua" nil  nil   nil I   L   () nil nil "Júma")
            ("jub" nil  nil   nil I   L   () nil nil "Wannu")
            ("juc" nil  nil   nil I   H   () nil nil "Jurchen")
            ("jud" nil  nil   nil I   L   () nil nil "Worodougou")
            ("juh" nil  nil   nil I   L   () nil nil "Hõne")
            ("jui" nil  nil   nil I   E   () nil nil "Ngadjuri")
            ("juk" nil  nil   nil I   L   () nil nil "Wapan")
            ("jul" nil  nil   nil I   L   () nil nil "Jirel")
            ("jum" nil  nil   nil I   L   () nil nil "Jumjum")
            ("jun" nil  nil   nil I   L   () nil nil "Juang")
            ("juo" nil  nil   nil I   L   () nil nil "Jiba")
            ("jup" nil  nil   nil I   L   () nil nil "Hupdë")
            ("jur" nil  nil   nil I   L   () nil nil "Jurúna")
            ("jus" nil  nil   nil I   L   () nil nil "Jumla Sign Language")
            ("jut" nil  nil   nil I   H   () nil "jysk" "Jutish")
            ("juu" nil  nil   nil I   L   () nil nil "Ju")
            ("juw" nil  nil   nil I   L   () nil nil "Wãpha")
            ("juy" nil  nil   nil I   L   () nil nil "Juray")
            ("jvd" nil  nil   nil I   L   () nil nil "Javindo")
            ("jvn" nil  nil   nil I   L   () nil nil "Javanese, Caribbean")
            ("jwi" nil  nil   nil I   L   () nil nil "Jwira-Pepesa")
            ("jya" nil  nil   nil I   L   () nil nil "Jiarong")
            ("jye" nil  nil   nil I   L   () "Arabic" nil "Arabic (Judeo-Yemeni)")
            ("jyy" nil  nil   nil I   L   () nil nil "Jaya")
            ;; K
            ("kaa" nil  "kaa" nil I   L   () nil "Қарақалпақ" "Karakalpak")
            ("kab" nil  "kab" nil I   L   () nil "Taqbaylit" "Kabyle")
            ("kac" nil  "kac" nil I   L   () nil nil "Jingpho")
            ("kad" nil  nil   nil I   L   () nil nil "Kadara")
            ("kae" nil  nil   nil I   E   () nil nil "Ketangalan")
            ("kaf" nil  nil   nil I   L   () nil nil "Katso")
            ("kag" nil  nil   nil I   L   () nil nil "Kajaman")
            ("kah" nil  nil   nil I   L   () nil nil "Kara (Central African Republic)")
            ("kai" nil  nil   nil I   L   () nil nil "Karekare")
            ("kaj" nil  nil   nil I   L   () nil "Kaje" "Jju")
            ("kak" nil  nil   nil I   L   () nil nil "Kallahan, Kayapa")
            ("kal" "kl" "kal" nil I   L   () "Eskimo–Aleut" "kalaallisut" "Kalaallisut")
            ("kam" nil  "kam" nil I   L   () nil nil "Kamba (Kenya)")
            ("kan" "kn" "kan" nil I   L   () "Dravidian" "ಕನ್ನಡ" "Kannada")
            ("kao" nil  nil   nil I   L   () nil nil "Xaasongaxango")
            ("kap" nil  nil   nil I   L   () nil "Бежкьа" "Bezhta")
            ("kaq" nil  nil   nil I   L   () nil nil "Capanahua")
            (nil   nil  "kar" nil C   nil () nil nil "Karen languages")
            ("kas" "ks" "kas" nil I   L   () "Indo-European" "كشميري" "Kashmiri")
            ("kat" "ka" "geo" nil I   L   () "Kartvelian" "ქართული" "Georgian")
            ("kau" "kr" "kau" nil M   L   ("kby" "knc"
                                           "krt" "bms" "kbl") "Nilo-Saharan" "kanuri" "Kanuri")
            ("kav" nil  nil   nil I   L   () nil nil "Katukína")
            ("kaw" nil  "kaw" nil I   A   () nil "Bhāṣa Kawi" "Kawi")
            ("kax" nil  nil   nil I   L   () nil nil "Kao")
            ("kay" nil  nil   nil I   L   () nil nil "Kamayurá")
            ("kaz" "kk" "kaz" nil I   L   () "Turkic" "Қазақша" "Kazakh")
            ("kba" nil  nil   nil I   E   () nil nil "Kalarko")
            ("kbb" nil  nil   nil I   E   () nil nil "Kaxuiâna")
            ("kbc" nil  nil   nil I   L   () nil nil "Kadiwéu")
            ("kbd" nil  "kbd" nil I   L   () nil "къэбэрдеибзэ" "Kabardian")
            ("kbe" nil  nil   nil I   L   () nil nil "Kanju")
            ("kbf" nil  nil   t   I   E   () nil nil "Kakauhua")
            ("kbg" nil  nil   nil I   L   () nil nil "Khamba")
            ("kbh" nil  nil   nil I   L   () nil nil "Camsá")
            ("kbi" nil  nil   nil I   L   () nil nil "Kaptiau")
            ("kbj" nil  nil   nil I   L   () nil nil "Kari")
            ("kbk" nil  nil   nil I   L   () nil nil "Koiari, Grass")
            ("kbl" nil  nil   nil I   L   () nil nil "Kanembu")
            ("kbm" nil  nil   nil I   L   () nil nil "Iwal")
            ("kbn" nil  nil   nil I   L   () nil nil "Kare (Central African Republic)")
            ("kbo" nil  nil   nil I   L   () nil nil "Keliko")
            ("kbp" nil  nil   nil I   L   () nil nil "Kabiyé")
            ("kbq" nil  nil   nil I   L   () nil nil "Kamano")
            ("kbr" nil  nil   nil I   L   () nil nil "Kafa")
            ("kbs" nil  nil   nil I   L   () nil nil "Kande")
            ("kbt" nil  nil   nil I   L   () nil nil "Abadi")
            ("kbu" nil  nil   nil I   L   () nil nil "Kabutra")
            ("kbv" nil  nil   nil I   L   () nil nil "Dera (Indonesia)")
            ("kbw" nil  nil   nil I   L   () nil nil "Kaiep")
            ("kbx" nil  nil   nil I   L   () nil nil "Ap Ma")
            ("kby" nil  nil   nil I   L   () nil nil "Kanuri, Manga")
            ("kbz" nil  nil   nil I   L   () nil nil "Duhwa")
            ("kca" nil  nil   nil I   L   () nil "ханты" "Khanty")
            ("kcb" nil  nil   nil I   L   () nil nil "Kawacha")
            ("kcc" nil  nil   nil I   L   () nil nil "Lubila")
            ("kcd" nil  nil   nil I   L   () nil nil "Kanum, Ngkâlmpw")
            ("kce" nil  nil   nil I   L   () nil nil "Kaivi")
            ("kcf" nil  nil   nil I   L   () nil nil "Ukaan")
            ("kcg" nil  nil   nil I   L   () nil nil "Tyap")
            ("kch" nil  nil   nil I   L   () nil nil "Vono")
            ("kci" nil  nil   nil I   L   () nil nil "Kamantan")
            ("kcj" nil  nil   nil I   L   () nil nil "Kobiana")
            ("kck" nil  nil   nil I   L   () nil nil "Kalanga")
            ("kcl" nil  nil   nil I   L   () nil nil "Kela (Papua New Guinea)")
            ("kcm" nil  nil   nil I   L   () nil nil "Gula (Central African Republic)")
            ("kcn" nil  nil   nil I   L   () nil nil "Nubi")
            ("kco" nil  nil   nil I   L   () nil nil "Kinalakna")
            ("kcp" nil  nil   nil I   L   () nil nil "Kanga")
            ("kcq" nil  nil   nil I   L   () nil nil "Kamo")
            ("kcr" nil  nil   nil I   L   () nil nil "Katla")
            ("kcs" nil  nil   nil I   L   () nil nil "Koenoem")
            ("kct" nil  nil   nil I   L   () nil nil "Kaian")
            ("kcu" nil  nil   nil I   L   () nil nil "Kami (Tanzania)")
            ("kcv" nil  nil   nil I   L   () nil nil "Kete")
            ("kcw" nil  nil   nil I   L   () nil nil "Kabwari")
            ("kcx" nil  nil   nil I   L   () nil nil "Kachama-Ganjule")
            ("kcy" nil  nil   nil I   L   () nil nil "Korandje")
            ("kcz" nil  nil   nil I   L   () nil nil "Konongo")
            ("kda" nil  nil   nil I   E   () nil nil "Worimi")
            ("kdc" nil  nil   nil I   L   () nil nil "Kutu")
            ("kdd" nil  nil   nil I   L   () nil nil "Yankunytjatjara")
            ("kde" nil  nil   nil I   L   () nil nil "Makonde")
            ("kdf" nil  nil   nil I   L   () nil nil "Mamusi")
            ("kdg" nil  nil   nil I   L   () nil nil "Seba")
            ("kdh" nil  nil   nil I   L   () nil nil "Tem")
            ("kdi" nil  nil   nil I   L   () nil nil "Kumam")
            ("kdj" nil  nil   nil I   L   () nil nil "Karamojong")
            ("kdk" nil  nil   nil I   L   () nil nil "Numee")
            ("kdl" nil  nil   nil I   L   () nil nil "Tsikimba")
            ("kdm" nil  nil   nil I   L   () nil nil "Kagoma")
            ("kdn" nil  nil   nil I   L   () nil nil "Kunda")
            ("kdp" nil  nil   nil I   L   () nil nil "Kaningdon-Nindem")
            ("kdq" nil  nil   nil I   L   () nil nil "Koch")
            ("kdr" nil  nil   nil I   L   () nil "Къарай" "Karaim")
            ("kds" nil  nil   t   nil nil () nil nil "Lahu Shi")
            ("kdt" nil  nil   nil I   L   () nil nil "Kuy")
            ("kdu" nil  nil   nil I   L   () nil nil "Kadaru")
            ("kdv" nil  nil   t   I   L   () nil nil "Kado")
            ("kdw" nil  nil   nil I   L   () nil nil "Koneraw")
            ("kdx" nil  nil   nil I   L   () nil nil "Kam")
            ("kdy" nil  nil   nil I   L   () nil nil "Keder")
            ("kdz" nil  nil   nil I   L   () nil nil "Kwaja")
            ("kea" nil  nil   nil I   L   () nil "kriolu kabuverdianu" "Kabuverdianu")
            ("keb" nil  nil   nil I   L   () nil nil "Kélé")
            ("kec" nil  nil   nil I   L   () nil nil "Keiga")
            ("ked" nil  nil   nil I   L   () nil nil "Kerewe")
            ("kee" nil  nil   nil I   L   () nil nil "Keres, Eastern")
            ("kef" nil  nil   nil I   L   () nil nil "Kpessi")
            ("keg" nil  nil   nil I   L   () nil nil "Tese")
            ("keh" nil  nil   nil I   L   () nil nil "Keak")
            ("kei" nil  nil   nil I   L   () nil nil "Kei")
            ("kej" nil  nil   nil I   L   () nil nil "Kadar")
            ("kek" nil  nil   nil I   L   () nil nil "Kekchí")
            ("kel" nil  nil   nil I   L   () nil nil "Kela (Democratic Republic of Congo)")
            ("kem" nil  nil   nil I   L   () nil nil "Kemak")
            ("ken" nil  nil   nil I   L   () nil nil "Kenyang")
            ("keo" nil  nil   nil I   L   () nil nil "Kakwa")
            ("kep" nil  nil   nil I   L   () nil nil "Kaikadi")
            ("keq" nil  nil   nil I   L   () nil nil "Kamar")
            ("ker" nil  nil   nil I   L   () nil nil "Kera")
            ("kes" nil  nil   nil I   L   () nil nil "Kugbo")
            ("ket" nil  nil   nil I   L   () nil nil "Ket")
            ("keu" nil  nil   nil I   L   () nil nil "Akebu")
            ("kev" nil  nil   nil I   L   () nil nil "Kanikkaran")
            ("kew" nil  nil   nil I   L   () nil nil "Kewa, West")
            ("kex" nil  nil   nil I   L   () nil nil "Kukna")
            ("key" nil  nil   nil I   L   () nil nil "Kupia")
            ("kez" nil  nil   nil I   L   () nil nil "Kukele")
            ("kfa" nil  nil   nil I   L   () nil "ಕೊಡವ" "Kodava")
            ("kfb" nil  nil   nil I   L   () nil nil "Kolami, Northwestern")
            ("kfc" nil  nil   nil I   L   () nil nil "Konda-Dora")
            ("kfd" nil  nil   nil I   L   () nil nil "Koraga, Korra")
            ("kfe" nil  nil   nil I   L   () nil nil "Kota (India)")
            ("kff" nil  nil   nil I   L   () nil nil "Koya")
            ("kfg" nil  nil   nil I   L   () nil nil "Kudiya")
            ("kfh" nil  nil   nil I   L   () nil nil "Kurichiya")
            ("kfi" nil  nil   nil I   L   () nil nil "Kurumba")
            ("kfj" nil  nil   nil I   L   () nil nil "Kemiehua")
            ("kfk" nil  nil   nil I   L   () nil nil "Kinnauri")
            ("kfl" nil  nil   nil I   L   () nil nil "Kung")
            ("kfm" nil  nil   nil I   L   () nil nil "Khunsari")
            ("kfn" nil  nil   nil I   L   () nil nil "Kuk")
            ("kfo" nil  nil   nil I   L   () nil nil "Koro (Côte d'Ivoire)")
            ("kfp" nil  nil   nil I   L   () nil nil "Korwa")
            ("kfq" nil  nil   nil I   L   () nil nil "Korku")
            ("kfr" nil  nil   nil I   L   () nil "કચ્ચિ" "Kachchi")
            ("kfs" nil  nil   nil I   L   () nil nil "Bilaspuri")
            ("kft" nil  nil   nil I   L   () nil nil "Kanjari")
            ("kfu" nil  nil   nil I   L   () nil nil "Katkari")
            ("kfv" nil  nil   nil I   L   () nil nil "Kurmukar")
            ("kfw" nil  nil   nil I   L   () nil nil "Naga, Kharam")
            ("kfx" nil  nil   nil I   L   () nil nil "Pahari, Kullu")
            ("kfy" nil  nil   nil I   L   () nil nil "Kumauni")
            ("kfz" nil  nil   nil I   L   () nil nil "Koromfé")
            ("kga" nil  nil   nil I   L   () nil nil "Koyaga")
            ("kgb" nil  nil   nil I   L   () nil nil "Kawe")
            ("kgc" nil  nil   t   I   L   () nil nil "Kasseng")
            ("kgd" nil  nil   t   I   L   () nil nil "Kataang")
            ("kge" nil  nil   nil I   L   () nil nil "Komering")
            ("kgf" nil  nil   nil I   L   () nil nil "Kube")
            ("kgg" nil  nil   nil I   L   () nil nil "Kusunda")
            ("kgh" nil  nil   t   I   L   () nil nil "Kalinga, Upper Tanudan")
            ("kgi" nil  nil   nil I   L   () nil nil "Selangor Sign Language")
            ("kgj" nil  nil   nil I   L   () nil nil "Kham, Gamale")
            ("kgk" nil  nil   nil I   L   () nil nil "Kaiwá")
            ("kgl" nil  nil   nil I   E   () nil nil "Kunggari")
            ("kgm" nil  nil   t   I   E   () nil nil "Karipúna")
            ("kgn" nil  nil   nil I   L   () nil nil "Karingani")
            ("kgo" nil  nil   nil I   L   () nil nil "Krongo")
            ("kgp" nil  nil   nil I   L   () nil nil "Kaingáng")
            ("kgq" nil  nil   nil I   L   () nil nil "Kamoro")
            ("kgr" nil  nil   nil I   L   () nil nil "Abun")
            ("kgs" nil  nil   nil I   L   () nil nil "Kumbainggar")
            ("kgt" nil  nil   nil I   L   () nil nil "Somyev")
            ("kgu" nil  nil   nil I   L   () nil nil "Kobol")
            ("kgv" nil  nil   nil I   L   () nil nil "Karas")
            ("kgw" nil  nil   nil I   L   () nil nil "Karon Dori")
            ("kgx" nil  nil   nil I   L   () nil nil "Kamaru")
            ("kgy" nil  nil   nil I   L   () nil nil "Kyerung")
            ("kha" nil  "kha" nil I   L   () nil "Khasi" "Khasi")
            ("khb" nil  nil   nil I   L   () nil nil "Lü")
            ("khc" nil  nil   nil I   L   () nil nil "Tukang Besi North")
            ("khd" nil  nil   nil I   L   () nil nil "Kanum, Bädi")
            ("khe" nil  nil   nil I   L   () nil nil "Korowai")
            ("khf" nil  nil   nil I   L   () nil nil "Khuen")
            ("khg" nil  nil   nil I   L   () nil nil "Tibetan, Khams")
            ("khh" nil  nil   nil I   L   () nil nil "Kehu")
            (nil   nil  "khi" nil C   nil () nil nil "Khoisan languages")
            ("khj" nil  nil   nil I   L   () nil nil "Kuturmi")
            ("khk" nil  nil   nil I   L   () nil nil "Mongolian, Halh")
            ("khl" nil  nil   nil I   L   () nil nil "Lusi")
            ("khm" "km" "khm" nil I   L   () "Austroasiatic" "ខ្មែរ" "Khmer")
            ("khn" nil  nil   nil I   L   () nil nil "Khandesi")
            ("kho" nil  "kho" nil I   A   () nil nil "Khotanese")
            ("khp" nil  nil   nil I   L   () nil nil "Kapori")
            ("khq" nil  nil   nil I   L   () nil nil "Songhay, Koyra Chiini")
            ("khr" nil  nil   nil I   L   () nil nil "Kharia")
            ("khs" nil  nil   nil I   L   () nil nil "Kasua")
            ("kht" nil  nil   nil I   L   () nil nil "Khamti")
            ("khu" nil  nil   nil I   L   () nil nil "Nkhumbi")
            ("khv" nil  nil   nil I   L   () nil nil "Khvarshi")
            ("khw" nil  nil   nil I   L   () nil nil "Khowar")
            ("khx" nil  nil   nil I   L   () nil nil "Kanu")
            ("khy" nil  nil   nil I   L   () nil nil "Kele (Democratic Republic of Congo)")
            ("khz" nil  nil   nil I   L   () nil nil "Keapara")
            ("kia" nil  nil   nil I   L   () nil nil "Kim")
            ("kib" nil  nil   nil I   L   () nil nil "Koalib")
            ("kic" nil  nil   nil I   L   () nil "kikapú" "Kickapoo")
            ("kid" nil  nil   nil I   L   () nil nil "Koshin")
            ("kie" nil  nil   nil I   L   () nil nil "Kibet")
            ("kif" nil  nil   nil I   L   () nil nil "Parbate, Eastern")
            ("kig" nil  nil   nil I   L   () nil nil "Kimaama")
            ("kih" nil  nil   nil I   L   () nil nil "Kilmeri")
            ("kii" nil  nil   nil I   E   () nil nil "Kitsai")
            ("kij" nil  nil   nil I   L   () nil nil "Kilivila")
            ("kik" "ki" "kik" nil I   L   () "Niger–Congo" "Gĩkũyũ" "Kikuyu")
            ("kil" nil  nil   nil I   L   () nil nil "Kariya")
            ("kim" nil  nil   nil I   L   () nil "Тоъфа" "Karagas")
            ("kin" "rw" "kin" nil I   L   () "Niger–Congo" "kinyaRwanda" "Kinyarwanda")
            ("kio" nil  nil   nil I   L   () nil "Cáuijo̱:gà" "Kiowa")
            ("kip" nil  nil   nil I   L   () nil nil "Kham, Sheshi")
            ("kiq" nil  nil   nil I   L   () nil nil "Kosadle")
            ("kir" "ky" "kir" nil I   L   () "Turkic" "Кыргыз" "Kyrgyz")
            ("kis" nil  nil   nil I   L   () nil nil "Kis")
            ("kit" nil  nil   nil I   L   () nil nil "Agob")
            ("kiu" nil  nil   nil I   L   () nil nil "Kirmanjki (individual language)")
            ("kiv" nil  nil   nil I   L   () nil nil "Kimbu")
            ("kiw" nil  nil   nil I   L   () nil nil "Kiwai, Northeast")
            ("kix" nil  nil   nil I   L   () nil nil "Khiamniungan Naga")
            ("kiy" nil  nil   nil I   L   () nil nil "Kirikiri")
            ("kiz" nil  nil   nil I   L   () nil nil "Kisi")
            ("kja" nil  nil   nil I   L   () nil nil "Mlap")
            ("kjb" nil  nil   nil I   L   () nil nil "Kanjobal, Eastern")
            ("kjc" nil  nil   nil I   L   () nil nil "Konjo, Coastal")
            ("kjd" nil  nil   nil I   L   () nil nil "Kiwai, Southern")
            ("kje" nil  nil   nil I   L   () nil nil "Kisar")
            ("kjf" nil  nil   t   I   L   () nil nil "Khalaj")
            ("kjg" nil  nil   nil I   L   () nil nil "Khmu")
            ("kjh" nil  nil   nil I   L   () nil "Хакасча" "Khakas")
            ("kji" nil  nil   nil I   L   () nil nil "Zabana")
            ("kjj" nil  nil   nil I   L   () nil nil "Khinalugh")
            ("kjk" nil  nil   nil I   L   () nil nil "Konjo, Highland")
            ("kjl" nil  nil   nil I   L   () nil nil "Parbate, Western")
            ("kjm" nil  nil   nil I   L   () nil nil "Kháng")
            ("kjn" nil  nil   nil I   L   () nil nil "Kunjen")
            ("kjo" nil  nil   nil I   L   () nil nil "Kinnauri, Harijan")
            ("kjp" nil  nil   nil I   L   () nil nil "Karen, Pwo Eastern")
            ("kjq" nil  nil   nil I   L   () nil nil "Keres, Western")
            ("kjr" nil  nil   nil I   L   () nil nil "Kurudu")
            ("kjs" nil  nil   nil I   L   () nil nil "Kewa, East")
            ("kjt" nil  nil   nil I   L   () nil nil "Karen, Phrae Pwo")
            ("kju" nil  nil   nil I   L   () nil nil "Kashaya")
            ("kjv" nil  nil   nil I   H   () nil "Kajkavski" "Kaikavian literary language (Kajkavian)")
            ("kjx" nil  nil   nil I   L   () nil nil "Ramopa")
            ("kjy" nil  nil   nil I   L   () nil nil "Erave")
            ("kjz" nil  nil   nil I   L   () nil nil "Bumthangkha")
            ("kka" nil  nil   nil I   L   () nil nil "Kakanda")
            ("kkb" nil  nil   nil I   L   () nil nil "Kwerisa")
            ("kkc" nil  nil   nil I   L   () nil nil "Odoodee")
            ("kkd" nil  nil   nil I   L   () nil nil "Kinuku")
            ("kke" nil  nil   nil I   L   () nil nil "Kakabe")
            ("kkf" nil  nil   nil I   L   () nil nil "Kalaktang Monpa")
            ("kkg" nil  nil   nil I   L   () nil nil "Kalinga, Mabaka Valley")
            ("kkh" nil  nil   nil I   L   () nil nil "Khün")
            ("kki" nil  nil   nil I   L   () nil nil "Kagulu")
            ("kkj" nil  nil   nil I   L   () nil nil "Kako")
            ("kkk" nil  nil   nil I   L   () nil nil "Kokota")
            ("kkl" nil  nil   nil I   L   () nil nil "Yale, Kosarek")
            ("kkm" nil  nil   nil I   L   () nil nil "Kiong")
            ("kkn" nil  nil   nil I   L   () nil nil "Kon Keu")
            ("kko" nil  nil   nil I   L   () nil nil "Karko")
            ("kkp" nil  nil   nil I   L   () nil nil "Gugubera")
            ("kkq" nil  nil   nil I   L   () nil nil "Kaiku")
            ("kkr" nil  nil   nil I   L   () nil nil "Kir-Balar")
            ("kks" nil  nil   nil I   L   () nil nil "Giiwo")
            ("kkt" nil  nil   nil I   L   () nil nil "Koi")
            ("kku" nil  nil   nil I   L   () nil nil "Tumi")
            ("kkv" nil  nil   nil I   L   () nil nil "Kangean")
            ("kkw" nil  nil   nil I   L   () nil nil "Teke-Kukuya")
            ("kkx" nil  nil   nil I   L   () nil nil "Kohin")
            ("kky" nil  nil   nil I   L   () nil nil "Guguyimidjir")
            ("kkz" nil  nil   nil I   L   () nil "Dene Dzage" "Kaska")
            ("kla" nil  nil   nil I   E   () nil nil "Klamath-Modoc")
            ("klb" nil  nil   nil I   L   () nil nil "Kiliwa")
            ("klc" nil  nil   nil I   L   () nil nil "Kolbila")
            ("kld" nil  nil   nil I   L   () nil nil "Gamilaraay")
            ("kle" nil  nil   nil I   L   () nil nil "Kulung (Nepal)")
            ("klf" nil  nil   nil I   L   () nil nil "Kendeje")
            ("klg" nil  nil   nil I   L   () nil nil "Kalagan, Tagakaulu")
            ("klh" nil  nil   nil I   L   () nil nil "Weliki")
            ("kli" nil  nil   nil I   L   () nil nil "Kalumpang")
            ("klj" nil  nil   nil I   L   () nil "Qalayce" "Khalaj, Turkic")
            ("klk" nil  nil   nil I   L   () nil nil "Kono (Nigeria)")
            ("kll" nil  nil   nil I   L   () nil nil "Kalagan, Kagan")
            ("klm" nil  nil   nil I   L   () nil nil "Kolom")
            ("kln" nil  nil   nil M   L   ("enb" "eyo" "niq"
                                           "oki" "pko" "sgc"
                                           "spy" "tec" "tuy") nil nil "Kalenjin")
            ("klo" nil  nil   nil I   L   () nil nil "Kapya")
            ("klp" nil  nil   nil I   L   () nil nil "Kamasa")
            ("klq" nil  nil   nil I   L   () nil nil "Rumu")
            ("klr" nil  nil   nil I   L   () nil nil "Khaling")
            ("kls" nil  nil   nil I   L   () nil nil "Kalasha")
            ("klt" nil  nil   nil I   L   () nil nil "Nukna")
            ("klu" nil  nil   nil I   L   () nil nil "Klao")
            ("klv" nil  nil   nil I   L   () nil nil "Maskelynes")
            ("klw" nil  nil   nil I   L   () nil nil "Lindu")
            ("klx" nil  nil   nil I   L   () nil nil "Koluwawa")
            ("kly" nil  nil   nil I   L   () nil nil "Kalao")
            ("klz" nil  nil   nil I   L   () nil nil "Kabola")
            ("kma" nil  nil   nil I   L   () nil nil "Konni")
            ("kmb" nil  "kmb" nil I   L   () nil nil "Kimbundu")
            ("kmc" nil  nil   nil I   L   () nil nil "Dong, Southern")
            ("kmd" nil  nil   nil I   L   () nil nil "Kalinga, Madukayang")
            ("kme" nil  nil   nil I   L   () nil nil "Bakole")
            ("kmf" nil  nil   nil I   L   () nil nil "Kare (Papua New Guinea)")
            ("kmg" nil  nil   nil I   L   () nil nil "Kâte")
            ("kmh" nil  nil   nil I   L   () nil nil "Kalam")
            ("kmi" nil  nil   nil I   L   () nil nil "Kami (Nigeria)")
            ("kmj" nil  nil   nil I   L   () nil nil "Kumarbhag Paharia")
            ("kmk" nil  nil   nil I   L   () nil nil "Kalinga, Limos")
            ("kml" nil  nil   nil I   L   () nil nil "Kalinga, Lower Tanudan")
            ("kmm" nil  nil   nil I   L   () nil nil "Kom (India)")
            ("kmn" nil  nil   nil I   L   () nil nil "Awtuw")
            ("kmo" nil  nil   nil I   L   () nil nil "Kwoma")
            ("kmp" nil  nil   nil I   L   () nil nil "Gimme")
            ("kmq" nil  nil   nil I   L   () nil nil "Kwama")
            ("kmr" nil  nil   nil I   L   () nil nil "Kurdish, northern")
            ("kms" nil  nil   nil I   L   () nil nil "Kamasau")
            ("kmt" nil  nil   nil I   L   () nil nil "Kemtuik")
            ("kmu" nil  nil   nil I   L   () nil nil "Kanite")
            ("kmv" nil  nil   nil I   L   () nil nil "Karipúna Creole French")
            ("kmw" nil  nil   nil I   L   () nil nil "Komo (Democratic Republic of Congo)")
            ("kmx" nil  nil   nil I   L   () nil nil "Waboda")
            ("kmy" nil  nil   nil I   L   () nil nil "Koma")
            ("kmz" nil  nil   nil I   L   () nil nil "Khorasani Turkish")
            ("kna" nil  nil   nil I   L   () nil nil "Dera (Nigeria)")
            ("knb" nil  nil   nil I   L   () nil nil "Kalinga, Lubuagan")
            ("knc" nil  nil   nil I   L   () nil nil "Kanuri, Central")
            ("knd" nil  nil   nil I   L   () nil nil "Konda")
            ("kne" nil  nil   nil I   L   () nil nil "Kankanaey")
            ("knf" nil  nil   nil I   L   () nil nil "Mankanya")
            ("kng" nil  nil   nil I   L   () nil nil "Koongo")
            ("knh" nil  nil   t   nil nil () nil nil "Kayan River Kenyah")
            ("kni" nil  nil   nil I   L   () nil nil "Kanufi")
            ("knj" nil  nil   nil I   L   () nil nil "Kanjobal, Western")
            ("knk" nil  nil   nil I   L   () nil nil "Kuranko")
            ("knl" nil  nil   nil I   L   () nil nil "Keninjal")
            ("knm" nil  nil   nil I   L   () nil nil "Kanamarí")
            ("knn" nil  nil   nil I   L   () nil "ಕೊಂಕಣಿ" "Konkani (specific)")
            ("kno" nil  nil   nil I   L   () nil nil "Kono (Sierra Leone)")
            ("knp" nil  nil   nil I   L   () nil nil "Kwanja")
            ("knq" nil  nil   nil I   L   () nil nil "Kintaq")
            ("knr" nil  nil   nil I   L   () nil nil "Kaningra")
            ("kns" nil  nil   nil I   L   () nil nil "Kensiu")
            ("knt" nil  nil   nil I   L   () nil nil "Katukína, Panoan")
            ("knu" nil  nil   nil I   L   () nil nil "Kono (Guinea)")
            ("knv" nil  nil   nil I   L   () nil nil "Tabo")
            ("knw" nil  nil   nil I   L   () nil nil "Kung-Ekoka")
            ("knx" nil  nil   nil I   L   () nil nil "Kendayan")
            ("kny" nil  nil   nil I   L   () nil nil "Kanyok")
            ("knz" nil  nil   nil I   L   () nil nil "Kalamsé")
            ("koa" nil  nil   nil I   L   () nil nil "Konomala")
            ("kob" nil  nil   t   nil nil () nil nil "Kohoroxitari")
            ("koc" nil  nil   nil I   E   () nil nil "Kpati")
            ("kod" nil  nil   nil I   L   () nil nil "Kodi")
            ("koe" nil  nil   nil I   L   () nil nil "Kacipo-Balesi")
            ("kof" nil  nil   nil I   E   () nil nil "Kubi")
            ("kog" nil  nil   nil I   L   () nil nil "Cogui")
            ("koh" nil  nil   nil I   L   () nil nil "Koyo")
            ("koi" nil  nil   nil I   L   () nil "перым-коми" "Komi-Permyak")
            ("koj" nil  nil   t   I   L   () nil nil "Sara Dunjo")
            ("kok" nil  "kok" nil M   L   ("gom" "knn") nil "कॊंकणि" "Konkani (generic)")
            ("kol" nil  nil   nil I   L   () nil nil "Kol (Papua New Guinea)")
            ("kom" "kv" "kom" nil M   L   ("koi" "kpv") "Uralic" "коми" "Komi")
            ("kon" "kg" "kon" nil M   L   ("kng" "kwy" "ldi") "Niger–Congo" "Kikongo" "Kongo")
            ("koo" nil  nil   nil I   L   () nil nil "Konjo")
            ("kop" nil  nil   nil I   L   () nil nil "Kwato")
            ("koq" nil  nil   nil I   L   () nil nil "Kota (Gabon)")
            ("kor" "ko" "kor" nil I   L   () "Koreanic" "한국어" "Korean")
            ("kos" nil  "kos" nil I   L   () nil "Kosrae" "Kosraean")
            ("kot" nil  nil   nil I   L   () nil nil "Lagwan")
            ("kou" nil  nil   nil I   L   () nil nil "Koke")
            ("kov" nil  nil   nil I   L   () nil nil "Kudu-Camo")
            ("kow" nil  nil   nil I   L   () nil nil "Kugama")
            ("kox" nil  nil   t   I   E   () nil nil "Coxima")
            ("koy" nil  nil   nil I   L   () nil nil "Koyukon")
            ("koz" nil  nil   nil I   L   () nil nil "Korak")
            ("kpa" nil  nil   nil I   L   () nil nil "Kutto")
            ("kpb" nil  nil   nil I   L   () nil nil "Kurumba, Mullu")
            ("kpc" nil  nil   nil I   L   () nil nil "Curripaco")
            ("kpd" nil  nil   nil I   L   () nil nil "Koba")
            ("kpe" nil  "kpe" nil M   L   ("gkp" "xpe") nil "kpele" "Kpelle")
            ("kpf" nil  nil   nil I   L   () nil nil "Komba")
            ("kpg" nil  nil   nil I   L   () nil nil "Kapingamarangi")
            ("kph" nil  nil   nil I   L   () nil nil "Kplang")
            ("kpi" nil  nil   nil I   L   () nil nil "Kofei")
            ("kpj" nil  nil   nil I   L   () nil nil "Karajá")
            ("kpk" nil  nil   nil I   L   () nil nil "Kpan")
            ("kpl" nil  nil   nil I   L   () nil nil "Kpala")
            ("kpm" nil  nil   nil I   L   () nil nil "Koho")
            ("kpn" nil  nil   nil I   E   () nil nil "Kepkiriwát")
            ("kpo" nil  nil   nil I   L   () nil "Akpɔssɔ" "Ikposo")
            ("kpp" nil  nil   t   I   L   () nil nil "Karen, Paku")
            ("kpq" nil  nil   nil I   L   () nil nil "Korupun-Sela")
            ("kpr" nil  nil   nil I   L   () nil nil "Korafe")
            ("kps" nil  nil   nil I   L   () nil nil "Tehit")
            ("kpt" nil  nil   nil I   L   () nil nil "Karata")
            ("kpu" nil  nil   nil I   L   () nil nil "Kafoa")
            ("kpv" nil  nil   nil I   L   () nil nil "Komi-Zyrian")
            ("kpw" nil  nil   nil I   L   () nil nil "Kobon")
            ("kpx" nil  nil   nil I   L   () nil nil "Koiali, Mountain")
            ("kpy" nil  nil   nil I   L   () nil "нымылг'эн йилыйыл, чав'чывэн йилыйил" "Koryak")
            ("kpz" nil  nil   nil I   L   () nil nil "Kupsabiny")
            ("kqa" nil  nil   nil I   L   () nil nil "Mum")
            ("kqb" nil  nil   nil I   L   () nil nil "Kovai")
            ("kqc" nil  nil   nil I   L   () nil nil "Doromu")
            ("kqd" nil  nil   nil I   L   () nil nil "Koy Sanjaq Surat")
            ("kqe" nil  nil   nil I   L   () nil nil "Kalagan")
            ("kqf" nil  nil   nil I   L   () nil nil "Kakabai")
            ("kqg" nil  nil   nil I   L   () nil nil "Khe")
            ("kqh" nil  nil   nil I   L   () nil nil "Kisankasa")
            ("kqi" nil  nil   nil I   L   () nil nil "Koitabu")
            ("kqj" nil  nil   nil I   L   () nil nil "Koromira")
            ("kqk" nil  nil   nil I   L   () nil nil "Gbe, Kotafon")
            ("kql" nil  nil   nil I   L   () nil nil "Kyenele")
            ("kqm" nil  nil   nil I   L   () nil nil "Khisa")
            ("kqn" nil  nil   nil I   L   () nil nil "Kaonde")
            ("kqo" nil  nil   nil I   L   () nil nil "Krahn, Eastern")
            ("kqp" nil  nil   nil I   L   () nil nil "Kimré")
            ("kqq" nil  nil   nil I   L   () nil nil "Krenak")
            ("kqr" nil  nil   nil I   L   () nil nil "Kimaragang")
            ("kqs" nil  nil   nil I   L   () nil nil "Kissi, Northern")
            ("kqt" nil  nil   nil I   L   () nil nil "Kadazan, Klias River")
            ("kqu" nil  nil   nil I   E   () nil nil "Seroa")
            ("kqv" nil  nil   nil I   L   () nil nil "Okolod")
            ("kqw" nil  nil   nil I   L   () nil nil "Kandas")
            ("kqx" nil  nil   nil I   L   () nil nil "Mser")
            ("kqy" nil  nil   nil I   L   () nil nil "Koorete")
            ("kqz" nil  nil   nil I   E   () nil nil "Korana")
            ("kra" nil  nil   nil I   L   () nil nil "Kumhali")
            ("krb" nil  nil   nil I   E   () nil nil "Karkin")
            ("krc" nil  "krc" nil I   L   () nil "Къарачай-Малкъар" "Karachay-Balkar")
            ("krd" nil  nil   nil I   L   () nil nil "Kairui-Midiki")
            ("kre" nil  nil   nil I   L   () nil nil "Kreen-Akarore")
            ("krf" nil  nil   nil I   L   () nil nil "Koro (Vanuatu)")
            ("krg" nil  nil   t   nil nil () nil nil "North Korowai")
            ("krh" nil  nil   nil I   L   () nil nil "Kurama")
            ("kri" nil  nil   nil I   L   () nil nil "Krio")
            ("krj" nil  nil   nil I   L   () nil nil "Kinaray-A")
            ("krk" nil  nil   nil I   E   () nil nil "Kerek")
            ("krl" nil  "krl" nil I   L   () nil "karjala" "Karelian")
            ("krm" nil  nil   t   I   L   () nil nil "Krim")
            ("krn" nil  nil   nil I   L   () nil nil "Sapo")
            (nil   nil  "kro" nil C   nil () nil nil "Kru languages")
            ("krp" nil  nil   nil I   L   () nil nil "Korop")
            ("krq" nil  nil   t   nil nil () nil nil "Krui")
            ("krr" nil  nil   nil I   L   () nil nil "Kru'ng 2")
            ("krs" nil  nil   nil I   L   () nil nil "Gbaya (Sudan)")
            ("krt" nil  nil   nil I   L   () nil nil "Kanuri, Tumari")
            ("kru" nil  "kru" nil I   L   () nil nil "Kurukh")
            ("krv" nil  nil   nil I   L   () nil nil "Kravet")
            ("krw" nil  nil   nil I   L   () nil nil "Krahn, Western")
            ("krx" nil  nil   nil I   L   () nil nil "Karon")
            ("kry" nil  nil   nil I   L   () nil nil "Kryts")
            ("krz" nil  nil   nil I   L   () nil nil "Kanum, Sota")
            ("ksa" nil  nil   t   I   L   () nil nil "Shuwa-Zamani")
            ("ksb" nil  nil   nil I   L   () nil nil "Shambala")
            ("ksc" nil  nil   nil I   L   () nil nil "Kalinga, Southern")
            ("ksd" nil  nil   nil I   L   () nil nil "Kuanua")
            ("kse" nil  nil   nil I   L   () nil nil "Kuni")
            ("ksf" nil  nil   nil I   L   () nil nil "Bafia")
            ("ksg" nil  nil   nil I   L   () nil nil "Kusaghe")
            ("ksh" nil  nil   nil I   L   () nil "Kölsch" "Colognian")
            ("ksi" nil  nil   nil I   L   () nil nil "Krisa")
            ("ksj" nil  nil   nil I   L   () nil nil "Uare")
            ("ksk" nil  nil   nil I   L   () nil "Kaáⁿze" "Kansa")
            ("ksl" nil  nil   nil I   L   () nil nil "Kumalu")
            ("ksm" nil  nil   nil I   L   () nil nil "Kumba")
            ("ksn" nil  nil   nil I   L   () nil nil "Kasiguranin")
            ("kso" nil  nil   nil I   L   () nil nil "Kofa")
            ("ksp" nil  nil   nil I   L   () nil nil "Kaba")
            ("ksq" nil  nil   nil I   L   () nil nil "Kwaami")
            ("ksr" nil  nil   nil I   L   () nil nil "Borong")
            ("kss" nil  nil   nil I   L   () nil nil "Kisi, Southern")
            ("kst" nil  nil   nil I   L   () nil nil "Winyé")
            ("ksu" nil  nil   nil I   L   () nil nil "Khamyang")
            ("ksv" nil  nil   nil I   L   () nil nil "Kusu")
            ("ksw" nil  nil   nil I   L   () nil nil "Karen, S'gaw")
            ("ksx" nil  nil   nil I   L   () nil nil "Kedang")
            ("ksy" nil  nil   nil I   L   () nil nil "Kharia Thar")
            ("ksz" nil  nil   nil I   L   () nil nil "Koraku")
            ("kta" nil  nil   nil I   L   () nil nil "Katua")
            ("ktb" nil  nil   nil I   L   () nil nil "Kambaata")
            ("ktc" nil  nil   nil I   L   () nil nil "Kholok")
            ("ktd" nil  nil   nil I   L   () nil nil "Kokata")
            ("kte" nil  nil   nil I   L   () nil nil "Nubri")
            ("ktf" nil  nil   nil I   L   () nil nil "Kwami")
            ("ktg" nil  nil   nil I   E   () nil nil "Kalkutungu")
            ("kth" nil  nil   nil I   L   () nil nil "Karanga")
            ("kti" nil  nil   nil I   L   () nil nil "Muyu, North")
            ("ktj" nil  nil   nil I   L   () nil nil "Krumen, Plapo")
            ("ktk" nil  nil   nil I   E   () nil nil "Kaniet")
            ("ktl" nil  nil   nil I   L   () nil nil "Koroshi")
            ("ktm" nil  nil   nil I   L   () nil nil "Kurti")
            ("ktn" nil  nil   nil I   L   () nil nil "Karitiâna")
            ("kto" nil  nil   nil I   L   () nil nil "Kuot")
            ("ktp" nil  nil   nil I   L   () nil nil "Kaduo")
            ("ktq" nil  nil   nil I   E   () nil nil "Katabaga")
            ("ktr" nil  nil   t   I   L   () nil nil "Kota Marudu Tinagas")
            ("kts" nil  nil   nil I   L   () nil nil "Muyu, South")
            ("ktt" nil  nil   nil I   L   () nil nil "Ketum")
            ("ktu" nil  nil   nil I   L   () nil nil "Kituba (Democratic Republic of Congo)")
            ("ktv" nil  nil   nil I   L   () nil nil "Katu, Eastern")
            ("ktw" nil  nil   nil I   E   () nil nil "Kato")
            ("ktx" nil  nil   nil I   L   () nil nil "Kaxararí")
            ("kty" nil  nil   nil I   L   () nil nil "Kango (Bas-Uélé District)")
            ("ktz" nil  nil   nil I   L   () nil nil "Ju/'hoan")
            ("kua" "kj" "kua" nil I   L   () "Niger–Congo" "kuanyama" "Kuanyama")
            ("kub" nil  nil   nil I   L   () nil nil "Kutep")
            ("kuc" nil  nil   nil I   L   () nil nil "Kwinsu")
            ("kud" nil  nil   nil I   L   () nil nil "'Auhelawa")
            ("kue" nil  nil   nil I   L   () nil nil "Kuman")
            ("kuf" nil  nil   nil I   L   () nil nil "Katu, Western")
            ("kug" nil  nil   nil I   L   () nil nil "Kupa")
            ("kuh" nil  nil   nil I   L   () nil nil "Kushi")
            ("kui" nil  nil   nil I   L   () nil nil "Kuikúro-Kalapálo")
            ("kuj" nil  nil   nil I   L   () nil nil "Kuria")
            ("kuk" nil  nil   nil I   L   () nil nil "Kepo'")
            ("kul" nil  nil   nil I   L   () nil nil "Kulere")
            ("kum" nil  "kum" nil I   L   () nil "Къумукъ" "Kumyk")
            ("kun" nil  nil   nil I   L   () nil nil "Kunama")
            ("kuo" nil  nil   nil I   L   () nil nil "Kumukio")
            ("kup" nil  nil   nil I   L   () nil nil "Kunimaipa")
            ("kuq" nil  nil   nil I   L   () nil nil "Karipuná")
            ("kur" "ku" "kur" nil M   L   ("ckb" "kmr" "sdh") "Indo-European" "Kurdî" "Kurdish")
            ("kus" nil  nil   nil I   L   () nil nil "Kusaal")
            ("kut" nil  "kut" nil I   L   () nil "Ktunaxa" "Kutenai")
            ("kuu" nil  nil   nil I   L   () nil nil "Kuskokwim, Upper")
            ("kuv" nil  nil   nil I   L   () nil nil "Kur")
            ("kuw" nil  nil   nil I   L   () nil nil "Kpagua")
            ("kux" nil  nil   nil I   L   () nil nil "Kukatja")
            ("kuy" nil  nil   nil I   L   () nil nil "Kuuku-Ya'u")
            ("kuz" nil  nil   nil I   E   () nil nil "Kunza")
            ("kva" nil  nil   nil I   L   () nil nil "Bagvalal")
            ("kvb" nil  nil   nil I   L   () nil nil "Kubu")
            ("kvc" nil  nil   nil I   L   () nil nil "Kove")
            ("kvd" nil  nil   nil I   L   () nil nil "Kui (Indonesia)")
            ("kve" nil  nil   nil I   L   () nil nil "Kalabakan")
            ("kvf" nil  nil   nil I   L   () nil nil "Kabalai")
            ("kvg" nil  nil   nil I   L   () nil nil "Kuni-Boazi")
            ("kvh" nil  nil   nil I   L   () nil nil "Komodo")
            ("kvi" nil  nil   nil I   L   () nil nil "Kwang")
            ("kvj" nil  nil   nil I   L   () nil nil "Psikye")
            ("kvk" nil  nil   nil I   L   () nil nil "Korean Sign Language")
            ("kvl" nil  nil   nil I   L   () nil nil "Karen, Brek")
            ("kvm" nil  nil   nil I   L   () nil nil "Kendem")
            ("kvn" nil  nil   nil I   L   () nil nil "Kuna, Border")
            ("kvo" nil  nil   nil I   L   () nil nil "Dobel")
            ("kvp" nil  nil   nil I   L   () nil nil "Kompane")
            ("kvq" nil  nil   nil I   L   () nil nil "Karen, Geba")
            ("kvr" nil  nil   nil I   L   () nil nil "Kerinci")
            ("kvs" nil  nil   t   I   L   () nil nil "Kunggara")
            ("kvt" nil  nil   nil I   L   () nil nil "Karen, Lahta")
            ("kvu" nil  nil   nil I   L   () nil nil "Karen, Yinbaw")
            ("kvv" nil  nil   nil I   L   () nil nil "Kola")
            ("kvw" nil  nil   nil I   L   () nil nil "Wersing")
            ("kvx" nil  nil   nil I   L   () nil nil "Koli, Parkari")
            ("kvy" nil  nil   nil I   L   () nil nil "Karen, Yintale")
            ("kvz" nil  nil   nil I   L   () nil nil "Tsakwambo")
            ("kwa" nil  nil   nil I   L   () nil nil "Dâw")
            ("kwb" nil  nil   nil I   L   () nil nil "Kwa")
            ("kwc" nil  nil   nil I   L   () nil nil "Likwala")
            ("kwd" nil  nil   nil I   L   () nil nil "Kwaio")
            ("kwe" nil  nil   nil I   L   () nil nil "Kwerba")
            ("kwf" nil  nil   nil I   L   () nil nil "Kwara'ae")
            ("kwg" nil  nil   nil I   L   () nil nil "Kaba Deme")
            ("kwh" nil  nil   nil I   L   () nil nil "Kowiai")
            ("kwi" nil  nil   nil I   L   () nil nil "Awa-Cuaiquer")
            ("kwj" nil  nil   nil I   L   () nil nil "Kwanga")
            ("kwk" nil  nil   nil I   L   () nil "Kwak̓wala" "Kwakiutl")
            ("kwl" nil  nil   nil I   L   () nil nil "Kofyar")
            ("kwm" nil  nil   nil I   L   () nil nil "Kwambi")
            ("kwn" nil  nil   nil I   L   () nil nil "Kwangali")
            ("kwo" nil  nil   nil I   L   () nil nil "Kwomtari")
            ("kwp" nil  nil   nil I   L   () nil nil "Kodia")
            ("kwq" nil  nil   t   I   L   () nil nil "Kwak")
            ("kwr" nil  nil   nil I   L   () nil nil "Kwer")
            ("kws" nil  nil   nil I   L   () nil nil "Kwese")
            ("kwt" nil  nil   nil I   L   () nil nil "Kwesten")
            ("kwu" nil  nil   nil I   L   () nil nil "Kwakum")
            ("kwv" nil  nil   nil I   L   () nil nil "Kaba Na")
            ("kww" nil  nil   nil I   L   () nil nil "Kwinti")
            ("kwx" nil  nil   nil I   L   () nil nil "Khirwar")
            ("kwy" nil  nil   nil I   L   () nil nil "Kongo, San Salvador")
            ("kwz" nil  nil   nil I   E   () nil nil "Kwadi")
            ("kxa" nil  nil   nil I   L   () nil nil "Kairiru")
            ("kxb" nil  nil   nil I   L   () nil nil "Krobu")
            ("kxc" nil  nil   nil I   L   () nil nil "Komso")
            ("kxd" nil  nil   nil I   L   () nil nil "Brunei")
            ("kxe" nil  nil   t   I   L   () nil nil "Kakihum")
            ("kxf" nil  nil   nil I   L   () nil nil "Karen, Manumanaw")
            ("kxg" nil  nil   t   nil nil () nil nil "Katingan")
            ("kxh" nil  nil   nil I   L   () nil nil "Karo (Ethiopia)")
            ("kxi" nil  nil   nil I   L   () nil nil "Keningau Murut")
            ("kxj" nil  nil   nil I   L   () nil nil "Kulfa")
            ("kxk" nil  nil   nil I   L   () nil nil "Karen, Zayein")
            ("kxl" nil  nil   t   I   L   () nil nil "Kurux, Nepali")
            ("kxm" nil  nil   nil I   L   () nil "ខ្មែរលើ" "Khmer, Northern")
            ("kxn" nil  nil   nil I   L   () nil nil "Kanowit")
            ("kxo" nil  nil   nil I   E   () nil nil "Kanoé")
            ("kxp" nil  nil   nil I   L   () nil nil "Koli, Wadiyara")
            ("kxq" nil  nil   nil I   L   () nil nil "Kanum, Smärky")
            ("kxr" nil  nil   nil I   L   () nil nil "Koro (Papua New Guinea)")
            ("kxs" nil  nil   nil I   L   () nil nil "Kangjia")
            ("kxt" nil  nil   nil I   L   () nil nil "Koiwat")
            ("kxu" nil  nil   t   I   L   () nil nil "Kui (India)")
            ("kxv" nil  nil   nil I   L   () nil nil "Kuvi")
            ("kxw" nil  nil   nil I   L   () nil nil "Konai")
            ("kxx" nil  nil   nil I   L   () nil nil "Likuba")
            ("kxy" nil  nil   nil I   L   () nil nil "Kayong")
            ("kxz" nil  nil   nil I   L   () nil nil "Kerewo")
            ("kya" nil  nil   nil I   L   () nil nil "Kwaya")
            ("kyb" nil  nil   nil I   L   () nil nil "Kalinga, Butbut")
            ("kyc" nil  nil   nil I   L   () nil nil "Kyaka")
            ("kyd" nil  nil   nil I   L   () nil nil "Karey")
            ("kye" nil  nil   nil I   L   () nil nil "Krache")
            ("kyf" nil  nil   nil I   L   () nil nil "Kouya")
            ("kyg" nil  nil   nil I   L   () nil nil "Keyagana")
            ("kyh" nil  nil   nil I   L   () nil nil "Karok")
            ("kyi" nil  nil   nil I   L   () nil nil "Kiput")
            ("kyj" nil  nil   nil I   L   () nil nil "Karao")
            ("kyk" nil  nil   nil I   L   () nil nil "Kamayo")
            ("kyl" nil  nil   nil I   L   () nil nil "Kalapuya")
            ("kym" nil  nil   nil I   L   () nil nil "Kpatili")
            ("kyn" nil  nil   nil I   L   () nil nil "Karolanos")
            ("kyo" nil  nil   nil I   L   () nil nil "Kelon")
            ("kyp" nil  nil   nil I   L   () nil nil "Kang")
            ("kyq" nil  nil   nil I   L   () nil nil "Kenga")
            ("kyr" nil  nil   nil I   L   () nil nil "Kuruáya")
            ("kys" nil  nil   nil I   L   () nil nil "Kayan, Baram")
            ("kyt" nil  nil   nil I   L   () nil nil "Kayagar")
            ("kyu" nil  nil   nil I   L   () nil nil "Kayah, Western")
            ("kyv" nil  nil   nil I   L   () nil nil "Kayort")
            ("kyw" nil  nil   nil I   L   () nil nil "Kudmali")
            ("kyx" nil  nil   nil I   L   () nil nil "Rapoisi")
            ("kyy" nil  nil   nil I   L   () nil nil "Kambaira")
            ("kyz" nil  nil   nil I   L   () nil nil "Kayabí")
            ("kza" nil  nil   nil I   L   () nil nil "Karaboro, Western")
            ("kzb" nil  nil   nil I   L   () nil nil "Kaibobo")
            ("kzc" nil  nil   nil I   L   () nil nil "Kulango, Bondoukou")
            ("kzd" nil  nil   nil I   L   () nil nil "Kadai")
            ("kze" nil  nil   nil I   L   () nil nil "Kosena")
            ("kzf" nil  nil   nil I   L   () nil nil "Kaili, Da'a")
            ("kzg" nil  nil   nil I   L   () nil nil "Kikai")
            ("kzh" nil  nil   t   I   L   () nil nil "Kenuzi-Dongola")
            ("kzi" nil  nil   nil I   L   () nil nil "Kelabit")
            ("kzj" nil  nil   t   I   L   () nil nil "Kadazan, Coastal")
            ("kzk" nil  nil   nil I   E   () nil nil "Kazukuru")
            ("kzl" nil  nil   nil I   L   () nil nil "Kayeli")
            ("kzm" nil  nil   nil I   L   () nil nil "Kais")
            ("kzn" nil  nil   nil I   L   () nil nil "Kokola")
            ("kzo" nil  nil   nil I   L   () nil nil "Kaningi")
            ("kzp" nil  nil   nil I   L   () nil nil "Kaidipang")
            ("kzq" nil  nil   nil I   L   () nil nil "Kaike")
            ("kzr" nil  nil   nil I   L   () nil nil "Karang")
            ("kzs" nil  nil   nil I   L   () nil nil "Dusun, Sugut")
            ("kzt" nil  nil   t   I   L   () nil nil "Dusun, Tambunan")
            ("kzu" nil  nil   nil I   L   () nil nil "Kayupulau")
            ("kzv" nil  nil   nil I   L   () nil nil "Komyandaret")
            ("kzw" nil  nil   nil I   E   () nil nil "Karirí-Xocó")
            ("kzx" nil  nil   nil I   E   () nil nil "Kamarian")
            ("kzy" nil  nil   nil I   L   () nil nil "Kango (Tshopo District)")
            ("kzz" nil  nil   nil I   L   () nil nil "Kalabra")
            ;; L
            ("laa" nil  nil   nil I   L   () nil nil "Subanun, Lapuyan")
            ("lab" nil  nil   nil I   A   () nil nil "Linear A")
            ("lac" nil  nil   nil I   L   () nil nil "Lacandon")
            ("lad" nil  "lad" nil I   L   () nil "ג'ודיאו–איספאנייול" "Ladino")
            ("lae" nil  nil   nil I   L   () nil nil "Pattani")
            ("laf" nil  nil   nil I   L   () nil nil "Lafofa")
            ("lag" nil  nil   nil I   L   () nil nil "Langi")
            ("lah" nil  "lah" nil M   L   ("ltg" "lvs") nil "ਲਹਿੰਦੀ" "Lahnda")
            ("lai" nil  nil   nil I   L   () nil nil "Lambya")
            ("laj" nil  nil   nil I   L   () nil nil "Lango (Uganda)")
            ("lak" nil  nil   t   I   L   () nil nil "Laka (Nigeria)")
            ("lal" nil  nil   nil I   L   () nil nil "Lalia")
            ("lam" nil  "lam" nil I   L   () nil nil "Lamba")
            ("lan" nil  nil   nil I   L   () nil nil "Laru")
            ("lao" "lo" "lao" nil I   L   () "Tai–Kadai" "ພາສາລາວ" "Lao")
            ("lap" nil  nil   nil I   L   () nil nil "Laka (Chad)")
            ("laq" nil  nil   nil I   L   () nil nil "Qabiao")
            ("lar" nil  nil   nil I   L   () nil nil "Larteh")
            ("las" nil  nil   nil I   L   () nil nil "Lama (Togo)")
            ("lat" "la" "lat" nil I   A   () "Indo-European" "lingua Latīna" "Latin")
            ("lau" nil  nil   nil I   L   () nil nil "Laba")
            ("lav" "lv" "lav" nil M   L   () "Indo-European" "latviešu" "Latvian")
            ("law" nil  nil   nil I   L   () nil nil "Lauje")
            ("lax" nil  nil   nil I   L   () nil nil "Tiwa")
            ("lay" nil  nil   nil I   L   () nil nil "Lama (Myanmar)")
            ("laz" nil  nil   nil I   E   () nil nil "Aribwatsa")
            ("lba" nil  nil   t   I   E   () nil nil "Lui")
            ("lbb" nil  nil   nil I   L   () nil nil "Label")
            ("lbc" nil  nil   nil I   L   () nil nil "Lakkia")
            ("lbe" nil  nil   nil I   L   () nil "лакку" "Lak")
            ("lbf" nil  nil   nil I   L   () nil nil "Tinani")
            ("lbg" nil  nil   nil I   L   () nil nil "Laopang")
            ("lbi" nil  nil   nil I   L   () nil nil "La'bi")
            ("lbj" nil  nil   nil I   L   () nil nil "Ladakhi")
            ("lbk" nil  nil   nil I   L   () nil nil "Central Bontok")
            ("lbl" nil  nil   nil I   L   () nil nil "Libon Bikol")
            ("lbm" nil  nil   nil I   L   () nil nil "Lodhi")
            ("lbn" nil  nil   nil I   L   () nil nil "Lamet")
            ("lbo" nil  nil   nil I   L   () nil nil "Laven")
            ("lbq" nil  nil   nil I   L   () nil nil "Wampar")
            ("lbr" nil  nil   nil I   L   () nil nil "Lorung, Northern")
            ("lbs" nil  nil   nil I   L   () nil nil "Libyan Sign Language")
            ("lbt" nil  nil   nil I   L   () nil nil "Lachi")
            ("lbu" nil  nil   nil I   L   () nil nil "Labu")
            ("lbv" nil  nil   nil I   L   () nil nil "Lavatbura-Lamusong")
            ("lbw" nil  nil   nil I   L   () nil nil "Tolaki")
            ("lbx" nil  nil   nil I   L   () nil nil "Lawangan")
            ("lby" nil  nil   nil I   E   () nil nil "Lamu-Lamu")
            ("lbz" nil  nil   nil I   L   () nil nil "Lardil")
            ("lcc" nil  nil   nil I   L   () nil nil "Legenyem")
            ("lcd" nil  nil   nil I   L   () nil nil "Lola")
            ("lce" nil  nil   nil I   L   () nil nil "Loncong")
            ("lcf" nil  nil   nil I   L   () nil nil "Lubu")
            ("lch" nil  nil   nil I   L   () nil nil "Luchazi")
            ("lcl" nil  nil   nil I   L   () nil nil "Lisela")
            ("lcm" nil  nil   nil I   L   () nil nil "Tungag")
            ("lcp" nil  nil   nil I   L   () nil nil "Lawa, Western")
            ("lcq" nil  nil   nil I   L   () nil nil "Luhu")
            ("lcs" nil  nil   nil I   L   () nil nil "Lisabata-Nuniali")
            ("lda" nil  nil   nil I   L   () nil nil "Kla-Dan")
            ("ldb" nil  nil   nil I   L   () nil nil "Idun")
            ("ldd" nil  nil   nil I   L   () nil nil "Luri")
            ("ldg" nil  nil   nil I   L   () nil nil "Lenyima")
            ("ldh" nil  nil   nil I   L   () nil nil "Lamja-Dengsa-Tola")
            ("ldi" nil  nil   nil I   L   () nil nil "Laari")
            ("ldj" nil  nil   nil I   L   () nil nil "Lemoro")
            ("ldk" nil  nil   nil I   L   () nil nil "Leelau")
            ("ldl" nil  nil   nil I   L   () nil nil "Kaan")
            ("ldm" nil  nil   nil I   L   () nil nil "Landoma")
            ("ldn" nil  nil   nil I   C   () nil nil "Láadan")
            ("ldo" nil  nil   nil I   L   () nil nil "Loo")
            ("ldp" nil  nil   nil I   L   () nil nil "Tso")
            ("ldq" nil  nil   nil I   L   () nil nil "Lufu")
            ("lea" nil  nil   nil I   L   () nil nil "Lega-Shabunda")
            ("leb" nil  nil   nil I   L   () nil nil "Lala-Bisa")
            ("lec" nil  nil   nil I   L   () nil nil "Leco")
            ("led" nil  nil   nil I   L   () nil nil "Lendu")
            ("lee" nil  nil   nil I   L   () nil nil "Lyélé")
            ("lef" nil  nil   nil I   L   () nil nil "Lelemi")
            ("leg" nil  nil   t   I   L   () nil nil "Lengua")
            ("leh" nil  nil   nil I   L   () nil nil "Lenje")
            ("lei" nil  nil   nil I   L   () nil nil "Lemio")
            ("lej" nil  nil   nil I   L   () nil nil "Lengola")
            ("lek" nil  nil   nil I   L   () nil nil "Leipon")
            ("lel" nil  nil   nil I   L   () nil nil "Lele (Democratic Republic of Congo)")
            ("lem" nil  nil   nil I   L   () nil nil "Nomaande")
            ("len" nil  nil   nil I   E   () nil nil "Lenca")
            ("leo" nil  nil   nil I   L   () nil nil "Leti (Cameroon)")
            ("lep" nil  nil   nil I   L   () nil nil "Lepcha")
            ("leq" nil  nil   nil I   L   () nil nil "Lembena")
            ("ler" nil  nil   nil I   L   () nil nil "Lenkau")
            ("les" nil  nil   nil I   L   () nil nil "Lese")
            ("let" nil  nil   nil I   L   () nil nil "Lesing-Gelimi")
            ("leu" nil  nil   nil I   L   () nil nil "Kara (Papua New Guinea)")
            ("lev" nil  nil   nil I   L   () nil nil "Lamma")
            ("lew" nil  nil   nil I   L   () nil nil "Kaili, Ledo")
            ("lex" nil  nil   nil I   L   () nil nil "Luang")
            ("ley" nil  nil   nil I   L   () nil nil "Lemolang")
            ("lez" nil  "lez" nil I   L   () nil "лезги" "Lezghian")
            ("lfa" nil  nil   nil I   L   () nil nil "Lefa")
            ("lfn" nil  nil   nil I   C   () nil nil "Lingua Franca Nova")
            ("lga" nil  nil   nil I   L   () nil nil "Lungga")
            ("lgb" nil  nil   nil I   L   () nil nil "Laghu")
            ("lgg" nil  nil   nil I   L   () nil nil "Lugbara")
            ("lgh" nil  nil   nil I   L   () nil nil "Laghuu")
            ("lgi" nil  nil   nil I   L   () nil nil "Lengilu")
            ("lgk" nil  nil   nil I   L   () nil nil "Lingarak")
            ("lgl" nil  nil   nil I   L   () nil nil "Wala")
            ("lgm" nil  nil   nil I   L   () nil nil "Lega-Mwenga")
            ("lgn" nil  nil   nil I   L   () nil nil "Opuuo")
            ("lgo" nil  nil   nil I   L   () "Nilo-Saharan" nil "Lango")
            ("lgq" nil  nil   nil I   L   () nil nil "Logba")
            ("lgr" nil  nil   nil I   L   () nil nil "Lengo")
            ("lgs" nil  nil   nil I   L   () nil nil "Guinea-Bissau Sign Language")
            ("lgt" nil  nil   nil I   L   () nil nil "Pahi")
            ("lgu" nil  nil   nil I   L   () nil nil "Longgu")
            ("lgz" nil  nil   nil I   L   () nil nil "Ligenza")
            ("lha" nil  nil   nil I   L   () nil nil "Laha (Viet Nam)")
            ("lhh" nil  nil   nil I   L   () nil nil "Laha (Indonesia)")
            ("lhi" nil  nil   nil I   L   () nil nil "Lahu Shi")
            ("lhl" nil  nil   nil I   L   () nil nil "Lohar, Lahul")
            ("lhm" nil  nil   nil I   L   () nil nil "Lhomi")
            ("lhn" nil  nil   nil I   L   () nil nil "Lahanan")
            ("lhp" nil  nil   nil I   L   () nil nil "Lhokpu")
            ("lhs" nil  nil   nil I   E   () nil nil "Mlahsö")
            ("lht" nil  nil   nil I   L   () nil nil "Toga")
            ("lhu" nil  nil   nil I   L   () nil "Laˇhuˍ hkawˇ" "Lahu")
            ("lia" nil  nil   nil I   L   () nil nil "Limba, West-Central")
            ("lib" nil  nil   nil I   L   () nil nil "Likum")
            ("lic" nil  nil   nil I   L   () nil nil "Hlai")
            ("lid" nil  nil   nil I   L   () nil nil "Nyindrou")
            ("lie" nil  nil   nil I   L   () nil nil "Likila")
            ("lif" nil  nil   nil I   L   () nil "ᤛᤡᤖᤡᤈᤨᤅ" "Limbu")
            ("lig" nil  nil   nil I   L   () nil nil "Ligbi")
            ("lih" nil  nil   nil I   L   () nil nil "Lihir")
            ("lii" nil  nil   t   I   L   () nil nil "Lingkhim")
            ("lij" nil  nil   nil I   L   () nil "líguru" "Ligurian")
            ("lik" nil  nil   nil I   L   () nil nil "Lika")
            ("lil" nil  nil   nil I   L   () nil "Sƛ’aƛ’imxǝc" "Lillooet")
            ("lim" "li" "lim" nil I   L   () "Indo-European" "Lèmburgs" "Limburgish, Limburgan")
            ("lin" "ln" "lin" nil I   L   () "Niger–Congo" "Lingala" "Lingala")
            ("lio" nil  nil   nil I   L   () nil nil "Liki")
            ("lip" nil  nil   nil I   L   () nil nil "Sekpele")
            ("liq" nil  nil   nil I   L   () nil nil "Libido")
            ("lir" nil  nil   nil I   L   () nil nil "Liberian English")
            ("lis" nil  nil   nil I   L   () nil nil "Lisu")
            ("lit" "lt" "lit" nil I   L   () "Indo-European" "lietuvių" "Lithuanian")
            ("liu" nil  nil   nil I   L   () nil nil "Logorik")
            ("liv" nil  nil   nil I   L   () nil nil "Liv(onian)")
            ("liw" nil  nil   nil I   L   () nil nil "Lembak")
            ("lix" nil  nil   nil I   L   () nil nil "Liabuku")
            ("liy" nil  nil   nil I   L   () nil nil "Banda-Bambari")
            ("liz" nil  nil   nil I   L   () nil nil "Libinza")
            ("lja" nil  nil   nil I   E   () nil nil "Golpa")
            ("lje" nil  nil   nil I   L   () nil nil "Rampi")
            ("lji" nil  nil   nil I   L   () nil nil "Laiyolo")
            ("ljl" nil  nil   nil I   L   () nil nil "Li'o")
            ("ljp" nil  nil   nil I   L   () nil nil "Lampung")
            ("ljw" nil  nil   nil I   L   () nil nil "Yirandali")
            ("ljx" nil  nil   nil I   E   () nil nil "Yuru")
            ("lka" nil  nil   nil I   L   () nil nil "Lakalei")
            ("lkb" nil  nil   nil I   L   () nil nil "Kabras")
            ("lkc" nil  nil   nil I   L   () nil nil "Kucong")
            ("lkd" nil  nil   nil I   L   () nil nil "Lakondê")
            ("lke" nil  nil   nil I   L   () nil nil "Kenyi")
            ("lkh" nil  nil   nil I   L   () nil nil "Lakha")
            ("lki" nil  nil   nil I   L   () nil "له کی له کستان" "Laki")
            ("lkj" nil  nil   nil I   L   () nil nil "Remun")
            ("lkl" nil  nil   nil I   L   () nil nil "Laeko-Libuat")
            ("lkm" nil  nil   nil I   E   () nil nil "Kalaamaya")
            ("lkn" nil  nil   nil I   L   () nil nil "Lakona")
            ("lko" nil  nil   nil I   L   () nil nil "Khayo")
            ("lkr" nil  nil   nil I   L   () nil nil "Päri")
            ("lks" nil  nil   nil I   L   () nil nil "Kisa")
            ("lkt" nil  nil   nil I   L   () nil "Lakȟótiyapi" "Lakota")
            ("lku" nil  nil   nil I   E   () nil nil "Kungkari")
            ("lky" nil  nil   nil I   L   () nil nil "Lokoya")
            ("lla" nil  nil   nil I   L   () nil nil "Lala-Roba")
            ("llb" nil  nil   nil I   L   () nil nil "Lolo")
            ("llc" nil  nil   nil I   L   () nil nil "Lele (Guinea)")
            ("lld" nil  nil   nil I   L   () nil "ladin" "Ladin")
            ("lle" nil  nil   nil I   L   () nil nil "Lele (Papua New Guinea)")
            ("llf" nil  nil   nil I   E   () nil nil "Hermit")
            ("llg" nil  nil   nil I   L   () nil nil "Lole")
            ("llh" nil  nil   nil I   L   () nil nil "Lamu")
            ("lli" nil  nil   nil I   L   () nil nil "Teke-Laali")
            ("llj" nil  nil   nil I   E   () nil nil "Ladji Ladji")
            ("llk" nil  nil   nil I   E   () nil nil "Lelak")
            ("lll" nil  nil   nil I   L   () nil nil "Lilau")
            ("llm" nil  nil   nil I   L   () nil nil "Lasalimu")
            ("lln" nil  nil   nil I   L   () nil nil "Lele (Chad)")
            ("llo" nil  nil   t   I   L   () nil nil "Khlor")
            ("llp" nil  nil   nil I   L   () nil nil "Efate, North")
            ("llq" nil  nil   nil I   L   () nil nil "Lolak")
            ("lls" nil  nil   nil I   L   () nil nil "Lithuanian Sign Language")
            ("llu" nil  nil   nil I   L   () nil nil "Lau")
            ("llx" nil  nil   nil I   L   () nil nil "Lauan")
            ("lma" nil  nil   nil I   L   () nil nil "Limba, East")
            ("lmb" nil  nil   nil I   L   () nil nil "Merei")
            ("lmc" nil  nil   nil I   E   () nil nil "Limilngan")
            ("lmd" nil  nil   nil I   L   () nil nil "Lumun")
            ("lme" nil  nil   nil I   L   () nil nil "Pévé")
            ("lmf" nil  nil   nil I   L   () nil nil "Lembata, South")
            ("lmg" nil  nil   nil I   L   () nil nil "Lamogai")
            ("lmh" nil  nil   nil I   L   () nil nil "Lambichhong")
            ("lmi" nil  nil   nil I   L   () nil nil "Lombi")
            ("lmj" nil  nil   nil I   L   () nil nil "Lembata, West")
            ("lmk" nil  nil   nil I   L   () nil nil "Lamkang")
            ("lml" nil  nil   nil I   L   () nil nil "Hano")
            ("lmm" nil  nil   t   I   L   () nil nil "Lamam")
            ("lmn" nil  nil   nil I   L   () nil nil "Lambadi")
            ("lmo" nil  nil   nil I   L   () nil "lumbard" "Lombard")
            ("lmp" nil  nil   nil I   L   () nil nil "Limbum")
            ("lmq" nil  nil   nil I   L   () nil nil "Lamatuka")
            ("lmr" nil  nil   nil I   L   () nil nil "Lamalera")
            ("lms" nil  nil   t   nil nil () nil nil "Limousin")
            ("lmt" nil  nil   t   nil nil () nil nil "Lematang")
            ("lmu" nil  nil   nil I   L   () nil nil "Lamenu")
            ("lmv" nil  nil   nil I   L   () nil nil "Lomaiviti")
            ("lmw" nil  nil   nil I   L   () nil nil "Miwok, Lake")
            ("lmx" nil  nil   nil I   L   () nil nil "Laimbue")
            ("lmy" nil  nil   nil I   L   () nil nil "Lamboya")
            ("lmz" nil  nil   t   I   E   () nil nil "Lumbee")
            ("lna" nil  nil   nil I   L   () nil nil "Langbashe")
            ("lnb" nil  nil   nil I   L   () nil nil "Mbalanhu")
            ("lnc" nil  nil   t   nil nil () nil nil "Languedocien")
            ("lnd" nil  nil   nil I   L   () nil nil "Lundayeh")
            ("lng" nil  nil   nil I   A   () nil nil "Langobardic")
            ("lnh" nil  nil   nil I   L   () nil nil "Lanoh")
            ("lni" nil  nil   nil I   L   () nil nil "Lantanai")
            ("lnj" nil  nil   nil I   E   () nil nil "Leningitij")
            ("lnl" nil  nil   nil I   L   () nil nil "Banda, South Central")
            ("lnm" nil  nil   nil I   L   () nil nil "Langam")
            ("lnn" nil  nil   nil I   L   () nil nil "Lorediakarkar")
            ("lno" nil  nil   t   I   L   () nil nil "Lango (Sudan)")
            ("lns" nil  nil   nil I   L   () nil nil "Lamnso'")
            ("lnt" nil  nil   t   nil nil () nil nil "Lintang")
            ("lnu" nil  nil   nil I   L   () nil nil "Longuda")
            ("lnw" nil  nil   nil I   E   () nil nil "Lanima")
            ("lnz" nil  nil   nil I   L   () nil nil "Lonzo")
            ("loa" nil  nil   nil I   L   () nil nil "Loloda")
            ("lob" nil  nil   nil I   L   () nil nil "Lobi")
            ("loc" nil  nil   nil I   L   () nil nil "Inonhan")
            ("lod" nil  nil   t   nil nil () nil nil "Berawan")
            ("loe" nil  nil   nil I   L   () nil nil "Saluan, Coastal")
            ("lof" nil  nil   nil I   L   () nil nil "Logol")
            ("log" nil  nil   nil I   L   () nil nil "Logo")
            ("loh" nil  nil   nil I   L   () nil nil "Narim")
            ("loi" nil  nil   nil I   L   () nil nil "Loma (Côte d'Ivoire)")
            ("loj" nil  nil   nil I   L   () nil nil "Lou")
            ("lok" nil  nil   nil I   L   () nil nil "Loko")
            ("lol" nil  "lol" nil I   L   () nil nil "Mongo")
            ("lom" nil  nil   nil I   L   () nil "Lö(g)ömàgòòi" "Loma (Liberia)")
            ("lon" nil  nil   nil I   L   () nil nil "Lomwe, Malawi")
            ("loo" nil  nil   nil I   L   () nil nil "Lombo")
            ("lop" nil  nil   nil I   L   () nil nil "Lopa")
            ("loq" nil  nil   nil I   L   () nil nil "Lobala")
            ("lor" nil  nil   nil I   L   () nil nil "Téén")
            ("los" nil  nil   nil I   L   () nil nil "Loniu")
            ("lot" nil  nil   nil I   L   () nil nil "Otuho")
            ("lou" nil  nil   nil I   L   () nil nil "Louisiana Creole French")
            ("lov" nil  nil   nil I   L   () nil nil "Lopi")
            ("low" nil  nil   nil I   L   () nil nil "Lobu, Tampias")
            ("lox" nil  nil   nil I   L   () nil nil "Loun")
            ("loy" nil  nil   nil I   L   () nil nil "Lowa")
            ("loz" nil  "loz" nil I   L   () nil "siLozi" "Lozi")
            ("lpa" nil  nil   nil I   L   () nil nil "Lelepa")
            ("lpe" nil  nil   nil I   L   () nil nil "Lepki")
            ("lpn" nil  nil   nil I   L   () nil nil "Long Phuri Naga")
            ("lpo" nil  nil   nil I   L   () nil nil "Lipo")
            ("lpx" nil  nil   nil I   L   () nil nil "Lopit")
            ("lqr" nil  nil   nil I   L   () "Nilo-Saharan" nil "Logir")
            ("lra" nil  nil   nil I   L   () nil nil "Lara'")
            ("lrc" nil  nil   nil I   L   () nil nil "Luri, Northern")
            ("lre" nil  nil   nil I   E   () nil nil "Laurentian")
            ("lrg" nil  nil   nil I   E   () nil nil "Laragia")
            ("lri" nil  nil   nil I   L   () nil nil "Marachi")
            ("lrk" nil  nil   nil I   L   () nil nil "Loarki")
            ("lrl" nil  nil   nil I   L   () nil nil "Lari")
            ("lrm" nil  nil   nil I   L   () nil nil "Marama")
            ("lrn" nil  nil   nil I   L   () nil nil "Lorang")
            ("lro" nil  nil   nil I   L   () nil nil "Laro")
            ("lrr" nil  nil   nil I   L   () nil nil "Lorung, Southern")
            ("lrt" nil  nil   nil I   L   () nil nil "Larantuka Malay")
            ("lrv" nil  nil   nil I   L   () nil nil "Larevat")
            ("lrz" nil  nil   nil I   L   () nil nil "Lemerig")
            ("lsa" nil  nil   nil I   L   () nil nil "Lasgerdi")
            ("lsb" nil  nil   nil I   L   () nil nil "Burundian Sign Language")
            ("lsc" nil  nil   nil I   L   () nil nil "Albarradas Sign Language")
            ("lsd" nil  nil   nil I   L   () nil nil "Lishana Deni")
            ("lse" nil  nil   nil I   L   () nil nil "Lusengo")
            ("lsg" nil  nil   t   I   L   () nil nil "Lyons Sign Language")
            ("lsh" nil  nil   nil I   L   () nil nil "Lish")
            ("lsi" nil  nil   nil I   L   () nil nil "Lashi")
            ("lsl" nil  nil   nil I   L   () nil nil "Latvian Sign Language")
            ("lsm" nil  nil   nil I   L   () nil nil "Saamia")
            ("lsn" nil  nil   nil I   L   () nil nil "Tibetan Sign Language")
            ("lso" nil  nil   nil I   L   () nil nil "Laos Sign Language")
            ("lsp" nil  nil   nil I   L   () nil nil "Panamanian Sign Language")
            ("lsr" nil  nil   nil I   L   () nil nil "Aruop")
            ("lss" nil  nil   nil I   L   () nil nil "Lasi")
            ("lst" nil  nil   nil I   L   () nil nil "Trinidad and Tobago Sign Language")
            ("lsv" nil  nil   nil I   L   () nil nil "Sivia Sign Language")
            ("lsw" nil  nil   nil I   L   () nil nil "Seychelles Sign Language")
            ("lsy" nil  nil   nil I   L   () nil nil "Mauritian Sign Language")
            ("ltc" nil  nil   nil I   H   () "Chinese" nil "Late Middle Chinese")
            ("ltg" nil  nil   nil I   L   () nil nil "Latgalian")
            ("lth" nil  nil   nil I   L   () "Nilo-Saharan" nil "Thur")
            ("lti" nil  nil   nil I   L   () nil nil "Leti (Indonesia)")
            ("ltn" nil  nil   nil I   L   () nil nil "Latundê")
            ("lto" nil  nil   nil I   L   () nil nil "Tsotso")
            ("lts" nil  nil   nil I   L   () nil nil "Tachoni")
            ("ltu" nil  nil   nil I   L   () nil nil "Latu")
            ("ltz" "lb" "ltz" nil I   L   () "Indo-European" "Lëtzebuergesch" "Luxembourgish; Letzeburgesch")
            ("lua" nil  "lua" nil I   L   () "Niger–Congo" "lwaà:" "Luba-Lulua")
            ("lub" "lu" "lub" nil I   L   () "Niger–Congo" nil "Luba-Katanga")
            ("luc" nil  nil   nil I   L   () nil nil "Aringa")
            ("lud" nil  nil   nil I   L   () nil "lüüdi" "Ludian")
            ("lue" nil  nil   nil I   L   () nil nil "Luvale")
            ("luf" nil  nil   nil I   L   () nil nil "Laua")
            ("lug" "lg" "lug" nil I   L   () "Niger–Congo" "Luganda" "Luganda")
            ("lui" nil  "lui" nil I   E   () nil nil "Luiseno")
            ("luj" nil  nil   nil I   L   () nil nil "Luna")
            ("luk" nil  nil   nil I   L   () nil nil "Lunanakha")
            ("lul" nil  nil   nil I   L   () nil nil "Olu'bo")
            ("lum" nil  nil   nil I   L   () nil nil "Luimbi")
            ("lun" nil  "lun" nil I   L   () nil "chiLunda" "Lunda")
            ("luo" nil  "luo" nil I   L   () nil "Dholuo" "Luo (Kenya and Tanzania)")
            ("lup" nil  nil   nil I   L   () nil nil "Lumbu")
            ("luq" nil  nil   nil I   L   () nil "Lucumí" "Lucumi")
            ("lur" nil  nil   nil I   L   () nil nil "Laura")
            ("lus" nil  "lus" nil I   L   () nil nil "Lushai")
            ("lut" nil  nil   nil I   L   () nil "Dəxʷləšucid" "Lushootseed")
            ("luu" nil  nil   nil I   L   () nil nil "Lumba-Yakkha")
            ("luv" nil  nil   nil I   L   () nil nil "Luwati")
            ("luw" nil  nil   nil I   L   () nil nil "Luo")
            ("luy" nil  nil   nil M   L   ("bxk" "ida"
                                           "lkb" "lko" "lks"
                                           "lri" "lrm" "lsm"
                                           "lto" "lts" "lwg"
                                           "nle" "nyd" "rag") nil nil "Luyia")
            ("luz" nil  nil   nil I   L   () nil nil "Luri, Southern")
            ("lva" nil  nil   nil I   L   () nil nil "Maku'a")
            ("lvi" nil  nil   nil I   L   () "Austronesian" nil "Lavi")
            ("lvk" nil  nil   nil I   L   () nil nil "Lavukaleve")
            ("lvl" nil  nil   nil I   L   () "Niger–Congo" nil "Lwel")
            ("lvs" nil  nil   nil I   L   () nil nil "Standard Latvian")
            ("lvu" nil  nil   nil I   L   () nil nil "Levuka")
            ("lwa" nil  nil   nil I   L   () nil nil "Lwalu")
            ("lwe" nil  nil   nil I   L   () nil nil "Lewo Eleng")
            ("lwg" nil  nil   nil I   L   () nil nil "Wanga")
            ("lwh" nil  nil   nil I   L   () nil nil "Lachi, White")
            ("lwl" nil  nil   nil I   L   () nil nil "Lawa, Eastern")
            ("lwm" nil  nil   nil I   L   () nil nil "Laomian")
            ("lwo" nil  nil   nil I   L   () nil nil "Luwo")
            ("lws" nil  nil   nil I   L   () nil nil "Malawian Sign Language")
            ("lwt" nil  nil   nil I   L   () nil nil "Lewotobi")
            ("lwu" nil  nil   nil I   L   () nil nil "Lawu")
            ("lww" nil  nil   nil I   L   () nil nil "Lewo")
            ("lxm" nil  nil   nil I   L   () "Austronesian" nil "Lakurumau")
            ("lya" nil  nil   nil I   L   () nil nil "Layakha")
            ("lyg" nil  nil   nil I   L   () nil nil "Lyngngam")
            ("lyn" nil  nil   nil I   L   () nil nil "Luyana")
            ("lzh" nil  nil   nil I   H   () "Chinese" nil "Literary Chinese")
            ("lzl" nil  nil   nil I   L   () nil nil "Litzlitz")
            ("lzn" nil  nil   nil I   L   () nil nil "Leinong Naga")
            ("lzz" nil  nil   nil I   L   () nil "ლაზური" "Laz")
            ;; M
            ("maa" nil  nil   nil I   L   () nil nil "Mazatec, San Jerónimo Tecóatl")
            ("mab" nil  nil   nil I   L   () nil nil "Mixtec, Yutanduchi")
            ("mad" nil  "mad" nil I   L   () nil "Basa Mathura" "Madurese")
            ("mae" nil  nil   nil I   L   () nil nil "Bo-Rukul")
            ("maf" nil  nil   nil I   L   () nil nil "Mafa")
            ("mag" nil  "mag" nil I   L   () nil "मगही" "Magahi")
            ("mah" "mh" "mah" nil I   L   () "Austronesian" "Kajin M̧ajeļ" "Marshall")
            ("mai" nil  "mai" nil I   L   () nil "मैथिली" "Maithili")
            ("maj" nil  nil   nil I   L   () nil nil "Mazatec, Jalapa De Díaz")
            ("mak" nil  "mak" nil I   L   () nil nil "Makasar")
            ("mal" "ml" "mal" nil I   L   () "Dravidian" "മലയാളം" "Malayalam")
            ("mam" nil  nil   nil I   L   () nil nil "Mam, Northern")
            ("man" nil  "man" nil M   L   ("emk"
                                           "mku" "mlq" "mnk"
                                           "msc" "mwk" "myq") nil nil "Mandingo")
            (nil   nil  "map" nil C   nil () nil nil "Austronesian languages")
            ("maq" nil  nil   nil I   L   () nil nil "Mazatec, Chiquihuitlán")
            ("mar" "mr" "mar" nil I   L   () "Indo-European" "मराठी" "Marathi")
            ("mas" nil  "mas" nil I   L   () nil "ɔl Maa" "Masai")
            ("mat" nil  nil   nil I   L   () nil nil "Matlatzinca, San Francisco")
            ("mau" nil  nil   nil I   L   () nil nil "Mazatec, Huautla")
            ("mav" nil  nil   nil I   L   () nil nil "Sateré-Mawé")
            ("maw" nil  nil   nil I   L   () nil nil "Mampruli")
            ("max" nil  nil   nil I   L   () nil nil "Malay, North Moluccan")
            ("maz" nil  nil   nil I   L   () nil nil "Mazahua Central")
            ("mba" nil  nil   nil I   L   () nil nil "Higaonon")
            ("mbb" nil  nil   nil I   L   () nil nil "Manobo, Western Bukidnon")
            ("mbc" nil  nil   nil I   L   () nil nil "Macushi")
            ("mbd" nil  nil   nil I   L   () nil nil "Manobo, Dibabawon")
            ("mbe" nil  nil   nil I   E   () nil nil "Molale")
            ("mbf" nil  nil   nil I   L   () nil nil "Malay, Baba")
            ("mbg" nil  nil   t   nil nil () nil nil "Northern Nambikuára")
            ("mbh" nil  nil   nil I   L   () nil nil "Mangseng")
            ("mbi" nil  nil   nil I   L   () nil nil "Manobo, Ilianen")
            ("mbj" nil  nil   nil I   L   () nil nil "Nadëb")
            ("mbk" nil  nil   nil I   L   () nil nil "Malol")
            ("mbl" nil  nil   nil I   L   () nil nil "Maxakalí")
            ("mbm" nil  nil   nil I   L   () nil nil "Ombamba")
            ("mbn" nil  nil   nil I   L   () nil nil "Macaguán")
            ("mbo" nil  nil   nil I   L   () nil nil "Mbo (Cameroon)")
            ("mbp" nil  nil   nil I   L   () nil nil "Malayo")
            ("mbq" nil  nil   nil I   L   () nil nil "Maisin")
            ("mbr" nil  nil   nil I   L   () nil nil "Nukak Makú")
            ("mbs" nil  nil   nil I   L   () nil nil "Manobo, Sarangani")
            ("mbt" nil  nil   nil I   L   () nil nil "Manobo, Matigsalug")
            ("mbu" nil  nil   nil I   L   () nil nil "Mbula-Bwazza")
            ("mbv" nil  nil   nil I   L   () nil nil "Mbulungish")
            ("mbw" nil  nil   nil I   L   () nil nil "Maring")
            ("mbx" nil  nil   nil I   L   () nil nil "Mari (East Sepik Province)")
            ("mby" nil  nil   nil I   L   () nil nil "Memoni")
            ("mbz" nil  nil   nil I   L   () nil nil "Mixtec, Amoltepec")
            ("mca" nil  nil   nil I   L   () nil nil "Maca")
            ("mcb" nil  nil   nil I   L   () nil nil "Machiguenga")
            ("mcc" nil  nil   nil I   L   () nil nil "Bitur")
            ("mcd" nil  nil   nil I   L   () nil nil "Sharanahua")
            ("mce" nil  nil   nil I   L   () nil nil "Mixtec, Itundujia")
            ("mcf" nil  nil   nil I   L   () nil nil "Matsés")
            ("mcg" nil  nil   nil I   L   () nil nil "Mapoyo")
            ("mch" nil  nil   nil I   L   () nil nil "Maquiritari")
            ("mci" nil  nil   nil I   L   () nil nil "Mese")
            ("mcj" nil  nil   nil I   L   () nil nil "Mvanip")
            ("mck" nil  nil   nil I   L   () nil nil "Mbunda")
            ("mcl" nil  nil   nil I   E   () nil nil "Macaguaje")
            ("mcm" nil  nil   nil I   L   () nil nil "Malaccan Creole Portuguese")
            ("mcn" nil  nil   nil I   L   () nil nil "Masana")
            ("mco" nil  nil   nil I   L   () nil nil "Mixe, Coatlán")
            ("mcp" nil  nil   nil I   L   () nil nil "Makaa")
            ("mcq" nil  nil   nil I   L   () nil nil "Ese")
            ("mcr" nil  nil   nil I   L   () nil nil "Menya")
            ("mcs" nil  nil   nil I   L   () nil nil "Mambai")
            ("mct" nil  nil   nil I   L   () nil nil "Mengisa")
            ("mcu" nil  nil   nil I   L   () nil nil "Mambila, Cameroon")
            ("mcv" nil  nil   nil I   L   () nil nil "Minanibai")
            ("mcw" nil  nil   nil I   L   () nil nil "Mawa (Chad)")
            ("mcx" nil  nil   nil I   L   () nil nil "Mpiemo")
            ("mcy" nil  nil   nil I   L   () nil nil "Watut, South")
            ("mcz" nil  nil   nil I   L   () nil nil "Mawan")
            ("mda" nil  nil   nil I   L   () nil nil "Mada (Nigeria)")
            ("mdb" nil  nil   nil I   L   () nil nil "Morigi")
            ("mdc" nil  nil   nil I   L   () nil nil "Male (Papua New Guinea)")
            ("mdd" nil  nil   nil I   L   () nil nil "Mbum")
            ("mde" nil  nil   nil I   L   () nil nil "Maba (Chad)")
            ("mdf" nil  "mdf" nil I   L   () nil "мокша" "Moksha")
            ("mdg" nil  nil   nil I   L   () nil nil "Massalat")
            ("mdh" nil  nil   nil I   L   () nil nil "Maguindanao")
            ("mdi" nil  nil   nil I   L   () nil nil "Mamvu")
            ("mdj" nil  nil   nil I   L   () nil nil "Mangbetu")
            ("mdk" nil  nil   nil I   L   () nil nil "Mangbutu")
            ("mdl" nil  nil   nil I   L   () nil nil "Maltese Sign Language")
            ("mdm" nil  nil   nil I   L   () nil nil "Mayogo")
            ("mdn" nil  nil   nil I   L   () nil nil "Mbati")
            ("mdo" nil  nil   t   nil nil () nil nil "Southwest Gbaya")
            ("mdp" nil  nil   nil I   L   () nil nil "Mbala")
            ("mdq" nil  nil   nil I   L   () nil nil "Mbole")
            ("mdr" nil  "mdr" nil I   L   () nil nil "Mandar")
            ("mds" nil  nil   nil I   L   () nil nil "Maria (Papua New Guinea)")
            ("mdt" nil  nil   nil I   L   () nil nil "Mbere")
            ("mdu" nil  nil   nil I   L   () nil nil "Mboko")
            ("mdv" nil  nil   nil I   L   () nil nil "Mixtec, Santa Lucía Monteverde")
            ("mdw" nil  nil   nil I   L   () nil nil "Mbosi")
            ("mdx" nil  nil   nil I   L   () nil nil "Dizi")
            ("mdy" nil  nil   nil I   L   () nil nil "Male (Ethiopia)")
            ("mdz" nil  nil   nil I   L   () nil nil "Suruí Do Pará")
            ("mea" nil  nil   nil I   L   () nil nil "Menka")
            ("meb" nil  nil   nil I   L   () nil nil "Ikobi-Mena")
            ("mec" nil  nil   nil I   L   () nil nil "Mara")
            ("med" nil  nil   nil I   L   () nil nil "Melpa")
            ("mee" nil  nil   nil I   L   () nil nil "Mengen")
            ("mef" nil  nil   nil I   L   () nil nil "Megam")
            ("meg" nil  nil   t   I   L   () nil nil "Mea")
            ("meh" nil  nil   nil I   L   () nil nil "Mixtec, Southwestern Tlaxiaco")
            ("mei" nil  nil   nil I   L   () nil nil "Midob")
            ("mej" nil  nil   nil I   L   () nil nil "Meyah")
            ("mek" nil  nil   nil I   L   () nil nil "Mekeo")
            ("mel" nil  nil   nil I   L   () nil nil "Melanau")
            ("mem" nil  nil   nil I   E   () nil nil "Mangala")
            ("men" nil  "men" nil I   L   () nil "Mɛnde" "Mende (Sierra Leone)")
            ("meo" nil  nil   nil I   L   () nil nil "Malay, Kedah")
            ("mep" nil  nil   nil I   L   () nil nil "Miriwung")
            ("meq" nil  nil   nil I   L   () nil nil "Merey")
            ("mer" nil  nil   nil I   L   () nil nil "Meru")
            ("mes" nil  nil   nil I   L   () nil nil "Masmaje")
            ("met" nil  nil   nil I   L   () nil nil "Mato")
            ("meu" nil  nil   nil I   L   () nil nil "Motu")
            ("mev" nil  nil   nil I   L   () nil nil "Mann")
            ("mew" nil  nil   nil I   L   () nil nil "Maaka")
            ("mey" nil  nil   nil I   L   () nil nil "Hassaniyya")
            ("mez" nil  nil   nil I   L   () nil "Omāēqnomenew" "Menominee")
            ("mfa" nil  nil   nil I   L   () nil nil "Malay, Pattani")
            ("mfb" nil  nil   nil I   L   () nil nil "Lom")
            ("mfc" nil  nil   nil I   L   () nil nil "Mba")
            ("mfd" nil  nil   nil I   L   () nil nil "Mendankwe-Nkwen")
            ("mfe" nil  nil   nil I   L   () nil "morisyin" "Morisyen")
            ("mff" nil  nil   nil I   L   () nil nil "Naki")
            ("mfg" nil  nil   nil I   L   () nil nil "Mixifore")
            ("mfh" nil  nil   nil I   L   () nil nil "Matal")
            ("mfi" nil  nil   nil I   L   () nil nil "Wandala")
            ("mfj" nil  nil   nil I   L   () nil nil "Mefele")
            ("mfk" nil  nil   nil I   L   () nil nil "Mofu, North")
            ("mfl" nil  nil   nil I   L   () nil nil "Putai")
            ("mfm" nil  nil   nil I   L   () nil nil "Marghi South")
            ("mfn" nil  nil   nil I   L   () nil nil "Mbembe, Cross River")
            ("mfo" nil  nil   nil I   L   () nil nil "Mbe")
            ("mfp" nil  nil   nil I   L   () nil nil "Malay, Makassar")
            ("mfq" nil  nil   nil I   L   () nil nil "Moba")
            ("mfr" nil  nil   nil I   L   () nil nil "Marithiel")
            ("mfs" nil  nil   nil I   L   () nil nil "Mexican Sign Language")
            ("mft" nil  nil   nil I   L   () nil nil "Mokerang")
            ("mfu" nil  nil   nil I   L   () nil nil "Mbwela")
            ("mfv" nil  nil   nil I   L   () nil nil "Mandjak")
            ("mfw" nil  nil   nil I   E   () nil nil "Mulaha")
            ("mfx" nil  nil   nil I   L   () nil nil "Melo")
            ("mfy" nil  nil   nil I   L   () nil "Caíta" "Mayo")
            ("mfz" nil  nil   nil I   L   () nil nil "Mabaan")
            ("mga" nil  "mga" nil I   H   () nil nil "Irish, Middle (900-1200)")
            ("mgb" nil  nil   nil I   L   () nil nil "Mararit")
            ("mgc" nil  nil   nil I   L   () nil nil "Morokodo")
            ("mgd" nil  nil   nil I   L   () nil nil "Moru")
            ("mge" nil  nil   nil I   L   () nil nil "Mango")
            ("mgf" nil  nil   nil I   L   () nil nil "Maklew")
            ("mgg" nil  nil   nil I   L   () nil nil "Mpongmpong")
            ("mgh" nil  nil   nil I   L   () nil nil "Makhuwa-Meetto")
            ("mgi" nil  nil   nil I   L   () nil nil "Lijili")
            ("mgj" nil  nil   nil I   L   () nil nil "Abureni")
            ("mgk" nil  nil   nil I   L   () nil nil "Mawes")
            ("mgl" nil  nil   nil I   L   () nil nil "Maleu-Kilenge")
            ("mgm" nil  nil   nil I   L   () nil nil "Mambae")
            ("mgn" nil  nil   nil I   L   () nil nil "Mbangi")
            ("mgo" nil  nil   nil I   L   () nil nil "Meta'")
            ("mgp" nil  nil   nil I   L   () nil nil "Magar, Eastern")
            ("mgq" nil  nil   nil I   L   () nil nil "Malila")
            ("mgr" nil  nil   nil I   L   () nil nil "Mambwe-Lungu")
            ("mgs" nil  nil   nil I   L   () nil nil "Manda (Tanzania)")
            ("mgt" nil  nil   nil I   L   () nil nil "Mongol")
            ("mgu" nil  nil   nil I   L   () nil nil "Mailu")
            ("mgv" nil  nil   nil I   L   () nil nil "Matengo")
            ("mgw" nil  nil   nil I   L   () nil nil "Matumbi")
            ("mgx" nil  nil   t   I   L   () nil nil "Omati")
            ("mgy" nil  nil   nil I   L   () nil nil "Mbunga")
            ("mgz" nil  nil   nil I   L   () nil nil "Mbugwe")
            ("mha" nil  nil   nil I   L   () nil nil "Manda (India)")
            ("mhb" nil  nil   nil I   L   () nil nil "Mahongwe")
            ("mhc" nil  nil   nil I   L   () nil nil "Mocho")
            ("mhd" nil  nil   nil I   L   () nil nil "Mbugu")
            ("mhe" nil  nil   nil I   L   () nil nil "Besisi")
            ("mhf" nil  nil   nil I   L   () nil nil "Mamaa")
            ("mhg" nil  nil   nil I   L   () nil nil "Margu")
            ("mhh" nil  nil   t   I   L   () nil nil "Maskoy Pidgin")
            ("mhi" nil  nil   nil I   L   () nil nil "Ma'di")
            ("mhj" nil  nil   nil I   L   () nil nil "Mogholi")
            ("mhk" nil  nil   nil I   L   () nil nil "Mungaka")
            ("mhl" nil  nil   nil I   L   () nil nil "Mauwake")
            ("mhm" nil  nil   nil I   L   () nil nil "Makhuwa-Moniga")
            ("mhn" nil  nil   nil I   L   () nil nil "Mócheno")
            ("mho" nil  nil   nil I   L   () nil nil "Mashi (Zambia)")
            ("mhp" nil  nil   nil I   L   () nil nil "Malay, Balinese")
            ("mhq" nil  nil   nil I   L   () nil "Rų́ʼeta:re" "Mandan")
            ("mhr" nil  nil   nil I   L   () nil nil "Mari, Eastern")
            ("mhs" nil  nil   nil I   L   () nil nil "Buru (Indonesia)")
            ("mht" nil  nil   nil I   L   () nil nil "Mandahuaca")
            ("mhu" nil  nil   nil I   L   () nil nil "Digaro-Mishmi")
            ("mhv" nil  nil   t   nil nil () nil nil "Arakanese")
            ("mhw" nil  nil   nil I   L   () nil nil "Mbukushu")
            ("mhx" nil  nil   nil I   L   () nil nil "Maru")
            ("mhy" nil  nil   nil I   L   () nil nil "Ma'anyan")
            ("mhz" nil  nil   nil I   L   () nil nil "Mor (Mor Islands)")
            ("mia" nil  nil   nil I   L   () nil nil "Miami")
            ("mib" nil  nil   nil I   L   () nil nil "Mixtec, Atatláhuca")
            ("mic" nil  "mic" nil I   L   () nil "Mi'gmaq" "Micmac")
            ("mid" nil  nil   nil I   L   () nil "Mandāyì" "Mandaic")
            ("mie" nil  nil   nil I   L   () nil nil "Mixtec, Ocotepec")
            ("mif" nil  nil   nil I   L   () nil nil "Mofu-Gudur")
            ("mig" nil  nil   nil I   L   () nil nil "Mixtec, San Miguel El Grande")
            ("mih" nil  nil   nil I   L   () nil nil "Mixtec, Chayuco")
            ("mii" nil  nil   nil I   L   () nil nil "Mixtec, Chigmecatitlán")
            ("mij" nil  nil   nil I   L   () nil nil "Abar")
            ("mik" nil  nil   nil I   L   () nil nil "Mikasuki")
            ("mil" nil  nil   nil I   L   () nil nil "Mixtec, Peñoles")
            ("mim" nil  nil   nil I   L   () nil nil "Mixtec, Alacatlatzala")
            ("min" nil  "min" nil I   L   () nil "Baso Minangkabau" "Minangkabau")
            ("mio" nil  nil   nil I   L   () nil nil "Mixtec, Pinotepa Nacional")
            ("mip" nil  nil   nil I   L   () nil nil "Mixtec, Apasco-Apoala")
            ("miq" nil  nil   nil I   L   () nil nil "Mískito")
            ("mir" nil  nil   nil I   L   () nil nil "Mixe, Isthmus")
            ("mis" nil  "mis" nil S   S   () nil nil "Uncoded languages")
            ("mit" nil  nil   nil I   L   () nil nil "Mixtec, Southern Puebla")
            ("miu" nil  nil   nil I   L   () nil nil "Mixtec, Cacaloxtepec")
            ("miv" nil  nil   t   nil nil () nil nil "Mimi")
            ("miw" nil  nil   nil I   L   () nil nil "Akoye")
            ("mix" nil  nil   nil I   L   () nil nil "Mixtec, Mixtepec")
            ("miy" nil  nil   nil I   L   () nil nil "Mixtec, Ayutla")
            ("miz" nil  nil   nil I   L   () nil nil "Mixtec, Coatzospan")
            ("mja" nil  nil   t   I   L   () nil nil "Mahei")
            ("mjb" nil  nil   nil I   L   () "Trans–New Guinea" nil "Makalero")
            ("mjc" nil  nil   nil I   L   () nil nil "Mixtec, San Juan Colorado")
            ("mjd" nil  nil   nil I   L   () nil nil "Maidu, Northwest (Konkow)")
            ("mje" nil  nil   nil I   E   () nil nil "Muskum")
            ("mjg" nil  nil   nil I   L   () nil nil "Tu")
            ("mjh" nil  nil   nil I   L   () nil nil "Mwera (Nyasa)")
            ("mji" nil  nil   nil I   L   () nil nil "Kim Mun")
            ("mjj" nil  nil   nil I   L   () nil nil "Mawak")
            ("mjk" nil  nil   nil I   L   () nil nil "Matukar")
            ("mjl" nil  nil   nil I   L   () nil nil "Mandeali")
            ("mjm" nil  nil   nil I   L   () nil nil "Medebur")
            ("mjn" nil  nil   nil I   L   () nil nil "Ma (Papua New Guinea)")
            ("mjo" nil  nil   nil I   L   () nil nil "Malankuravan")
            ("mjp" nil  nil   nil I   L   () nil nil "Malapandaram")
            ("mjq" nil  nil   nil I   E   () nil nil "Malaryan")
            ("mjr" nil  nil   nil I   L   () nil nil "Malavedan")
            ("mjs" nil  nil   nil I   L   () nil nil "Miship")
            ("mjt" nil  nil   nil I   L   () nil nil "Sauria Paharia")
            ("mju" nil  nil   nil I   L   () nil nil "Manna-Dora")
            ("mjv" nil  nil   nil I   L   () nil nil "Mannan")
            ("mjw" nil  nil   nil I   L   () nil nil "Karbi")
            ("mjx" nil  nil   nil I   L   () nil nil "Mahali")
            ("mjy" nil  nil   nil I   E   () nil nil "Mahican")
            ("mjz" nil  nil   nil I   L   () nil nil "Majhi")
            ("mka" nil  nil   nil I   L   () nil nil "Mbre")
            ("mkb" nil  nil   nil I   L   () nil nil "Mal Paharia")
            ("mkc" nil  nil   nil I   L   () nil nil "Siliput")
            ("mkd" "mk" "mac" nil I   L   () "Indo-European" "македонски" "Macedonian")
            ("mke" nil  nil   nil I   L   () nil nil "Mawchi")
            ("mkf" nil  nil   nil I   L   () nil nil "Miya")
            ("mkg" nil  nil   nil I   L   () nil nil "Mak (China)")
            (nil   nil  "mkh" nil C   nil () nil nil "Mon-Khmer languages")
            ("mki" nil  nil   nil I   L   () nil nil "Dhatki")
            ("mkj" nil  nil   nil I   L   () nil nil "Mokilese")
            ("mkk" nil  nil   nil I   L   () nil nil "Byep")
            ("mkl" nil  nil   nil I   L   () nil nil "Mokole")
            ("mkm" nil  nil   nil I   L   () nil nil "Moklen")
            ("mkn" nil  nil   nil I   L   () nil nil "Malay, Kupang")
            ("mko" nil  nil   nil I   L   () nil nil "Mingang Doso")
            ("mkp" nil  nil   nil I   L   () nil nil "Moikodi")
            ("mkq" nil  nil   nil I   E   () nil nil "Miwok, Bay")
            ("mkr" nil  nil   nil I   L   () nil nil "Malas")
            ("mks" nil  nil   nil I   L   () nil nil "Mixtec, Silacayoapan")
            ("mkt" nil  nil   nil I   L   () nil nil "Vamale")
            ("mku" nil  nil   nil I   L   () nil nil "Maninka, Konyanka")
            ("mkv" nil  nil   nil I   L   () nil nil "Mafea")
            ("mkw" nil  nil   nil I   L   () nil nil "Kituba (Congo)")
            ("mkx" nil  nil   nil I   L   () nil nil "Manobo, Cinamiguin")
            ("mky" nil  nil   nil I   L   () nil nil "Makian, East")
            ("mkz" nil  nil   nil I   L   () nil nil "Makasae")
            ("mla" nil  nil   nil I   L   () nil nil "Malo")
            ("mlb" nil  nil   nil I   L   () nil nil "Mbule")
            ("mlc" nil  nil   nil I   L   () nil nil "Cao Lan")
            ("mld" nil  nil   t   I   L   () nil nil "Malakhel")
            ("mle" nil  nil   nil I   L   () nil nil "Manambu")
            ("mlf" nil  nil   nil I   L   () nil nil "Mal")
            ("mlg" "mg" "mlg" nil M   L   ("bhr" "bmm" "bzc"
                                           "msh" "plt" "skg"
                                           "tdx" "tkg" "txy"
                                           "xmv" "xmw" "bjq") "Austronesian" "Malagasy" "Malagasy")
            ("mlh" nil  nil   nil I   L   () nil nil "Mape")
            ("mli" nil  nil   nil I   L   () nil nil "Malimpung")
            ("mlj" nil  nil   nil I   L   () nil nil "Miltu")
            ("mlk" nil  nil   nil I   L   () nil nil "Malakote")
            ("mll" nil  nil   nil I   L   () nil nil "Malua Bay")
            ("mlm" nil  nil   nil I   L   () nil nil "Mulam")
            ("mln" nil  nil   nil I   L   () nil nil "Malango")
            ("mlo" nil  nil   nil I   L   () nil nil "Mlomp")
            ("mlp" nil  nil   nil I   L   () nil nil "Bargam")
            ("mlq" nil  nil   nil I   L   () nil nil "Maninkakan, Western")
            ("mlr" nil  nil   nil I   L   () nil nil "Vame")
            ("mls" nil  nil   nil I   L   () nil nil "Masalit")
            ("mlt" "mt" "mlt" nil I   L   () "Afro-Asiatic" "bil-Malta" "Maltese")
            ("mlu" nil  nil   nil I   L   () nil nil "To'abaita")
            ("mlv" nil  nil   nil I   L   () nil nil "Motlav")
            ("mlw" nil  nil   nil I   L   () nil nil "Moloko")
            ("mlx" nil  nil   nil I   L   () nil nil "Malfaxal")
            ("mly" nil  nil   t   nil nil () nil nil "Malay (individual language)")
            ("mlz" nil  nil   nil I   L   () nil nil "Malaynon")
            ("mma" nil  nil   nil I   L   () nil nil "Mama")
            ("mmb" nil  nil   nil I   L   () nil nil "Momina")
            ("mmc" nil  nil   nil I   L   () nil nil "Mazahua, Michoacán")
            ("mmd" nil  nil   nil I   L   () nil nil "Maonan")
            ("mme" nil  nil   nil I   L   () nil nil "Mae")
            ("mmf" nil  nil   nil I   L   () nil nil "Mundat")
            ("mmg" nil  nil   nil I   L   () nil nil "Ambrym, North")
            ("mmh" nil  nil   nil I   L   () nil nil "Mehináku")
            ("mmi" nil  nil   nil I   L   () nil nil "Musar")
            ("mmj" nil  nil   nil I   L   () nil nil "Majhwar")
            ("mmk" nil  nil   nil I   L   () nil nil "Mukha-Dora")
            ("mml" nil  nil   nil I   L   () nil nil "Man Met")
            ("mmm" nil  nil   nil I   L   () nil nil "Maii")
            ("mmn" nil  nil   nil I   L   () nil nil "Mamanwa")
            ("mmo" nil  nil   nil I   L   () nil nil "Buang, Mangga")
            ("mmp" nil  nil   nil I   L   () nil nil "Musan")
            ("mmq" nil  nil   nil I   L   () nil nil "Musak")
            ("mmr" nil  nil   nil I   L   () nil nil "Hmong, Western Xiangxi")
            ("mms" nil  nil   t   I   L   () nil nil "Mam, Southern")
            ("mmt" nil  nil   nil I   L   () nil nil "Malalamai")
            ("mmu" nil  nil   nil I   L   () nil nil "Mmaala")
            ("mmv" nil  nil   nil I   E   () nil nil "Miriti")
            ("mmw" nil  nil   nil I   L   () nil nil "Emae")
            ("mmx" nil  nil   nil I   L   () nil nil "Madak")
            ("mmy" nil  nil   nil I   L   () nil nil "Migaama")
            ("mmz" nil  nil   nil I   L   () nil nil "Mabaale")
            ("mna" nil  nil   nil I   L   () nil nil "Mbula")
            ("mnb" nil  nil   nil I   L   () nil nil "Muna")
            ("mnc" nil  "mnc" nil I   L   () nil "ᠮᠠᠨᠵᡠ ᡤᡳᠰᡠᠨ" "Manchu")
            ("mnd" nil  nil   nil I   L   () nil nil "Mondé")
            ("mne" nil  nil   nil I   L   () nil nil "Naba")
            ("mnf" nil  nil   nil I   L   () nil nil "Mundani")
            ("mng" nil  nil   nil I   L   () nil nil "Mnong, Eastern")
            ("mnh" nil  nil   nil I   L   () nil nil "Mono (Democratic Republic of Congo)")
            ("mni" nil  "mni" nil I   L   () "Sino-Tibetan" "মৈইতৈইলোন" "Meitei")
            ("mnj" nil  nil   nil I   L   () nil nil "Munji")
            ("mnk" nil  nil   nil I   L   () nil nil "Mandinka")
            ("mnl" nil  nil   nil I   L   () nil nil "Tiale")
            ("mnm" nil  nil   nil I   L   () nil nil "Mapena")
            ("mnn" nil  nil   nil I   L   () nil nil "Mnong, Southern")
            (nil   nil  "mno" nil C   nil () nil nil "Manobo languages")
            ("mnp" nil  nil   nil I   L   () "Chinese" "闽北" "Min Bei Chinese")
            ("mnq" nil  nil   nil I   L   () nil nil "Minriq")
            ("mnr" nil  nil   nil I   L   () nil nil "Mono (USA)")
            ("mns" nil  nil   nil I   L   () "Uralic" "ма̄ньси" "Mansi")
            ("mnt" nil  nil   t   I   E   () nil nil "Maykulan")
            ("mnu" nil  nil   nil I   L   () nil nil "Mer")
            ("mnv" nil  nil   nil I   L   () nil nil "Rennell-Belona")
            ("mnw" nil  nil   nil I   L   () nil "ဘာသာမန်" "Mon")
            ("mnx" nil  nil   nil I   L   () nil nil "Manikion")
            ("mny" nil  nil   nil I   L   () nil nil "Manyawa")
            ("mnz" nil  nil   nil I   L   () nil nil "Moni")
            ("moa" nil  nil   nil I   L   () nil nil "Mwan")
            ("mob" nil  nil   t   nil nil () nil nil "Moinba")
            ("moc" nil  nil   nil I   L   () nil nil "Mocoví")
            ("mod" nil  nil   nil I   E   () nil nil "Mobilian")
            ("moe" nil  nil   nil I   L   () nil nil "Montagnais")
            ("mof" nil  nil   t   I   E   () nil nil "Mohegan-Montauk-Narragansett")
            ("mog" nil  nil   nil I   L   () nil nil "Mongondow")
            ("moh" nil  "moh" nil I   L   () nil "Kanien’kéha" "Mohawk")
            ("moi" nil  nil   nil I   L   () nil nil "Mboi")
            ("moj" nil  nil   nil I   L   () nil nil "Monzombo")
            ("mok" nil  nil   nil I   L   () nil nil "Morori")
            ("mol" nil  nil   t   nil nil () nil nil "Moldavian")
            ("mom" nil  nil   nil I   E   () nil nil "Monimbo")
            ("mon" "mn" "mon" nil M   L   ("khk" "mvf") "Mongolic" "монгол" "Mongolian")
            ("moo" nil  nil   nil I   L   () nil nil "Monom")
            ("mop" nil  nil   nil I   L   () nil nil "Mopán Maya")
            ("moq" nil  nil   nil I   L   () nil nil "Mor (Bomberai Peninsula)")
            ("mor" nil  nil   nil I   L   () nil nil "Moro")
            ("mos" nil  "mos" nil I   L   () nil "Mòoré" "Mossi")
            ("mot" nil  nil   nil I   L   () nil nil "Barí")
            ("mou" nil  nil   nil I   L   () nil nil "Mogum")
            ("mov" nil  nil   nil I   L   () nil nil "Mohave")
            ("mow" nil  nil   nil I   L   () nil nil "Moi (Congo)")
            ("mox" nil  nil   nil I   L   () nil nil "Molima")
            ("moy" nil  nil   nil I   L   () nil nil "Shekkacho")
            ("moz" nil  nil   nil I   L   () nil nil "Mukulu")
            ("mpa" nil  nil   nil I   L   () nil nil "Mpoto")
            ("mpb" nil  nil   nil I   L   () nil nil "Mullukmulluk")
            ("mpc" nil  nil   nil I   L   () nil nil "Mangarayi")
            ("mpd" nil  nil   nil I   L   () nil nil "Machinere")
            ("mpe" nil  nil   nil I   L   () nil nil "Majang")
            ("mpf" nil  nil   t   I   L   () nil nil "Mam, Tajumulco")
            ("mpg" nil  nil   nil I   L   () nil nil "Marba")
            ("mph" nil  nil   nil I   L   () nil nil "Maung")
            ("mpi" nil  nil   nil I   L   () nil nil "Mpade")
            ("mpj" nil  nil   nil I   L   () nil nil "Martu Wangka")
            ("mpk" nil  nil   nil I   L   () nil nil "Mbara (Chad)")
            ("mpl" nil  nil   nil I   L   () nil nil "Watut, Middle")
            ("mpm" nil  nil   nil I   L   () nil nil "Mixtec, Yosondúa")
            ("mpn" nil  nil   nil I   L   () nil nil "Mindiri")
            ("mpo" nil  nil   nil I   L   () nil nil "Miu")
            ("mpp" nil  nil   nil I   L   () nil nil "Migabac")
            ("mpq" nil  nil   nil I   L   () nil nil "Matís")
            ("mpr" nil  nil   nil I   L   () nil nil "Vangunu")
            ("mps" nil  nil   nil I   L   () nil nil "Dadibi")
            ("mpt" nil  nil   nil I   L   () nil nil "Mian")
            ("mpu" nil  nil   nil I   L   () nil nil "Makuráp")
            ("mpv" nil  nil   nil I   L   () nil nil "Munkip")
            ("mpw" nil  nil   nil I   L   () nil nil "Mapidian")
            ("mpx" nil  nil   nil I   L   () nil nil "Misima-Paneati")
            ("mpy" nil  nil   nil I   L   () nil nil "Mapia")
            ("mpz" nil  nil   nil I   L   () nil nil "Mpi")
            ("mqa" nil  nil   nil I   L   () nil nil "Maba (Indonesia)")
            ("mqb" nil  nil   nil I   L   () nil nil "Mbuko")
            ("mqc" nil  nil   nil I   L   () nil nil "Mangole")
            ("mqd" nil  nil   t   nil nil () nil nil "Madang")
            ("mqe" nil  nil   nil I   L   () nil nil "Matepi")
            ("mqf" nil  nil   nil I   L   () nil nil "Momuna")
            ("mqg" nil  nil   nil I   L   () nil nil "Malay, Kota Bangun Kutai")
            ("mqh" nil  nil   nil I   L   () nil nil "Mixtec, Tlazoyaltepec")
            ("mqi" nil  nil   nil I   L   () nil nil "Mariri")
            ("mqj" nil  nil   nil I   L   () nil nil "Mamasa")
            ("mqk" nil  nil   nil I   L   () nil nil "Manobo, Rajah Kabunsuwan")
            ("mql" nil  nil   nil I   L   () nil nil "Mbelime")
            ("mqm" nil  nil   nil I   L   () nil nil "Marquesan, South")
            ("mqn" nil  nil   nil I   L   () nil nil "Moronene")
            ("mqo" nil  nil   nil I   L   () nil nil "Modole")
            ("mqp" nil  nil   nil I   L   () nil nil "Manipa")
            ("mqq" nil  nil   nil I   L   () nil nil "Minokok")
            ("mqr" nil  nil   nil I   L   () nil nil "Mander")
            ("mqs" nil  nil   nil I   L   () nil nil "Makian, West")
            ("mqt" nil  nil   nil I   L   () nil nil "Mok")
            ("mqu" nil  nil   nil I   L   () nil nil "Mandari")
            ("mqv" nil  nil   nil I   L   () nil nil "Mosimo")
            ("mqw" nil  nil   nil I   L   () nil nil "Murupi")
            ("mqx" nil  nil   nil I   L   () nil nil "Mamuju")
            ("mqy" nil  nil   nil I   L   () nil nil "Manggarai")
            ("mqz" nil  nil   nil I   L   () nil nil "Malasanga")
            ("mra" nil  nil   nil I   L   () nil nil "Mlabri")
            ("mrb" nil  nil   nil I   L   () nil nil "Marino")
            ("mrc" nil  nil   nil I   L   () nil nil "Maricopa")
            ("mrd" nil  nil   nil I   L   () nil nil "Magar, Western")
            ("mre" nil  nil   nil I   E   () nil nil "Martha's Vineyard Sign Language")
            ("mrf" nil  nil   nil I   L   () nil nil "Elseng")
            ("mrg" nil  nil   nil I   L   () nil nil "Miri")
            ("mrh" nil  nil   nil I   L   () nil nil "Chin, Mara")
            ("mri" "mi" "mao" nil I   L   () "Austronesian" "te reo Māori" "Maori")
            ("mrj" nil  nil   nil I   L   () nil nil "Mari, Western")
            ("mrk" nil  nil   nil I   L   () nil nil "Hmwaveke")
            ("mrl" nil  nil   nil I   L   () nil nil "Mortlockese")
            ("mrm" nil  nil   nil I   L   () nil nil "Merlav")
            ("mrn" nil  nil   nil I   L   () nil nil "Cheke Holo")
            ("mro" nil  nil   nil I   L   () nil nil "Mru")
            ("mrp" nil  nil   nil I   L   () nil nil "Morouas")
            ("mrq" nil  nil   nil I   L   () nil nil "Marquesan, North")
            ("mrr" nil  nil   nil I   L   () nil nil "Maria (India)")
            ("mrs" nil  nil   nil I   L   () nil nil "Maragus")
            ("mrt" nil  nil   nil I   L   () nil nil "Marghi Central")
            ("mru" nil  nil   nil I   L   () nil nil "Mono (Cameroon)")
            ("mrv" nil  nil   nil I   L   () nil nil "Mangareva")
            ("mrw" nil  nil   nil I   L   () nil "Austronesian" "Maranao")
            ("mrx" nil  nil   nil I   L   () nil nil "Maremgi")
            ("mry" nil  nil   nil I   L   () nil nil "Mandaya, Karaga")
            ("mrz" nil  nil   nil I   L   () nil nil "Marind")
            ("msa" "ms" "may" nil M   L   ("bjn"
                                           "btj" "bve" "bvu"
                                           "coa" "dup" "hji"
                                           "ind" "jak" "jax"
                                           "kvb" "kvr" "kxd"
                                           "lce" "lcf" "liw"
                                           "max" "meo" "mfa"
                                           "mfb" "min" "mqg"
                                           "msi" "mui" "orn"
                                           "ors" "pel" "pse"
                                           "tmw" "urk" "vkk"
                                           "vkt" "xmm" "zlm"
                                           "zmi" "zsm" "mly") "Austronesian" "bahasa Melayu" "Malay (generic)")
            ("msb" nil  nil   nil I   L   () nil nil "Masbatenyo")
            ("msc" nil  nil   nil I   L   () nil nil "Maninka, Sankaran")
            ("msd" nil  nil   nil I   L   () nil nil "Yucatec Maya Sign Language")
            ("mse" nil  nil   nil I   L   () nil nil "Musey")
            ("msf" nil  nil   nil I   L   () nil nil "Mekwei")
            ("msg" nil  nil   nil I   L   () nil nil "Moraid")
            ("msh" nil  nil   nil I   L   () nil nil "Malagasy, Masikoro")
            ("msi" nil  nil   nil I   L   () nil nil "Malay, Sabah")
            ("msj" nil  nil   nil I   L   () nil nil "Ma (Democratic Republic of Congo)")
            ("msk" nil  nil   nil I   L   () nil nil "Mansaka")
            ("msl" nil  nil   nil I   L   () nil nil "Molof")
            ("msm" nil  nil   nil I   L   () nil nil "Manobo, Agusan")
            ("msn" nil  nil   nil I   L   () nil nil "Mosina")
            ("mso" nil  nil   nil I   L   () nil nil "Mombum")
            ("msp" nil  nil   nil I   E   () nil nil "Maritsauá")
            ("msq" nil  nil   nil I   L   () nil nil "Caac")
            ("msr" nil  nil   nil I   L   () nil nil "Mongolian Sign Language")
            ("mss" nil  nil   nil I   L   () nil nil "Masela, West")
            ("mst" nil  nil   t   I   L   () nil nil "Mandaya, Cataelano")
            ("msu" nil  nil   nil I   L   () nil nil "Musom")
            ("msv" nil  nil   nil I   L   () nil nil "Maslam")
            ("msw" nil  nil   nil I   L   () nil nil "Mansoanka")
            ("msx" nil  nil   nil I   L   () nil nil "Moresada")
            ("msy" nil  nil   nil I   L   () nil nil "Aruamu")
            ("msz" nil  nil   nil I   L   () nil nil "Momare")
            ("mta" nil  nil   nil I   L   () nil nil "Manobo, Cotabato")
            ("mtb" nil  nil   nil I   L   () nil nil "Anyin Morofo")
            ("mtc" nil  nil   nil I   L   () nil nil "Munit")
            ("mtd" nil  nil   nil I   L   () nil nil "Mualang")
            ("mte" nil  nil   nil I   L   () nil nil "Mono (Solomon Islands)")
            ("mtf" nil  nil   nil I   L   () nil nil "Murik")
            ("mtg" nil  nil   nil I   L   () nil nil "Una")
            ("mth" nil  nil   nil I   L   () nil nil "Munggui")
            ("mti" nil  nil   nil I   L   () nil nil "Maiwa (Papua New Guinea)")
            ("mtj" nil  nil   nil I   L   () nil nil "Moskona")
            ("mtk" nil  nil   nil I   L   () nil nil "Mbe'")
            ("mtl" nil  nil   nil I   L   () nil nil "Montol")
            ("mtm" nil  nil   nil I   E   () nil nil "Mator")
            ("mtn" nil  nil   nil I   E   () nil nil "Matagalpa")
            ("mto" nil  nil   nil I   L   () nil nil "Mixe, Totontepec")
            ("mtp" nil  nil   nil I   L   () nil nil "Wichí Lhamtés Nocten")
            ("mtq" nil  nil   nil I   L   () nil "Mường" "Muong")
            ("mtr" nil  nil   nil I   L   () nil nil "Mewari")
            ("mts" nil  nil   nil I   L   () nil nil "Yora")
            ("mtt" nil  nil   nil I   L   () nil nil "Mota")
            ("mtu" nil  nil   nil I   L   () nil nil "Mixtec, Tututepec")
            ("mtv" nil  nil   nil I   L   () nil nil "Asaro'o")
            ("mtw" nil  nil   nil I   L   () nil nil "Magahat")
            ("mtx" nil  nil   nil I   L   () nil nil "Mixtec, Tidaá")
            ("mty" nil  nil   nil I   L   () nil nil "Nabi")
            ("mtz" nil  nil   t   I   L   () nil nil "Tacanec")
            ("mua" nil  nil   nil I   L   () nil nil "Mundang")
            ("mub" nil  nil   nil I   L   () nil nil "Mubi")
            ("muc" nil  nil   nil I   L   () nil nil "Mbu'")
            ("mud" nil  nil   nil I   L   () nil nil "Aleut, Mednyj")
            ("mue" nil  nil   nil I   L   () nil nil "Media Lengua")
            ("mug" nil  nil   nil I   L   () nil nil "Musgu")
            ("muh" nil  nil   nil I   L   () nil nil "Mündü")
            ("mui" nil  nil   nil I   L   () nil nil "Musi")
            ("muj" nil  nil   nil I   L   () nil nil "Mabire")
            ("muk" nil  nil   nil I   L   () nil nil "Mugom")
            ("mul" nil  "mul" nil S   S   () nil "*" "(Multiple)")
            ("mum" nil  nil   nil I   L   () nil nil "Maiwala")
            (nil   nil  "mun" nil C   nil () nil nil "Munda languages")
            ("muo" nil  nil   nil I   L   () nil nil "Nyong")
            ("mup" nil  nil   nil I   L   () nil nil "Malvi")
            ("muq" nil  nil   nil I   L   () nil nil "Hmong, Eastern Xiangxi")
            ("mur" nil  nil   nil I   L   () nil nil "Murle")
            ("mus" nil  "mus" nil I   L   () nil "Mvskokē" "Creek")
            ("mut" nil  nil   nil I   L   () nil nil "Muria, Western")
            ("muu" nil  nil   nil I   L   () nil nil "Yaaku")
            ("muv" nil  nil   nil I   L   () nil nil "Muthuvan")
            ("muw" nil  nil   t   nil nil () nil nil "Mundari")
            ("mux" nil  nil   nil I   L   () nil nil "Mbo-Ung")
            ("muy" nil  nil   nil I   L   () nil nil "Muyang")
            ("muz" nil  nil   nil I   L   () nil nil "Mursi")
            ("mva" nil  nil   nil I   L   () nil nil "Manam")
            ("mvb" nil  nil   nil I   E   () nil nil "Mattole")
            ("mvc" nil  nil   t   I   L   () nil nil "Mam, Central")
            ("mvd" nil  nil   nil I   L   () nil nil "Mamboru")
            ("mve" nil  nil   nil I   L   () nil nil "Marwari (Pakistan)")
            ("mvf" nil  nil   nil I   L   () nil nil "Mongolian, Peripheral")
            ("mvg" nil  nil   nil I   L   () nil nil "Mixtec, Yucuañe")
            ("mvh" nil  nil   nil I   L   () nil nil "Mire")
            ("mvi" nil  nil   nil I   L   () nil nil "Miyako")
            ("mvj" nil  nil   t   I   L   () nil nil "Mam, Todos Santos Cuchumatán")
            ("mvk" nil  nil   nil I   L   () nil nil "Mekmek")
            ("mvl" nil  nil   nil I   E   () nil nil "Mbara (Australia)")
            ("mvm" nil  nil   t   I   L   () nil nil "Muya")
            ("mvn" nil  nil   nil I   L   () nil nil "Minaveha")
            ("mvo" nil  nil   nil I   L   () nil nil "Marovo")
            ("mvp" nil  nil   nil I   L   () nil nil "Duri")
            ("mvq" nil  nil   nil I   L   () nil nil "Moere")
            ("mvr" nil  nil   nil I   L   () nil nil "Marau")
            ("mvs" nil  nil   nil I   L   () nil nil "Massep")
            ("mvt" nil  nil   nil I   L   () nil nil "Mpotovoro")
            ("mvu" nil  nil   nil I   L   () nil nil "Marfa")
            ("mvv" nil  nil   nil I   L   () nil nil "Tagal Murut")
            ("mvw" nil  nil   nil I   L   () nil nil "Machinga")
            ("mvx" nil  nil   nil I   L   () nil nil "Meoswar")
            ("mvy" nil  nil   nil I   L   () nil nil "Kohistani, Indus")
            ("mvz" nil  nil   nil I   L   () nil nil "Mesqan")
            ("mwa" nil  nil   nil I   L   () nil nil "Mwatebu")
            ("mwb" nil  nil   nil I   L   () nil nil "Juwal")
            ("mwc" nil  nil   nil I   L   () nil nil "Are")
            ("mwd" nil  nil   t   I   L   () nil nil "Mudbura")
            ("mwe" nil  nil   nil I   L   () nil nil "Mwera (Chimwera)")
            ("mwf" nil  nil   nil I   L   () nil nil "Murrinh-Patha")
            ("mwg" nil  nil   nil I   L   () nil nil "Aiklep")
            ("mwh" nil  nil   nil I   L   () nil nil "Mouk-Aria")
            ("mwi" nil  nil   nil I   L   () nil nil "Labo")
            ("mwj" nil  nil   t   I   L   () nil nil "Maligo")
            ("mwk" nil  nil   nil I   L   () nil nil "Maninkakan, Kita")
            ("mwl" nil  "mwl" nil I   L   () nil "mirandês" "Mirandese")
            ("mwm" nil  nil   nil I   L   () nil nil "Sar")
            ("mwn" nil  nil   nil I   L   () nil nil "Nyamwanga")
            ("mwo" nil  nil   nil I   L   () nil nil "Maewo, Central")
            ("mwp" nil  nil   nil I   L   () nil nil "Kala Lagaw Ya")
            ("mwq" nil  nil   nil I   L   () nil nil "Chin, Mün")
            ("mwr" nil  "mwr" nil M   L   ("dhd" "mtr" "mve"
                                           "rwr" "swv" "wry") nil "मारवाड़ी" "Marwari")
            ("mws" nil  nil   nil I   L   () nil nil "Mwimbi-Muthambi")
            ("mwt" nil  nil   nil I   L   () nil nil "Moken")
            ("mwu" nil  nil   nil I   E   () nil nil "Mittu")
            ("mwv" nil  nil   nil I   L   () nil nil "Mentawai")
            ("mww" nil  nil   nil I   L   () nil nil "Hmong Daw")
            ("mwx" nil  nil   t   I   L   () nil nil "Mediak")
            ("mwy" nil  nil   t   I   L   () nil nil "Mosiro")
            ("mwz" nil  nil   nil I   L   () nil nil "Moingi")
            ("mxa" nil  nil   nil I   L   () nil nil "Mixtec, Northwest Oaxaca")
            ("mxb" nil  nil   nil I   L   () nil nil "Mixtec, Tezoatlán")
            ("mxc" nil  nil   nil I   L   () nil nil "Manyika")
            ("mxd" nil  nil   nil I   L   () nil nil "Modang")
            ("mxe" nil  nil   nil I   L   () nil nil "Mele-Fila")
            ("mxf" nil  nil   nil I   L   () nil nil "Malgbe")
            ("mxg" nil  nil   nil I   L   () nil nil "Mbangala")
            ("mxh" nil  nil   nil I   L   () nil nil "Mvuba")
            ("mxi" nil  nil   nil I   H   () nil "مُزَرَب" "Mozarabic")
            ("mxj" nil  nil   nil I   L   () nil nil "Miju-Mishmi")
            ("mxk" nil  nil   nil I   L   () nil nil "Monumbo")
            ("mxl" nil  nil   nil I   L   () nil nil "Gbe, Maxi")
            ("mxm" nil  nil   nil I   L   () nil nil "Meramera")
            ("mxn" nil  nil   nil I   L   () nil nil "Moi (Indonesia)")
            ("mxo" nil  nil   nil I   L   () nil nil "Mbowe")
            ("mxp" nil  nil   nil I   L   () nil nil "Mixe, Tlahuitoltepec")
            ("mxq" nil  nil   nil I   L   () nil nil "Mixe, Juquila")
            ("mxr" nil  nil   nil I   L   () nil nil "Kayan, Murik")
            ("mxs" nil  nil   nil I   L   () nil nil "Mixtec, Huitepec")
            ("mxt" nil  nil   nil I   L   () nil nil "Mixtec, Jamiltepec")
            ("mxu" nil  nil   nil I   L   () nil nil "Mada (Cameroon)")
            ("mxv" nil  nil   nil I   L   () nil nil "Mixtec, Metlatónoc")
            ("mxw" nil  nil   nil I   L   () nil nil "Namo")
            ("mxx" nil  nil   nil I   L   () nil nil "Mahou")
            ("mxy" nil  nil   nil I   L   () nil nil "Mixtec, Southeastern Nochixtlán")
            ("mxz" nil  nil   nil I   L   () nil nil "Masela, Central")
            ("mya" "my" "bur" nil I   L   () "Sino-Tibetan" "မြန်မာစာ" "Burmese")
            ("myb" nil  nil   nil I   L   () nil nil "Mbay")
            ("myc" nil  nil   nil I   L   () nil nil "Mayeka")
            ("myd" nil  nil   t   I   L   () nil nil "Maramba")
            ("mye" nil  nil   nil I   L   () nil nil "Myene")
            ("myf" nil  nil   nil I   L   () nil nil "Bambassi")
            ("myg" nil  nil   nil I   L   () nil nil "Manta")
            ("myh" nil  nil   nil I   L   () nil nil "Makah")
            ("myi" nil  nil   t   I   L   () nil nil "Mina (India)")
            ("myj" nil  nil   nil I   L   () nil nil "Mangayat")
            ("myk" nil  nil   nil I   L   () nil nil "Senoufo, Mamara")
            ("myl" nil  nil   nil I   L   () nil nil "Moma")
            ("mym" nil  nil   nil I   L   () nil nil "Me'en")
            (nil   nil  "myn" nil C   nil () nil nil "Mayan languages")
            ("myo" nil  nil   nil I   L   () nil nil "Anfillo")
            ("myp" nil  nil   nil I   L   () nil "Hi'aiti'ihi'" "Pirahã")
            ("myq" nil  nil   t   I   L   () nil nil "Maninka, Forest")
            ("myr" nil  nil   nil I   L   () nil nil "Muniche")
            ("mys" nil  nil   nil I   E   () nil nil "Mesmes")
            ("myt" nil  nil   t   I   L   () nil nil "Mandaya, Sangab")
            ("myu" nil  nil   nil I   L   () nil nil "Mundurukú")
            ("myv" nil  "myv" nil I   L   () nil "эрзя" "Erzya")
            ("myw" nil  nil   nil I   L   () nil nil "Muyuw")
            ("myx" nil  nil   nil I   L   () nil nil "Masaba")
            ("myy" nil  nil   nil I   L   () nil nil "Macuna")
            ("myz" nil  nil   nil I   H   () nil nil "Mandaic, Classical")
            ("mza" nil  nil   nil I   L   () nil nil "Mixtec, Santa María Zacatepec")
            ("mzb" nil  nil   nil I   L   () nil "تومزابت" "Tumzabt")
            ("mzc" nil  nil   nil I   L   () nil nil "Madagascar Sign Language")
            ("mzd" nil  nil   nil I   L   () nil nil "Malimba")
            ("mze" nil  nil   nil I   L   () nil nil "Morawa")
            ("mzf" nil  nil   t   nil nil () nil nil "Aiku")
            ("mzg" nil  nil   nil I   L   () nil nil "Monastic Sign Language")
            ("mzh" nil  nil   nil I   L   () nil nil "Wichí Lhamtés Güisnay")
            ("mzi" nil  nil   nil I   L   () nil nil "Mazatec, Ixcatlán")
            ("mzj" nil  nil   nil I   L   () nil nil "Manya")
            ("mzk" nil  nil   nil I   L   () nil nil "Mambila, Nigeria")
            ("mzl" nil  nil   nil I   L   () nil nil "Mixe, Mazatlán")
            ("mzm" nil  nil   nil I   L   () nil nil "Mumuye")
            ("mzn" nil  nil   nil I   L   () nil nil "Mazanderani")
            ("mzo" nil  nil   nil I   E   () nil nil "Matipuhy")
            ("mzp" nil  nil   nil I   L   () nil nil "Movima")
            ("mzq" nil  nil   nil I   L   () nil nil "Mori Atas")
            ("mzr" nil  nil   nil I   L   () nil nil "Marúbo")
            ("mzs" nil  nil   nil I   L   () nil nil "Macanese")
            ("mzt" nil  nil   nil I   L   () nil nil "Mintil")
            ("mzu" nil  nil   nil I   L   () nil nil "Inapang")
            ("mzv" nil  nil   nil I   L   () nil nil "Manza")
            ("mzw" nil  nil   nil I   L   () nil nil "Deg")
            ("mzx" nil  nil   nil I   L   () nil nil "Mawayana")
            ("mzy" nil  nil   nil I   L   () nil nil "Mozambican Sign Language")
            ("mzz" nil  nil   nil I   L   () nil nil "Maiadomu")
            ;; N
            ("naa" nil  nil   nil I   L   () nil nil "Namla")
            ("nab" nil  nil   nil I   L   () nil nil "Nambikuára, Southern")
            ("nac" nil  nil   nil I   L   () nil nil "Narak")
            ("nad" nil  nil   t   I   L   () nil nil "Nijadali")
            ("nae" nil  nil   nil I   E   () nil nil "Naka'ela")
            ("naf" nil  nil   nil I   L   () nil nil "Nabak")
            ("nag" nil  nil   nil I   L   () nil nil "Naga Pidgin")
            (nil   nil  "nah" nil C   nil () nil nil "Nahuatl languages")
            (nil   nil  "nai" nil C   nil () nil nil "North American Indian languages")
            ("naj" nil  nil   nil I   L   () nil nil "Nalu")
            ("nak" nil  nil   nil I   L   () nil nil "Nakanai")
            ("nal" nil  nil   nil I   L   () nil nil "Nalik")
            ("nam" nil  nil   nil I   L   () nil nil "Nangikurrunggurr")
            ("nan" nil  nil   nil I   L   () "Chinese" nil "Min Nan Chinese")
            ("nao" nil  nil   nil I   L   () nil nil "Naaba")
            ("nap" nil  "nap" nil I   L   () nil "nnapulitano" "Neapolitan")
            ("naq" nil  nil   nil I   L   () nil "Khoekhoegowab" "Nama (Namibia)")
            ("nar" nil  nil   nil I   L   () nil nil "Iguta")
            ("nas" nil  nil   nil I   L   () nil nil "Naasioi")
            ("nat" nil  nil   nil I   L   () nil nil "Hungworo")
            ("nau" "na" "nau" nil I   L   () "Austronesian" "ekakairũ naoero" "Nauru")
            ("nav" "nv" "nav" nil I   L   () "Dené–Yeniseian" "Diné bizaad" "Navajo")
            ("naw" nil  nil   nil I   L   () nil nil "Nawuri")
            ("nax" nil  nil   nil I   L   () nil nil "Nakwi")
            ("nay" nil  nil   nil I   E   () nil nil "Narrinyeri")
            ("naz" nil  nil   nil I   L   () nil nil "Nahuatl, Coatepec")
            ("nba" nil  nil   nil I   L   () nil nil "Nyemba")
            ("nbb" nil  nil   nil I   L   () nil nil "Ndoe")
            ("nbc" nil  nil   nil I   L   () nil nil "Naga, Chang")
            ("nbd" nil  nil   nil I   L   () nil nil "Ngbinda")
            ("nbe" nil  nil   nil I   L   () nil nil "Naga, Konyak")
            ("nbf" nil  nil   t   I   L   () nil nil "Naxi")
            ("nbg" nil  nil   nil I   L   () nil nil "Nagarchal")
            ("nbh" nil  nil   nil I   L   () nil nil "Ngamo")
            ("nbi" nil  nil   nil I   L   () nil nil "Naga, Mao")
            ("nbj" nil  nil   nil I   L   () nil nil "Ngarinman")
            ("nbk" nil  nil   nil I   L   () nil nil "Nake")
            ("nbl" "nr" "nbl" nil I   L   () "Niger–Congo" "Ndébélé" "Ndebele, South")
            ("nbm" nil  nil   nil I   L   () nil nil "Ngbaka Ma'bo")
            ("nbn" nil  nil   nil I   L   () nil nil "Kuri")
            ("nbo" nil  nil   nil I   L   () nil nil "Nkukoli")
            ("nbp" nil  nil   nil I   L   () nil nil "Nnam")
            ("nbq" nil  nil   nil I   L   () nil nil "Nggem")
            ("nbr" nil  nil   nil I   L   () nil nil "Numana-Nunku-Gbantu-Numbu")
            ("nbs" nil  nil   nil I   L   () nil nil "Namibian Sign Language")
            ("nbt" nil  nil   nil I   L   () nil nil "Na")
            ("nbu" nil  nil   nil I   L   () nil nil "Naga, Rongmei")
            ("nbv" nil  nil   nil I   L   () nil nil "Ngamambo")
            ("nbw" nil  nil   nil I   L   () nil nil "Ngbandi, Southern")
            ("nbx" nil  nil   t   I   E   () nil nil "Ngura")
            ("nby" nil  nil   nil I   L   () nil nil "Ningera")
            ("nca" nil  nil   nil I   L   () nil nil "Iyo")
            ("ncb" nil  nil   nil I   L   () nil nil "Nicobarese, Central")
            ("ncc" nil  nil   nil I   L   () nil nil "Ponam")
            ("ncd" nil  nil   nil I   L   () nil nil "Nachhiring")
            ("nce" nil  nil   nil I   L   () nil nil "Yale")
            ("ncf" nil  nil   nil I   L   () nil nil "Notsi")
            ("ncg" nil  nil   nil I   L   () nil "Nisǥa’a" "Nisga'a")
            ("nch" nil  nil   nil I   L   () nil nil "Nahuatl, Central Huasteca")
            ("nci" nil  nil   nil I   H   () nil nil "Nahuatl, Classical")
            ("ncj" nil  nil   nil I   L   () nil nil "Nahuatl, Northern Puebla")
            ("nck" nil  nil   nil I   L   () nil nil "Nakara")
            ("ncl" nil  nil   nil I   L   () nil nil "Nahuatl, Michoacán")
            ("ncm" nil  nil   nil I   L   () nil nil "Nambo")
            ("ncn" nil  nil   nil I   L   () nil nil "Nauna")
            ("nco" nil  nil   nil I   L   () nil nil "Sibe")
            ("ncp" nil  nil   t   I   L   () nil nil "Ndaktup")
            ("ncq" nil  nil   nil I   L   () "Austroasiatic" nil "Northern Katang")
            ("ncr" nil  nil   nil I   L   () nil nil "Ncane")
            ("ncs" nil  nil   nil I   L   () nil nil "Nicaraguan Sign Language")
            ("nct" nil  nil   nil I   L   () nil nil "Naga, Chothe")
            ("ncu" nil  nil   nil I   L   () nil nil "Chumburung")
            ("ncx" nil  nil   nil I   L   () nil nil "Nahuatl, Central Puebla")
            ("ncz" nil  nil   nil I   E   () nil nil "Natchez")
            ("nda" nil  nil   nil I   L   () nil nil "Ndasa")
            ("ndb" nil  nil   nil I   L   () nil nil "Kenswei Nsei")
            ("ndc" nil  nil   nil I   L   () nil nil "Ndau")
            ("ndd" nil  nil   nil I   L   () nil nil "Nde-Nsele-Nta")
            ("nde" "nd" "nde" nil I   L   () "Niger–Congo" "isiNdebele" "Ndebele, North")
            ("ndf" nil  nil   nil I   H   () nil nil "Nadruvian")
            ("ndg" nil  nil   nil I   L   () nil nil "Ndengereko")
            ("ndh" nil  nil   nil I   L   () nil nil "Ndali")
            ("ndi" nil  nil   nil I   L   () nil nil "Samba Leko")
            ("ndj" nil  nil   nil I   L   () nil nil "Ndamba")
            ("ndk" nil  nil   nil I   L   () nil nil "Ndaka")
            ("ndl" nil  nil   nil I   L   () nil nil "Ndolo")
            ("ndm" nil  nil   nil I   L   () nil nil "Ndam")
            ("ndn" nil  nil   nil I   L   () nil nil "Ngundi")
            ("ndo" "ng" "ndo" nil I   L   () "Niger–Congo" "O(shi)wambo" "Ndonga")
            ("ndp" nil  nil   nil I   L   () nil nil "Ndo")
            ("ndq" nil  nil   nil I   L   () nil nil "Ndombe")
            ("ndr" nil  nil   nil I   L   () nil nil "Ndoola")
            ("nds" nil  "nds" nil I   L   () nil "Plattdüütsch; Neddersass'sch" "Low German; Low Saxon")
            ("ndt" nil  nil   nil I   L   () nil nil "Ndunga")
            ("ndu" nil  nil   nil I   L   () nil nil "Dugun")
            ("ndv" nil  nil   nil I   L   () nil nil "Ndut")
            ("ndw" nil  nil   nil I   L   () nil nil "Ndobo")
            ("ndx" nil  nil   nil I   L   () nil nil "Nduga")
            ("ndy" nil  nil   nil I   L   () nil nil "Lutos")
            ("ndz" nil  nil   nil I   L   () nil nil "Ndogo")
            ("nea" nil  nil   nil I   L   () nil nil "Ngad'a, Eastern")
            ("neb" nil  nil   nil I   L   () nil nil "Toura (Côte d'Ivoire)")
            ("nec" nil  nil   nil I   L   () nil nil "Nedebang")
            ("ned" nil  nil   nil I   L   () nil nil "Nde-Gbite")
            ("nee" nil  nil   nil I   L   () nil nil "Kumak")
            ("nef" nil  nil   nil I   L   () nil nil "Nefamese")
            ("neg" nil  nil   nil I   L   () nil nil "Negidal")
            ("neh" nil  nil   nil I   L   () nil nil "Nyenkha")
            ("nei" nil  nil   nil I   A   () nil nil "Hittite, Neo-")
            ("nej" nil  nil   nil I   L   () nil nil "Neko")
            ("nek" nil  nil   nil I   L   () nil nil "Neku")
            ("nem" nil  nil   nil I   L   () nil nil "Nemi")
            ("nen" nil  nil   nil I   L   () nil nil "Nengone")
            ("neo" nil  nil   nil I   L   () nil nil "Ná-Meo")
            ("nep" "ne" "nep" nil M   L   ("dty" "npi") "Indo-European" "नेपाली" "Nepali")
            ("neq" nil  nil   nil I   L   () nil nil "Mixe, North Central")
            ("ner" nil  nil   nil I   L   () nil nil "Yahadian")
            ("nes" nil  nil   nil I   L   () nil nil "Kinnauri, Bhoti")
            ("net" nil  nil   nil I   L   () nil nil "Nete")
            ("neu" nil  nil   nil I   C   () nil nil "Neo")
            ("nev" nil  nil   nil I   L   () nil nil "Nyaheun")
            ("new" nil  "new" nil I   L   () nil "नेपाल भाषा" "Newari; Nepal Bhasa")
            ("nex" nil  nil   nil I   L   () nil nil "Neme")
            ("ney" nil  nil   nil I   L   () nil nil "Neyo")
            ("nez" nil  nil   nil I   L   () nil nil "Nez Perce")
            ("nfa" nil  nil   nil I   L   () nil nil "Dhao")
            ("nfd" nil  nil   nil I   L   () nil nil "Ndun")
            ("nfg" nil  nil   t   I   L   () nil nil "Nyeng")
            ("nfk" nil  nil   t   I   L   () nil nil "Shakara")
            ("nfl" nil  nil   nil I   L   () nil nil "Ayiwo")
            ("nfr" nil  nil   nil I   L   () nil nil "Nafaanra")
            ("nfu" nil  nil   nil I   L   () nil nil "Mfumte")
            ("nga" nil  nil   nil I   L   () nil nil "Ngbaka")
            ("ngb" nil  nil   nil I   L   () nil nil "Ngbandi, Northern")
            ("ngc" nil  nil   nil I   L   () nil nil "Ngombe (Democratic Republic of Congo)")
            ("ngd" nil  nil   nil I   L   () nil nil "Ngando (Central African Republic)")
            ("nge" nil  nil   nil I   L   () nil nil "Ngemba")
            ("ngg" nil  nil   nil I   L   () nil nil "Ngbaka Manza")
            ("ngh" nil  nil   nil I   L   () nil nil "N/u")
            ("ngi" nil  nil   nil I   L   () nil nil "Ngizim")
            ("ngj" nil  nil   nil I   L   () nil nil "Ngie")
            ("ngk" nil  nil   nil I   L   () nil nil "Ngalkbun")
            ("ngl" nil  nil   nil I   L   () nil nil "Lomwe")
            ("ngm" nil  nil   nil I   L   () nil nil "Ngatik Men's Creole")
            ("ngn" nil  nil   nil I   L   () nil nil "Ngwo")
            ("ngo" nil  nil   t   I   L   () nil nil "Ngoni")
            ("ngp" nil  nil   nil I   L   () nil nil "Ngulu")
            ("ngq" nil  nil   nil I   L   () nil nil "Ngurimi")
            ("ngr" nil  nil   nil I   L   () nil nil "Nanggu")
            ("ngs" nil  nil   nil I   L   () nil nil "Gvoko")
            ("ngt" nil  nil   nil I   L   () nil nil "Ngeq")
            ("ngu" nil  nil   nil I   L   () nil nil "Nahuatl, Guerrero")
            ("ngv" nil  nil   nil I   E   () nil nil "Nagumi")
            ("ngw" nil  nil   nil I   L   () nil nil "Ngwaba")
            ("ngx" nil  nil   nil I   L   () nil nil "Nggwahyi")
            ("ngy" nil  nil   nil I   L   () nil nil "Tibea")
            ("ngz" nil  nil   nil I   L   () nil nil "Ngungwel")
            ("nha" nil  nil   nil I   L   () nil nil "Nhanda")
            ("nhb" nil  nil   nil I   L   () nil nil "Beng")
            ("nhc" nil  nil   nil I   E   () nil nil "Nahuatl, Tabasco")
            ("nhd" nil  nil   nil I   L   () nil nil "Chiripá")
            ("nhe" nil  nil   nil I   L   () nil nil "Nahuatl, Eastern Huasteca")
            ("nhf" nil  nil   nil I   L   () nil nil "Nhuwala")
            ("nhg" nil  nil   nil I   L   () nil nil "Nahuatl, Tetelcingo")
            ("nhh" nil  nil   nil I   L   () nil nil "Nahari")
            ("nhi" nil  nil   nil I   L   () nil nil "Nahuatl, Tenango")
            ("nhj" nil  nil   t   nil nil () nil nil "Tlalitzlipa Nahuatl")
            ("nhk" nil  nil   nil I   L   () nil nil "Nahuatl, Isthmus-Cosoleacaque")
            ("nhm" nil  nil   nil I   L   () nil nil "Nahuatl, Morelos")
            ("nhn" nil  nil   nil I   L   () nil nil "Nahuatl, Central")
            ("nho" nil  nil   nil I   L   () nil nil "Takuu")
            ("nhp" nil  nil   nil I   L   () nil nil "Nahuatl, Isthmus-Pajapan")
            ("nhq" nil  nil   nil I   L   () nil nil "Nahuatl, Huaxcaleca")
            ("nhr" nil  nil   nil I   L   () nil nil "Naro")
            ("nhs" nil  nil   t   nil nil () nil nil "Southeastern Puebla Nahuatl")
            ("nht" nil  nil   nil I   L   () nil nil "Nahuatl, Ometepec")
            ("nhu" nil  nil   nil I   L   () nil nil "Noone")
            ("nhv" nil  nil   nil I   L   () nil nil "Nahuatl, Temascaltepec")
            ("nhw" nil  nil   nil I   L   () nil nil "Nahuatl, Western Huasteca")
            ("nhx" nil  nil   nil I   L   () nil nil "Nahuatl, Isthmus-Mecayapan")
            ("nhy" nil  nil   nil I   L   () nil nil "Nahuatl, Northern Oaxaca")
            ("nhz" nil  nil   nil I   L   () nil nil "Nahuatl, Santa María La Alta")
            ("nia" nil  "nia" nil I   L   () nil nil "Nias")
            ("nib" nil  nil   nil I   L   () nil nil "Nakama")
            (nil   nil  "nic" nil C   nil () nil nil "Niger-Kordofanian languages")
            ("nid" nil  nil   nil I   E   () nil nil "Ngandi")
            ("nie" nil  nil   nil I   L   () nil nil "Niellim")
            ("nif" nil  nil   nil I   L   () nil nil "Nek")
            ("nig" nil  nil   nil I   E   () nil nil "Ngalakan")
            ("nih" nil  nil   nil I   L   () nil nil "Nyiha")
            ("nii" nil  nil   nil I   L   () nil nil "Nii")
            ("nij" nil  nil   nil I   L   () nil nil "Ngaju")
            ("nik" nil  nil   nil I   L   () nil nil "Nicobarese, Southern")
            ("nil" nil  nil   nil I   L   () nil nil "Nila")
            ("nim" nil  nil   nil I   L   () nil nil "Nilamba")
            ("nin" nil  nil   nil I   L   () nil nil "Ninzo")
            ("nio" nil  nil   nil I   L   () nil "нганасаны" "Nganasan")
            ("niq" nil  nil   nil I   L   () nil nil "Nandi")
            ("nir" nil  nil   nil I   L   () nil nil "Nimboran")
            ("nis" nil  nil   nil I   L   () nil nil "Nimi")
            ("nit" nil  nil   nil I   L   () nil nil "Kolami, Southeastern")
            ("niu" nil  "niu" nil I   L   () nil "ko e vagahau Niuē" "Niuean")
            ("niv" nil  nil   nil I   L   () nil "нивхгу" "Gilyak; Nivkh")
            ("niw" nil  nil   nil I   L   () nil nil "Nimo")
            ("nix" nil  nil   nil I   L   () nil nil "Hema")
            ("niy" nil  nil   nil I   L   () nil nil "Ngiti")
            ("niz" nil  nil   nil I   L   () nil nil "Ningil")
            ("nja" nil  nil   nil I   L   () nil nil "Nzanyi")
            ("njb" nil  nil   nil I   L   () "Naga" nil "Naga, Nocte")
            ("njd" nil  nil   nil I   L   () nil nil "Ndonde Hamba")
            ("njh" nil  nil   nil I   L   () "Naga" nil "Naga, Lotha")
            ("nji" nil  nil   nil I   L   () nil nil "Gudanji")
            ("njj" nil  nil   nil I   L   () nil nil "Njen")
            ("njl" nil  nil   nil I   L   () nil nil "Njalgulgule")
            ("njm" nil  nil   nil I   L   () "Naga" nil "Naga, Angami")
            ("njn" nil  nil   nil I   L   () "Naga" nil "Naga, Liangmai")
            ("njo" nil  nil   nil I   L   () "Naga" nil "Naga, Ao")
            ("njr" nil  nil   nil I   L   () nil nil "Njerep")
            ("njs" nil  nil   nil I   L   () nil nil "Nisa")
            ("njt" nil  nil   nil I   L   () nil nil "Ndyuka-Trio Pidgin")
            ("nju" nil  nil   nil I   L   () nil nil "Ngadjunmaya")
            ("njx" nil  nil   nil I   L   () nil nil "Kunyi")
            ("njy" nil  nil   nil I   L   () nil nil "Njyem")
            ("njz" nil  nil   nil I   L   () nil nil "Nyishi")
            ("nka" nil  nil   nil I   L   () nil nil "Nkoya")
            ("nkb" nil  nil   nil I   L   () nil nil "Naga, Khoibu")
            ("nkc" nil  nil   nil I   L   () nil nil "Nkongho")
            ("nkd" nil  nil   nil I   L   () nil nil "Koireng")
            ("nke" nil  nil   nil I   L   () nil nil "Duke")
            ("nkf" nil  nil   nil I   L   () nil nil "Naga, Inpui")
            ("nkg" nil  nil   nil I   L   () nil nil "Nekgini")
            ("nkh" nil  nil   nil I   L   () nil nil "Naga, Khezha")
            ("nki" nil  nil   nil I   L   () nil nil "Naga, Thangal")
            ("nkj" nil  nil   nil I   L   () nil nil "Nakai")
            ("nkk" nil  nil   nil I   L   () nil nil "Nokuku")
            ("nkm" nil  nil   nil I   L   () nil nil "Namat")
            ("nkn" nil  nil   nil I   L   () nil nil "Nkangala")
            ("nko" nil  nil   nil I   L   () nil nil "Nkonya")
            ("nkp" nil  nil   nil I   E   () nil nil "Niuatoputapu")
            ("nkq" nil  nil   nil I   L   () nil nil "Nkami")
            ("nkr" nil  nil   nil I   L   () nil nil "Nukuoro")
            ("nks" nil  nil   nil I   L   () nil nil "Asmat, North")
            ("nkt" nil  nil   nil I   L   () nil nil "Nyika (Tanzania)")
            ("nku" nil  nil   nil I   L   () nil nil "Kulango, Bouna")
            ("nkv" nil  nil   nil I   L   () nil nil "Nyika (Malawi and Zambia)")
            ("nkw" nil  nil   nil I   L   () nil nil "Nkutu")
            ("nkx" nil  nil   nil I   L   () nil nil "Nkoroo")
            ("nky" nil  nil   t   I   L   () nil nil "Naga, Khiamniungan")
            ("nkz" nil  nil   nil I   L   () nil nil "Nkari")
            ("nla" nil  nil   nil I   L   () nil nil "Ngombale")
            ("nlc" nil  nil   nil I   L   () nil nil "Nalca")
            ("nld" "nl" "dut" nil I   L   () "Indo-European" "Nederlands" "Dutch")
            ("nle" nil  nil   nil I   L   () nil nil "Nyala, East")
            ("nlg" nil  nil   nil I   L   () nil nil "Gela")
            ("nli" nil  nil   nil I   L   () nil nil "Grangali")
            ("nlj" nil  nil   nil I   L   () nil nil "Nyali")
            ("nlk" nil  nil   nil I   L   () nil nil "Yali, Ninia")
            ("nll" nil  nil   nil I   L   () nil nil "Nihali")
            ("nlm" nil  nil   nil I   L   () "Indo-European" nil "Mankiyali")
            ("nln" nil  nil   t   I   L   () nil nil "Nahuatl, Durango")
            ("nlo" nil  nil   nil I   L   () nil nil "Ngul")
            ("nlq" nil  nil   nil I   L   () nil nil "Lao Naga")
            ("nlr" nil  nil   t   I   L   () nil nil "Ngarla")
            ("nlu" nil  nil   nil I   L   () nil nil "Nchumbulu")
            ("nlv" nil  nil   nil I   L   () nil nil "Nahuatl, Orizaba")
            ("nlw" nil  nil   nil I   E   () nil nil "Walangama")
            ("nlx" nil  nil   nil I   L   () nil nil "Nahali")
            ("nly" nil  nil   nil I   L   () nil nil "Nyamal")
            ("nlz" nil  nil   nil I   L   () nil nil "Nalögo")
            ("nma" nil  nil   nil I   L   () nil nil "Naga, Maram")
            ("nmb" nil  nil   nil I   L   () nil nil "Nambas, Big")
            ("nmc" nil  nil   nil I   L   () nil nil "Ngam")
            ("nmd" nil  nil   nil I   L   () nil nil "Ndumu")
            ("nme" nil  nil   nil I   L   () nil nil "Naga, Mzieme")
            ("nmf" nil  nil   nil I   L   () nil nil "Naga, Tangkhul")
            ("nmg" nil  nil   nil I   L   () nil nil "Ngumba")
            ("nmh" nil  nil   nil I   L   () nil nil "Naga, Monsang")
            ("nmi" nil  nil   nil I   L   () nil nil "Nyam")
            ("nmj" nil  nil   nil I   L   () nil nil "Ngombe (Central African Republic)")
            ("nmk" nil  nil   nil I   L   () nil nil "Namakura")
            ("nml" nil  nil   nil I   L   () nil nil "Ndemli")
            ("nmm" nil  nil   nil I   L   () nil nil "Manangba")
            ("nmn" nil  nil   nil I   L   () nil nil "!Xóõ")
            ("nmo" nil  nil   nil I   L   () nil nil "Naga, Moyon")
            ("nmp" nil  nil   nil I   E   () nil nil "Nimanbur")
            ("nmq" nil  nil   nil I   L   () nil nil "Nambya")
            ("nmr" nil  nil   nil I   E   () nil nil "Nimbari")
            ("nms" nil  nil   nil I   L   () nil nil "Letemboi")
            ("nmt" nil  nil   nil I   L   () nil nil "Namonuito")
            ("nmu" nil  nil   nil I   L   () nil nil "Maidu, Northeast")
            ("nmv" nil  nil   nil I   E   () nil nil "Ngamini")
            ("nmw" nil  nil   nil I   L   () nil nil "Nimoa")
            ("nmx" nil  nil   nil I   L   () nil nil "Nama (Papua New Guinea)")
            ("nmy" nil  nil   nil I   L   () nil nil "Namuyi")
            ("nmz" nil  nil   nil I   L   () nil nil "Nawdm")
            ("nna" nil  nil   nil I   L   () nil nil "Nyangumarta")
            ("nnb" nil  nil   nil I   L   () nil nil "Nande")
            ("nnc" nil  nil   nil I   L   () nil nil "Nancere")
            ("nnd" nil  nil   nil I   L   () nil nil "Ambae, West")
            ("nne" nil  nil   nil I   L   () nil nil "Ngandyera")
            ("nnf" nil  nil   nil I   L   () nil nil "Ngaing")
            ("nng" nil  nil   nil I   L   () nil nil "Naga, Maring")
            ("nnh" nil  nil   nil I   L   () nil nil "Ngiemboon")
            ("nni" nil  nil   nil I   L   () nil nil "Nuaulu, North")
            ("nnj" nil  nil   nil I   L   () nil nil "Nyangatom")
            ("nnk" nil  nil   nil I   L   () nil nil "Nankina")
            ("nnl" nil  nil   nil I   L   () nil nil "Naga, Northern Rengma")
            ("nnm" nil  nil   nil I   L   () nil nil "Namia")
            ("nnn" nil  nil   nil I   L   () nil nil "Ngete")
            ("nno" "nn" "nno" nil I   L   () "Indo-European" "Nynorsk" "Norwegian Nynorsk")
            ("nnp" nil  nil   nil I   L   () nil nil "Naga, Wancho")
            ("nnq" nil  nil   nil I   L   () nil nil "Ngindo")
            ("nnr" nil  nil   nil I   E   () nil nil "Narungga")
            ("nns" nil  nil   t   I   L   () nil nil "Ningye")
            ("nnt" nil  nil   nil I   E   () nil nil "Nanticoke")
            ("nnu" nil  nil   nil I   L   () nil nil "Dwang")
            ("nnv" nil  nil   nil I   E   () nil nil "Nugunu (Australia)")
            ("nnw" nil  nil   nil I   L   () nil nil "Nuni, Southern")
            ("nnx" nil  nil   t   I   L   () nil nil "Ngong")
            ("nny" nil  nil   nil I   E   () nil nil "Nyangga")
            ("nnz" nil  nil   nil I   L   () nil nil "Nda'nda'")
            ("noa" nil  nil   nil I   L   () nil nil "Woun Meu")
            ("nob" "nb" "nob" nil I   L   () "Indo-European" "Bokmål" "Norwegian Bokmål")
            ("noc" nil  nil   nil I   L   () nil nil "Nuk")
            ("nod" nil  nil   nil I   L   () nil "ล้านนา" "Thai (Northern)")
            ("noe" nil  nil   nil I   L   () nil nil "Nimadi")
            ("nof" nil  nil   nil I   L   () nil nil "Nomane")
            ("nog" nil  "nog" nil I   L   () nil "Ногай" "Nogai")
            ("noh" nil  nil   nil I   L   () nil nil "Nomu")
            ("noi" nil  nil   nil I   L   () nil nil "Noiri")
            ("noj" nil  nil   nil I   L   () nil nil "Nonuya")
            ("nok" nil  nil   nil I   E   () nil nil "Nooksack")
            ("nol" nil  nil   nil I   E   () nil nil "Nomlaki")
            ("nom" nil  nil   t   I   E   () nil nil "Nocamán")
            ("non" nil  "non" nil I   H   () nil "norrǿna" "Norse, Old")
            ("noo" nil  nil   t   I   L   () nil "Nuučaan̓uł" "Nootka")
            ("nop" nil  nil   nil I   L   () nil nil "Numanggang")
            ("noq" nil  nil   nil I   L   () nil nil "Ngongo")
            ("nor" "no" "nor" nil M   L   ("nno" "nob") "Indo-European" "norsk" "Norwegian")
            ("nos" nil  nil   nil I   L   () nil nil "Yi, Eastern Nisu")
            ("not" nil  nil   nil I   L   () nil nil "Nomatsiguenga")
            ("nou" nil  nil   nil I   L   () nil nil "Ewage-Notu")
            ("nov" nil  nil   nil I   C   () nil "novial" "Novial")
            ("now" nil  nil   nil I   L   () nil nil "Nyambo")
            ("noy" nil  nil   nil I   L   () nil nil "Noy")
            ("noz" nil  nil   nil I   L   () nil nil "Nayi")
            ("npa" nil  nil   nil I   L   () nil nil "Nar Phu")
            ("npb" nil  nil   nil I   L   () nil nil "Nupbikha")
            ("npg" nil  nil   nil I   L   () nil nil "Ponyo-Gongwang Naga")
            ("nph" nil  nil   nil I   L   () nil nil "Naga, Phom")
            ("npi" nil  nil   nil I   L   () nil nil "Nepali (individual language)")
            ("npl" nil  nil   nil I   L   () nil nil "Southeastern Puebla Nahuatl")
            ("npn" nil  nil   nil I   L   () nil nil "Mondropolon")
            ("npo" nil  nil   nil I   L   () nil nil "Naga, Pochuri")
            ("nps" nil  nil   nil I   L   () nil nil "Nipsan")
            ("npu" nil  nil   nil I   L   () nil nil "Naga, Puimei")
            ("npx" nil  nil   nil I   L   () "Austronesian" nil "Noipx")
            ("npy" nil  nil   nil I   L   () nil nil "Napu")
            ("nqg" nil  nil   nil I   L   () nil nil "Ede Nago")
            ("nqk" nil  nil   nil I   L   () nil nil "Ede Nago, Kura")
            ("nql" nil  nil   nil I   L   () "Niger–Congo" nil "Ngendelengo")
            ("nqm" nil  nil   nil I   L   () nil nil "Ndom")
            ("nqn" nil  nil   nil I   L   () nil nil "Nen")
            ("nqo" nil  "nqo" nil I   L   () nil nil "N'Ko")
            ("nqq" nil  nil   nil I   L   () nil nil "Kyan-Karyaw Naga")
            ("nqt" nil  nil   nil I   L   () "Afro-Asiatic" nil "Nteng")
            ("nqy" nil  nil   nil I   L   () nil nil "Akyaung Ari Naga")
            ("nra" nil  nil   nil I   L   () nil nil "Ngom")
            ("nrb" nil  nil   nil I   L   () nil nil "Nara")
            ("nrc" nil  nil   nil I   A   () nil nil "Noric")
            ("nre" nil  nil   nil I   L   () nil nil "Naga, Southern Rengma")
            ("nrf" nil  nil   nil I   L   () "Indo-European" nil "Guernésiais, Jèrriais")
            ("nrg" nil  nil   nil I   L   () nil nil "Narango")
            ("nri" nil  nil   nil I   L   () nil nil "Naga, Chokri")
            ("nrk" nil  nil   nil I   L   () nil nil "Ngarla")
            ("nrl" nil  nil   nil I   L   () nil nil "Ngarluma")
            ("nrm" nil  nil   nil I   L   () nil nil "Narom")
            ("nrn" nil  nil   nil I   E   () nil nil "Norn")
            ("nrp" nil  nil   nil I   A   () nil nil "North Picene")
            ("nrr" nil  nil   nil I   E   () nil nil "Norra")
            ("nrt" nil  nil   nil I   E   () nil nil "Northern Kalapuya")
            ("nru" nil  nil   nil I   L   () nil nil "Narua")
            ("nrx" nil  nil   nil I   E   () nil nil "Ngurmbur")
            ("nrz" nil  nil   nil I   L   () nil nil "Lala")
            ("nsa" nil  nil   nil I   L   () nil nil "Naga, Sangtam")
            ("nsb" nil  nil   nil I   E   () nil nil "Lower Nossob")
            ("nsc" nil  nil   nil I   L   () nil nil "Nshi")
            ("nsd" nil  nil   nil I   L   () nil nil "Southern Nisu")
            ("nse" nil  nil   nil I   L   () nil nil "Nsenga")
            ("nsf" nil  nil   nil I   L   () nil nil "Northwestern Nisu")
            ("nsg" nil  nil   nil I   L   () nil nil "Ngasa")
            ("nsh" nil  nil   nil I   L   () nil nil "Ngoshie")
            ("nsi" nil  nil   nil I   L   () nil nil "Nigerian Sign Language")
            ("nsk" nil  nil   nil I   L   () nil "ᓇᔅᑲᐱ" "Naskapi")
            ("nsl" nil  nil   nil I   L   () nil nil "Norwegian Sign Language")
            ("nsm" nil  nil   nil I   L   () nil nil "Naga, Sumi")
            ("nsn" nil  nil   nil I   L   () nil nil "Nehan")
            ("nso" nil  "nso" nil I   L   () nil "sePêdi" "Northern Sotho, Pedi; Sepedi")
            ("nsp" nil  nil   nil I   L   () nil nil "Nepalese Sign Language")
            ("nsq" nil  nil   nil I   L   () nil nil "Miwok, Northern Sierra")
            ("nsr" nil  nil   nil I   L   () nil nil "Maritime Sign Language")
            ("nss" nil  nil   nil I   L   () nil nil "Nali")
            ("nst" nil  nil   nil I   L   () nil nil "Naga, Tase")
            ("nsu" nil  nil   nil I   L   () nil nil "Sierra Negra Nahuatl")
            ("nsv" nil  nil   nil I   L   () nil nil "Southwestern Nisu")
            ("nsw" nil  nil   nil I   L   () nil nil "Navut")
            ("nsx" nil  nil   nil I   L   () nil nil "Nsongo")
            ("nsy" nil  nil   nil I   L   () nil nil "Nasal")
            ("nsz" nil  nil   nil I   L   () nil nil "Nisenan")
            ("ntd" nil  nil   nil I   L   () "Austronesian" nil "Northern Tidung")
            ("nte" nil  nil   nil I   L   () nil nil "Nathembo")
            ("ntg" nil  nil   nil I   E   () nil nil "Ngantangarra")
            ("nti" nil  nil   nil I   L   () nil nil "Natioro")
            ("ntj" nil  nil   nil I   L   () nil nil "Ngaanyatjarra")
            ("ntk" nil  nil   nil I   L   () nil nil "Ikoma")
            ("ntm" nil  nil   nil I   L   () nil nil "Nateni")
            ("nto" nil  nil   nil I   L   () nil nil "Ntomba")
            ("ntp" nil  nil   nil I   L   () nil nil "Tepehuan, Northern")
            ("ntr" nil  nil   nil I   L   () nil nil "Delo")
            ("nts" nil  nil   t   I   E   () nil nil "Natagaimas")
            ("ntu" nil  nil   nil I   L   () nil nil "Natügu")
            ("ntw" nil  nil   nil I   E   () nil nil "Nottoway")
            ("ntx" nil  nil   nil I   L   () nil nil "Tangkhul Naga (Myanmar)")
            ("nty" nil  nil   nil I   L   () nil nil "Mantsi")
            ("ntz" nil  nil   nil I   L   () nil nil "Natanzi")
            ("nua" nil  nil   nil I   L   () nil nil "Yuaga")
            (nil   nil  "nub" nil C   nil () nil nil "Nubian languages")
            ("nuc" nil  nil   nil I   E   () nil nil "Nukuini")
            ("nud" nil  nil   nil I   L   () nil nil "Ngala")
            ("nue" nil  nil   nil I   L   () nil nil "Ngundu")
            ("nuf" nil  nil   nil I   L   () nil nil "Nusu")
            ("nug" nil  nil   nil I   E   () nil nil "Nungali")
            ("nuh" nil  nil   nil I   L   () nil nil "Ndunda")
            ("nui" nil  nil   nil I   L   () nil nil "Ngumbi")
            ("nuj" nil  nil   nil I   L   () nil nil "Nyole")
            ("nuk" nil  nil   nil I   L   () nil nil "Nuu-chah-nulth")
            ("nul" nil  nil   nil I   E   () nil nil "Nusa Laut")
            ("num" nil  nil   nil I   L   () nil nil "Niuafo'ou")
            ("nun" nil  nil   nil I   L   () nil nil "Nung (Burma)")
            ("nuo" nil  nil   nil I   L   () nil nil "Nguôn")
            ("nup" nil  nil   nil I   L   () nil nil "Nupe-Nupe-Tako")
            ("nuq" nil  nil   nil I   L   () nil nil "Nukumanu")
            ("nur" nil  nil   nil I   L   () nil nil "Nukuria")
            ("nus" nil  nil   nil I   L   () nil nil "Nuer")
            ("nut" nil  nil   nil I   L   () nil nil "Nung (Viet Nam)")
            ("nuu" nil  nil   nil I   L   () nil nil "Ngbundu")
            ("nuv" nil  nil   nil I   L   () nil nil "Nuni, Northern")
            ("nuw" nil  nil   nil I   L   () nil nil "Nguluwan")
            ("nux" nil  nil   nil I   L   () nil nil "Mehek")
            ("nuy" nil  nil   nil I   L   () nil nil "Nunggubuyu")
            ("nuz" nil  nil   nil I   L   () nil nil "Nahuatl, Tlamacazapa")
            ("nvh" nil  nil   nil I   L   () nil nil "Nasarian")
            ("nvm" nil  nil   nil I   L   () nil nil "Namiae")
            ("nvo" nil  nil   nil I   L   () nil nil "Nyokon")
            ("nwa" nil  nil   nil I   E   () nil nil "Nawathinehena")
            ("nwb" nil  nil   nil I   L   () nil nil "Nyabwa")
            ("nwc" nil  "nwc" nil I   H   () nil nil "Classical Newari; Old Newari")
            ("nwe" nil  nil   nil I   L   () nil nil "Ngwe")
            ("nwg" nil  nil   nil I   E   () nil nil "Ngayawung")
            ("nwi" nil  nil   nil I   L   () nil nil "Tanna, Southwest")
            ("nwm" nil  nil   nil I   L   () nil nil "Nyamusa-Molo")
            ("nwo" nil  nil   nil I   E   () nil nil "Nauo")
            ("nwr" nil  nil   nil I   L   () nil nil "Nawaru")
            ("nww" nil  nil   nil I   L   () "Niger–Congo" nil "Ndwewe")
            ("nwx" nil  nil   nil I   H   () nil nil "Newar, Middle")
            ("nwy" nil  nil   nil I   E   () nil nil "Nottoway-Meherrin")
            ("nxa" nil  nil   nil I   L   () nil nil "Nauete")
            ("nxd" nil  nil   nil I   L   () nil nil "Ngando (Democratic Republic of Congo)")
            ("nxe" nil  nil   nil I   L   () nil nil "Nage")
            ("nxg" nil  nil   nil I   L   () nil nil "Ngad'a")
            ("nxi" nil  nil   nil I   L   () nil nil "Nindi")
            ("nxj" nil  nil   t   nil nil () nil nil "Nyadu")
            ("nxk" nil  nil   nil I   L   () nil nil "Koki Naga")
            ("nxl" nil  nil   nil I   L   () nil nil "Nuaulu, South")
            ("nxm" nil  nil   nil I   A   () nil nil "Numidian")
            ("nxn" nil  nil   nil I   E   () nil nil "Ngawun")
            ("nxo" nil  nil   nil I   L   () "Niger–Congo" nil "Ndambomo")
            ("nxq" nil  nil   nil I   L   () nil nil "Naxi")
            ("nxr" nil  nil   nil I   L   () nil nil "Ninggerum")
            ("nxu" nil  nil   t   I   E   () nil nil "Narau")
            ("nxx" nil  nil   nil I   L   () nil nil "Nafri")
            ("nya" "ny" "nya" nil I   L   () "Niger–Congo" "chiCheŵa" "Chichewa; Nyanja")
            ("nyb" nil  nil   nil I   L   () nil nil "Nyangbo")
            ("nyc" nil  nil   nil I   L   () nil nil "Nyanga-li")
            ("nyd" nil  nil   nil I   L   () nil nil "Nyore")
            ("nye" nil  nil   nil I   L   () nil nil "Nyengo")
            ("nyf" nil  nil   nil I   L   () nil nil "Giryama")
            ("nyg" nil  nil   nil I   L   () nil nil "Nyindu")
            ("nyh" nil  nil   nil I   L   () nil nil "Nyigina")
            ("nyi" nil  nil   nil I   L   () nil nil "Ama (Sudan)")
            ("nyj" nil  nil   nil I   L   () nil nil "Nyanga")
            ("nyk" nil  nil   nil I   L   () nil nil "Nyaneka")
            ("nyl" nil  nil   nil I   L   () nil nil "Nyeu")
            ("nym" nil  "nym" nil I   L   () nil "Kinyamwezi" "Nyamwezi")
            ("nyn" nil  "nyn" nil I   L   () nil nil "Nyankole")
            ("nyo" nil  "nyo" nil I   L   () nil "Orunyoro" "Nyoro")
            ("nyp" nil  nil   nil I   E   () nil nil "Nyang'i")
            ("nyq" nil  nil   nil I   L   () nil nil "Nayini")
            ("nyr" nil  nil   nil I   L   () nil nil "Nyiha (Malawi)")
            ("nys" nil  nil   nil I   L   () "Pama–Nyungan" "Noongar" "Nyungah")
            ("nyt" nil  nil   nil I   E   () nil nil "Nyawaygi")
            ("nyu" nil  nil   nil I   L   () nil nil "Nyungwe")
            ("nyv" nil  nil   nil I   E   () nil nil "Nyulnyul")
            ("nyw" nil  nil   nil I   L   () nil nil "Nyaw")
            ("nyx" nil  nil   nil I   E   () nil nil "Nganyaywana")
            ("nyy" nil  nil   nil I   L   () nil nil "Nyakyusa")
            ("nza" nil  nil   nil I   L   () nil nil "Mbembe, Tigon")
            ("nzb" nil  nil   nil I   L   () nil nil "Njebi")
            ("nzd" nil  nil   nil I   L   () "Niger–Congo" nil "Nzadi")
            ("nzi" nil  "nzi" nil I   L   () nil nil "Nzima")
            ("nzk" nil  nil   nil I   L   () nil nil "Nzakara")
            ("nzm" nil  nil   nil I   L   () nil nil "Naga, Zeme")
            ("nzr" nil  nil   nil I   L   () "Afro-Asiatic" nil "Dir-Nyamzak-Mbarimi")
            ("nzs" nil  nil   nil I   L   () nil nil "New Zealand Sign Language")
            ("nzu" nil  nil   nil I   L   () nil nil "Teke-Nzikou")
            ("nzy" nil  nil   nil I   L   () nil nil "Nzakambay")
            ("nzz" nil  nil   nil I   L   () nil nil "Nanga Dama Dogon")
            ;; O
            ("oaa" nil  nil   nil I   L   () nil nil "Orok")
            ("oac" nil  nil   nil I   L   () nil nil "Oroch")
            ("oar" nil  nil   nil I   A   () nil nil "Aramaic, Old")
            ("oav" nil  nil   nil I   H   () nil nil "Avar, Old")
            ("obi" nil  nil   nil I   E   () nil nil "Obispeño")
            ("obk" nil  nil   nil I   L   () nil nil "Southern Bontok")
            ("obl" nil  nil   nil I   L   () nil nil "Oblo")
            ("obm" nil  nil   nil I   A   () nil nil "Moabite")
            ("obo" nil  nil   nil I   L   () nil nil "Manobo, Obo")
            ("obr" nil  nil   nil I   H   () nil nil "Burmese, Old")
            ("obt" nil  nil   nil I   H   () nil nil "Breton, Old")
            ("obu" nil  nil   nil I   L   () nil nil "Obulom")
            ("oca" nil  nil   nil I   L   () nil nil "Ocaina")
            ("occ" nil  nil   t   nil nil () nil nil "Occidental")
            ("och" nil  nil   nil I   A   () "Chinese" nil "Old Chinese")
            ("oci" "oc" "oci" nil I   L   () "Indo-European" "occitan" "Occitan (post 1500)")
            ("ocm" nil  nil   nil I   H   () "Austronesian" nil "Cham, Old")
            ("oco" nil  nil   nil I   H   () nil nil "Cornish, Old")
            ("ocu" nil  nil   nil I   L   () nil nil "Matlatzinca, Atzingo")
            ("oda" nil  nil   nil I   L   () nil nil "Odut")
            ("odk" nil  nil   nil I   L   () nil nil "Od")
            ("odt" nil  nil   nil I   H   () nil nil "Dutch, Old")
            ("odu" nil  nil   nil I   L   () nil nil "Odual")
            ("ofo" nil  nil   nil I   E   () nil nil "Ofo")
            ("ofs" nil  nil   nil I   H   () "Friesisch" nil "Frisian, Old")
            ("ofu" nil  nil   nil I   L   () nil nil "Efutop")
            ("ogb" nil  nil   nil I   L   () nil nil "Ogbia")
            ("ogc" nil  nil   nil I   L   () nil nil "Ogbah")
            ("oge" nil  nil   nil I   H   () nil nil "Georgian, Old")
            ("ogg" nil  nil   nil I   L   () nil nil "Ogbogolo")
            ("ogn" nil  nil   t   nil nil () nil nil "Ogan")
            ("ogo" nil  nil   nil I   L   () nil nil "Khana")
            ("ogu" nil  nil   nil I   L   () nil nil "Ogbronuagum")
            ("oht" nil  nil   nil I   A   () nil nil "Hittite, Old")
            ("ohu" nil  nil   nil I   H   () nil nil "Hungarian, Old")
            ("oia" nil  nil   nil I   L   () nil nil "Oirata")
            ("oie" nil  nil   nil I   L   () "Nilo-Saharan" nil "Okolie")
            ("oin" nil  nil   nil I   L   () nil nil "One, Inebu")
            ("ojb" nil  nil   nil I   L   () nil "Anishinaabemowin (Ojibwemowin)" "Ojibwa, Northwestern")
            ("ojc" nil  nil   nil I   L   () nil "Anishinaabemowin (Ojibwemowin)" "Ojibwa, Central")
            ("ojg" nil  nil   nil I   L   () nil "Nishnaabemwin (Jibwemwin)" "Ojibwa, Eastern")
            ("oji" "oj" "oji" nil M   L   ("ciw"
                                           "ojb" "ojc" "ojg"
                                           "ojs" "ojw" "otw") "Cree" "ᐊᓂᔑᓇᐯᒧᐎᓐ (Anishinaabemowin)" "Ojibwa")
            ("ojp" nil  nil   nil I   H   () nil nil "Japanese, Old")
            ("ojs" nil  nil   nil I   L   () nil "ᐊᓂᔑᓂᓂᒧᐎᓐ (Anishininiimowin)" "Ojibwa, Severn")
            ("ojv" nil  nil   nil I   L   () nil nil "Ontong Java")
            ("ojw" nil  nil   nil I   L   () nil "Anihšināpēmowin (Nakawēmowin)" "Ojibwa, Western")
            ("oka" nil  nil   nil I   L   () nil nil "Okanagan")
            ("okb" nil  nil   nil I   L   () nil nil "Okobo")
            ("okc" nil  nil   nil I   L   () "Niger–Congo" nil "Kobo")
            ("okd" nil  nil   nil I   L   () nil nil "Okodia")
            ("oke" nil  nil   nil I   L   () nil nil "Okpe (Southwestern Edo)")
            ("okg" nil  nil   nil I   E   () nil nil "Koko Babangk")
            ("okh" nil  nil   nil I   L   () nil nil "Koresh-e Rostam")
            ("oki" nil  nil   nil I   L   () nil nil "Okiek")
            ("okj" nil  nil   nil I   E   () nil nil "Oko-Juwoi")
            ("okk" nil  nil   nil I   L   () nil nil "One, Kwamtim")
            ("okl" nil  nil   nil I   E   () nil nil "Old Kentish Sign Language")
            ("okm" nil  nil   nil I   H   () nil nil "Korean, Middle (10th–16th centuries)")
            ("okn" nil  nil   nil I   L   () nil nil "Oki-No-Erabu")
            ("oko" nil  nil   nil I   H   () nil nil "Korean, Old (3rd–9th centuries)")
            ("okr" nil  nil   nil I   L   () nil nil "Kirike")
            ("oks" nil  nil   nil I   L   () nil nil "Oko-Eni-Osayen")
            ("oku" nil  nil   nil I   L   () nil nil "Oku")
            ("okv" nil  nil   nil I   L   () nil nil "Orokaiva")
            ("okx" nil  nil   nil I   L   () nil nil "Okpe (Northwestern Edo)")
            ("okz" nil  nil   nil I   H   () "Austroasiatic" nil "Khmer, Old")
            ("ola" nil  nil   nil I   L   () nil nil "Walungge")
            ("old" nil  nil   nil I   L   () nil nil "Mochi")
            ("ole" nil  nil   nil I   L   () nil nil "Olekha")
            ("olk" nil  nil   nil I   E   () nil nil "Olkol")
            ("olm" nil  nil   nil I   L   () nil nil "Oloma")
            ("olo" nil  nil   nil I   L   () nil nil "Livvi")
            ("olr" nil  nil   nil I   L   () nil nil "Olrat")
            ("olt" nil  nil   nil I   H   () "Indo-European" nil "Old Lithuanian")
            ("olu" nil  nil   nil I   L   () "Niger–Congo" nil "Kuvale")
            ("oma" nil  nil   nil I   L   () nil nil "Omaha-Ponca")
            ("omb" nil  nil   nil I   L   () nil nil "Ambae, East")
            ("omc" nil  nil   nil I   E   () nil nil "Mochica")
            ("ome" nil  nil   t   I   E   () nil nil "Omejes")
            ("omg" nil  nil   nil I   L   () nil nil "Omagua")
            ("omi" nil  nil   nil I   L   () nil nil "Omi")
            ("omk" nil  nil   nil I   E   () nil nil "Omok")
            ("oml" nil  nil   nil I   L   () nil nil "Ombo")
            ("omn" nil  nil   nil I   A   () nil nil "Minoan")
            ("omo" nil  nil   nil I   L   () nil nil "Utarmbung")
            ("omp" nil  nil   nil I   H   () nil nil "Manipuri, Old")
            ("omr" nil  nil   nil I   H   () nil nil "Marathi, Old")
            ("omt" nil  nil   nil I   L   () nil nil "Omotik")
            ("omu" nil  nil   nil I   E   () nil nil "Omurano")
            ("omw" nil  nil   nil I   L   () nil nil "Tairora, South")
            ("omx" nil  nil   nil I   H   () nil nil "Mon, Old")
            ("omy" nil  nil   nil I   H   () "Austronesian" nil "Malay, Old")
            ("ona" nil  nil   nil I   L   () nil nil "Ona")
            ("onb" nil  nil   nil I   L   () nil nil "Lingao")
            ("one" nil  nil   nil I   L   () nil "Onʌyota’a:ka" "Oneida")
            ("ong" nil  nil   nil I   L   () nil nil "Olo")
            ("oni" nil  nil   nil I   L   () nil nil "Onin")
            ("onj" nil  nil   nil I   L   () nil nil "Onjob")
            ("onk" nil  nil   nil I   L   () nil nil "One, Kabore")
            ("onn" nil  nil   nil I   L   () nil nil "Onobasulu")
            ("ono" nil  nil   nil I   L   () nil "Onǫta’kéka’" "Onondaga")
            ("onp" nil  nil   nil I   L   () nil nil "Sartang")
            ("onr" nil  nil   nil I   L   () nil nil "One, Northern")
            ("ons" nil  nil   nil I   L   () nil nil "Ono")
            ("ont" nil  nil   nil I   L   () nil nil "Ontenu")
            ("onu" nil  nil   nil I   L   () nil nil "Unua")
            ("onw" nil  nil   nil I   H   () nil nil "Nubian, Old")
            ("onx" nil  nil   nil I   L   () nil nil "Onin Based Pidgin")
            ("ood" nil  nil   nil I   L   () nil "O'odham" "Tohono O'odham")
            ("oog" nil  nil   nil I   L   () nil nil "Ong")
            ("oon" nil  nil   nil I   L   () nil nil "Önge")
            ("oor" nil  nil   nil I   L   () nil nil "Oorlams")
            ("oos" nil  nil   nil I   A   () nil nil "Ossetic, Old")
            ("opa" nil  nil   nil I   L   () nil nil "Okpamheri")
            ("ope" nil  nil   t   nil nil () nil nil "Old Persian")
            ("opk" nil  nil   nil I   L   () nil nil "Kopkaka")
            ("opm" nil  nil   nil I   L   () nil nil "Oksapmin")
            ("opo" nil  nil   nil I   L   () nil nil "Opao")
            ("opt" nil  nil   nil I   E   () nil nil "Opata")
            ("opy" nil  nil   nil I   L   () nil nil "Ofayé")
            ("ora" nil  nil   nil I   L   () nil nil "Oroha")
            ("orc" nil  nil   nil I   L   () nil nil "Orma")
            ("ore" nil  nil   nil I   L   () nil nil "Orejón")
            ("org" nil  nil   nil I   L   () nil nil "Oring")
            ("orh" nil  nil   nil I   L   () nil nil "Oroqen")
            ("ori" "or" "ori" nil M   L   ("ory" "spv") "Indo-European" "ଓଡ଼ିଆ" "Oriya")
            ("ork" nil  nil   t   nil nil () nil nil "Orokaiva")
            ("orm" "om" "orm" nil M   L   ("gax"
                                           "gaz" "hae" "orc") "Afro-Asiatic" "Oromoo" "Oromo")
            ("orn" nil  nil   nil I   L   () nil nil "Orang Kanaq")
            ("oro" nil  nil   nil I   L   () nil nil "Orokolo")
            ("orr" nil  nil   nil I   L   () nil nil "Oruma")
            ("ors" nil  nil   nil I   L   () nil nil "Orang Seletar")
            ("ort" nil  nil   nil I   L   () nil nil "Oriya, Adivasi")
            ("oru" nil  nil   nil I   L   () nil nil "Ormuri")
            ("orv" nil  nil   nil I   H   () nil nil "Russian, Old")
            ("orw" nil  nil   nil I   L   () nil nil "Oro Win")
            ("orx" nil  nil   nil I   L   () nil nil "Oro")
            ("ory" nil  nil   nil I   L   () nil nil "Oriya (individual language)")
            ("orz" nil  nil   nil I   L   () nil nil "Ormu")
            ("osa" nil  "osa" nil I   L   () nil "𐓏𐓘𐓻𐓘𐓻𐓟" "Osage")
            ("osc" nil  nil   nil I   A   () nil nil "Oscan")
            ("osi" nil  nil   nil I   L   () nil nil "Osing")
            ("osn" nil  nil   nil I   H   () "Austronesian" nil "Sundanese, Old")
            ("oso" nil  nil   nil I   L   () nil nil "Ososo")
            ("osp" nil  nil   nil I   H   () nil nil "Spanish, Old")
            ("oss" "os" "oss" nil I   L   () "Indo-European" "ирон" "Ossetian")
            ("ost" nil  nil   nil I   L   () nil nil "Osatu")
            ("osu" nil  nil   nil I   L   () nil nil "One, Southern")
            ("osx" nil  nil   nil I   H   () nil nil "Saxon, Old")
            ("ota" nil  "ota" nil I   H   () nil "لسان عثمانى" "Turkish, Ottoman (1500–1928)")
            ("otb" nil  nil   nil I   H   () nil nil "Tibetan, Old")
            ("otd" nil  nil   nil I   L   () nil nil "Dohoi")
            ("ote" nil  nil   nil I   L   () nil nil "Otomi, Mezquital")
            ("oti" nil  nil   nil I   E   () nil nil "Oti")
            ("otk" nil  nil   nil I   H   () nil nil "Turkish, Old")
            ("otl" nil  nil   nil I   L   () nil nil "Otomi, Tilapa")
            ("otm" nil  nil   nil I   L   () nil nil "Otomi, Eastern Highland")
            ("otn" nil  nil   nil I   L   () nil nil "Otomi, Tenango")
            (nil   nil  "oto" nil C   nil () nil nil "Otomian languages")
            ("otq" nil  nil   nil I   L   () nil nil "Otomi, Querétaro")
            ("otr" nil  nil   nil I   L   () nil nil "Otoro")
            ("ots" nil  nil   nil I   L   () nil nil "Otomi, Estado de México")
            ("ott" nil  nil   nil I   L   () nil nil "Otomi, Temoaya")
            ("otu" nil  nil   nil I   E   () nil nil "Otuke")
            ("otw" nil  nil   nil I   L   () nil "Nishnaabemwin (Daawaamwin)" "Ottawa")
            ("otx" nil  nil   nil I   L   () nil nil "Otomi, Texcatepec")
            ("oty" nil  nil   nil I   A   () nil nil "Tamil, Old")
            ("otz" nil  nil   nil I   L   () nil nil "Otomi, Ixtenco")
            ("oua" nil  nil   nil I   L   () nil nil "Tagargrent")
            ("oub" nil  nil   nil I   L   () nil nil "Glio-Oubi")
            ("oue" nil  nil   nil I   L   () nil nil "Oune")
            ("oui" nil  nil   nil I   H   () nil nil "Uighur, Old")
            ("oum" nil  nil   nil I   E   () nil nil "Ouma")
            ("oun" nil  nil   t   I   L   () nil nil "!O!ung")
            ("ovd" nil  nil   nil I   L   () "Indo-European" nil "Elfdalian, Övdalian")
            ("owi" nil  nil   nil I   L   () nil nil "Owiniga")
            ("owl" nil  nil   nil I   H   () nil nil "Welsh, Old")
            ("oyb" nil  nil   nil I   L   () nil nil "Oy")
            ("oyd" nil  nil   nil I   L   () nil nil "Oyda")
            ("oym" nil  nil   nil I   L   () nil nil "Wayampi")
            ("oyy" nil  nil   nil I   L   () nil nil "Oya'oya")
            ("ozm" nil  nil   nil I   L   () nil nil "Koonzime")
            ;; P
            (nil   nil  "paa" nil C   nil () nil nil "Papuan languages")
            ("pab" nil  nil   nil I   L   () nil nil "Parecís")
            ("pac" nil  nil   nil I   L   () nil nil "Pacoh")
            ("pad" nil  nil   nil I   L   () nil nil "Paumarí")
            ("pae" nil  nil   nil I   L   () nil nil "Pagibete")
            ("paf" nil  nil   nil I   E   () nil nil "Paranawát")
            ("pag" nil  "pag" nil I   L   () nil "Pangasinán" "Pangasinan")
            ("pah" nil  nil   nil I   L   () nil nil "Tenharim")
            ("pai" nil  nil   nil I   L   () nil nil "Pe")
            ("paj" nil  nil   t   nil nil () nil nil "Ipeka-Tapuia")
            ("pak" nil  nil   nil I   L   () nil nil "Parakanã")
            ("pal" nil  "pal" nil I   A   () nil nil "Pahlavi")
            ("pam" nil  "pam" nil I   L   () nil "Kapampangan" "Pampanga")
            ("pan" "pa" "pan" nil I   L   () "Indo-European" "ਪੰਜਾਬੀ" "Panjabi; Punjabi")
            ("pao" nil  nil   nil I   L   () nil nil "Paiute, Northern")
            ("pap" nil  "pap" nil I   L   () nil "Papiamentu" "Papiamento")
            ("paq" nil  nil   nil I   L   () nil nil "Parya")
            ("par" nil  nil   nil I   L   () nil nil "Panamint")
            ("pas" nil  nil   nil I   L   () nil nil "Papasena")
            ("pat" nil  nil   t   I   L   () nil nil "Papitalai")
            ("pau" nil  "pau" nil I   L   () nil "tekoi ra Belau" "Palauan")
            ("pav" nil  nil   nil I   L   () nil nil "Pakaásnovos")
            ("paw" nil  nil   nil I   L   () nil nil "Pawnee")
            ("pax" nil  nil   nil I   E   () nil nil "Pankararé")
            ("pay" nil  nil   nil I   L   () nil nil "Pech")
            ("paz" nil  nil   nil I   E   () nil nil "Pankararú")
            ("pbb" nil  nil   nil I   L   () nil nil "Páez")
            ("pbc" nil  nil   nil I   L   () nil nil "Patamona")
            ("pbe" nil  nil   nil I   L   () nil nil "Popoloca, Mezontla")
            ("pbf" nil  nil   nil I   L   () nil nil "Popoloca, Coyotepec")
            ("pbg" nil  nil   nil I   E   () nil nil "Paraujano")
            ("pbh" nil  nil   nil I   L   () nil nil "Eñepa")
            ("pbi" nil  nil   nil I   L   () nil nil "Parkwa")
            ("pbl" nil  nil   nil I   L   () nil nil "Mak (Nigeria)")
            ("pbm" nil  nil   nil I   L   () "Oto-Manguean" nil "Puebla Mazatec")
            ("pbn" nil  nil   nil I   L   () nil nil "Kpasam")
            ("pbo" nil  nil   nil I   L   () nil nil "Papel")
            ("pbp" nil  nil   nil I   L   () nil nil "Badyara")
            ("pbr" nil  nil   nil I   L   () nil nil "Pangwa")
            ("pbs" nil  nil   nil I   L   () nil nil "Pame, Central")
            ("pbt" nil  nil   nil I   L   () nil nil "Pashto, Southern")
            ("pbu" nil  nil   nil I   L   () nil nil "Pashto, Northern")
            ("pbv" nil  nil   nil I   L   () nil nil "Pnar")
            ("pby" nil  nil   nil I   L   () nil nil "Pyu")
            ("pbz" nil  nil   t   I   L   () nil nil "Palu")
            ("pca" nil  nil   nil I   L   () nil nil "Popoloca, Santa Inés Ahuatempan")
            ("pcb" nil  nil   nil I   L   () nil nil "Pear")
            ("pcc" nil  nil   nil I   L   () nil nil "Bouyei")
            ("pcd" nil  nil   nil I   L   () nil "picard" "Picard")
            ("pce" nil  nil   nil I   L   () nil nil "Palaung, Pale")
            ("pcf" nil  nil   nil I   L   () nil nil "Paliyan")
            ("pcg" nil  nil   nil I   L   () nil nil "Paniya")
            ("pch" nil  nil   nil I   L   () nil nil "Pardhan")
            ("pci" nil  nil   nil I   L   () nil nil "Duruwa")
            ("pcj" nil  nil   nil I   L   () nil nil "Parenga")
            ("pck" nil  nil   nil I   L   () nil nil "Chin, Paite")
            ("pcl" nil  nil   nil I   L   () nil nil "Pardhi")
            ("pcm" nil  nil   nil I   L   () nil nil "Pidgin, Nigerian")
            ("pcn" nil  nil   nil I   L   () nil nil "Piti")
            ("pcp" nil  nil   nil I   L   () nil nil "Pacahuara")
            ("pcr" nil  nil   t   I   L   () nil nil "Panang")
            ("pcw" nil  nil   nil I   L   () nil nil "Pyapun")
            ("pda" nil  nil   nil I   L   () nil nil "Anam")
            ("pdc" nil  nil   nil I   L   () nil "Pennsilfaani-Deitsch" "German, Pennsylvania")
            ("pdi" nil  nil   nil I   L   () nil nil "Pa Di")
            ("pdn" nil  nil   nil I   L   () nil nil "Podena")
            ("pdo" nil  nil   nil I   L   () nil nil "Padoe")
            ("pdt" nil  nil   nil I   L   () nil "Plautdietsch" "Plautdietsch")
            ("pdu" nil  nil   nil I   L   () nil nil "Kayan")
            ("pea" nil  nil   nil I   L   () nil nil "Indonesian, Peranakan")
            ("peb" nil  nil   nil I   E   () nil nil "Pomo, Eastern")
            ("pec" nil  nil   t   nil nil () nil nil "Southern Pesisir")
            ("ped" nil  nil   nil I   L   () nil nil "Mala (Papua New Guinea)")
            ("pee" nil  nil   nil I   L   () nil nil "Taje")
            ("pef" nil  nil   nil I   E   () nil nil "Pomo, Northeastern")
            ("peg" nil  nil   nil I   L   () nil nil "Pengo")
            ("peh" nil  nil   nil I   L   () nil nil "Bonan")
            ("pei" nil  nil   nil I   L   () nil nil "Chichimeca-Jonaz")
            ("pej" nil  nil   nil I   E   () nil nil "Pomo, Northern")
            ("pek" nil  nil   nil I   L   () nil nil "Penchal")
            ("pel" nil  nil   nil I   L   () nil nil "Pekal")
            ("pem" nil  nil   nil I   L   () nil nil "Phende")
            ("pen" nil  nil   t   nil nil () nil nil "Penesak")
            ("peo" nil  "peo" nil I   H   () nil nil "Persian, Old (ca.600-400 B.C.)")
            ("pep" nil  nil   nil I   L   () nil nil "Kunja")
            ("peq" nil  nil   nil I   L   () nil nil "Pomo, Southern")
            ("pes" nil  nil   nil I   L   () nil "فارسی" "Western Farsi")
            ("pev" nil  nil   nil I   L   () nil nil "Pémono")
            ("pex" nil  nil   nil I   L   () nil nil "Petats")
            ("pey" nil  nil   nil I   L   () nil nil "Petjo")
            ("pez" nil  nil   nil I   L   () nil nil "Penan, Eastern")
            ("pfa" nil  nil   nil I   L   () nil nil "Pááfang")
            ("pfe" nil  nil   nil I   L   () nil nil "Peere")
            ("pfl" nil  nil   nil I   L   () nil "Pälzisch" "Pfaelzisch")
            ("pga" nil  nil   nil I   L   () nil nil "Arabic, Sudanese Creole")
            ("pgd" nil  nil   nil I   H   () "Indo-European" nil "Gāndhārī")
            ("pgg" nil  nil   nil I   L   () nil nil "Pangwali")
            ("pgi" nil  nil   nil I   L   () nil nil "Pagi")
            ("pgk" nil  nil   nil I   L   () nil nil "Rerep")
            ("pgl" nil  nil   nil I   A   () nil nil "Primitive Irish")
            ("pgn" nil  nil   nil I   A   () nil nil "Paelignian")
            ("pgs" nil  nil   nil I   L   () nil nil "Pangseng")
            ("pgu" nil  nil   nil I   L   () nil nil "Pagu")
            ("pgy" nil  nil   t   I   L   () nil nil "Pongyong")
            ("pgz" nil  nil   nil I   L   () nil nil "Papua New Guinean Sign Language")
            ("pha" nil  nil   nil I   L   () nil nil "Pa-Hng")
            ("phd" nil  nil   nil I   L   () nil nil "Phudagi")
            ("phg" nil  nil   nil I   L   () nil nil "Phuong")
            ("phh" nil  nil   nil I   L   () nil nil "Phula")
            (nil   nil  "phi" nil C   nil () nil nil "Philippine languages")
            ("phj" nil  nil   nil I   L   () "Sino-Tibetan" nil "Pahari")
            ("phk" nil  nil   nil I   L   () nil nil "Phake")
            ("phl" nil  nil   nil I   L   () nil nil "Phalura")
            ("phm" nil  nil   nil I   L   () nil nil "Phimbi")
            ("phn" nil  "phn" nil I   A   () nil nil "Phoenician")
            ("pho" nil  nil   nil I   L   () nil nil "Phunoi")
            ("phq" nil  nil   nil I   L   () nil nil "Phana'")
            ("phr" nil  nil   nil I   L   () nil nil "Pahari-Potwari")
            ("pht" nil  nil   nil I   L   () nil nil "Phu Thai")
            ("phu" nil  nil   nil I   L   () nil nil "Phuan")
            ("phv" nil  nil   nil I   L   () nil nil "Pahlavani")
            ("phw" nil  nil   nil I   L   () nil nil "Phangduwali")
            ("pia" nil  nil   nil I   L   () nil nil "Pima Bajo")
            ("pib" nil  nil   nil I   L   () nil nil "Yine")
            ("pic" nil  nil   nil I   L   () nil nil "Pinji")
            ("pid" nil  nil   nil I   L   () nil nil "Piaroa")
            ("pie" nil  nil   nil I   E   () nil nil "Piro")
            ("pif" nil  nil   nil I   L   () nil nil "Pingelapese")
            ("pig" nil  nil   nil I   L   () nil nil "Pisabo")
            ("pih" nil  nil   nil I   L   () nil "Norfuk" "Pitcairn-Norfolk")
            ("pii" nil  nil   t   I   L   () nil nil "Pini")
            ("pij" nil  nil   nil I   E   () nil nil "Pijao")
            ("pil" nil  nil   nil I   L   () nil nil "Yom")
            ("pim" nil  nil   nil I   E   () nil nil "Powhatan")
            ("pin" nil  nil   nil I   L   () nil nil "Piame")
            ("pio" nil  nil   nil I   L   () nil nil "Piapoco")
            ("pip" nil  nil   nil I   L   () nil nil "Pero")
            ("pir" nil  nil   nil I   L   () nil nil "Piratapuyo")
            ("pis" nil  nil   nil I   L   () nil nil "Pijin")
            ("pit" nil  nil   nil I   E   () nil nil "Pitta Pitta")
            ("piu" nil  nil   nil I   L   () nil nil "Pintupi-Luritja")
            ("piv" nil  nil   nil I   L   () nil nil "Pileni")
            ("piw" nil  nil   nil I   L   () nil nil "Pimbwe")
            ("pix" nil  nil   nil I   L   () nil nil "Piu")
            ("piy" nil  nil   nil I   L   () nil nil "Piya-Kwonci")
            ("piz" nil  nil   nil I   L   () nil nil "Pije")
            ("pjt" nil  nil   nil I   L   () nil nil "Pitjantjatjara")
            ("pka" nil  nil   nil I   H   () nil nil "Ardhamāgadhī Prākrit")
            ("pkb" nil  nil   nil I   L   () nil nil "Pokomo, Upper")
            ("pkc" nil  nil   nil I   A   () nil nil "Paekche")
            ("pkg" nil  nil   nil I   L   () nil nil "Pak-Tong")
            ("pkh" nil  nil   nil I   L   () nil nil "Pankhu")
            ("pkn" nil  nil   nil I   L   () nil nil "Pakanha")
            ("pko" nil  nil   nil I   L   () nil nil "Pökoot")
            ("pkp" nil  nil   nil I   L   () nil nil "Pukapuka")
            ("pkr" nil  nil   nil I   L   () nil nil "Attapady Kurumba")
            ("pks" nil  nil   nil I   L   () nil nil "Pakistan Sign Language")
            ("pkt" nil  nil   nil I   L   () nil nil "Maleng")
            ("pku" nil  nil   nil I   L   () nil nil "Paku")
            ("pla" nil  nil   nil I   L   () nil nil "Miani")
            ("plb" nil  nil   nil I   L   () nil nil "Polonombauk")
            ("plc" nil  nil   nil I   L   () nil nil "Palawano, Central")
            ("pld" nil  nil   nil I   L   () nil nil "Polari")
            ("ple" nil  nil   nil I   L   () nil nil "Palu'e")
            ("plg" nil  nil   nil I   L   () nil nil "Pilagá")
            ("plh" nil  nil   nil I   L   () nil nil "Paulohi")
            ("pli" "pi" "pli" nil I   A   () "Indo-European" "पालि, पाळि" "Pali")
            ("plj" nil  nil   t   I   L   () nil nil "Polci")
            ("plk" nil  nil   nil I   L   () nil nil "Shina, Kohistani")
            ("pll" nil  nil   nil I   L   () nil nil "Palaung, Shwe")
            ("plm" nil  nil   t   nil nil () nil nil "Palembang")
            ("pln" nil  nil   nil I   L   () nil nil "Palenquero")
            ("plo" nil  nil   nil I   L   () nil nil "Popoluca, Oluta")
            ("plp" nil  nil   t   I   L   () nil nil "Palpa")
            ("plq" nil  nil   nil I   A   () nil nil "Palaic")
            ("plr" nil  nil   nil I   L   () nil nil "Senoufo, Palaka")
            ("pls" nil  nil   nil I   L   () nil nil "Popoloca, San Marcos Tlalcoyalco")
            ("plt" nil  nil   nil I   L   () nil nil "Malagasy, Plateau")
            ("plu" nil  nil   nil I   L   () nil nil "Palikúr")
            ("plv" nil  nil   nil I   L   () nil nil "Palawano, Southwest")
            ("plw" nil  nil   nil I   L   () nil nil "Palawano, Brooke's Point")
            ("ply" nil  nil   nil I   L   () nil nil "Bolyu")
            ("plz" nil  nil   nil I   L   () nil nil "Paluan")
            ("pma" nil  nil   nil I   L   () nil nil "Paama")
            ("pmb" nil  nil   nil I   L   () nil nil "Pambia")
            ("pmc" nil  nil   t   I   E   () nil nil "Palumata")
            ("pmd" nil  nil   nil I   E   () nil nil "Pallanganmiddang")
            ("pme" nil  nil   nil I   L   () nil nil "Pwaamei")
            ("pmf" nil  nil   nil I   L   () nil nil "Pamona")
            ("pmh" nil  nil   nil I   H   () nil nil "Māhārāṣṭri Prākrit")
            ("pmi" nil  nil   nil I   L   () nil nil "Pumi, Northern")
            ("pmj" nil  nil   nil I   L   () nil nil "Pumi, Southern")
            ("pmk" nil  nil   t   I   E   () nil nil "Pamlico")
            ("pml" nil  nil   nil I   E   () nil nil "Lingua Franca")
            ("pmm" nil  nil   nil I   L   () nil nil "Pomo")
            ("pmn" nil  nil   nil I   L   () nil nil "Pam")
            ("pmo" nil  nil   nil I   L   () nil nil "Pom")
            ("pmq" nil  nil   nil I   L   () nil nil "Pame, Northern")
            ("pmr" nil  nil   nil I   L   () nil nil "Paynamar")
            ("pms" nil  nil   nil I   L   () nil "piemontèis" "Piedmontese")
            ("pmt" nil  nil   nil I   L   () nil nil "Tuamotuan")
            ("pmu" nil  nil   t   I   L   () nil nil "Panjabi, Mirpur")
            ("pmw" nil  nil   nil I   L   () nil nil "Miwok, Plains")
            ("pmx" nil  nil   nil I   L   () nil nil "Naga, Poumei")
            ("pmy" nil  nil   nil I   L   () nil nil "Papuan Malay")
            ("pmz" nil  nil   nil I   E   () nil nil "Pame, Southern")
            ("pna" nil  nil   nil I   L   () nil nil "Punan Bah-Biau")
            ("pnb" nil  nil   nil I   L   () nil nil "Panjabi, Western")
            ("pnc" nil  nil   nil I   L   () nil nil "Pannei")
            ("pnd" nil  nil   nil I   L   () nil nil "Mpinda")
            ("pne" nil  nil   nil I   L   () nil nil "Penan, Western")
            ("png" nil  nil   nil I   L   () nil nil "Pongu")
            ("pnh" nil  nil   nil I   L   () nil nil "Penrhyn")
            ("pni" nil  nil   nil I   L   () nil nil "Aoheng")
            ("pnj" nil  nil   nil I   E   () nil nil "Pinjarup")
            ("pnk" nil  nil   nil I   L   () nil nil "Paunaka")
            ("pnl" nil  nil   nil I   L   () "Niger–Congo" nil "Paleni")
            ("pnm" nil  nil   nil I   L   () nil nil "Punan Batu 1")
            ("pnn" nil  nil   nil I   L   () nil nil "Pinai-Hagahai")
            ("pno" nil  nil   nil I   E   () nil nil "Panobo")
            ("pnp" nil  nil   nil I   L   () nil nil "Pancana")
            ("pnq" nil  nil   nil I   L   () nil nil "Pana (Burkina Faso)")
            ("pnr" nil  nil   nil I   L   () nil nil "Panim")
            ("pns" nil  nil   nil I   L   () nil nil "Ponosakan")
            ("pnt" nil  nil   nil I   L   () nil "Ποντιακά" "Pontic")
            ("pnu" nil  nil   nil I   L   () nil nil "Bunu, Jiongnai")
            ("pnv" nil  nil   nil I   L   () nil nil "Pinigura")
            ("pnw" nil  nil   nil I   L   () nil nil "Panytyima")
            ("pnx" nil  nil   nil I   L   () nil nil "Phong-Kniang")
            ("pny" nil  nil   nil I   L   () nil nil "Pinyin")
            ("pnz" nil  nil   nil I   L   () nil nil "Pana (Central African Republic)")
            ("poa" nil  nil   t   I   L   () nil nil "Pokomam, Eastern")
            ("pob" nil  nil   t   I   L   () nil nil "Pokomchí, Western")
            ("poc" nil  nil   nil I   L   () nil nil "Pokomam, Central")
            ("pod" nil  nil   t   I   E   () nil nil "Ponares")
            ("poe" nil  nil   nil I   L   () nil nil "Popoloca, San Juan Atzingo")
            ("pof" nil  nil   nil I   L   () nil nil "Poke")
            ("pog" nil  nil   nil I   E   () nil nil "Potiguára")
            ("poh" nil  nil   nil I   L   () nil nil "Pokomchí, Eastern")
            ("poi" nil  nil   nil I   L   () nil nil "Popoluca, Highland")
            ("poj" nil  nil   t   nil nil () nil nil "Lower Pokomo")
            ("pok" nil  nil   nil I   L   () nil nil "Pokangá")
            ("pol" "pl" "pol" nil I   L   () "Indo-European" "polski" "Polish")
            ("pom" nil  nil   nil I   L   () nil nil "Pomo, Southeastern")
            ("pon" nil  "pon" nil I   L   () nil nil "Pohnpeian")
            ("poo" nil  nil   nil I   E   () nil nil "Pomo, Central")
            ("pop" nil  nil   nil I   L   () nil nil "Pwapwa")
            ("poq" nil  nil   nil I   L   () nil nil "Popoluca, Texistepec")
            ("por" "pt" "por" nil I   L   () "Indo-European" "português" "Portuguese")
            ("pos" nil  nil   nil I   L   () nil nil "Popoluca, Sayula")
            ("pot" nil  nil   nil I   L   () nil "Neshnabémwen (Bodéwadmimwen)" "Potawatomi")
            ("pou" nil  nil   t   I   L   () nil nil "Pokomam, Southern")
            ("pov" nil  nil   nil I   L   () nil nil "Crioulo, Upper Guinea")
            ("pow" nil  nil   nil I   L   () nil nil "Popoloca, San Felipe Otlaltepec")
            ("pox" nil  nil   nil I   E   () nil nil "Polabian")
            ("poy" nil  nil   nil I   L   () nil nil "Pogolo")
            ("ppa" nil  nil   t   I   L   () nil nil "Pao")
            ("ppe" nil  nil   nil I   L   () nil nil "Papi")
            ("ppi" nil  nil   nil I   L   () nil "aka'ala" "Paipai")
            ("ppk" nil  nil   nil I   L   () nil nil "Uma")
            ("ppl" nil  nil   nil I   L   () nil nil "Pipil")
            ("ppm" nil  nil   nil I   L   () nil nil "Papuma")
            ("ppn" nil  nil   nil I   L   () nil nil "Papapana")
            ("ppo" nil  nil   nil I   L   () nil nil "Folopa")
            ("ppp" nil  nil   nil I   L   () nil nil "Pelende")
            ("ppq" nil  nil   nil I   L   () nil nil "Pei")
            ("ppr" nil  nil   t   I   L   () nil nil "Piru")
            ("pps" nil  nil   nil I   L   () nil nil "Popoloca, San Luís Temalacayuca")
            ("ppt" nil  nil   nil I   L   () nil nil "Pare")
            ("ppu" nil  nil   nil I   E   () nil nil "Papora")
            ("ppv" nil  nil   t   I   L   () nil nil "Papavô")
            ("pqa" nil  nil   nil I   L   () nil nil "Pa'a")
            ("pqm" nil  nil   nil I   L   () nil "Peskotomuhkati" "Malecite-Passamaquoddy")
            (nil   nil  "pra" nil C   nil () nil nil "Prakrit languages")
            ("prb" nil  nil   t   I   L   () nil nil "Lua'")
            ("prc" nil  nil   nil I   L   () nil nil "Parachi")
            ("prd" nil  nil   nil I   L   () nil "(فارسی (دری" "Dari (Persian)")
            ("pre" nil  nil   nil I   L   () nil nil "Principense")
            ("prf" nil  nil   nil I   L   () nil nil "Paranan")
            ("prg" nil  nil   nil I   L   () nil "Prūsiska" "Prussian")
            ("prh" nil  nil   nil I   L   () nil nil "Porohanon")
            ("pri" nil  nil   nil I   L   () nil nil "Paicî")
            ("prk" nil  nil   nil I   L   () nil nil "Parauk")
            ("prl" nil  nil   nil I   L   () nil nil "Peruvian Sign Language")
            ("prm" nil  nil   nil I   L   () nil nil "Kibiri")
            ("prn" nil  nil   nil I   L   () nil nil "Prasuni")
            ("pro" nil  "pro" nil I   H   () nil nil "Provençal, Old (to 1500)")
            ("prp" nil  nil   t   I   L   () nil "فارسی" "Persian[clarification needed]")
            ("prq" nil  nil   nil I   L   () nil nil "Ashéninka Perené")
            ("prr" nil  nil   nil I   E   () nil nil "Puri")
            ("prs" nil  nil   nil I   L   () nil "(فارسی (دری" "Persian (Dari)")
            ("prt" nil  nil   nil I   L   () nil nil "Phai")
            ("pru" nil  nil   nil I   L   () nil nil "Puragi")
            ("prv" nil  nil   t   nil nil () nil nil "Provençal")
            ("prw" nil  nil   nil I   L   () nil nil "Parawen")
            ("prx" nil  nil   nil I   L   () nil nil "Purik")
            ("pry" nil  nil   t   I   L   () nil nil "Pray 3")
            ("prz" nil  nil   nil I   L   () nil nil "Providencia Sign Language")
            ("psa" nil  nil   nil I   L   () nil nil "Awyu, Asue")
            ("psc" nil  nil   nil I   L   () nil nil "Persian Sign Language")
            ("psd" nil  nil   nil I   L   () nil nil "Plains Indian Sign Language")
            ("pse" nil  nil   nil I   L   () nil nil "Central Malay")
            ("psg" nil  nil   nil I   L   () nil nil "Penang Sign Language")
            ("psh" nil  nil   nil I   L   () nil nil "Pashayi, Southwest")
            ("psi" nil  nil   nil I   L   () nil nil "Pashayi, Southeast")
            ("psl" nil  nil   nil I   L   () nil nil "Puerto Rican Sign Language")
            ("psm" nil  nil   nil I   E   () nil nil "Pauserna")
            ("psn" nil  nil   nil I   L   () nil nil "Panasuan")
            ("pso" nil  nil   nil I   L   () nil nil "Polish Sign Language")
            ("psp" nil  nil   nil I   L   () nil nil "Philippine Sign Language")
            ("psq" nil  nil   nil I   L   () nil nil "Pasi")
            ("psr" nil  nil   nil I   L   () nil nil "Portuguese Sign Language")
            ("pss" nil  nil   nil I   L   () nil nil "Kaulong")
            ("pst" nil  nil   nil I   L   () nil nil "Pashto, Central")
            ("psu" nil  nil   nil I   H   () nil nil "Sauraseni Prākrit")
            ("psw" nil  nil   nil I   L   () nil nil "Port Sandwich")
            ("psy" nil  nil   nil I   E   () nil nil "Piscataway")
            ("pta" nil  nil   nil I   L   () nil nil "Pai Tavytera")
            ("pth" nil  nil   nil I   E   () nil nil "Pataxó-Hãhaãi")
            ("pti" nil  nil   nil I   L   () nil nil "Pintiini")
            ("ptn" nil  nil   nil I   L   () nil nil "Patani")
            ("pto" nil  nil   nil I   L   () nil nil "Zo'é")
            ("ptp" nil  nil   nil I   L   () nil nil "Patep")
            ("ptq" nil  nil   nil I   L   () "Dravidian" nil "Pattapu")
            ("ptr" nil  nil   nil I   L   () nil nil "Piamatsina")
            ("ptt" nil  nil   nil I   L   () nil nil "Enrekang")
            ("ptu" nil  nil   nil I   L   () nil nil "Bambam")
            ("ptv" nil  nil   nil I   L   () nil nil "Port Vato")
            ("ptw" nil  nil   nil I   E   () nil nil "Pentlatch")
            ("pty" nil  nil   nil I   L   () nil nil "Pathiya")
            ("pua" nil  nil   nil I   L   () nil nil "Purepecha, Western Highland")
            ("pub" nil  nil   nil I   L   () nil nil "Purum")
            ("puc" nil  nil   nil I   L   () nil nil "Punan Merap")
            ("pud" nil  nil   nil I   L   () nil nil "Punan Aput")
            ("pue" nil  nil   nil I   E   () nil nil "Puelche")
            ("puf" nil  nil   nil I   L   () nil nil "Punan Merah")
            ("pug" nil  nil   nil I   L   () nil nil "Phuie")
            ("pui" nil  nil   nil I   L   () nil nil "Puinave")
            ("puj" nil  nil   nil I   L   () nil nil "Punan Tubu")
            ("puk" nil  nil   t   I   L   () nil nil "Pu Ko")
            ("pum" nil  nil   nil I   L   () nil nil "Puma")
            ("pun" nil  nil   t   nil nil () nil nil "Pubian")
            ("puo" nil  nil   nil I   L   () nil nil "Puoc")
            ("pup" nil  nil   nil I   L   () nil nil "Pulabu")
            ("puq" nil  nil   nil I   E   () nil nil "Puquina")
            ("pur" nil  nil   nil I   L   () nil nil "Puruborá")
            ("pus" "ps" "pus" nil M   L   ("pbt" "pbu" "pst") "Indo-European" "پښتو" "Pushto")
            ("put" nil  nil   nil I   L   () nil nil "Putoh")
            ("puu" nil  nil   nil I   L   () nil nil "Punu")
            ("puw" nil  nil   nil I   L   () nil nil "Puluwatese")
            ("pux" nil  nil   nil I   L   () nil nil "Puari")
            ("puy" nil  nil   nil I   E   () nil nil "Purisimeño")
            ("puz" nil  nil   t   I   L   () nil nil "Naga, Purum")
            ("pwa" nil  nil   nil I   L   () nil nil "Pawaia")
            ("pwb" nil  nil   nil I   L   () nil nil "Panawa")
            ("pwg" nil  nil   nil I   L   () nil nil "Gapapaiwa")
            ("pwi" nil  nil   nil I   E   () nil nil "Patwin")
            ("pwm" nil  nil   nil I   L   () nil nil "Molbog")
            ("pwn" nil  nil   nil I   L   () nil nil "Paiwan")
            ("pwo" nil  nil   nil I   L   () nil nil "Karen, Pwo Western")
            ("pwr" nil  nil   nil I   L   () nil nil "Powari")
            ("pww" nil  nil   nil I   L   () nil nil "Karen, Pwo Northern")
            ("pxm" nil  nil   nil I   L   () nil nil "Mixe, Quetzaltepec")
            ("pye" nil  nil   nil I   L   () nil nil "Krumen, Pye")
            ("pym" nil  nil   nil I   L   () nil nil "Fyam")
            ("pyn" nil  nil   nil I   L   () nil nil "Poyanáwa")
            ("pys" nil  nil   nil I   L   () nil nil "Paraguayan Sign Language")
            ("pyu" nil  nil   nil I   L   () nil nil "Puyuma")
            ("pyx" nil  nil   nil I   A   () nil nil "Pyu (Burma)")
            ("pyy" nil  nil   nil I   L   () nil nil "Pyen")
            ("pze" nil  nil   nil I   L   () "Afro-Asiatic" nil "Pesse")
            ("pzh" nil  nil   nil I   L   () "Austronesian" nil "Pazeh")
            ("pzn" nil  nil   nil I   L   () nil nil "Para Naga")
            ;; Q
            ("qua" nil  nil   nil I   L   () nil nil "Quapaw")
            ("qub" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Huallaga Huánuco")
            ("quc" nil  nil   nil I   L   () "Maya" "k'iche’" "K’iche’")
            ("qud" nil  nil   nil I   L   () "Quichua" nil "Quichua, Calderón Highland")
            ("que" "qu" "que" nil M   L   ("qub" "qud"
                                           "quf" "qug" "quh"
                                           "quk" "qul" "qup"
                                           "qur" "qus" "quw"
                                           "qux" "quy" "quz"
                                           "qva" "qvc" "qve"
                                           "qvh" "qvi" "qvj"
                                           "qvl" "qvm" "qvn"
                                           "qvo" "qvp" "qvs"
                                           "qvw" "qvz" "qwa"
                                           "qwc" "qwh" "qws"
                                           "qxa" "qxc" "qxh"
                                           "qxl" "qxn" "qxo"
                                           "qxp" "qxr" "qxt"
                                           "qxu" "qxw" "cqu") "Quechuan" "Runa Simi" "Quechua")
            ("quf" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Lambayeque")
            ("qug" nil  nil   nil I   L   () "Quichua" nil "Quichua, Chimborazo Highland")
            ("quh" nil  nil   nil I   L   () "Quechuan" nil "Quechua, South Bolivian")
            ("qui" nil  nil   nil I   L   () nil nil "Quileute")
            ("quj" nil  nil   t   I   L   () "Quiché" nil "Quiché, Joyabaj")
            ("quk" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Chachapoyas")
            ("qul" nil  nil   nil I   L   () "Quechuan" nil "Quechua, North Bolivian")
            ("qum" nil  nil   nil I   L   () nil nil "Sipacapense")
            ("qun" nil  nil   nil I   E   () nil nil "Quinault")
            ("qup" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Southern Pastaza")
            ("quq" nil  nil   nil I   L   () nil nil "Quinqui")
            ("qur" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Yanahuanca Pasco")
            ("qus" nil  nil   nil I   L   () "Quichua" nil "Quichua, Santiago del Estero")
            ("qut" nil  nil   t   I   L   () "Quiché" nil "Quiché, West Central")
            ("quu" nil  nil   t   I   L   () "Quiché" nil "Quiché, Eastern")
            ("quv" nil  nil   nil I   L   () nil nil "Sacapulteco")
            ("quw" nil  nil   nil I   L   () "Quichua" nil "Quichua, Tena Lowland")
            ("qux" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Yauyos")
            ("quy" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Ayacucho")
            ("quz" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Cusco")
            ("qva" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Ambo-Pasco")
            ("qvc" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Cajamarca")
            ("qve" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Eastern Apurímac")
            ("qvh" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Huamalíes-Dos de Mayo Huánuco")
            ("qvi" nil  nil   nil I   L   () "Quichua" nil "Quichua, Imbabura Highland")
            ("qvj" nil  nil   nil I   L   () "Quichua" nil "Quichua, Loja Highland")
            ("qvl" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Cajatambo North Lima")
            ("qvm" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Margos-Yarowilca-Lauricocha")
            ("qvn" nil  nil   nil I   L   () "Quechuan" nil "Quechua, North Junín")
            ("qvo" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Napo Lowland")
            ("qvp" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Pacaraos")
            ("qvs" nil  nil   nil I   L   () "Quechuan" nil "Quechua, San Martín")
            ("qvw" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Huaylla Wanca")
            ("qvy" nil  nil   nil I   L   () nil nil "Queyu")
            ("qvz" nil  nil   nil I   L   () "Quichua" nil "Quichua, Northern Pastaza")
            ("qwa" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Corongo Ancash")
            ("qwc" nil  nil   nil I   H   () "Quechuan" nil "Quechua, Classical")
            ("qwh" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Huaylas Ancash")
            ("qwm" nil  nil   nil I   E   () nil nil "Kuman (Russia)")
            ("qws" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Sihuas Ancash")
            ("qwt" nil  nil   nil I   E   () nil nil "Kwalhioqua-Tlatskanai")
            ("qxa" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Chiquián Ancash")
            ("qxc" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Chincha")
            ("qxh" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Panao Huánuco")
            ("qxi" nil  nil   t   I   L   () "Quiché" nil "Quiché, San Andrés")
            ("qxl" nil  nil   nil I   L   () "Quichua" nil "Quichua, Salasaca Highland")
            ("qxn" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Northern Conchucos Ancash")
            ("qxo" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Southern Conchucos Ancash")
            ("qxp" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Puno")
            ("qxq" nil  nil   nil I   L   () nil "Qaşqaycə" "Qashqa'i")
            ("qxr" nil  nil   nil I   L   () "Quichua" nil "Quichua, Cañar Highland")
            ("qxs" nil  nil   nil I   L   () nil nil "Qiang, Southern")
            ("qxt" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Santa Ana de Tusi Pasco")
            ("qxu" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Arequipa-La Unión")
            ("qxw" nil  nil   nil I   L   () "Quechuan" nil "Quechua, Jauja Wanca")
            ("qya" nil  nil   nil I   C   () nil nil "Quenya")
            ("qyp" nil  nil   nil I   E   () nil nil "Quiripi")
            ;; R
            ("raa" nil  nil   nil I   L   () nil nil "Dungmali")
            ("rab" nil  nil   nil I   L   () nil nil "Camling")
            ("rac" nil  nil   nil I   L   () nil nil "Rasawa")
            ("rad" nil  nil   nil I   L   () nil nil "Rade")
            ("rae" nil  nil   t   nil nil () nil nil "Ranau")
            ("raf" nil  nil   nil I   L   () nil nil "Meohang, Western")
            ("rag" nil  nil   nil I   L   () nil nil "Logooli")
            ("rah" nil  nil   nil I   L   () nil nil "Rabha")
            ("rai" nil  nil   nil I   L   () nil nil "Ramoaaina")
            ("raj" nil  "raj" nil M   L   ("bgq" "gda" "gju"
                                           "hoj" "mup" "wbr") nil nil "Rajasthani")
            ("rak" nil  nil   nil I   L   () nil nil "Tulu-Bohuai")
            ("ral" nil  nil   nil I   L   () nil nil "Ralte")
            ("ram" nil  nil   nil I   L   () nil nil "Canela")
            ("ran" nil  nil   nil I   L   () nil nil "Riantana")
            ("rao" nil  nil   nil I   L   () nil nil "Rao")
            ("rap" nil  "rap" nil I   L   () nil "rapanui" "Rapanui")
            ("raq" nil  nil   nil I   L   () nil nil "Saam")
            ("rar" nil  "rar" nil I   L   () nil "Māori Kūki 'Āirani" "Rarotongan")
            ("ras" nil  nil   nil I   L   () nil nil "Tegali")
            ("rat" nil  nil   nil I   L   () nil nil "Razajerdi")
            ("rau" nil  nil   nil I   L   () nil nil "Raute")
            ("rav" nil  nil   nil I   L   () nil nil "Sampang")
            ("raw" nil  nil   nil I   L   () nil nil "Rawang")
            ("rax" nil  nil   nil I   L   () nil nil "Rang")
            ("ray" nil  nil   nil I   L   () nil nil "Rapa")
            ("raz" nil  nil   nil I   L   () nil nil "Rahambuu")
            ("rbb" nil  nil   nil I   L   () nil nil "Palaung, Rumai")
            ("rbk" nil  nil   nil I   L   () nil nil "Northern Bontok")
            ("rbl" nil  nil   nil I   L   () nil nil "Miraya Bikol")
            ("rbp" nil  nil   nil I   E   () nil nil "Barababaraba")
            ("rcf" nil  nil   nil I   L   () nil "Kréol Rénioné" "Réunion Creole French")
            ("rdb" nil  nil   nil I   L   () nil nil "Rudbari")
            ("rea" nil  nil   nil I   L   () nil nil "Rerau")
            ("reb" nil  nil   nil I   L   () nil nil "Rembong")
            ("ree" nil  nil   nil I   L   () nil nil "Kayan, Rejang")
            ("reg" nil  nil   nil I   L   () nil nil "Kara (Tanzania)")
            ("rei" nil  nil   nil I   L   () nil nil "Reli")
            ("rej" nil  nil   nil I   L   () nil nil "Rejang")
            ("rel" nil  nil   nil I   L   () nil nil "Rendille")
            ("rem" nil  nil   nil I   E   () nil nil "Remo")
            ("ren" nil  nil   nil I   L   () nil nil "Rengao")
            ("rer" nil  nil   nil I   E   () nil nil "Rer Bare")
            ("res" nil  nil   nil I   L   () nil nil "Reshe")
            ("ret" nil  nil   nil I   L   () nil nil "Retta")
            ("rey" nil  nil   nil I   L   () nil nil "Reyesano")
            ("rga" nil  nil   nil I   L   () nil nil "Roria")
            ("rge" nil  nil   nil I   L   () nil nil "Romano-Greek")
            ("rgk" nil  nil   nil I   E   () nil nil "Rangkas")
            ("rgn" nil  nil   nil I   L   () nil "Rumagnòl" "Romagnol")
            ("rgr" nil  nil   nil I   L   () nil nil "Resígaro")
            ("rgs" nil  nil   nil I   L   () nil nil "Roglai, Southern")
            ("rgu" nil  nil   nil I   L   () nil nil "Ringgou")
            ("rhg" nil  nil   nil I   L   () "Indo-European" "Ruáingga" "Rohingya")
            ("rhp" nil  nil   nil I   L   () nil nil "Yahang")
            ("ria" nil  nil   nil I   L   () nil nil "Riang (India)")
            ("rib" nil  nil   nil I   L   () nil nil "Bribri Sign Language")
            ("rie" nil  nil   t   I   L   () nil nil "Rien")
            ("rif" nil  nil   nil I   L   () nil nil "Tarifit")
            ("ril" nil  nil   nil I   L   () nil nil "Riang (Myanmar)")
            ("rim" nil  nil   nil I   L   () nil nil "Nyaturu")
            ("rin" nil  nil   nil I   L   () nil nil "Nungu")
            ("rir" nil  nil   nil I   L   () nil nil "Ribun")
            ("rit" nil  nil   nil I   L   () nil nil "Ritarungo")
            ("riu" nil  nil   nil I   L   () nil nil "Riung")
            ("rjb" nil  nil   t   nil nil () nil nil "Rajbanshi")
            ("rjg" nil  nil   nil I   L   () nil nil "Rajong")
            ("rji" nil  nil   nil I   L   () nil nil "Raji")
            ("rjs" nil  nil   nil I   L   () nil nil "Rajbanshi")
            ("rka" nil  nil   nil I   L   () nil nil "Kraol")
            ("rkb" nil  nil   nil I   L   () nil nil "Rikbaktsa")
            ("rkh" nil  nil   nil I   L   () nil nil "Rakahanga-Manihiki")
            ("rki" nil  nil   nil I   L   () nil nil "Rakhine")
            ("rkm" nil  nil   nil I   L   () nil nil "Marka")
            ("rkt" nil  nil   nil I   L   () "Indo-European" "রংপুরী" "Rangpuri")
            ("rkw" nil  nil   nil I   E   () nil nil "Arakwal")
            ("rma" nil  nil   nil I   L   () nil nil "Rama")
            ("rmb" nil  nil   nil I   L   () nil nil "Rembarunga")
            ("rmc" nil  nil   nil I   L   () nil nil "Romani, Carpathian")
            ("rmd" nil  nil   nil I   E   () nil nil "Traveller Danish")
            ("rme" nil  nil   nil I   L   () nil "Romanichal" "Angloromani")
            ("rmf" nil  nil   nil I   L   () nil "Romanó Kaló" "Romani, Kalo Finnish")
            ("rmg" nil  nil   nil I   L   () nil nil "Traveller Norwegian")
            ("rmh" nil  nil   nil I   L   () nil nil "Murkim")
            ("rmi" nil  nil   nil I   L   () nil nil "Lomavren")
            ("rmk" nil  nil   nil I   L   () nil nil "Romkun")
            ("rml" nil  nil   nil I   L   () nil nil "Romani, Baltic")
            ("rmm" nil  nil   nil I   L   () nil nil "Roma")
            ("rmn" nil  nil   nil I   L   () nil nil "Romani, Balkan")
            ("rmo" nil  nil   nil I   L   () nil "Sinto" "Romani, Sinte")
            ("rmp" nil  nil   nil I   L   () nil nil "Rempi")
            ("rmq" nil  nil   nil I   L   () nil nil "Caló")
            ("rmr" nil  nil   t   I   L   () nil "caló" "Caló")
            ("rms" nil  nil   nil I   L   () nil nil "Romanian Sign Language")
            ("rmt" nil  nil   nil I   L   () nil nil "Domari")
            ("rmu" nil  nil   nil I   L   () nil nil "Romani, Tavringer")
            ("rmv" nil  nil   nil I   C   () nil nil "Romanova")
            ("rmw" nil  nil   nil I   L   () nil nil "Romani, Welsh")
            ("rmx" nil  nil   nil I   L   () nil nil "Romam")
            ("rmy" nil  nil   nil I   L   () nil nil "Romani, Vlax")
            ("rmz" nil  nil   nil I   L   () nil nil "Marma")
            ("rna" nil  nil   t   I   E   () nil nil "Runa")
            ("rnb" nil  nil   nil I   L   () nil nil "Brunca Sign Language")
            ("rnd" nil  nil   nil I   L   () nil nil "Ruund")
            ("rng" nil  nil   nil I   L   () nil nil "Ronga")
            ("rnl" nil  nil   nil I   L   () nil nil "Ranglong")
            ("rnn" nil  nil   nil I   L   () nil nil "Roon")
            ("rnp" nil  nil   nil I   L   () nil nil "Rongpo")
            ("rnr" nil  nil   nil I   E   () nil nil "Nari Nari")
            ("rnw" nil  nil   nil I   L   () nil nil "Rungwa")
            (nil   nil  "roa" nil C   nil () nil nil "Romance languages")
            ("rob" nil  nil   nil I   L   () nil nil "Tae'")
            ("roc" nil  nil   nil I   L   () nil nil "Roglai, Cacgia")
            ("rod" nil  nil   nil I   L   () nil nil "Rogo")
            ("roe" nil  nil   nil I   L   () nil nil "Ronji")
            ("rof" nil  nil   nil I   L   () nil nil "Rombo")
            ("rog" nil  nil   nil I   L   () nil nil "Roglai, Northern")
            ("roh" "rm" "roh" nil I   L   () "Indo-European" "rumantsch" "Raeto-Romance")
            ("rol" nil  nil   nil I   L   () nil nil "Romblomanon")
            ("rom" nil  "rom" nil M   L   ("rmc"
                                           "rmf" "rml" "rmn"
                                           "rmo" "rmw" "rmy") nil "रोमानो" "Romany")
            ("ron" "ro" "rum" nil I   L   () "Indo-European" "română" "Romanian")
            ("roo" nil  nil   nil I   L   () nil nil "Rotokas")
            ("rop" nil  nil   nil I   L   () nil nil "Kriol")
            ("ror" nil  nil   nil I   L   () nil nil "Rongga")
            ("rou" nil  nil   nil I   L   () nil nil "Runga")
            ("row" nil  nil   nil I   L   () nil nil "Dela-Oenale")
            ("rpn" nil  nil   nil I   L   () nil nil "Repanbitip")
            ("rpt" nil  nil   nil I   L   () nil nil "Rapting")
            ("rri" nil  nil   nil I   L   () nil nil "Ririo")
            ("rrm" nil  nil   nil I   E   () "Austronesian" "Ta Rē Moriori" "Moriori")
            ("rro" nil  nil   nil I   L   () nil nil "Waima")
            ("rrt" nil  nil   nil I   E   () nil nil "Arritinngithigh")
            ("rsb" nil  nil   nil I   L   () nil nil "Romano-Serbian")
            ("rsi" nil  nil   t   I   L   () nil nil "Rennellese Sign Language")
            ("rsk" nil  nil   nil I   L   () "Indo-European" "Руски" "Rusyn, Ruthenian")
            ("rsl" nil  nil   nil I   L   () nil nil "Russian Sign Language")
            ("rsm" nil  nil   nil I   L   () nil nil "Miriwoong Sign Language")
            ("rsn" nil  nil   nil I   L   () nil nil "Rwandan Sign Language")
            ("rsw" nil  nil   nil I   L   () "Niger–Congo" nil "Rishiwa")
            ("rtc" nil  nil   nil I   L   () nil nil "Rungtu Chin")
            ("rth" nil  nil   nil I   L   () nil nil "Ratahan")
            ("rtm" nil  nil   nil I   L   () nil nil "Rotuman")
            ("rts" nil  nil   nil I   E   () "Uralic" nil "Yurats")
            ("rtw" nil  nil   nil I   L   () nil nil "Rathawi")
            ("rub" nil  nil   nil I   L   () nil nil "Gungu")
            ("ruc" nil  nil   nil I   L   () nil nil "Ruli")
            ("rue" nil  nil   nil I   L   () nil "русин" "Rusyn")
            ("ruf" nil  nil   nil I   L   () nil nil "Luguru")
            ("rug" nil  nil   nil I   L   () nil nil "Roviana")
            ("ruh" nil  nil   nil I   L   () nil nil "Ruga")
            ("rui" nil  nil   nil I   L   () nil nil "Rufiji")
            ("ruk" nil  nil   nil I   L   () nil nil "Che")
            ("run" "rn" "run" nil I   L   () "Niger–Congo" "kiRundi" "Rundi")
            ("ruo" nil  nil   nil I   L   () nil "istroromånă" "Romanian, Istro")
            ("rup" nil  "rup" nil I   L   () nil "Armăneashce" "Aromanian")
            ("ruq" nil  nil   nil I   L   () nil "meglenoromană" "Romanian, Megleno")
            ("rus" "ru" "rus" nil I   L   () "Indo-European" "русский" "Russian")
            ("rut" nil  nil   nil I   L   () nil nil "Rutul")
            ("ruu" nil  nil   nil I   L   () nil nil "Lobu, Lanas")
            ("ruy" nil  nil   nil I   L   () nil nil "Mala (Nigeria)")
            ("ruz" nil  nil   nil I   L   () nil nil "Ruma")
            ("rwa" nil  nil   nil I   L   () nil nil "Rawo")
            ("rwk" nil  nil   nil I   L   () nil nil "Rwa")
            ("rwl" nil  nil   nil I   L   () "Niger–Congo" nil "Ruwila")
            ("rwm" nil  nil   nil I   L   () nil nil "Amba (Uganda)")
            ("rwo" nil  nil   nil I   L   () nil nil "Rawa")
            ("rwr" nil  nil   nil I   L   () nil nil "Marwari (India)")
            ("rws" nil  nil   t   nil nil () nil nil "Rawas")
            ("rxd" nil  nil   nil I   L   () nil nil "Ngardi")
            ("rxw" nil  nil   nil I   E   () nil nil "Karuwali")
            ("ryn" nil  nil   nil I   L   () nil nil "Amami-Oshima, Northern")
            ("rys" nil  nil   nil I   L   () nil nil "Yaeyama")
            ("ryu" nil  nil   nil I   L   () nil "うちなーぐち" "Okinawan, Central")
            ("rzh" nil  nil   nil I   L   () nil nil "Razihi")
            ;; S
            ("saa" nil  nil   nil I   L   () nil nil "Saba")
            ("sab" nil  nil   nil I   L   () nil nil "Buglere")
            ("sac" nil  nil   nil I   L   () nil nil "Mesquakie")
            ("sad" nil  "sad" nil I   L   () nil nil "Sandawe")
            ("sae" nil  nil   nil I   L   () nil nil "Sabanês")
            ("saf" nil  nil   nil I   L   () nil nil "Safaliba")
            ("sag" "sg" "sag" nil I   L   () "Creole" "sängö" "Sango")
            ("sah" nil  "sah" nil I   L   () nil "Саха" "Sakha")
            (nil   nil  "sai" nil C   nil () nil nil "South American Indian languages")
            ("saj" nil  nil   nil I   L   () nil nil "Sahu")
            ("sak" nil  nil   nil I   L   () nil nil "Sake")
            (nil   nil  "sal" nil C   nil () nil nil "Salishan languages")
            ("sam" nil  "sam" nil I   E   () nil "ܐܪܡܝܐ" "Aramaic, Samaritan")
            ("san" "sa" "san" nil I   A   ("cls" "vsn") "Indo-European" "संस्कृतम्" "Sanskrit")
            ("sao" nil  nil   nil I   L   () nil nil "Sause")
            ("sap" nil  nil   t   I   L   () nil nil "Sanapaná")
            ("saq" nil  nil   nil I   L   () nil nil "Samburu")
            ("sar" nil  nil   nil I   E   () nil nil "Saraveca")
            ("sas" nil  "sas" nil I   L   () nil nil "Sasak")
            ("sat" nil  "sat" nil I   L   () nil "संथाली" "Santali")
            ("sau" nil  nil   nil I   L   () nil nil "Saleman")
            ("sav" nil  nil   nil I   L   () nil nil "Saafi-Saafi")
            ("saw" nil  nil   nil I   L   () nil nil "Sawi")
            ("sax" nil  nil   nil I   L   () nil nil "Sa")
            ("say" nil  nil   nil I   L   () nil nil "Saya")
            ("saz" nil  nil   nil I   L   () nil nil "Saurashtra")
            ("sba" nil  nil   nil I   L   () nil nil "Ngambay")
            ("sbb" nil  nil   nil I   L   () nil nil "Simbo")
            ("sbc" nil  nil   nil I   L   () nil nil "Kele (Papua New Guinea)")
            ("sbd" nil  nil   nil I   L   () nil nil "Samo, Southern")
            ("sbe" nil  nil   nil I   L   () nil nil "Saliba")
            ("sbf" nil  nil   nil I   L   () nil nil "Shabo")
            ("sbg" nil  nil   nil I   L   () nil nil "Seget")
            ("sbh" nil  nil   nil I   L   () nil nil "Sori-Harengan")
            ("sbi" nil  nil   nil I   L   () nil nil "Seti")
            ("sbj" nil  nil   nil I   L   () nil nil "Surbakhal")
            ("sbk" nil  nil   nil I   L   () nil nil "Safwa")
            ("sbl" nil  nil   nil I   L   () nil nil "Sambal, Botolan")
            ("sbm" nil  nil   nil I   L   () nil nil "Sagala")
            ("sbn" nil  nil   nil I   L   () nil nil "Sindhi Bhil")
            ("sbo" nil  nil   nil I   L   () nil nil "Sabüm")
            ("sbp" nil  nil   nil I   L   () nil nil "Sangu (Tanzania)")
            ("sbq" nil  nil   nil I   L   () nil nil "Sileibi")
            ("sbr" nil  nil   nil I   L   () nil nil "Sembakung Murut")
            ("sbs" nil  nil   nil I   L   () nil nil "Subiya")
            ("sbt" nil  nil   nil I   L   () nil nil "Kimki")
            ("sbu" nil  nil   nil I   L   () nil nil "Stod Bhoti")
            ("sbv" nil  nil   nil I   A   () nil nil "Sabine")
            ("sbw" nil  nil   nil I   L   () nil nil "Simba")
            ("sbx" nil  nil   nil I   L   () nil nil "Seberuang")
            ("sby" nil  nil   nil I   L   () nil nil "Soli")
            ("sbz" nil  nil   nil I   L   () nil nil "Sara Kaba")
            ("sca" nil  nil   t   I   L   () nil nil "Sansu")
            ("scb" nil  nil   nil I   L   () nil nil "Chut")
            ("scc" nil  nil   t   nil nil () nil nil "Serbian")
            ("sce" nil  nil   nil I   L   () nil nil "Dongxiang")
            ("scf" nil  nil   nil I   L   () nil nil "San Miguel Creole French")
            ("scg" nil  nil   nil I   L   () nil nil "Sanggau")
            ("sch" nil  nil   nil I   L   () nil nil "Sakechep")
            ("sci" nil  nil   nil I   L   () nil nil "Sri Lankan Creole Malay")
            ("sck" nil  nil   nil I   L   () nil nil "Sadri")
            ("scl" nil  nil   nil I   L   () nil nil "Shina")
            ("scn" nil  "scn" nil I   L   () nil "sicilianu" "Sicilian")
            ("sco" nil  "sco" nil I   L   () nil "Scots" "Scots")
            ("scp" nil  nil   nil I   L   () nil nil "Helambu Sherpa")
            ("scq" nil  nil   nil I   L   () nil nil "Sa'och")
            ("scr" nil  nil   t   nil nil () nil nil "Croatian")
            ("scs" nil  nil   nil I   L   () nil "Sahtú Got’ine" "Slavey, North")
            ("sct" nil  nil   nil I   L   () "Austroasiatic" nil "Southern Katang")
            ("scu" nil  nil   nil I   L   () nil nil "Shumcho")
            ("scv" nil  nil   nil I   L   () nil nil "Sheni")
            ("scw" nil  nil   nil I   L   () nil nil "Sha")
            ("scx" nil  nil   nil I   A   () nil nil "Sicel")
            ("sda" nil  nil   nil I   L   () nil nil "Toraja-Sa'dan")
            ("sdb" nil  nil   nil I   L   () nil nil "Shabak")
            ("sdc" nil  nil   nil I   L   () nil "sassaresu" "Sardinian, Sassarese")
            ("sdd" nil  nil   t   nil nil () nil nil "Semendo")
            ("sde" nil  nil   nil I   L   () nil nil "Surubu")
            ("sdf" nil  nil   nil I   L   () nil nil "Sarli")
            ("sdg" nil  nil   nil I   L   () nil nil "Savi")
            ("sdh" nil  nil   nil I   L   () nil nil "Kurdish, Southern")
            ("sdi" nil  nil   t   nil nil () nil nil "Sindang Kelingi")
            ("sdj" nil  nil   nil I   L   () nil nil "Suundi")
            ("sdk" nil  nil   nil I   L   () nil nil "Sos Kundi")
            ("sdl" nil  nil   nil I   L   () nil nil "Saudi Arabian Sign Language")
            ("sdm" nil  nil   t   I   L   () nil nil "Semandang")
            ("sdn" nil  nil   nil I   L   () nil nil "Sardinian, Gallurese")
            ("sdo" nil  nil   nil I   L   () nil nil "Bukar Sadong")
            ("sdp" nil  nil   nil I   L   () nil nil "Sherdukpen")
            ("sdq" nil  nil   nil I   L   () nil nil "Semandang")
            ("sdr" nil  nil   nil I   L   () nil nil "Sadri, Oraon")
            ("sds" nil  nil   nil I   E   () nil nil "Sened")
            ("sdt" nil  nil   nil I   E   () nil nil "Shuadit")
            ("sdu" nil  nil   nil I   L   () nil nil "Sarudu")
            ("sdx" nil  nil   nil I   L   () nil nil "Sibu")
            ("sdz" nil  nil   nil I   L   () nil nil "Sallands")
            ("sea" nil  nil   nil I   L   () nil nil "Semai")
            ("seb" nil  nil   nil I   L   () nil nil "Senoufo, Shempire")
            ("sec" nil  nil   nil I   L   () nil "Shashishalhem" "Sechelt")
            ("sed" nil  nil   nil I   L   () nil nil "Sedang")
            ("see" nil  nil   nil I   L   () "Iroquoian" "Onõtowáka" "Seneca")
            ("sef" nil  nil   nil I   L   () nil nil "Senoufo, Cebaara")
            ("seg" nil  nil   nil I   L   () nil nil "Segeju")
            ("seh" nil  nil   nil I   L   () nil nil "Sena")
            ("sei" nil  nil   nil I   L   () nil "Cmiique Iitom" "Seri")
            ("sej" nil  nil   nil I   L   () nil nil "Sene")
            ("sek" nil  nil   nil I   L   () nil "Tsek’ehne" "Sekani")
            ("sel" nil  "sel" nil I   L   () nil "шӧльӄумыт" "Selkup")
            (nil   nil  "sem" nil C   nil () nil nil "Semitic languages")
            ("sen" nil  nil   nil I   L   () nil nil "Sénoufo, Nanerigé")
            ("seo" nil  nil   nil I   L   () nil nil "Suarmin")
            ("sep" nil  nil   nil I   L   () nil nil "Sénoufo, Sìcìté")
            ("seq" nil  nil   nil I   L   () nil nil "Sénoufo, Senara")
            ("ser" nil  nil   nil I   L   () nil nil "Serrano")
            ("ses" nil  nil   nil I   L   () nil nil "Songhai, Koyraboro Senni")
            ("set" nil  nil   nil I   L   () nil nil "Sentani")
            ("seu" nil  nil   nil I   L   () nil nil "Serui-Laut")
            ("sev" nil  nil   nil I   L   () nil nil "Senoufo, Nyarafolo")
            ("sew" nil  nil   nil I   L   () nil nil "Sewa Bay")
            ("sey" nil  nil   nil I   L   () nil nil "Secoya")
            ("sez" nil  nil   nil I   L   () nil nil "Chin, Senthang")
            ("sfb" nil  nil   nil I   L   () nil nil "Langue des signes de Belgique Francophone")
            ("sfe" nil  nil   nil I   L   () nil nil "Eastern Subanen")
            ("sfm" nil  nil   nil I   L   () nil nil "Small Flowery Miao")
            ("sfs" nil  nil   nil I   L   () nil nil "South African Sign Language")
            ("sfw" nil  nil   nil I   L   () nil nil "Sehwi")
            ("sga" nil  "sga" nil I   H   () nil nil "Old Irish")
            ("sgb" nil  nil   nil I   L   () nil nil "Ayta, Mag-Anchi")
            ("sgc" nil  nil   nil I   L   () nil nil "Kipsigis")
            ("sgd" nil  nil   nil I   L   () nil nil "Surigaonon")
            ("sge" nil  nil   nil I   L   () nil nil "Segai")
            ("sgg" nil  nil   nil I   L   () nil nil "Swiss-German Sign Language")
            ("sgh" nil  nil   nil I   L   () nil nil "Shughni")
            ("sgi" nil  nil   nil I   L   () nil nil "Suga")
            ("sgj" nil  nil   nil I   L   () nil nil "Surgujia")
            ("sgk" nil  nil   nil I   L   () nil nil "Sangkong")
            ("sgl" nil  nil   t   I   L   () nil nil "Sanglechi-Ishkashimi")
            ("sgm" nil  nil   nil I   E   () nil nil "Singa")
            (nil   nil  "sgn" nil C   nil () nil nil "Sign Languages")
            ("sgo" nil  nil   t   I   L   () nil nil "Songa")
            ("sgp" nil  nil   nil I   L   () nil nil "Singpho")
            ("sgr" nil  nil   nil I   L   () nil nil "Sangisari")
            ("sgs" nil  nil   nil I   L   () nil nil "Samogitian")
            ("sgt" nil  nil   nil I   L   () nil nil "Brokpake")
            ("sgu" nil  nil   nil I   L   () nil nil "Salas")
            ("sgw" nil  nil   nil I   L   () nil nil "Sebat Bet Gurage")
            ("sgx" nil  nil   nil I   L   () nil nil "Sierra Leone Sign Language")
            ("sgy" nil  nil   nil I   L   () nil nil "Sanglechi")
            ("sgz" nil  nil   nil I   L   () nil nil "Sursurunga")
            ("sha" nil  nil   nil I   L   () nil nil "Shall-Zwall")
            ("shb" nil  nil   nil I   L   () nil nil "Ninam")
            ("shc" nil  nil   nil I   L   () nil nil "Sonde")
            ("shd" nil  nil   nil I   L   () nil nil "Kundal Shahi")
            ("she" nil  nil   nil I   L   () nil nil "Sheko")
            ("shg" nil  nil   nil I   L   () nil nil "Shua")
            ("shh" nil  nil   nil I   L   () nil "Sosoni'" "Shoshoni")
            ("shi" nil  nil   nil I   L   () nil "تشلحيت" "Tachelhit")
            ("shj" nil  nil   nil I   L   () nil nil "Shatt")
            ("shk" nil  nil   nil I   L   () nil nil "Shilluk")
            ("shl" nil  nil   nil I   L   () nil nil "Shendu")
            ("shm" nil  nil   nil I   L   () nil nil "Shahrudi")
            ("shn" nil  "shn" nil I   L   () nil nil "Shan")
            ("sho" nil  nil   nil I   L   () nil nil "Shanga")
            ("shp" nil  nil   nil I   L   () nil nil "Shipibo-Conibo")
            ("shq" nil  nil   nil I   L   () nil nil "Sala")
            ("shr" nil  nil   nil I   L   () nil nil "Shi")
            ("shs" nil  nil   nil I   L   () nil "Secwepemctsin" "Shuswap")
            ("sht" nil  nil   nil I   E   () nil nil "Shasta")
            ("shu" nil  nil   nil I   L   () "Arabic" nil "Arabic (Chadian)")
            ("shv" nil  nil   nil I   L   () nil nil "Shehri")
            ("shw" nil  nil   nil I   L   () nil nil "Shwai")
            ("shx" nil  nil   nil I   L   () nil nil "She")
            ("shy" nil  nil   nil I   L   () nil "Tachawit" "Tachawit")
            ("shz" nil  nil   nil I   L   () nil nil "Senoufo, Syenara")
            ("sia" nil  nil   nil I   E   () nil nil "Sami, Akkala")
            ("sib" nil  nil   nil I   L   () nil nil "Kenyah, Sebob")
            ("sic" nil  nil   t   I   L   () nil nil "Malinguat")
            ("sid" nil  "sid" nil I   L   () nil "Sidámo" "Sidamo")
            ("sie" nil  nil   nil I   L   () nil nil "Simaa")
            ("sif" nil  nil   nil I   L   () nil nil "Siamou")
            ("sig" nil  nil   nil I   L   () nil nil "Paasaal")
            ("sih" nil  nil   nil I   L   () nil nil "Zire")
            ("sii" nil  nil   nil I   L   () nil nil "Shom Peng")
            ("sij" nil  nil   nil I   L   () nil nil "Numbami")
            ("sik" nil  nil   nil I   L   () nil nil "Sikiana")
            ("sil" nil  nil   nil I   L   () nil nil "Sisaala, Tumulung")
            ("sim" nil  nil   nil I   L   () nil nil "Mende (Papua New Guinea)")
            ("sin" "si" "sin" nil I   L   () "Indo-European" "සිංහල" "Sinhala; Sinhalese")
            (nil   nil  "sio" nil C   nil () nil nil "Siouan languages")
            ("sip" nil  nil   nil I   L   () nil nil "Sikkimese")
            ("siq" nil  nil   nil I   L   () nil nil "Sonia")
            ("sir" nil  nil   nil I   L   () nil nil "Siri")
            ("sis" nil  nil   nil I   E   () nil nil "Siuslaw")
            (nil   nil  "sit" nil C   nil () nil nil "Sino-Tibetan languages")
            ("siu" nil  nil   nil I   L   () nil nil "Sinagen")
            ("siv" nil  nil   nil I   L   () nil nil "Sumariup")
            ("siw" nil  nil   nil I   L   () nil nil "Siwai")
            ("six" nil  nil   nil I   L   () nil nil "Sumau")
            ("siy" nil  nil   nil I   L   () nil nil "Sivandi")
            ("siz" nil  nil   nil I   L   () nil nil "Siwi")
            ("sja" nil  nil   nil I   L   () nil nil "Epena")
            ("sjb" nil  nil   nil I   L   () nil nil "Sajau Basap")
            ("sjd" nil  nil   nil I   L   () nil "самь" "Sami, Kildin")
            ("sje" nil  nil   nil I   L   () nil nil "Sami, Pite")
            ("sjg" nil  nil   nil I   L   () nil nil "Assangori")
            ("sjk" nil  nil   nil I   E   () nil nil "Sami, Kemi")
            ("sjl" nil  nil   nil I   L   () nil nil "Sajalong")
            ("sjm" nil  nil   nil I   L   () nil nil "Mapun")
            ("sjn" nil  nil   nil I   C   () nil nil "Sindarin")
            ("sjo" nil  nil   nil I   L   () nil "ᠰᡞᠪᡝ ᡤᡞᠰᡠᠨ" "Xibe")
            ("sjp" nil  nil   nil I   L   () nil nil "Surajpuri")
            ("sjr" nil  nil   nil I   L   () nil nil "Siar-Lak")
            ("sjs" nil  nil   nil I   E   () nil nil "Senhaja De Srair")
            ("sjt" nil  nil   nil I   L   () nil nil "Sami, Ter")
            ("sju" nil  nil   nil I   L   () nil nil "Sami, Ume")
            ("sjw" nil  nil   nil I   L   () nil "Shaawanwaʼ" "Shawnee")
            ("ska" nil  nil   nil I   L   () nil nil "Skagit")
            ("skb" nil  nil   nil I   L   () nil nil "Saek")
            ("skc" nil  nil   nil I   L   () nil nil "Sauk")
            ("skd" nil  nil   nil I   L   () nil nil "Miwok, Southern Sierra")
            ("ske" nil  nil   nil I   L   () nil nil "Seke (Vanuatu)")
            ("skf" nil  nil   nil I   L   () nil nil "Sakirabiá")
            ("skg" nil  nil   nil I   L   () nil nil "Malagasy, Sakalava")
            ("skh" nil  nil   nil I   L   () nil nil "Sikule")
            ("ski" nil  nil   nil I   L   () nil nil "Sika")
            ("skj" nil  nil   nil I   L   () nil nil "Seke (Nepal)")
            ("skk" nil  nil   t   I   L   () nil nil "Sok")
            ("skl" nil  nil   t   nil nil () nil nil "Selako")
            ("skm" nil  nil   nil I   L   () nil nil "Sakam")
            ("skn" nil  nil   nil I   L   () nil nil "Subanon, Kolibugan")
            ("sko" nil  nil   nil I   L   () nil nil "Seko Tengah")
            ("skp" nil  nil   nil I   L   () nil nil "Sekapan")
            ("skq" nil  nil   nil I   L   () nil nil "Sininkere")
            ("skr" nil  nil   nil I   L   () nil nil "Saraiki")
            ("sks" nil  nil   nil I   L   () nil nil "Maia")
            ("skt" nil  nil   nil I   L   () nil nil "Sakata")
            ("sku" nil  nil   nil I   L   () nil nil "Sakao")
            ("skv" nil  nil   nil I   L   () nil nil "Skou")
            ("skw" nil  nil   nil I   E   () nil nil "Skepi Creole Dutch")
            ("skx" nil  nil   nil I   L   () nil nil "Seko Padang")
            ("sky" nil  nil   nil I   L   () nil nil "Sikaiana")
            ("skz" nil  nil   nil I   L   () nil nil "Sekar")
            (nil   nil  "sla" nil C   nil () nil nil "Slavic languages")
            ("slb" nil  nil   t   nil nil () nil nil "Kahumamahon Saluan")
            ("slc" nil  nil   nil I   L   () nil nil "Sáliba")
            ("sld" nil  nil   nil I   L   () nil nil "Sissala")
            ("sle" nil  nil   nil I   L   () nil nil "Sholaga")
            ("slf" nil  nil   nil I   L   () nil nil "Swiss-Italian Sign Language")
            ("slg" nil  nil   nil I   L   () nil nil "Selungai Murut")
            ("slh" nil  nil   nil I   L   () nil nil "Salish, Southern Puget Sound")
            ("sli" nil  nil   nil I   L   () nil nil "Silesian, Lower")
            ("slj" nil  nil   nil I   L   () nil nil "Salumá")
            ("slk" "sk" "slo" nil I   L   () "Indo-European" "slovenčina" "Slovak")
            ("sll" nil  nil   nil I   L   () nil nil "Salt-Yui")
            ("slm" nil  nil   nil I   L   () nil nil "Sama, Pangutaran")
            ("sln" nil  nil   nil I   E   () nil nil "Salinan")
            ("slp" nil  nil   nil I   L   () nil nil "Lamaholot")
            ("slq" nil  nil   t   I   E   () nil nil "Salchuq")
            ("slr" nil  nil   nil I   L   () nil "Salar" "Salar")
            ("sls" nil  nil   nil I   L   () nil nil "Singapore Sign Language")
            ("slt" nil  nil   nil I   L   () nil nil "Sila")
            ("slu" nil  nil   nil I   L   () nil nil "Selaru")
            ("slv" "sl" "slv" nil I   L   () "Indo-European" "slovenščina" "Slovenian")
            ("slw" nil  nil   nil I   L   () nil nil "Sialum")
            ("slx" nil  nil   nil I   L   () nil nil "Salampasu")
            ("sly" nil  nil   nil I   L   () nil nil "Selayar")
            ("slz" nil  nil   nil I   L   () nil nil "Ma'ya")
            ("sma" nil  "sma" nil I   L   () nil "saemi" "Southern Sami")
            ("smb" nil  nil   nil I   L   () nil nil "Simbari")
            ("smc" nil  nil   nil I   E   () nil nil "Som")
            ("smd" nil  nil   t   I   L   () nil nil "Sama")
            ("sme" "se" "sme" nil I   L   () "Uralic" "sámi" "Northern Sami")
            ("smf" nil  nil   nil I   L   () nil nil "Auwe")
            ("smg" nil  nil   nil I   L   () nil nil "Simbali")
            ("smh" nil  nil   nil I   L   () nil nil "Samei")
            (nil   nil  "smi" nil C   nil () nil nil "Sami languages")
            ("smj" nil  "smj" nil I   L   () nil "sámi" "Lule Sami")
            ("smk" nil  nil   nil I   L   () nil nil "Bolinao")
            ("sml" nil  nil   nil I   L   () nil nil "Sama, Central")
            ("smm" nil  nil   nil I   L   () nil nil "Musasa")
            ("smn" nil  "smn" nil I   L   () nil "säämi" "Inari Sami")
            ("smo" "sm" "smo" nil I   L   () "Austronesian" "gagana Samoa" "Samoan")
            ("smp" nil  nil   nil I   E   () nil nil "Samaritan")
            ("smq" nil  nil   nil I   L   () nil nil "Samo")
            ("smr" nil  nil   nil I   L   () nil nil "Simeulue")
            ("sms" nil  "sms" nil I   L   () nil "sää'm" "Skolt Sami")
            ("smt" nil  nil   nil I   L   () nil nil "Simte")
            ("smu" nil  nil   nil I   E   () nil nil "Somray")
            ("smv" nil  nil   nil I   L   () nil nil "Samvedi")
            ("smw" nil  nil   nil I   L   () nil nil "Sumbawa")
            ("smx" nil  nil   nil I   L   () nil nil "Samba")
            ("smy" nil  nil   nil I   L   () nil nil "Semnani")
            ("smz" nil  nil   nil I   L   () nil nil "Simeku")
            ("sna" "sn" "sna" nil I   L   () "Niger–Congo" "chiShona" "Shona")
            ("snb" nil  nil   t   I   L   () nil nil "Sebuyau")
            ("snc" nil  nil   nil I   L   () nil nil "Sinaugoro")
            ("snd" "sd" "snd" nil I   L   () "Indo-European" "سندھی" "Sindhi")
            ("sne" nil  nil   nil I   L   () nil nil "Jagoi")
            ("snf" nil  nil   nil I   L   () nil nil "Noon")
            ("sng" nil  nil   nil I   L   () nil nil "Sanga (Democratic Republic of Congo)")
            ("snh" nil  nil   t   I   E   () nil nil "Shinabo")
            ("sni" nil  nil   nil I   E   () nil nil "Sensi")
            ("snj" nil  nil   nil I   L   () nil nil "Sango, Riverain")
            ("snk" nil  "snk" nil I   L   () nil "Soninkanxaane" "Soninke")
            ("snl" nil  nil   nil I   L   () nil nil "Sangil")
            ("snm" nil  nil   nil I   L   () nil nil "Ma'di, Southern")
            ("snn" nil  nil   nil I   L   () nil nil "Siona")
            ("sno" nil  nil   nil I   L   () nil nil "Snohomish")
            ("snp" nil  nil   nil I   L   () nil nil "Siane")
            ("snq" nil  nil   nil I   L   () nil nil "Sangu (Gabon)")
            ("snr" nil  nil   nil I   L   () nil nil "Sihan")
            ("sns" nil  nil   nil I   L   () nil nil "South West Bay")
            ("snu" nil  nil   nil I   L   () nil nil "Senggi")
            ("snv" nil  nil   nil I   L   () nil nil "Sa'ban")
            ("snw" nil  nil   nil I   L   () nil nil "Selee")
            ("snx" nil  nil   nil I   L   () nil nil "Sam")
            ("sny" nil  nil   nil I   L   () nil nil "Saniyo-Hiyewe")
            ("snz" nil  nil   nil I   L   () nil nil "Sinsauru")
            ("soa" nil  nil   nil I   L   () nil nil "Thai Song")
            ("sob" nil  nil   nil I   L   () nil nil "Sobei")
            ("soc" nil  nil   nil I   L   () nil nil "So (Democratic Republic of Congo)")
            ("sod" nil  nil   nil I   L   () nil nil "Songoora")
            ("soe" nil  nil   nil I   L   () nil nil "Songomeno")
            ("sog" nil  "sog" nil I   A   () nil nil "Sogdian")
            ("soh" nil  nil   nil I   L   () nil nil "Aka")
            ("soi" nil  nil   nil I   L   () nil nil "Sonha")
            ("soj" nil  nil   nil I   L   () nil nil "Soi")
            ("sok" nil  nil   nil I   L   () nil nil "Sokoro")
            ("sol" nil  nil   nil I   L   () nil nil "Solos")
            ("som" "so" "som" nil I   L   () "Afro-Asiatic" "Soomaaliga" "Somali")
            (nil   nil  "son" nil C   nil () nil nil "Songhai languages")
            ("soo" nil  nil   nil I   L   () nil nil "Songo")
            ("sop" nil  nil   nil I   L   () nil nil "Songe")
            ("soq" nil  nil   nil I   L   () nil nil "Kanasi")
            ("sor" nil  nil   nil I   L   () nil nil "Somrai")
            ("sos" nil  nil   nil I   L   () nil nil "Seeku")
            ("sot" "st" "sot" nil I   L   () "Niger–Congo" "Sesotho" "Sotho, Southern")
            ("sou" nil  nil   nil I   L   () nil nil "Thai, Southern")
            ("sov" nil  nil   nil I   L   () nil nil "Sonsorol")
            ("sow" nil  nil   nil I   L   () nil nil "Sowanda")
            ("sox" nil  nil   nil I   L   () nil nil "So (Cameroon)")
            ("soy" nil  nil   nil I   L   () nil nil "Miyobe")
            ("soz" nil  nil   nil I   L   () nil nil "Temi")
            ("spa" "es" "spa" nil I   L   () "Indo-European" "español" "Spanish (Castilian)")
            ("spb" nil  nil   nil I   L   () nil nil "Sepa (Indonesia)")
            ("spc" nil  nil   nil I   L   () nil nil "Sapé")
            ("spd" nil  nil   nil I   L   () nil nil "Saep")
            ("spe" nil  nil   nil I   L   () nil nil "Sepa (Papua New Guinea)")
            ("spg" nil  nil   nil I   L   () nil nil "Sian")
            ("spi" nil  nil   nil I   L   () nil nil "Saponi")
            ("spk" nil  nil   nil I   L   () nil nil "Sengo")
            ("spl" nil  nil   nil I   L   () nil nil "Selepet")
            ("spm" nil  nil   nil I   L   () nil nil "Sepen")
            ("spn" nil  nil   nil I   L   () "Mascoian" nil "Sanapaná")
            ("spo" nil  nil   nil I   L   () nil nil "Spokane")
            ("spp" nil  nil   nil I   L   () nil nil "Senoufo, Supyire")
            ("spq" nil  nil   nil I   L   () nil nil "Spanish, Loreto-Ucayali")
            ("spr" nil  nil   nil I   L   () nil nil "Saparua")
            ("sps" nil  nil   nil I   L   () nil nil "Saposa")
            ("spt" nil  nil   nil I   L   () nil nil "Spiti Bhoti")
            ("spu" nil  nil   nil I   L   () nil nil "Sapuan")
            ("spv" nil  nil   nil I   L   () nil nil "Sambalpuri")
            ("spx" nil  nil   nil I   A   () nil nil "South Picene")
            ("spy" nil  nil   nil I   L   () nil nil "Sabaot")
            ("sqa" nil  nil   nil I   L   () nil nil "Shama-Sambuga")
            ("sqh" nil  nil   nil I   L   () nil nil "Shau")
            ("sqi" "sq" "alb" nil M   L   ("aae"
                                           "aat" "aln" "als") "Indo-European" "Shqip" "Albanian")
            ("sqk" nil  nil   nil I   L   () nil nil "Albanian Sign Language")
            ("sqm" nil  nil   nil I   L   () nil nil "Suma")
            ("sqn" nil  nil   nil I   E   () nil nil "Susquehannock")
            ("sqo" nil  nil   nil I   L   () nil nil "Sorkhei")
            ("sqq" nil  nil   nil I   L   () nil nil "Sou")
            ("sqr" nil  nil   nil I   H   () nil nil "Siculo Arabic")
            ("sqs" nil  nil   nil I   L   () nil nil "Sri Lankan Sign Language")
            ("sqt" nil  nil   nil I   L   () nil nil "Soqotri")
            ("squ" nil  nil   nil I   L   () nil nil "Squamish")
            ("sqx" nil  nil   nil I   L   () nil nil "Kufr Qassem Sign Language (KQSL)")
            ("sra" nil  nil   nil I   L   () nil nil "Saruga")
            ("srb" nil  nil   nil I   L   () nil nil "Sora")
            ("src" nil  nil   nil I   L   () nil nil "Sardinian, Logudorese")
            ("srd" "sc" "srd" nil M   L   ("sdc"
                                           "sdn" "src" "sro") "Indo-European" "sardu" "Sardinian")
            ("sre" nil  nil   nil I   L   () nil nil "Sara")
            ("srf" nil  nil   nil I   L   () nil nil "Nafi")
            ("srg" nil  nil   nil I   L   () nil nil "Sulod")
            ("srh" nil  nil   nil I   L   () nil nil "Sarikoli")
            ("sri" nil  nil   nil I   L   () nil nil "Siriano")
            ("srj" nil  nil   t   nil nil () nil nil "Serawai")
            ("srk" nil  nil   nil I   L   () nil nil "Serudung Murut")
            ("srl" nil  nil   nil I   L   () nil nil "Isirawa")
            ("srm" nil  nil   nil I   L   () nil "saamáka" "Saramaccan")
            ("srn" nil  "srn" nil I   L   () nil "Sranang Tongo" "Sranan")
            ("sro" nil  nil   nil I   L   () nil nil "Sardinian, Campidanese")
            ("srp" "sr" "srp" nil I   L   () "Indo-European" "српски / srpski" "Serbian")
            ("srq" nil  nil   nil I   L   () nil nil "Sirionó")
            ("srr" nil  "srr" nil I   L   () nil nil "Serer")
            ("srs" nil  nil   nil I   L   () nil nil "Sarsi")
            ("srt" nil  nil   nil I   L   () nil nil "Sauri")
            ("sru" nil  nil   nil I   L   () nil nil "Suruí")
            ("srv" nil  nil   nil I   L   () nil nil "Sorsogon, Waray")
            ("srw" nil  nil   nil I   L   () nil nil "Serua")
            ("srx" nil  nil   nil I   L   () nil nil "Sirmauri")
            ("sry" nil  nil   nil I   L   () nil nil "Sera")
            ("srz" nil  nil   nil I   L   () nil nil "Shahmirzadi")
            (nil   nil  "ssa" nil C   nil () nil nil "Nilo-Saharan languages")
            ("ssb" nil  nil   nil I   L   () nil nil "Sama, Southern")
            ("ssc" nil  nil   nil I   L   () nil nil "Suba-Simbiti")
            ("ssd" nil  nil   nil I   L   () nil nil "Siroi")
            ("sse" nil  nil   nil I   L   () nil nil "Balangingi")
            ("ssf" nil  nil   nil I   L   () nil nil "Thao")
            ("ssg" nil  nil   nil I   L   () nil nil "Seimat")
            ("ssh" nil  nil   nil I   L   () nil nil "Arabic, Shihhi Spoken")
            ("ssi" nil  nil   nil I   L   () nil nil "Sansi")
            ("ssj" nil  nil   nil I   L   () nil nil "Sausi")
            ("ssk" nil  nil   nil I   L   () nil nil "Sunam")
            ("ssl" nil  nil   nil I   L   () nil nil "Sisaala, Western")
            ("ssm" nil  nil   nil I   L   () nil nil "Semnam")
            ("ssn" nil  nil   nil I   L   () nil nil "Sanye")
            ("sso" nil  nil   nil I   L   () nil nil "Sissano")
            ("ssp" nil  nil   nil I   L   () nil nil "Spanish Sign Language")
            ("ssq" nil  nil   nil I   L   () nil nil "So'a")
            ("ssr" nil  nil   nil I   L   () nil nil "Swiss-French Sign Language")
            ("sss" nil  nil   nil I   L   () nil nil "Sô")
            ("sst" nil  nil   nil I   L   () nil nil "Sinasina")
            ("ssu" nil  nil   nil I   L   () nil nil "Susuami")
            ("ssv" nil  nil   nil I   L   () nil nil "Shark Bay")
            ("ssw" "ss" "ssw" nil I   L   () "Niger–Congo" "siSwati" "Swati")
            ("ssx" nil  nil   nil I   L   () nil nil "Samberigi")
            ("ssy" nil  nil   nil I   L   () nil nil "Saho")
            ("ssz" nil  nil   nil I   L   () nil nil "Sengseng")
            ("sta" nil  nil   nil I   L   () nil nil "Settla")
            ("stb" nil  nil   nil I   L   () nil nil "Subanen, Northern")
            ("stc" nil  nil   t   I   L   () nil nil "Santa Cruz")
            ("std" nil  nil   nil I   L   () nil nil "Sentinel")
            ("ste" nil  nil   nil I   L   () nil nil "Liana-Seti")
            ("stf" nil  nil   nil I   L   () nil nil "Seta")
            ("stg" nil  nil   nil I   L   () nil nil "Trieng")
            ("sth" nil  nil   nil I   L   () nil nil "Shelta")
            ("sti" nil  nil   nil I   L   () nil nil "Stieng, Bulo")
            ("stj" nil  nil   nil I   L   () nil nil "Samo, Matya")
            ("stk" nil  nil   nil I   L   () nil nil "Arammba")
            ("stl" nil  nil   nil I   L   () nil nil "Stellingwerfs")
            ("stm" nil  nil   nil I   L   () nil nil "Setaman")
            ("stn" nil  nil   nil I   L   () nil nil "Owa")
            ("sto" nil  nil   nil I   L   () nil "Isga Iʼabi" "Stoney")
            ("stp" nil  nil   nil I   L   () nil nil "Tepehuan, Southeastern")
            ("stq" nil  nil   nil I   L   () nil "Seeltersk" "Saterland Frisian")
            ("str" nil  nil   nil I   L   () nil "xʷsenəčqən" "Salish, Straits")
            ("sts" nil  nil   nil I   L   () nil nil "Shumashti")
            ("stt" nil  nil   nil I   L   () nil nil "Stieng, Budeh")
            ("stu" nil  nil   nil I   L   () nil nil "Samtao")
            ("stv" nil  nil   nil I   L   () nil nil "Silt'e")
            ("stw" nil  nil   nil I   L   () nil nil "Satawalese")
            ("sty" nil  nil   nil I   L   () "Turkic" nil "Siberian Tatar")
            ("sua" nil  nil   nil I   L   () nil nil "Sulka")
            ("sub" nil  nil   nil I   L   () nil nil "Suku")
            ("suc" nil  nil   nil I   L   () nil nil "Subanon, Western")
            ("sue" nil  nil   nil I   L   () nil nil "Suena")
            ("suf" nil  nil   t   nil nil () nil nil "Tarpia")
            ("sug" nil  nil   nil I   L   () nil nil "Suganga")
            ("suh" nil  nil   t   nil nil () nil nil "Suba")
            ("sui" nil  nil   nil I   L   () nil nil "Suki")
            ("suj" nil  nil   nil I   L   () nil nil "Shubi")
            ("suk" nil  "suk" nil I   L   () nil nil "Sukuma")
            ("sul" nil  nil   t   I   L   () nil nil "Surigaonon")
            ("sum" nil  nil   t   I   L   () nil nil "Sumo-Mayangna")
            ("sun" "su" "sun" nil I   L   () "Austronesian" "basa Sunda" "Sundanese")
            ("suo" nil  nil   nil I   L   () "Skou" nil "Bouni")
            ("suq" nil  nil   nil I   L   () nil nil "Suri")
            ("sur" nil  nil   nil I   L   () nil nil "Mwaghavul")
            ("sus" nil  "sus" nil I   L   () nil nil "Susu")
            ("sut" nil  nil   nil I   E   () nil nil "Subtiaba")
            ("suu" nil  nil   t   nil nil () nil nil "Sungkai")
            ("suv" nil  nil   nil I   L   () nil nil "Sulung")
            ("suw" nil  nil   nil I   L   () nil nil "Sumbwa")
            ("sux" nil  "sux" nil I   A   () nil "eme-ĝir" "Sumerian")
            ("suy" nil  nil   nil I   L   () nil nil "Suyá")
            ("suz" nil  nil   nil I   L   () nil nil "Sunwar")
            ("sva" nil  nil   nil I   L   () nil "ლუშნუ" "Svan")
            ("svb" nil  nil   nil I   L   () nil nil "Ulau-Suain")
            ("svc" nil  nil   nil I   L   () nil nil "Vincentian Creole English")
            ("sve" nil  nil   nil I   L   () nil nil "Serili")
            ("svk" nil  nil   nil I   L   () nil nil "Slovakian Sign Language")
            ("svm" nil  nil   nil I   L   () nil nil "Slavomolisano")
            ("svr" nil  nil   t   I   L   () nil nil "Savara")
            ("svs" nil  nil   nil I   L   () nil nil "Savosavo")
            ("svx" nil  nil   nil I   H   () nil nil "Skalvian")
            ("swa" "sw" "swa" nil M   L   ("swc" "swh") "Niger–Congo" "kiswahili" "Swahili (generic)")
            ("swb" nil  nil   nil I   L   () nil "شِقُمُرِ" "Comorian")
            ("swc" nil  nil   nil I   L   () nil nil "Swahili, Congo")
            ("swe" "sv" "swe" nil I   L   () "Indo-European" "svenska" "Swedish")
            ("swf" nil  nil   nil I   L   () nil nil "Sere")
            ("swg" nil  nil   nil I   L   () nil "Schwäbisch" "Swabian")
            ("swh" nil  nil   nil I   L   () nil nil "Swahili (specific)")
            ("swi" nil  nil   nil I   L   () nil nil "Sui")
            ("swj" nil  nil   nil I   L   () nil nil "Sira")
            ("swk" nil  nil   nil I   L   () nil nil "Sena, Malawi")
            ("swl" nil  nil   nil I   L   () nil "svenskt teckenspråk" "Swedish Sign Language")
            ("swm" nil  nil   nil I   L   () nil nil "Samosa")
            ("swn" nil  nil   nil I   L   () nil nil "Sawknah")
            ("swo" nil  nil   nil I   L   () nil nil "Shanenawa")
            ("swp" nil  nil   nil I   L   () nil nil "Suau")
            ("swq" nil  nil   nil I   L   () nil nil "Sharwa")
            ("swr" nil  nil   nil I   L   () nil nil "Saweru")
            ("sws" nil  nil   nil I   L   () nil nil "Seluwasan")
            ("swt" nil  nil   nil I   L   () nil nil "Sawila")
            ("swu" nil  nil   nil I   L   () nil nil "Suwawa")
            ("swv" nil  nil   nil I   L   () nil nil "Shekhawati")
            ("sww" nil  nil   nil I   E   () nil nil "Sowa")
            ("swx" nil  nil   nil I   L   () nil nil "Suruahá")
            ("swy" nil  nil   nil I   L   () nil nil "Sarua")
            ("sxb" nil  nil   nil I   L   () nil nil "Suba")
            ("sxc" nil  nil   nil I   A   () nil nil "Sicanian")
            ("sxe" nil  nil   nil I   L   () nil nil "Sighu")
            ("sxg" nil  nil   nil I   L   () nil nil "Shixing")
            ("sxk" nil  nil   nil I   E   () nil nil "Kalapuya, Southern")
            ("sxl" nil  nil   nil I   E   () nil nil "Selian")
            ("sxm" nil  nil   nil I   L   () nil nil "Samre")
            ("sxn" nil  nil   nil I   L   () nil nil "Sangir")
            ("sxo" nil  nil   nil I   A   () nil nil "Sorothaptic")
            ("sxr" nil  nil   nil I   L   () nil nil "Saaroa")
            ("sxs" nil  nil   nil I   L   () nil nil "Sasaru")
            ("sxu" nil  nil   nil I   L   () nil "Sächsisch" "Saxon, Upper")
            ("sxw" nil  nil   nil I   L   () nil nil "Gbe, Saxwe")
            ("sya" nil  nil   nil I   L   () nil nil "Siang")
            ("syb" nil  nil   nil I   L   () nil nil "Subanen, Central")
            ("syc" nil  "syc" nil I   H   () nil nil "Syriac, Classical")
            ("syi" nil  nil   nil I   L   () nil nil "Seki")
            ("syk" nil  nil   nil I   L   () nil nil "Sukur")
            ("syl" nil  nil   nil I   L   () "Indo-European" "ꠍꠤꠟꠐꠤ" "Sylheti")
            ("sym" nil  nil   nil I   L   () nil nil "Samo, Maya")
            ("syn" nil  nil   nil I   L   () nil nil "Senaya")
            ("syo" nil  nil   nil I   L   () nil nil "Suoy")
            ("syr" nil  "syr" nil M   L   ("aii" "cld") nil "ܣܘܪܝܐܝܐ" "Syriac")
            ("sys" nil  nil   nil I   L   () nil nil "Sinyar")
            ("syw" nil  nil   nil I   L   () nil nil "Kagate")
            ("syx" nil  nil   nil I   L   () "Niger–Congo" nil "Samay")
            ("syy" nil  nil   nil I   L   () nil nil "Al-Sayyid Bedouin Sign Language")
            ("sza" nil  nil   nil I   L   () nil nil "Semelai")
            ("szb" nil  nil   nil I   L   () nil nil "Ngalum")
            ("szc" nil  nil   nil I   L   () nil nil "Semaq Beri")
            ("szd" nil  nil   t   I   E   () nil nil "Seru")
            ("sze" nil  nil   nil I   L   () nil nil "Seze")
            ("szg" nil  nil   nil I   L   () nil nil "Sengele")
            ("szk" nil  nil   t   nil nil () nil nil "Sizaki")
            ("szl" nil  nil   nil I   L   () nil "Ślůnsko godka" "Silesian")
            ("szn" nil  nil   nil I   L   () nil nil "Sula")
            ("szp" nil  nil   nil I   L   () nil nil "Suabo")
            ("szs" nil  nil   nil I   L   () nil nil "Solomon Islands Sign Language")
            ("szv" nil  nil   nil I   L   () nil nil "Isu (Fako Division)")
            ("szw" nil  nil   nil I   L   () nil nil "Sawai")
            ("szy" nil  nil   nil I   L   () "Austronesian" nil "Sakizaya")
            ;; T
            ("taa" nil  nil   nil I   L   () nil nil "Tanana, Lower")
            ("tab" nil  nil   nil I   L   () nil "табасаран" "Tabassaran")
            ("tac" nil  nil   nil I   L   () nil nil "Tarahumara, Lowland")
            ("tad" nil  nil   nil I   L   () nil nil "Tause")
            ("tae" nil  nil   nil I   L   () nil nil "Tariano")
            ("taf" nil  nil   nil I   L   () nil nil "Tapirapé")
            ("tag" nil  nil   nil I   L   () nil nil "Tagoi")
            ("tah" "ty" "tah" nil I   L   () "Austronesian" "te reo Tahiti" "Tahitian")
            (nil   nil  "tai" nil C   nil () nil nil "Tai languages")
            ("taj" nil  nil   nil I   L   () nil nil "Tamang, Eastern")
            ("tak" nil  nil   nil I   L   () nil nil "Tala")
            ("tal" nil  nil   nil I   L   () nil nil "Tal")
            ("tam" "ta" "tam" nil I   L   () "Dravidian" "தமிழ்" "Tamil")
            ("tan" nil  nil   nil I   L   () nil nil "Tangale")
            ("tao" nil  nil   nil I   L   () nil nil "Yami")
            ("tap" nil  nil   nil I   L   () nil nil "Taabwa")
            ("taq" nil  nil   nil I   L   () nil "تَمَاشَقْ" "Tamasheq")
            ("tar" nil  nil   nil I   L   () nil "Ralámuli" "Tarahumara, Central")
            ("tas" nil  nil   nil I   E   () nil nil "Tay Boi")
            ("tat" "tt" "tat" nil I   L   () "Turkic" "Tatarça" "Tatar")
            ("tau" nil  nil   nil I   L   () nil nil "Tanana, Upper")
            ("tav" nil  nil   nil I   L   () nil nil "Tatuyo")
            ("taw" nil  nil   nil I   L   () nil nil "Tai")
            ("tax" nil  nil   nil I   L   () nil nil "Tamki")
            ("tay" nil  nil   nil I   L   () nil nil "Atayal")
            ("taz" nil  nil   nil I   L   () nil nil "Tocho")
            ("tba" nil  nil   nil I   L   () nil nil "Tubarão")
            ("tbb" nil  nil   t   I   E   () nil nil "Tapeba")
            ("tbc" nil  nil   nil I   L   () nil nil "Takia")
            ("tbd" nil  nil   nil I   L   () nil nil "Kaki Ae")
            ("tbe" nil  nil   nil I   L   () nil nil "Tanimbili")
            ("tbf" nil  nil   nil I   L   () nil nil "Mandara")
            ("tbg" nil  nil   nil I   L   () nil nil "Tairora, North")
            ("tbh" nil  nil   nil I   E   () nil nil "Thurawal")
            ("tbi" nil  nil   nil I   L   () nil nil "Gaam")
            ("tbj" nil  nil   nil I   L   () nil nil "Tiang")
            ("tbk" nil  nil   nil I   L   () nil nil "Tagbanwa, Calamian")
            ("tbl" nil  nil   nil I   L   () nil nil "Tboli")
            ("tbm" nil  nil   nil I   L   () nil nil "Tagbu")
            ("tbn" nil  nil   nil I   L   () nil nil "Tunebo, Barro Negro")
            ("tbo" nil  nil   nil I   L   () nil nil "Tawala")
            ("tbp" nil  nil   nil I   L   () nil nil "Taworta")
            ("tbr" nil  nil   nil I   L   () nil nil "Tumtum")
            ("tbs" nil  nil   nil I   L   () nil nil "Tanguat")
            ("tbt" nil  nil   nil I   L   () nil nil "Tembo (Kitembo)")
            ("tbu" nil  nil   nil I   E   () nil nil "Tubar")
            ("tbv" nil  nil   nil I   L   () nil nil "Tobo")
            ("tbw" nil  nil   nil I   L   () nil "tabanawa" "Tagbanwa")
            ("tbx" nil  nil   nil I   L   () nil nil "Kapin")
            ("tby" nil  nil   nil I   L   () nil nil "Tabaru")
            ("tbz" nil  nil   nil I   L   () nil nil "Ditammari")
            ("tca" nil  nil   nil I   L   () nil nil "Ticuna")
            ("tcb" nil  nil   nil I   L   () nil nil "Tanacross")
            ("tcc" nil  nil   nil I   L   () nil nil "Datooga")
            ("tcd" nil  nil   nil I   L   () nil nil "Tafi")
            ("tce" nil  nil   nil I   L   () nil nil "Tutchone, Southern")
            ("tcf" nil  nil   nil I   L   () nil nil "Tlapanec, Malinaltepec")
            ("tcg" nil  nil   nil I   L   () nil nil "Tamagario")
            ("tch" nil  nil   nil I   L   () nil nil "Turks And Caicos Creole English")
            ("tci" nil  nil   nil I   L   () nil nil "Wára")
            ("tck" nil  nil   nil I   L   () nil nil "Tchitchege")
            ("tcl" nil  nil   nil I   E   () nil nil "Taman (Myanmar)")
            ("tcm" nil  nil   nil I   L   () nil nil "Tanahmerah")
            ("tcn" nil  nil   nil I   L   () nil nil "Tichurong")
            ("tco" nil  nil   nil I   L   () nil nil "Taungyo")
            ("tcp" nil  nil   nil I   L   () nil nil "Chin, Tawr")
            ("tcq" nil  nil   nil I   L   () nil nil "Kaiy")
            ("tcs" nil  nil   nil I   L   () nil nil "Torres Strait Creole")
            ("tct" nil  nil   nil I   L   () nil nil "T'en")
            ("tcu" nil  nil   nil I   L   () nil nil "Tarahumara, Southeastern")
            ("tcw" nil  nil   nil I   L   () nil nil "Tecpatlán Totonac")
            ("tcx" nil  nil   nil I   L   () nil "தோதா" "Toda")
            ("tcy" nil  nil   nil I   L   () nil "ತುಳು" "Tulu")
            ("tcz" nil  nil   nil I   L   () nil nil "Chin, Thado")
            ("tda" nil  nil   nil I   L   () nil nil "Tagdal")
            ("tdb" nil  nil   nil I   L   () nil nil "Panchpargania")
            ("tdc" nil  nil   nil I   L   () nil nil "Emberá-Tadó")
            ("tdd" nil  nil   nil I   L   () nil "ᥖᥭᥰᥖᥬᥳᥑᥨᥒᥰ" "Tai Nüa")
            ("tde" nil  nil   nil I   L   () nil nil "Tiranige Diga Dogon")
            ("tdf" nil  nil   nil I   L   () nil nil "Talieng")
            ("tdg" nil  nil   nil I   L   () nil nil "Tamang, Western")
            ("tdh" nil  nil   nil I   L   () nil nil "Thulung")
            ("tdi" nil  nil   nil I   L   () nil nil "Tomadino")
            ("tdj" nil  nil   nil I   L   () nil nil "Tajio")
            ("tdk" nil  nil   nil I   L   () nil nil "Tambas")
            ("tdl" nil  nil   nil I   L   () nil nil "Sur")
            ("tdm" nil  nil   nil I   L   () nil nil "Taruma")
            ("tdn" nil  nil   nil I   L   () nil nil "Tondano")
            ("tdo" nil  nil   nil I   L   () nil nil "Teme")
            ("tdq" nil  nil   nil I   L   () nil nil "Tita")
            ("tdr" nil  nil   nil I   L   () nil nil "Todrah")
            ("tds" nil  nil   nil I   L   () nil nil "Doutai")
            ("tdt" nil  nil   nil I   L   () nil nil "Tetun Dili")
            ("tdu" nil  nil   t   I   L   () nil nil "Dusun, Tempasuk")
            ("tdv" nil  nil   nil I   L   () nil nil "Toro")
            ("tdx" nil  nil   nil I   L   () nil nil "Malagasy, Tandroy-Mahafaly")
            ("tdy" nil  nil   nil I   L   () nil nil "Tadyawan")
            ("tea" nil  nil   nil I   L   () nil nil "Temiar")
            ("teb" nil  nil   nil I   E   () nil nil "Tetete")
            ("tec" nil  nil   nil I   L   () nil nil "Terik")
            ("ted" nil  nil   nil I   L   () nil nil "Krumen, Tepo")
            ("tee" nil  nil   nil I   L   () nil nil "Tepehua, Huehuetla")
            ("tef" nil  nil   nil I   L   () nil nil "Teressa")
            ("teg" nil  nil   nil I   L   () nil nil "Teke-Tege")
            ("teh" nil  nil   nil I   L   () nil nil "Tehuelche")
            ("tei" nil  nil   nil I   L   () nil nil "Torricelli")
            ("tek" nil  nil   nil I   L   () nil nil "Teke, Ibali")
            ("tel" "te" "tel" nil I   L   () "Dravidian" "తెలుగు" "Telugu")
            ("tem" nil  "tem" nil I   L   () nil nil "Time")
            ("ten" nil  nil   nil I   E   () nil nil "Tama (Colombia)")
            ("teo" nil  nil   nil I   L   () nil nil "Teso")
            ("tep" nil  nil   nil I   E   () nil nil "Tepecano")
            ("teq" nil  nil   nil I   L   () nil nil "Temein")
            ("ter" nil  "ter" nil I   L   () nil nil "Tereno")
            ("tes" nil  nil   nil I   L   () nil nil "Tengger")
            ("tet" nil  "tet" nil I   L   () nil "Tetun" "Tetum")
            ("teu" nil  nil   nil I   L   () nil nil "Soo")
            ("tev" nil  nil   nil I   L   () nil nil "Teor")
            ("tew" nil  nil   nil I   L   () nil nil "Tewa (USA)")
            ("tex" nil  nil   nil I   L   () nil nil "Tennet")
            ("tey" nil  nil   nil I   L   () nil nil "Tulishi")
            ("tez" nil  nil   nil I   L   () "Afro-Asiatic" nil "Tetserret")
            ("tfi" nil  nil   nil I   L   () nil nil "Gbe, Tofin")
            ("tfn" nil  nil   nil I   L   () nil nil "Tanaina")
            ("tfo" nil  nil   nil I   L   () nil nil "Tefaro")
            ("tfr" nil  nil   nil I   L   () nil nil "Teribe")
            ("tft" nil  nil   nil I   L   () nil nil "Ternate")
            ("tga" nil  nil   nil I   L   () nil nil "Sagalla")
            ("tgb" nil  nil   nil I   L   () nil nil "Tebilung")
            ("tgc" nil  nil   nil I   L   () nil nil "Tigak")
            ("tgd" nil  nil   nil I   L   () nil nil "Ciwogai")
            ("tge" nil  nil   nil I   L   () nil nil "Tamang, Eastern Gorkha")
            ("tgf" nil  nil   nil I   L   () nil nil "Chalikha")
            ("tgg" nil  nil   t   I   L   () nil nil "Tangga")
            ("tgh" nil  nil   nil I   L   () nil nil "Tobagonian Creole English")
            ("tgi" nil  nil   nil I   L   () nil nil "Lawunuia")
            ("tgj" nil  nil   nil I   L   () nil nil "Tagin")
            ("tgk" "tg" "tgk" nil I   L   () "Indo-European" "тоҷикӣ" "Tajik")
            ("tgl" "tl" "tgl" nil I   L   () "Austronesian" "Tagálog" "Tagalog")
            ("tgn" nil  nil   nil I   L   () nil nil "Tandaganon")
            ("tgo" nil  nil   nil I   L   () nil nil "Sudest")
            ("tgp" nil  nil   nil I   L   () nil nil "Tangoa")
            ("tgq" nil  nil   nil I   L   () nil nil "Tring")
            ("tgr" nil  nil   nil I   L   () nil nil "Tareng")
            ("tgs" nil  nil   nil I   L   () nil nil "Nume")
            ("tgt" nil  nil   nil I   L   () nil nil "Tagbanwa, Central")
            ("tgu" nil  nil   nil I   L   () nil nil "Tanggu")
            ("tgv" nil  nil   nil I   E   () nil nil "Tingui-Boto")
            ("tgw" nil  nil   nil I   L   () nil nil "Senoufo, Tagwana")
            ("tgx" nil  nil   nil I   L   () nil "Tāgizi" "Tagish")
            ("tgy" nil  nil   nil I   E   () nil nil "Togoyo")
            ("tgz" nil  nil   nil I   E   () nil nil "Tagalaka")
            ("tha" "th" "tha" nil I   L   () "Tai–Kadai" "ภาษาไทย" "Thai")
            ("thc" nil  nil   t   I   L   () nil nil "Tai Hang Tong")
            ("thd" nil  nil   nil I   L   () nil nil "Thayore")
            ("the" nil  nil   nil I   L   () nil nil "Tharu, Chitwania")
            ("thf" nil  nil   nil I   L   () nil nil "Thangmi")
            ("thh" nil  nil   nil I   L   () nil nil "Tarahumara, Northern")
            ("thi" nil  nil   nil I   L   () nil nil "Tai Long")
            ("thk" nil  nil   nil I   L   () nil nil "Tharaka")
            ("thl" nil  nil   nil I   L   () nil nil "Tharu, Dangaura")
            ("thm" nil  nil   nil I   L   () nil nil "Aheu")
            ("thn" nil  nil   nil I   L   () nil nil "Thachanadan")
            ("thp" nil  nil   nil I   L   () nil "Nłeʼkepmxcin" "Thompson")
            ("thq" nil  nil   nil I   L   () nil nil "Tharu, Kochila")
            ("thr" nil  nil   nil I   L   () nil nil "Tharu, Rana")
            ("ths" nil  nil   nil I   L   () nil nil "Thakali")
            ("tht" nil  nil   nil I   L   () nil "Tāłtān" "Tahltan")
            ("thu" nil  nil   nil I   L   () nil nil "Thuri")
            ("thv" nil  nil   nil I   L   () nil nil "Tamahaq, Tahaggart")
            ("thw" nil  nil   t   I   L   () nil nil "Thudam")
            ("thx" nil  nil   t   I   L   () nil nil "The")
            ("thy" nil  nil   nil I   L   () nil nil "Tha")
            ("thz" nil  nil   nil I   L   () nil nil "Tamajeq, Tayart")
            ("tia" nil  nil   nil I   L   () nil nil "Tamazight, Tidikelt")
            ("tic" nil  nil   nil I   L   () nil nil "Tira")
            ("tid" nil  nil   t   I   L   () nil nil "Tidong")
            ("tie" nil  nil   t   I   L   () nil nil "Tingal")
            ("tif" nil  nil   nil I   L   () nil nil "Tifal")
            ("tig" nil  "tig" nil I   L   () nil "Tigré" "Tigre")
            ("tih" nil  nil   nil I   L   () nil nil "Timugon Murut")
            ("tii" nil  nil   nil I   L   () nil nil "Tiene")
            ("tij" nil  nil   nil I   L   () nil nil "Tilung")
            ("tik" nil  nil   nil I   L   () nil nil "Tikar")
            ("til" nil  nil   nil I   E   () nil nil "Tillamook")
            ("tim" nil  nil   nil I   L   () nil nil "Timbe")
            ("tin" nil  nil   nil I   L   () nil nil "Tindi")
            ("tio" nil  nil   nil I   L   () nil nil "Teop")
            ("tip" nil  nil   nil I   L   () nil nil "Trimuris")
            ("tiq" nil  nil   nil I   L   () nil nil "Tiéfo")
            ("tir" "ti" "tir" nil I   L   () "Afro-Asiatic" "ትግርኛ" "Tigrinya")
            ("tis" nil  nil   nil I   L   () nil nil "Itneg, Masadiit")
            ("tit" nil  nil   nil I   L   () nil nil "Tinigua")
            ("tiu" nil  nil   nil I   L   () nil nil "Itneg, Adasen")
            ("tiv" nil  "tiv" nil I   L   () nil nil "Tiv")
            ("tiw" nil  nil   nil I   L   () nil nil "Tiwi")
            ("tix" nil  nil   nil I   L   () nil nil "Tiwa, Southern")
            ("tiy" nil  nil   nil I   L   () nil nil "Tiruray")
            ("tiz" nil  nil   nil I   L   () nil nil "Tai Hongjin")
            ("tja" nil  nil   nil I   L   () nil nil "Tajuasohn")
            ("tjg" nil  nil   nil I   L   () nil nil "Tunjung")
            ("tji" nil  nil   nil I   L   () nil nil "Tujia, Northern")
            ("tjj" nil  nil   nil I   L   () nil nil "Tjungundji")
            ("tjl" nil  nil   nil I   L   () nil nil "Tai Laing")
            ("tjm" nil  nil   nil I   E   () nil nil "Timucua")
            ("tjn" nil  nil   nil I   E   () nil nil "Tonjon")
            ("tjo" nil  nil   nil I   L   () nil nil "Tamazight, Temacine")
            ("tjp" nil  nil   nil I   L   () nil nil "Tjupany")
            ("tjs" nil  nil   nil I   L   () nil nil "Tujia, Southern")
            ("tju" nil  nil   nil I   E   () nil nil "Tjurruru")
            ("tjw" nil  nil   nil I   L   () nil nil "Djabwurrung")
            ("tka" nil  nil   nil I   E   () nil nil "Truká")
            ("tkb" nil  nil   nil I   L   () nil nil "Buksa")
            ("tkd" nil  nil   nil I   L   () nil nil "Tukudede")
            ("tke" nil  nil   nil I   L   () nil nil "Takwane")
            ("tkf" nil  nil   nil I   E   () nil nil "Tukumanféd")
            ("tkg" nil  nil   nil I   L   () nil nil "Tesaka Malagasy")
            ("tkk" nil  nil   t   I   L   () nil nil "Takpa")
            ("tkl" nil  "tkl" nil I   L   () nil nil "Tokelau")
            ("tkm" nil  nil   nil I   E   () nil nil "Takelma")
            ("tkn" nil  nil   nil I   L   () nil nil "Toku-No-Shima")
            ("tkp" nil  nil   nil I   L   () nil nil "Tikopia")
            ("tkq" nil  nil   nil I   L   () nil nil "Tee")
            ("tkr" nil  nil   nil I   L   () nil nil "Tsakhur")
            ("tks" nil  nil   nil I   L   () nil nil "Takestani")
            ("tkt" nil  nil   nil I   L   () nil nil "Tharu, Kathoriya")
            ("tku" nil  nil   nil I   L   () nil nil "Upper Necaxa Totonac")
            ("tkv" nil  nil   nil I   L   () "Austronesian" nil "Mur Pano")
            ("tkw" nil  nil   nil I   L   () nil nil "Teanu")
            ("tkx" nil  nil   nil I   L   () nil nil "Tangko")
            ("tkz" nil  nil   nil I   L   () nil nil "Takua")
            ("tla" nil  nil   nil I   L   () nil nil "Tepehuan, Southwestern")
            ("tlb" nil  nil   nil I   L   () nil nil "Tobelo")
            ("tlc" nil  nil   nil I   L   () nil nil "Totonac, Yecuatla")
            ("tld" nil  nil   nil I   L   () nil nil "Talaud")
            ("tle" nil  nil   t   nil nil () nil nil "Southern Marakwet")
            ("tlf" nil  nil   nil I   L   () nil nil "Telefol")
            ("tlg" nil  nil   nil I   L   () nil nil "Tofanma")
            ("tlh" nil  "tlh" nil I   C   () nil "tlhIngan Hol" "Klingon; tlhIngan-Hol")
            ("tli" nil  "tli" nil I   L   () nil "Lingít" "Tlingit")
            ("tlj" nil  nil   nil I   L   () nil nil "Talinga-Bwisi")
            ("tlk" nil  nil   nil I   L   () nil nil "Taloki")
            ("tll" nil  nil   nil I   L   () nil nil "Tetela")
            ("tlm" nil  nil   nil I   L   () nil nil "Tolomako")
            ("tln" nil  nil   nil I   L   () nil nil "Talondo'")
            ("tlo" nil  nil   nil I   L   () nil nil "Talodi")
            ("tlp" nil  nil   nil I   L   () nil nil "Totonac, Filomena Mata-Coahuitlán")
            ("tlq" nil  nil   nil I   L   () nil nil "Tai Loi")
            ("tlr" nil  nil   nil I   L   () nil nil "Talise")
            ("tls" nil  nil   nil I   L   () nil nil "Tambotalo")
            ("tlt" nil  nil   nil I   L   () nil nil "Teluti")
            ("tlu" nil  nil   nil I   L   () nil nil "Tulehu")
            ("tlv" nil  nil   nil I   L   () nil nil "Taliabu")
            ("tlw" nil  nil   t   I   L   () nil nil "Wemale, South")
            ("tlx" nil  nil   nil I   L   () nil nil "Khehek")
            ("tly" nil  nil   nil I   L   () nil "толышә" "Talysh")
            ("tlz" nil  nil   t   I   L   () nil nil "Toala'")
            ("tma" nil  nil   nil I   L   () nil nil "Tama (Chad)")
            ("tmb" nil  nil   nil I   L   () nil nil "Katbol")
            ("tmc" nil  nil   nil I   L   () nil nil "Tumak")
            ("tmd" nil  nil   nil I   L   () nil nil "Haruai")
            ("tme" nil  nil   nil I   E   () nil nil "Tremembé")
            ("tmf" nil  nil   nil I   L   () nil nil "Toba-Maskoy")
            ("tmg" nil  nil   nil I   E   () nil nil "Ternateño")
            ("tmh" nil  "tmh" nil M   L   ("taq"
                                           "thv" "thz" "ttq") nil nil "Tamashek")
            ("tmi" nil  nil   nil I   L   () nil nil "Tutuba")
            ("tmj" nil  nil   nil I   L   () nil nil "Samarokena")
            ("tmk" nil  nil   t   I   L   () nil nil "Tamang, Northwestern")
            ("tml" nil  nil   nil I   L   () nil nil "Citak, Tamnim")
            ("tmm" nil  nil   nil I   L   () nil nil "Tai Thanh")
            ("tmn" nil  nil   nil I   L   () nil nil "Taman (Indonesia)")
            ("tmo" nil  nil   nil I   L   () nil nil "Temoq")
            ("tmp" nil  nil   t   I   L   () nil nil "Tai Mène")
            ("tmq" nil  nil   nil I   L   () nil nil "Tumleo")
            ("tmr" nil  nil   nil I   E   () nil nil "Aramaic, Talmudic")
            ("tms" nil  nil   nil I   L   () nil nil "Tima")
            ("tmt" nil  nil   nil I   L   () nil nil "Tasmate")
            ("tmu" nil  nil   nil I   L   () nil nil "Iau")
            ("tmv" nil  nil   nil I   L   () nil nil "Tembo (Motembo)")
            ("tmw" nil  nil   nil I   L   () nil nil "Temuan")
            ("tmx" nil  nil   t   nil nil () nil nil "Tomyang")
            ("tmy" nil  nil   nil I   L   () nil nil "Tami")
            ("tmz" nil  nil   nil I   E   () nil nil "Tamanaku")
            ("tna" nil  nil   nil I   L   () nil nil "Tacana")
            ("tnb" nil  nil   nil I   L   () nil nil "Tunebo, Western")
            ("tnc" nil  nil   nil I   L   () nil nil "Tanimuca-Retuarã")
            ("tnd" nil  nil   nil I   L   () nil nil "Tunebo, Angosturas")
            ("tne" nil  nil   t   I   L   () nil nil "Kallahan, Tinoc")
            ("tnf" nil  nil   t   I   L   () nil nil "Tangshewi")
            ("tng" nil  nil   nil I   L   () nil nil "Tobanga")
            ("tnh" nil  nil   nil I   L   () nil nil "Maiani")
            ("tni" nil  nil   nil I   L   () nil nil "Tandia")
            ("tnj" nil  nil   t   nil nil () nil nil "Tanjong")
            ("tnk" nil  nil   nil I   L   () nil nil "Kwamera")
            ("tnl" nil  nil   nil I   L   () nil nil "Lenakel")
            ("tnm" nil  nil   nil I   L   () nil nil "Tabla")
            ("tnn" nil  nil   nil I   L   () nil nil "Tanna, North")
            ("tno" nil  nil   nil I   L   () nil nil "Toromono")
            ("tnp" nil  nil   nil I   L   () nil nil "Whitesands")
            ("tnq" nil  nil   nil I   E   () nil nil "Taino")
            ("tnr" nil  nil   nil I   L   () nil nil "Budik")
            ("tns" nil  nil   nil I   L   () nil nil "Tenis")
            ("tnt" nil  nil   nil I   L   () nil nil "Tontemboan")
            ("tnu" nil  nil   nil I   L   () nil nil "Tay Khang")
            ("tnv" nil  nil   nil I   L   () nil nil "Tangchangya")
            ("tnw" nil  nil   nil I   L   () nil nil "Tonsawang")
            ("tnx" nil  nil   nil I   L   () nil nil "Tanema")
            ("tny" nil  nil   nil I   L   () nil nil "Tongwe")
            ("tnz" nil  nil   nil I   L   () nil nil "Tonga (Thailand)")
            ("tob" nil  nil   nil I   L   () nil nil "Toba")
            ("toc" nil  nil   nil I   L   () nil nil "Totonac, Coyutla")
            ("tod" nil  nil   nil I   L   () nil nil "Toma")
            ("toe" nil  nil   t   I   E   () nil nil "Tomedes")
            ("tof" nil  nil   nil I   L   () nil nil "Gizrra")
            ("tog" nil  "tog" nil I   L   () nil "chiTonga" "Tonga (Nyasa)")
            ("toh" nil  nil   nil I   L   () nil nil "Gitonga")
            ("toi" nil  nil   nil I   L   () nil nil "Tonga (Zambia)")
            ("toj" nil  nil   nil I   L   () nil nil "Tojolabal")
            ("tok" nil  nil   nil I   C   () nil "toki pona" "Toki Pona")
            ("tol" nil  nil   nil I   E   () nil nil "Tolowa")
            ("tom" nil  nil   nil I   L   () nil nil "Tombulu")
            ("ton" "to" "ton" nil I   L   () "Austronesian" "faka-Tonga" "Tonga (Tonga Islands)")
            ("too" nil  nil   nil I   L   () nil nil "Totonac, Xicotepec De Juárez")
            ("top" nil  nil   nil I   L   () nil nil "Totonac, Papantla")
            ("toq" nil  nil   nil I   L   () nil nil "Toposa")
            ("tor" nil  nil   nil I   L   () nil nil "Banda, Togbo-Vara")
            ("tos" nil  nil   nil I   L   () nil nil "Totonac, Highland")
            ("tot" nil  nil   t   nil nil () nil nil "Patla-Chicontla Totonac")
            ("tou" nil  nil   nil I   L   () nil nil "Tho")
            ("tov" nil  nil   nil I   L   () nil nil "Taromi, Upper")
            ("tow" nil  nil   nil I   L   () nil nil "Jemez")
            ("tox" nil  nil   nil I   L   () nil nil "Tobian")
            ("toy" nil  nil   nil I   L   () nil nil "Topoiyo")
            ("toz" nil  nil   nil I   L   () nil nil "To")
            ("tpa" nil  nil   nil I   L   () nil nil "Taupota")
            ("tpc" nil  nil   nil I   L   () nil nil "Tlapanec, Azoyú")
            ("tpe" nil  nil   nil I   L   () nil nil "Tippera")
            ("tpf" nil  nil   nil I   L   () nil nil "Tarpia")
            ("tpg" nil  nil   nil I   L   () nil nil "Kula")
            ("tpi" nil  "tpi" nil I   L   () nil "Tok Pisin" "Tok Pisin")
            ("tpj" nil  nil   nil I   L   () nil nil "Tapieté")
            ("tpk" nil  nil   nil I   E   () nil nil "Tupinikin")
            ("tpl" nil  nil   nil I   L   () nil nil "Tlapanec, Tlacoapa")
            ("tpm" nil  nil   nil I   L   () nil nil "Tampulma")
            ("tpn" nil  nil   nil I   E   () nil "abáñe'enga" "Tupinambá")
            ("tpo" nil  nil   nil I   L   () nil nil "Tai Pao")
            ("tpp" nil  nil   nil I   L   () nil nil "Tepehua, Pisaflores")
            ("tpq" nil  nil   nil I   L   () nil nil "Tukpa")
            ("tpr" nil  nil   nil I   L   () nil nil "Tuparí")
            ("tpt" nil  nil   nil I   L   () nil nil "Tepehua, Tlachichilco")
            ("tpu" nil  nil   nil I   L   () nil nil "Tampuan")
            ("tpv" nil  nil   nil I   L   () nil nil "Tanapag")
            ("tpw" nil  nil   t   I   E   () nil nil "Tupí")
            ("tpx" nil  nil   nil I   L   () nil nil "Tlapanec, Acatepec")
            ("tpy" nil  nil   nil I   L   () nil nil "Trumaí")
            ("tpz" nil  nil   nil I   L   () nil nil "Tinputz")
            ("tqb" nil  nil   nil I   L   () nil nil "Tembé")
            ("tql" nil  nil   nil I   L   () nil nil "Lehali")
            ("tqm" nil  nil   nil I   L   () nil nil "Turumsa")
            ("tqn" nil  nil   nil I   L   () nil nil "Tenino")
            ("tqo" nil  nil   nil I   L   () nil nil "Toaripi")
            ("tqp" nil  nil   nil I   L   () nil nil "Tomoip")
            ("tqq" nil  nil   nil I   L   () nil nil "Tunni")
            ("tqr" nil  nil   nil I   E   () nil nil "Torona")
            ("tqt" nil  nil   nil I   L   () nil nil "Totonac, Ozumatlán")
            ("tqu" nil  nil   nil I   L   () nil nil "Touo")
            ("tqw" nil  nil   nil I   E   () nil nil "Tonkawa")
            ("tra" nil  nil   nil I   L   () nil nil "Tirahi")
            ("trb" nil  nil   nil I   L   () nil nil "Terebu")
            ("trc" nil  nil   nil I   L   () nil nil "Triqui, Copala")
            ("trd" nil  nil   nil I   L   () nil nil "Turi")
            ("tre" nil  nil   nil I   L   () nil nil "Tarangan, East")
            ("trf" nil  nil   nil I   L   () nil nil "Trinidadian Creole English")
            ("trg" nil  nil   nil I   L   () nil nil "Lishán Didán")
            ("trh" nil  nil   nil I   L   () nil nil "Turaka")
            ("tri" nil  nil   nil I   L   () nil nil "Trió")
            ("trj" nil  nil   nil I   L   () nil nil "Toram")
            ("trl" nil  nil   nil I   L   () nil nil "Traveller Scottish")
            ("trm" nil  nil   nil I   L   () nil nil "Tregami")
            ("trn" nil  nil   nil I   L   () nil nil "Trinitario")
            ("tro" nil  nil   nil I   L   () nil nil "Naga, Tarao")
            ("trp" nil  nil   nil I   L   () nil nil "Kok Borok")
            ("trq" nil  nil   nil I   L   () nil nil "Triqui, San Martín Itunyoso")
            ("trr" nil  nil   nil I   L   () nil nil "Taushiro")
            ("trs" nil  nil   nil I   L   () nil nil "Triqui, Chicahuaxtla")
            ("trt" nil  nil   nil I   L   () nil nil "Tunggare")
            ("tru" nil  nil   nil I   L   () nil nil "Turoyo")
            ("trv" nil  nil   nil I   L   () "Austronesian" "Kari Seediq" "Seediq")
            ("trw" nil  nil   nil I   L   () nil nil "Torwali")
            ("trx" nil  nil   nil I   L   () nil nil "Tringgus")
            ("try" nil  nil   nil I   E   () nil nil "Turung")
            ("trz" nil  nil   nil I   E   () nil nil "Torá")
            ("tsa" nil  nil   nil I   L   () nil nil "Tsaangi")
            ("tsb" nil  nil   nil I   L   () nil nil "Tsamai")
            ("tsc" nil  nil   nil I   L   () nil nil "Tswa")
            ("tsd" nil  nil   nil I   L   () nil nil "Tsakonian")
            ("tse" nil  nil   nil I   L   () nil nil "Tunisian Sign Language")
            ("tsf" nil  nil   t   I   L   () nil nil "Tamang, Southwestern")
            ("tsg" nil  nil   nil I   L   () nil nil "Tausug")
            ("tsh" nil  nil   nil I   L   () nil nil "Tsuvan")
            ("tsi" nil  "tsi" nil I   L   () nil "Sm’algyax̣" "Tsimshian")
            ("tsj" nil  nil   nil I   L   () nil nil "Tshangla")
            ("tsk" nil  nil   nil I   L   () nil nil "Tseku")
            ("tsl" nil  nil   nil I   L   () nil nil "Ts'ün-Lao")
            ("tsm" nil  nil   nil I   L   () nil nil "Turkish Sign Language")
            ("tsn" "tn" "tsn" nil I   L   () "Niger–Congo" "Setswana" "Tswana")
            ("tso" "ts" "tso" nil I   L   () "Niger–Congo" "Xitsonga" "Tsonga")
            ("tsp" nil  nil   nil I   L   () nil nil "Toussian, Northern")
            ("tsq" nil  nil   nil I   L   () nil nil "Thai Sign Language")
            ("tsr" nil  nil   nil I   L   () nil nil "Akei")
            ("tss" nil  nil   nil I   L   () nil nil "Taiwan Sign Language")
            ("tst" nil  nil   nil I   L   () nil nil "Tondi Songway Kiini")
            ("tsu" nil  nil   nil I   L   () nil nil "Tsou")
            ("tsv" nil  nil   nil I   L   () nil nil "Tsogo")
            ("tsw" nil  nil   nil I   L   () nil nil "Tsishingini")
            ("tsx" nil  nil   nil I   L   () nil nil "Mubami")
            ("tsy" nil  nil   nil I   L   () nil nil "Tebul Sign Language")
            ("tsz" nil  nil   nil I   L   () nil nil "Purepecha")
            ("tta" nil  nil   nil I   E   () nil nil "Tutelo")
            ("ttb" nil  nil   nil I   L   () nil nil "Gaa")
            ("ttc" nil  nil   nil I   L   () nil nil "Tektiteko")
            ("ttd" nil  nil   nil I   L   () nil nil "Tauade")
            ("tte" nil  nil   nil I   L   () nil nil "Bwanabwana")
            ("ttf" nil  nil   nil I   L   () nil nil "Tuotomb")
            ("ttg" nil  nil   nil I   L   () nil nil "Tutong 2")
            ("tth" nil  nil   nil I   L   () nil nil "Ta'oih, Upper")
            ("tti" nil  nil   nil I   L   () nil nil "Tobati")
            ("ttj" nil  nil   nil I   L   () "Niger-Congo" "Orutooro" "Tooro")
            ("ttk" nil  nil   nil I   L   () nil nil "Totoro")
            ("ttl" nil  nil   nil I   L   () nil nil "Totela")
            ("ttm" nil  nil   nil I   L   () nil nil "Tutchone, Northern")
            ("ttn" nil  nil   nil I   L   () nil nil "Towei")
            ("tto" nil  nil   nil I   L   () nil nil "Ta'oih, Lower")
            ("ttp" nil  nil   nil I   L   () nil nil "Tombelala")
            ("ttq" nil  nil   nil I   L   () nil "تَمَاجِقْ" "Tamajaq, Tawallammat")
            ("ttr" nil  nil   nil I   L   () nil nil "Tera")
            ("tts" nil  nil   nil I   L   () nil "ภาษาอีสาน" "Thai, Northeastern")
            ("ttt" nil  nil   nil I   L   () nil "Tati, тати" "Tat, Muslim")
            ("ttu" nil  nil   nil I   L   () nil nil "Torau")
            ("ttv" nil  nil   nil I   L   () nil nil "Titan")
            ("ttw" nil  nil   nil I   L   () nil nil "Kenyah, Tutoh")
            ("ttx" nil  nil   t   nil nil () nil nil "Tutong 1")
            ("tty" nil  nil   nil I   L   () nil nil "Sikaritai")
            ("ttz" nil  nil   nil I   L   () nil nil "Tsum")
            ("tua" nil  nil   nil I   L   () nil nil "Wiarumus")
            ("tub" nil  nil   nil I   E   () nil nil "Tübatulabal")
            ("tuc" nil  nil   nil I   L   () nil nil "Mutu")
            ("tud" nil  nil   nil I   E   () nil nil "Tuxá")
            ("tue" nil  nil   nil I   L   () nil nil "Tuyuca")
            ("tuf" nil  nil   nil I   L   () nil nil "Tunebo, Central")
            ("tug" nil  nil   nil I   L   () nil nil "Tunia")
            ("tuh" nil  nil   nil I   L   () nil nil "Taulil")
            ("tui" nil  nil   nil I   L   () nil nil "Tupuri")
            ("tuj" nil  nil   nil I   L   () nil nil "Tugutil")
            ("tuk" "tk" "tuk" nil I   L   () "Turkic" "Түркмен" "Turkmen")
            ("tul" nil  nil   nil I   L   () nil nil "Tula")
            ("tum" nil  "tum" nil I   L   () nil "chiTumbuka" "Tumbuka")
            ("tun" nil  nil   nil I   L   () nil nil "Tunica")
            ("tuo" nil  nil   nil I   L   () nil nil "Tucano")
            (nil   nil  "tup" nil C   nil () nil nil "Tupi languages")
            ("tuq" nil  nil   nil I   L   () nil nil "Tedaga")
            ("tur" "tr" "tur" nil I   L   () "Turkic" "Türkçe" "Turkish")
            ("tus" nil  nil   nil I   L   () nil "Skarù∙rę’" "Tuscarora")
            (nil   nil  "tut" nil C   nil () nil nil "Altaic languages")
            ("tuu" nil  nil   nil I   L   () nil nil "Tututni")
            ("tuv" nil  nil   nil I   L   () nil "Ng'aturk(w)ana" "Turkana")
            ("tux" nil  nil   nil I   E   () nil nil "Tuxináwa")
            ("tuy" nil  nil   nil I   L   () nil nil "Tugen, North")
            ("tuz" nil  nil   nil I   L   () nil nil "Turka")
            ("tva" nil  nil   nil I   L   () nil nil "Vaghua")
            ("tvd" nil  nil   nil I   L   () nil nil "Tsuvadi")
            ("tve" nil  nil   nil I   L   () nil nil "Te'un")
            ("tvi" nil  nil   nil I   L   () "Afro-Asiatic" nil "Tulai")
            ("tvk" nil  nil   nil I   L   () nil nil "Ambrym, Southeast")
            ("tvl" nil  "tvl" nil I   L   () nil "'gana Tuvalu" "Tuvalu")
            ("tvm" nil  nil   nil I   L   () nil nil "Tela-Masbuar")
            ("tvn" nil  nil   nil I   L   () nil nil "Tavoyan")
            ("tvo" nil  nil   nil I   L   () nil nil "Tidore")
            ("tvs" nil  nil   nil I   L   () nil nil "Taveta")
            ("tvt" nil  nil   nil I   L   () nil nil "Naga, Tutsa")
            ("tvu" nil  nil   nil I   L   () nil nil "Tunen")
            ("tvw" nil  nil   nil I   L   () nil nil "Sedoa")
            ("tvx" nil  nil   nil I   E   () nil nil "Taivoan")
            ("tvy" nil  nil   nil I   E   () nil nil "Pidgin, Timor")
            ("twa" nil  nil   nil I   E   () nil nil "Twana")
            ("twb" nil  nil   nil I   L   () nil nil "Tawbuid, Western")
            ("twc" nil  nil   nil I   E   () nil nil "Teshenawa")
            ("twd" nil  nil   nil I   L   () nil nil "Twents")
            ("twe" nil  nil   nil I   L   () nil nil "Tewa (Indonesia)")
            ("twf" nil  nil   nil I   L   () nil nil "Tiwa, Northern")
            ("twg" nil  nil   nil I   L   () nil nil "Tereweng")
            ("twh" nil  nil   nil I   L   () nil nil "Tai Dón")
            ("twi" "tw" "twi" nil I   L   () "Niger–Congo" "Twi" "Twi")
            ("twl" nil  nil   nil I   L   () nil nil "Tawara")
            ("twm" nil  nil   nil I   L   () nil nil "Tawang Monpa")
            ("twn" nil  nil   nil I   L   () nil nil "Twendi")
            ("two" nil  nil   nil I   L   () nil nil "Tswapong")
            ("twp" nil  nil   nil I   L   () nil nil "Ere")
            ("twq" nil  nil   nil I   L   () nil nil "Tasawaq")
            ("twr" nil  nil   nil I   L   () nil nil "Tarahumara, Southwestern")
            ("twt" nil  nil   nil I   E   () nil nil "Turiwára")
            ("twu" nil  nil   nil I   L   () nil nil "Termanu")
            ("tww" nil  nil   nil I   L   () nil nil "Tuwari")
            ("twx" nil  nil   nil I   L   () nil nil "Tewe")
            ("twy" nil  nil   nil I   L   () nil nil "Tawoyan")
            ("txa" nil  nil   nil I   L   () nil nil "Tombonuwo")
            ("txb" nil  nil   nil I   A   () nil nil "Tokharian B")
            ("txc" nil  nil   nil I   E   () nil nil "Tsetsaut")
            ("txe" nil  nil   nil I   L   () nil nil "Totoli")
            ("txg" nil  nil   nil I   A   () "Sino-Tibetan" "𗼇𗟲" "Tangut")
            ("txh" nil  nil   nil I   A   () nil nil "Thracian")
            ("txi" nil  nil   nil I   L   () nil nil "Ikpeng")
            ("txj" nil  nil   nil I   L   () "Nilo-Saharan" nil "Tarjumo")
            ("txm" nil  nil   nil I   L   () nil nil "Tomini")
            ("txn" nil  nil   nil I   L   () nil nil "Tarangan, West")
            ("txo" nil  nil   nil I   L   () nil nil "Toto")
            ("txq" nil  nil   nil I   L   () nil nil "Tii")
            ("txr" nil  nil   nil I   A   () nil nil "Tartessian")
            ("txs" nil  nil   nil I   L   () nil nil "Tonsea")
            ("txt" nil  nil   nil I   L   () nil nil "Citak")
            ("txu" nil  nil   nil I   L   () nil nil "Kayapó")
            ("txx" nil  nil   nil I   L   () nil nil "Tatana")
            ("txy" nil  nil   nil I   L   () nil nil "Malagasy, Tanosy")
            ("tya" nil  nil   nil I   L   () nil nil "Tauya")
            ("tye" nil  nil   nil I   L   () nil nil "Kyenga")
            ("tyh" nil  nil   nil I   L   () nil nil "O'du")
            ("tyi" nil  nil   nil I   L   () nil nil "Teke-Tsaayi")
            ("tyj" nil  nil   nil I   L   () nil nil "Tai Do")
            ("tyl" nil  nil   nil I   L   () nil nil "Thu Lao")
            ("tyn" nil  nil   nil I   L   () nil nil "Kombai")
            ("typ" nil  nil   nil I   E   () nil nil "Thaypan")
            ("tyr" nil  nil   nil I   L   () nil nil "Tai Daeng")
            ("tys" nil  nil   nil I   L   () nil nil "Tày Sa Pa")
            ("tyt" nil  nil   nil I   L   () nil nil "Tày Tac")
            ("tyu" nil  nil   nil I   L   () nil nil "Kua")
            ("tyv" nil  "tyv" nil I   L   () nil "Тыва" "Tuvinian")
            ("tyx" nil  nil   nil I   L   () nil nil "Teke-Tyee")
            ("tyy" nil  nil   nil I   L   () "Niger–Congo" nil "Tiyaa")
            ("tyz" nil  nil   nil I   L   () nil nil "Tày")
            ("tza" nil  nil   nil I   L   () nil nil "Tanzanian Sign Language")
            ("tzb" nil  nil   t   I   L   () nil nil "Tzeltal, Bachajón")
            ("tzc" nil  nil   t   I   L   () nil nil "Tzotzil, Chamula")
            ("tze" nil  nil   t   I   L   () nil nil "Tzotzil, Chenalhó")
            ("tzh" nil  nil   nil I   L   () nil nil "Tzeltal, Oxchuc")
            ("tzj" nil  nil   nil I   L   () nil nil "Tzutujil, Eastern")
            ("tzl" nil  nil   nil I   C   () nil nil "Talossan")
            ("tzm" nil  nil   nil I   L   () nil "ⵜⵎⴰⵣⵉⵖⵜ" "Tamazight, Central Atlas")
            ("tzn" nil  nil   nil I   L   () nil nil "Tugun")
            ("tzo" nil  nil   nil I   L   () nil nil "Tzotzil, Venustiano Carranza")
            ("tzs" nil  nil   t   I   L   () nil nil "Tzotzil, San Andrés Larrainzar")
            ("tzt" nil  nil   t   I   L   () nil nil "Tzutujil, Western")
            ("tzu" nil  nil   t   I   L   () nil nil "Tzotzil, Huixtán")
            ("tzx" nil  nil   nil I   L   () nil nil "Tabriak")
            ("tzz" nil  nil   t   I   L   () nil nil "Tzotzil, Zinacantán")
            ;; U
            ("uam" nil  nil   nil I   E   () nil nil "Uamué")
            ("uan" nil  nil   nil I   L   () nil nil "Kuan")
            ("uar" nil  nil   nil I   L   () nil nil "Tairuma")
            ("uba" nil  nil   nil I   L   () nil nil "Ubang")
            ("ubi" nil  nil   nil I   L   () nil nil "Ubi")
            ("ubl" nil  nil   nil I   L   () nil nil "Buhi'non Bikol")
            ("ubm" nil  nil   t   nil nil () nil nil "Upper Baram Kenyah")
            ("ubr" nil  nil   nil I   L   () nil nil "Ubir")
            ("ubu" nil  nil   nil I   L   () nil nil "Umbu-Ungu")
            ("uby" nil  nil   nil I   E   () nil "atʷaχəbza" "Ubykh")
            ("uda" nil  nil   nil I   L   () nil nil "Uda")
            ("ude" nil  nil   nil I   L   () nil nil "Udihe")
            ("udg" nil  nil   nil I   L   () nil nil "Muduga")
            ("udi" nil  nil   nil I   L   () nil "удин" "Udi")
            ("udj" nil  nil   nil I   L   () nil nil "Ujir")
            ("udl" nil  nil   nil I   L   () nil nil "Wuzlam")
            ("udm" nil  "udm" nil I   L   () nil "удмурт" "Udmurt")
            ("udu" nil  nil   nil I   L   () nil nil "Uduk")
            ("ues" nil  nil   nil I   L   () nil nil "Kioko")
            ("ufi" nil  nil   nil I   L   () nil nil "Ufim")
            ("uga" nil  "uga" nil I   A   () nil nil "Ugaritic")
            ("ugb" nil  nil   nil I   E   () nil nil "Kuku-Ugbanh")
            ("uge" nil  nil   nil I   L   () nil nil "Ughele")
            ("ugh" nil  nil   nil I   L   () "Northeast Caucasian" nil "Kubachi")
            ("ugn" nil  nil   nil I   L   () nil nil "Ugandan Sign Language")
            ("ugo" nil  nil   nil I   L   () nil nil "Ugong")
            ("ugy" nil  nil   nil I   L   () nil nil "Uruguayan Sign Language")
            ("uha" nil  nil   nil I   L   () nil nil "Uhami")
            ("uhn" nil  nil   nil I   L   () nil nil "Damal")
            ("uig" "ug" "uig" nil I   L   () "Turkic" "ئۇيغۇرچە" "Uighur")
            ("uis" nil  nil   nil I   L   () nil nil "Uisai")
            ("uiv" nil  nil   nil I   L   () nil nil "Iyive")
            ("uji" nil  nil   nil I   L   () nil nil "Tanjijili")
            ("uka" nil  nil   nil I   L   () nil nil "Kaburi")
            ("ukg" nil  nil   nil I   L   () nil nil "Ukuriguma")
            ("ukh" nil  nil   nil I   L   () nil nil "Ukhwejo")
            ("uki" nil  nil   nil I   L   () nil nil "Kui")
            ("ukk" nil  nil   nil I   L   () "Austroasiatic" nil "Muak Sa-aak")
            ("ukl" nil  nil   nil I   L   () nil nil "Ukrainian Sign Language")
            ("ukp" nil  nil   nil I   L   () nil nil "Ukpe-Bayobiri")
            ("ukq" nil  nil   nil I   L   () nil nil "Ukwa")
            ("ukr" "uk" "ukr" nil I   L   () "Indo-European" "українська" "Ukrainian")
            ("uks" nil  nil   nil I   L   () nil nil "Urubú-Kaapor Sign Language")
            ("uku" nil  nil   nil I   L   () nil nil "Ukue")
            ("ukv" nil  nil   nil I   L   () nil nil "Kuku")
            ("ukw" nil  nil   nil I   L   () nil nil "Ukwuani-Aboh-Ndoni")
            ("uky" nil  nil   nil I   E   () nil nil "Kuuk-Yak")
            ("ula" nil  nil   nil I   L   () nil nil "Fungwa")
            ("ulb" nil  nil   nil I   L   () nil nil "Ulukwumi")
            ("ulc" nil  nil   nil I   L   () nil nil "Ulch")
            ("ule" nil  nil   nil I   E   () nil nil "Lule")
            ("ulf" nil  nil   nil I   L   () nil nil "Usku")
            ("uli" nil  nil   nil I   L   () nil nil "Ulithian")
            ("ulk" nil  nil   nil I   L   () nil nil "Meriam")
            ("ull" nil  nil   nil I   L   () nil nil "Ullatan")
            ("ulm" nil  nil   nil I   L   () nil nil "Ulumanda'")
            ("uln" nil  nil   nil I   L   () nil nil "Unserdeutsch")
            ("ulu" nil  nil   nil I   L   () nil nil "Uma' Lung")
            ("ulw" nil  nil   nil I   L   () nil nil "Ulwa")
            ("uly" nil  nil   nil I   L   () "Afro-Asiatic" nil "Buli")
            ("uma" nil  nil   nil I   L   () nil nil "Umatilla")
            ("umb" nil  "umb" nil I   L   () nil nil "Umbundu")
            ("umc" nil  nil   nil I   A   () nil nil "Marrucinian")
            ("umd" nil  nil   nil I   E   () nil nil "Umbindhamu")
            ("umg" nil  nil   nil I   E   () nil nil "Umbuygamu")
            ("umi" nil  nil   nil I   L   () nil nil "Ukit")
            ("umm" nil  nil   nil I   L   () nil nil "Umon")
            ("umn" nil  nil   nil I   L   () nil nil "Makyan Naga")
            ("umo" nil  nil   nil I   E   () nil nil "Umotína")
            ("ump" nil  nil   nil I   L   () nil nil "Umpila")
            ("umr" nil  nil   nil I   E   () nil nil "Umbugarla")
            ("ums" nil  nil   nil I   L   () nil nil "Pendau")
            ("umu" nil  nil   nil I   L   () nil nil "Munsee")
            ("una" nil  nil   nil I   L   () nil nil "Watut, North")
            ("und" nil  "und" nil S   S   () nil nil "Undetermined")
            ("une" nil  nil   nil I   L   () nil nil "Uneme")
            ("ung" nil  nil   nil I   L   () nil nil "Ngarinyin")
            ("uni" nil  nil   nil I   L   () "Skou" nil "Uni")
            ("unk" nil  nil   nil I   L   () nil nil "Enawené-Nawé")
            ("unm" nil  nil   nil I   E   () nil nil "Unami")
            ("unn" nil  nil   nil I   L   () nil nil "Kurnai")
            ("unp" nil  nil   t   I   L   () nil nil "Worora")
            ("unr" nil  nil   nil I   L   () nil nil "Mundari")
            ("unu" nil  nil   nil I   L   () nil nil "Unubahe")
            ("unx" nil  nil   nil I   L   () nil nil "Munda")
            ("unz" nil  nil   nil I   L   () nil nil "Kaili, Unde")
            ("uok" nil  nil   t   I   L   () nil nil "Uokha")
            ("uon" nil  nil   nil I   E   () "Austronesian" nil "Kulon")
            ("upi" nil  nil   nil I   L   () nil nil "Umeda")
            ("upv" nil  nil   nil I   L   () nil nil "Uripiv-Wala-Rano-Atchin")
            ("ura" nil  nil   nil I   L   () nil nil "Urarina")
            ("urb" nil  nil   nil I   L   () nil nil "Urubú-Kaapor")
            ("urc" nil  nil   nil I   E   () nil nil "Urningangg")
            ("urd" "ur" "urd" nil I   L   () "Indo-European" "اردو" "Urdu")
            ("ure" nil  nil   nil I   L   () nil nil "Uru")
            ("urf" nil  nil   nil I   E   () nil nil "Uradhi")
            ("urg" nil  nil   nil I   L   () nil nil "Urigina")
            ("urh" nil  nil   nil I   L   () nil nil "Urhobo")
            ("uri" nil  nil   nil I   L   () nil nil "Urim")
            ("urk" nil  nil   nil I   L   () nil nil "Urak Lawoi'")
            ("url" nil  nil   nil I   L   () nil nil "Urali")
            ("urm" nil  nil   nil I   L   () nil nil "Urapmin")
            ("urn" nil  nil   nil I   L   () nil nil "Uruangnirin")
            ("uro" nil  nil   nil I   L   () nil nil "Ura (Papua New Guinea)")
            ("urp" nil  nil   nil I   L   () nil nil "Uru-Pa-In")
            ("urr" nil  nil   nil I   L   () nil nil "Lehalurup")
            ("urt" nil  nil   nil I   L   () nil nil "Urat")
            ("uru" nil  nil   nil I   E   () nil nil "Urumi")
            ("urv" nil  nil   nil I   E   () nil nil "Uruava")
            ("urw" nil  nil   nil I   L   () nil nil "Sop")
            ("urx" nil  nil   nil I   L   () nil nil "Urimo")
            ("ury" nil  nil   nil I   L   () nil nil "Orya")
            ("urz" nil  nil   nil I   L   () nil nil "Uru-Eu-Wau-Wau")
            ("usa" nil  nil   nil I   L   () nil nil "Usarufa")
            ("ush" nil  nil   nil I   L   () nil nil "Ushojo")
            ("usi" nil  nil   nil I   L   () nil nil "Usui")
            ("usk" nil  nil   nil I   L   () nil nil "Usaghade")
            ("usp" nil  nil   nil I   L   () nil nil "Uspanteco")
            ("uss" nil  nil   nil I   L   () "Niger–Congo" nil "us-Saare")
            ("usu" nil  nil   nil I   L   () nil nil "Uya")
            ("uta" nil  nil   nil I   L   () nil nil "Otank")
            ("ute" nil  nil   nil I   L   () nil nil "Ute-Southern Paiute")
            ("uth" nil  nil   nil I   L   () "Niger–Congo" nil "ut-Hun")
            ("utp" nil  nil   nil I   L   () nil nil "Amba (Solomon Islands)")
            ("utr" nil  nil   nil I   L   () nil nil "Etulo")
            ("utu" nil  nil   nil I   L   () nil nil "Utu")
            ("uum" nil  nil   nil I   L   () nil "Урум" "Urum")
            ("uun" nil  nil   t   I   L   () nil nil "Kulon-Pazeh")
            ("uur" nil  nil   nil I   L   () nil nil "Ura (Vanuatu)")
            ("uuu" nil  nil   nil I   L   () nil nil "U")
            ("uve" nil  nil   nil I   L   () nil nil "Uvean, West")
            ("uvh" nil  nil   nil I   L   () nil nil "Uri")
            ("uvl" nil  nil   nil I   L   () nil nil "Lote")
            ("uwa" nil  nil   nil I   L   () nil nil "Kuku-Uwanh")
            ("uya" nil  nil   nil I   L   () nil nil "Doko-Uyanga")
            ("uzb" "uz" "uzb" nil M   L   ("uzn" "uzs") "Turkic" "Ўзбек" "Uzbek")
            ("uzn" nil  nil   nil I   L   () nil nil "Uzbek, Northern")
            ("uzs" nil  nil   nil I   L   () nil nil "Uzbek, Southern")
            ;; V
            ("vaa" nil  nil   nil I   L   () nil nil "Vaagri Booli")
            ("vae" nil  nil   nil I   L   () nil nil "Vale")
            ("vaf" nil  nil   nil I   L   () nil nil "Vafsi")
            ("vag" nil  nil   nil I   L   () nil nil "Vagla")
            ("vah" nil  nil   nil I   L   () nil nil "Varhadi-Nagpuri")
            ("vai" nil  "vai" nil I   L   () nil nil "Vai")
            ("vaj" nil  nil   nil I   L   () nil nil "Vasekela Bushman")
            ("val" nil  nil   nil I   L   () nil nil "Vehes")
            ("vam" nil  nil   nil I   L   () nil nil "Vanimo")
            ("van" nil  nil   nil I   L   () nil nil "Valman")
            ("vao" nil  nil   nil I   L   () nil nil "Vao")
            ("vap" nil  nil   nil I   L   () nil nil "Vaiphei")
            ("var" nil  nil   nil I   L   () nil nil "Huarijio")
            ("vas" nil  nil   nil I   L   () nil nil "Vasavi")
            ("vau" nil  nil   nil I   L   () nil nil "Vanuma")
            ("vav" nil  nil   nil I   L   () nil nil "Varli")
            ("vay" nil  nil   nil I   L   () nil nil "Wayu")
            ("vbb" nil  nil   nil I   L   () nil nil "Babar, Southeast")
            ("vbk" nil  nil   nil I   L   () nil nil "Southwestern Bontok")
            ("vec" nil  nil   nil I   L   () nil "veneto" "Venetian")
            ("ved" nil  nil   nil I   L   () nil nil "Veddah")
            ("vel" nil  nil   nil I   L   () nil nil "Veluws")
            ("vem" nil  nil   nil I   L   () nil nil "Vemgo-Mabas")
            ("ven" "ve" "ven" nil I   L   () "Niger–Congo" "Tshivenḓa" "Venda")
            ("veo" nil  nil   nil I   E   () nil nil "Ventureño")
            ("vep" nil  nil   nil I   L   () nil "vepsä" "Veps")
            ("ver" nil  nil   nil I   L   () nil nil "Mom Jango")
            ("vgr" nil  nil   nil I   L   () nil nil "Vaghri")
            ("vgt" nil  nil   nil I   L   () nil nil "Vlaamse Gebarentaal")
            ("vic" nil  nil   nil I   L   () nil nil "Virgin Islands Creole English")
            ("vid" nil  nil   nil I   L   () nil nil "Vidunda")
            ("vie" "vi" "vie" nil I   L   () "Austroasiatic" "tiếng Việt" "Vietnamese")
            ("vif" nil  nil   nil I   L   () nil nil "Vili")
            ("vig" nil  nil   nil I   L   () nil nil "Viemo")
            ("vil" nil  nil   nil I   L   () nil nil "Vilela")
            ("vin" nil  nil   nil I   L   () nil nil "Vinza")
            ("vis" nil  nil   nil I   L   () nil nil "Vishavan")
            ("vit" nil  nil   nil I   L   () nil nil "Viti")
            ("viv" nil  nil   nil I   L   () nil nil "Iduna")
            ("vjk" nil  nil   nil I   L   () "Indo-European" nil "Bajjika")
            ("vka" nil  nil   nil I   E   () nil nil "Kariyarra")
            ("vki" nil  nil   t   I   L   () nil nil "Ija-Zuba")
            ("vkj" nil  nil   nil I   L   () nil nil "Kujarge")
            ("vkk" nil  nil   nil I   L   () nil nil "Kaur")
            ("vkl" nil  nil   nil I   L   () nil nil "Kulisusu")
            ("vkm" nil  nil   nil I   E   () nil nil "Kamakan")
            ("vkn" nil  nil   nil I   L   () "Niger–Congo" nil "Koro Nulu")
            ("vko" nil  nil   nil I   L   () nil nil "Kodeoha")
            ("vkp" nil  nil   nil I   L   () nil nil "Korlai Creole Portuguese")
            ("vkt" nil  nil   nil I   L   () nil nil "Malay, Tenggarong Kutai")
            ("vku" nil  nil   nil I   L   () nil nil "Kurrama")
            ("vky" nil  nil   nil I   L   () nil nil "Kayu Agung")
            ("vkz" nil  nil   nil I   L   () "Niger–Congo" nil "Koro Zuba")
            ("vlp" nil  nil   nil I   L   () nil nil "Valpei")
            ("vlr" nil  nil   t   I   L   () nil nil "Vatrata")
            ("vls" nil  nil   nil I   L   () nil "West-Vlams" "West Flemish")
            ("vma" nil  nil   nil I   L   () nil nil "Martuyhunira")
            ("vmb" nil  nil   nil I   E   () nil nil "Mbabaram")
            ("vmc" nil  nil   nil I   L   () nil nil "Mixtec, Juxtlahuaca")
            ("vmd" nil  nil   nil I   L   () nil nil "Koraga, Mudu")
            ("vme" nil  nil   nil I   L   () nil nil "Masela, East")
            ("vmf" nil  nil   nil I   L   () nil nil "Mainfränkisch")
            ("vmg" nil  nil   nil I   L   () nil nil "Minigir")
            ("vmh" nil  nil   nil I   L   () nil nil "Maraghei")
            ("vmi" nil  nil   nil I   E   () nil nil "Miwa")
            ("vmj" nil  nil   nil I   L   () nil nil "Mixtec, Ixtayutla")
            ("vmk" nil  nil   nil I   L   () nil nil "Makhuwa-Shirima")
            ("vml" nil  nil   nil I   E   () nil nil "Malgana")
            ("vmm" nil  nil   nil I   L   () nil nil "Mixtec, Mitlatongo")
            ("vmo" nil  nil   t   nil nil () nil nil "Muko-Muko")
            ("vmp" nil  nil   nil I   L   () nil nil "Mazatec, Soyaltepec")
            ("vmq" nil  nil   nil I   L   () nil nil "Mixtec, Soyaltepec")
            ("vmr" nil  nil   nil I   L   () nil nil "Marenje")
            ("vms" nil  nil   nil I   E   () nil nil "Moksela")
            ("vmu" nil  nil   nil I   E   () nil nil "Muluridyi")
            ("vmv" nil  nil   nil I   E   () nil nil "Maidu, Valley")
            ("vmw" nil  nil   nil I   L   () nil nil "Makhuwa")
            ("vmx" nil  nil   nil I   L   () nil nil "Mixtec, Tamazola")
            ("vmy" nil  nil   nil I   L   () nil nil "Mazatec, Ayautla")
            ("vmz" nil  nil   nil I   L   () nil nil "Mazatec, Mazatlán")
            ("vnk" nil  nil   nil I   L   () nil nil "Vano")
            ("vnm" nil  nil   nil I   L   () nil nil "Vinmavis")
            ("vnp" nil  nil   nil I   L   () nil nil "Vunapu")
            ("vol" "vo" "vol" nil I   C   () "constructed" "volapük" "Volapük")
            ("vor" nil  nil   nil I   L   () nil "Voro" "Voro")
            ("vot" nil  "vot" nil I   L   () nil "vaďďa" "Votic")
            ("vra" nil  nil   nil I   L   () nil nil "Vera'a")
            ("vro" nil  nil   nil I   L   () nil "võro" "Võro")
            ("vrs" nil  nil   nil I   L   () nil nil "Varisi")
            ("vrt" nil  nil   nil I   L   () nil nil "Burmbar")
            ("vsi" nil  nil   nil I   L   () nil nil "Moldova Sign Language")
            ("vsl" nil  nil   nil I   L   () nil nil "Venezuelan Sign Language")
            ("vsv" nil  nil   nil I   L   () nil nil "Valencian Sign Language")
            ("vto" nil  nil   nil I   L   () nil nil "Vitou")
            ("vum" nil  nil   nil I   L   () nil nil "Vumbu")
            ("vun" nil  nil   nil I   L   () nil nil "Vunjo")
            ("vut" nil  nil   nil I   L   () nil nil "Vute")
            ("vwa" nil  nil   nil I   L   () nil nil "Awa (China)")
            ;; W
            ("waa" nil  nil   nil I   L   () nil nil "Walla Walla")
            ("wab" nil  nil   nil I   L   () nil nil "Wab")
            ("wac" nil  nil   nil I   E   () nil nil "Wasco-Wishram")
            ("wad" nil  nil   nil I   L   () nil nil "Wandamen")
            ("wae" nil  nil   nil I   L   () nil nil "Walser")
            ("waf" nil  nil   nil I   E   () nil nil "Wakoná")
            ("wag" nil  nil   nil I   L   () nil nil "Wa'ema")
            ("wah" nil  nil   nil I   L   () nil nil "Watubela")
            ("wai" nil  nil   nil I   L   () nil nil "Wares")
            ("waj" nil  nil   nil I   L   () nil nil "Waffa")
            (nil   nil  "wak" nil C   nil () nil nil "Wakashan languages")
            ("wal" nil  "wal" nil I   L   () nil nil "Wolaitta")
            ("wam" nil  nil   nil I   E   () nil nil "Wampanoag")
            ("wan" nil  nil   nil I   L   () nil nil "Wan")
            ("wao" nil  nil   nil I   E   () nil nil "Wappo")
            ("wap" nil  nil   nil I   L   () nil nil "Wapishana")
            ("waq" nil  nil   nil I   L   () nil nil "Wageman")
            ("war" nil  "war" nil I   L   () nil "Winaray" "Waray (Philippines)")
            ("was" nil  "was" nil I   L   () nil nil "Washo")
            ("wat" nil  nil   nil I   L   () nil nil "Kaninuwa")
            ("wau" nil  nil   nil I   L   () nil nil "Waurá")
            ("wav" nil  nil   nil I   L   () nil nil "Waka")
            ("waw" nil  nil   nil I   L   () nil nil "Waiwai")
            ("wax" nil  nil   nil I   L   () nil nil "Watam")
            ("way" nil  nil   nil I   L   () nil nil "Wayana")
            ("waz" nil  nil   nil I   L   () nil nil "Wampur")
            ("wba" nil  nil   nil I   L   () nil nil "Warao")
            ("wbb" nil  nil   nil I   L   () nil nil "Wabo")
            ("wbe" nil  nil   nil I   L   () nil nil "Waritai")
            ("wbf" nil  nil   nil I   L   () nil nil "Wara")
            ("wbh" nil  nil   nil I   L   () nil nil "Wanda")
            ("wbi" nil  nil   nil I   L   () nil nil "Wanji")
            ("wbj" nil  nil   nil I   L   () nil nil "Alagwa")
            ("wbk" nil  nil   nil I   L   () nil nil "Waigali")
            ("wbl" nil  nil   nil I   L   () nil nil "Wakhi")
            ("wbm" nil  nil   nil I   L   () nil nil "Wa")
            ("wbp" nil  nil   nil I   L   () nil nil "Warlpiri")
            ("wbq" nil  nil   nil I   L   () nil nil "Waddar")
            ("wbr" nil  nil   nil I   L   () nil nil "Wagdi")
            ("wbs" nil  nil   nil I   L   () nil nil "West Bengal Sign Language")
            ("wbt" nil  nil   nil I   L   () nil nil "Wanman")
            ("wbv" nil  nil   nil I   L   () nil nil "Wajarri")
            ("wbw" nil  nil   nil I   L   () nil nil "Woi")
            ("wca" nil  nil   nil I   L   () nil nil "Yanomámi")
            ("wci" nil  nil   nil I   L   () nil nil "Gbe, Waci")
            ("wdd" nil  nil   nil I   L   () nil nil "Wandji")
            ("wdg" nil  nil   nil I   L   () nil nil "Wadaginam")
            ("wdj" nil  nil   nil I   L   () nil nil "Wadjiginy")
            ("wdk" nil  nil   nil I   E   () nil nil "Wadikali")
            ("wdt" nil  nil   nil I   L   () "Iroquoian" nil "Wendat")
            ("wdu" nil  nil   nil I   E   () nil nil "Wadjigu")
            ("wdy" nil  nil   nil I   E   () nil nil "Wadjabangayi")
            ("wea" nil  nil   nil I   E   () nil nil "Wewaw")
            ("wec" nil  nil   nil I   L   () nil nil "Wè Western")
            ("wed" nil  nil   nil I   L   () nil nil "Wedau")
            ("weg" nil  nil   nil I   L   () nil nil "Wergaia")
            ("weh" nil  nil   nil I   L   () nil nil "Weh")
            ("wei" nil  nil   nil I   L   () nil nil "Were")
            ("wem" nil  nil   nil I   L   () nil nil "Gbe, Weme")
            (nil   nil  "wen" nil C   nil () nil nil "Sorbian languages")
            ("weo" nil  nil   nil I   L   () nil nil "Wemale, North")
            ("wep" nil  nil   nil I   L   () nil nil "Westphalien")
            ("wer" nil  nil   nil I   L   () nil nil "Weri")
            ("wes" nil  nil   nil I   L   () nil nil "Pidgin, Cameroon")
            ("wet" nil  nil   nil I   L   () nil nil "Perai")
            ("weu" nil  nil   nil I   L   () nil nil "Welaung")
            ("wew" nil  nil   nil I   L   () nil nil "Wejewa")
            ("wfg" nil  nil   nil I   L   () nil nil "Yafi")
            ("wga" nil  nil   nil I   E   () nil nil "Wagaya")
            ("wgb" nil  nil   nil I   L   () nil nil "Wagawaga")
            ("wgg" nil  nil   nil I   E   () nil nil "Wangganguru")
            ("wgi" nil  nil   nil I   L   () nil nil "Wahgi")
            ("wgo" nil  nil   nil I   L   () nil nil "Waigeo")
            ("wgu" nil  nil   nil I   E   () nil nil "Wirangu")
            ("wgw" nil  nil   t   I   L   () nil nil "Wagawaga")
            ("wgy" nil  nil   nil I   L   () nil nil "Warrgamay")
            ("wha" nil  nil   nil I   L   () nil nil "Manusela")
            ("whg" nil  nil   nil I   L   () nil nil "Wahgi, North")
            ("whk" nil  nil   nil I   L   () nil nil "Kenyah, Wahau")
            ("whu" nil  nil   nil I   L   () nil nil "Kayan, Wahau")
            ("wib" nil  nil   nil I   L   () nil nil "Toussian, Southern")
            ("wic" nil  nil   nil I   E   () nil nil "Wichita")
            ("wie" nil  nil   nil I   E   () nil nil "Wik-Epa")
            ("wif" nil  nil   nil I   E   () nil nil "Wik-Keyangan")
            ("wig" nil  nil   nil I   L   () nil nil "Wik-Ngathana")
            ("wih" nil  nil   nil I   L   () nil nil "Wik-Me'anha")
            ("wii" nil  nil   nil I   L   () nil nil "Wiaki")
            ("wij" nil  nil   nil I   L   () nil nil "Wik-Iiyanh")
            ("wik" nil  nil   nil I   L   () nil nil "Wikalkan")
            ("wil" nil  nil   nil I   E   () nil nil "Wilawila")
            ("wim" nil  nil   nil I   L   () nil nil "Wik-Mungkan")
            ("win" nil  nil   nil I   L   () nil nil "Ho-Chunk")
            ("wir" nil  nil   nil I   E   () nil nil "Wiraféd")
            ("wit" nil  nil   t   I   L   () nil nil "Wintu")
            ("wiu" nil  nil   nil I   L   () nil nil "Wiru")
            ("wiv" nil  nil   nil I   L   () nil nil "Muduapa")
            ("wiw" nil  nil   t   I   L   () nil nil "Wirangu")
            ("wiy" nil  nil   nil I   E   () nil "Wiyot" "Wiyot")
            ("wja" nil  nil   nil I   L   () nil nil "Waja")
            ("wji" nil  nil   nil I   L   () nil nil "Warji")
            ("wka" nil  nil   nil I   E   () nil nil "Kw'adza")
            ("wkb" nil  nil   nil I   L   () nil nil "Kumbaran")
            ("wkd" nil  nil   nil I   L   () nil nil "Wakde")
            ("wkl" nil  nil   nil I   L   () nil nil "Kalanadi")
            ("wkr" nil  nil   nil I   L   () "Pama–Nyungan" nil "Keerray-Woorroong")
            ("wku" nil  nil   nil I   L   () nil nil "Kunduvadi")
            ("wkw" nil  nil   nil I   E   () nil nil "Wakawaka")
            ("wky" nil  nil   nil I   E   () nil nil "Wangkayutyuru")
            ("wla" nil  nil   nil I   L   () nil nil "Walio")
            ("wlc" nil  nil   nil I   L   () nil nil "Comorian, Mwali")
            ("wle" nil  nil   nil I   L   () nil nil "Wolane")
            ("wlg" nil  nil   nil I   L   () nil nil "Kunbarlang")
            ("wlh" nil  nil   nil I   L   () nil nil "Welaun")
            ("wli" nil  nil   nil I   L   () nil nil "Waioli")
            ("wlk" nil  nil   nil I   E   () nil nil "Wailaki")
            ("wll" nil  nil   nil I   L   () nil nil "Wali (Sudan)")
            ("wlm" nil  nil   nil I   H   () nil nil "Welsh, Middle")
            ("wln" "wa" "wln" nil I   L   () "Indo-European" "walon" "Walloon")
            ("wlo" nil  nil   nil I   L   () nil nil "Wolio")
            ("wlr" nil  nil   nil I   L   () nil nil "Wailapa")
            ("wls" nil  nil   nil I   L   () nil nil "Wallisian")
            ("wlu" nil  nil   nil I   E   () nil nil "Wuliwuli")
            ("wlv" nil  nil   nil I   L   () nil nil "Wichí Lhamtés Vejoz")
            ("wlw" nil  nil   nil I   L   () nil nil "Walak")
            ("wlx" nil  nil   nil I   L   () nil nil "Wali (Ghana)")
            ("wly" nil  nil   nil I   E   () nil nil "Waling")
            ("wma" nil  nil   nil I   E   () nil nil "Mawa (Nigeria)")
            ("wmb" nil  nil   nil I   L   () nil nil "Wambaya")
            ("wmc" nil  nil   nil I   L   () nil nil "Wamas")
            ("wmd" nil  nil   nil I   L   () nil nil "Mamaindé")
            ("wme" nil  nil   nil I   L   () nil nil "Wambule")
            ("wmg" nil  nil   nil I   L   () "Sino-Tibetan" nil "Minyag, Western")
            ("wmh" nil  nil   nil I   L   () nil nil "Waima'a")
            ("wmi" nil  nil   nil I   E   () nil nil "Wamin")
            ("wmm" nil  nil   nil I   L   () nil nil "Maiwa (Indonesia)")
            ("wmn" nil  nil   nil I   E   () nil nil "Waamwang")
            ("wmo" nil  nil   nil I   L   () nil nil "Wom (Papua New Guinea)")
            ("wms" nil  nil   nil I   L   () nil nil "Wambon")
            ("wmt" nil  nil   nil I   L   () nil nil "Walmajarri")
            ("wmw" nil  nil   nil I   L   () nil nil "Mwani")
            ("wmx" nil  nil   nil I   L   () nil nil "Womo")
            ("wnb" nil  nil   nil I   L   () nil nil "Wanambre")
            ("wnc" nil  nil   nil I   L   () nil nil "Wantoat")
            ("wnd" nil  nil   nil I   E   () nil nil "Wandarang")
            ("wne" nil  nil   nil I   L   () nil nil "Waneci")
            ("wng" nil  nil   nil I   L   () nil nil "Wanggom")
            ("wni" nil  nil   nil I   L   () nil nil "Comorian, Ndzwani")
            ("wnk" nil  nil   nil I   L   () nil nil "Wanukaka")
            ("wnm" nil  nil   nil I   E   () nil nil "Wanggamala")
            ("wnn" nil  nil   nil I   E   () nil nil "Wunumara")
            ("wno" nil  nil   nil I   L   () nil nil "Wano")
            ("wnp" nil  nil   nil I   L   () nil nil "Wanap")
            ("wnu" nil  nil   nil I   L   () nil nil "Usan")
            ("wnw" nil  nil   nil I   L   () nil nil "Wintu")
            ("wny" nil  nil   nil I   L   () nil nil "Wanyi")
            ("woa" nil  nil   nil I   L   () nil nil "Tyaraity")
            ("wob" nil  nil   nil I   L   () nil nil "Wè Northern")
            ("woc" nil  nil   nil I   L   () nil nil "Wogeo")
            ("wod" nil  nil   nil I   L   () nil nil "Wolani")
            ("woe" nil  nil   nil I   L   () nil nil "Woleaian")
            ("wof" nil  nil   nil I   L   () nil nil "Wolof, Gambian")
            ("wog" nil  nil   nil I   L   () nil nil "Wogamusin")
            ("woi" nil  nil   nil I   L   () nil nil "Kamang")
            ("wok" nil  nil   nil I   L   () nil nil "Longto")
            ("wol" "wo" "wol" nil I   L   () "Niger–Congo" "Wolof" "Wolof")
            ("wom" nil  nil   nil I   L   () nil nil "Wom (Nigeria)")
            ("won" nil  nil   nil I   L   () nil nil "Wongo")
            ("woo" nil  nil   nil I   L   () nil nil "Manombai")
            ("wor" nil  nil   nil I   L   () nil nil "Woria")
            ("wos" nil  nil   nil I   L   () nil nil "Hanga Hundi")
            ("wow" nil  nil   nil I   L   () nil nil "Wawonii")
            ("woy" nil  nil   nil I   E   () nil nil "Weyto")
            ("wpc" nil  nil   nil I   L   () nil nil "Maco")
            ("wra" nil  nil   t   I   L   () nil nil "Warapu")
            ("wrb" nil  nil   nil I   E   () nil nil "Warluwara")
            ("wrd" nil  nil   t   I   L   () nil nil "Warduji")
            ("wre" nil  nil   t   nil nil () nil nil "Ware")
            ("wrg" nil  nil   nil I   E   () nil nil "Warungu")
            ("wrh" nil  nil   nil I   E   () nil nil "Wiradhuri")
            ("wri" nil  nil   nil I   E   () nil nil "Wariyangga")
            ("wrk" nil  nil   nil I   L   () nil nil "Garrwa")
            ("wrl" nil  nil   nil I   L   () nil nil "Warlmanpa")
            ("wrm" nil  nil   nil I   L   () nil nil "Warumungu")
            ("wrn" nil  nil   nil I   L   () nil nil "Warnang")
            ("wro" nil  nil   nil I   E   () nil nil "Worrorra")
            ("wrp" nil  nil   nil I   L   () nil nil "Waropen")
            ("wrr" nil  nil   nil I   L   () nil nil "Wardaman")
            ("wrs" nil  nil   nil I   L   () nil nil "Waris")
            ("wru" nil  nil   nil I   L   () nil nil "Waru")
            ("wrv" nil  nil   nil I   L   () nil nil "Waruna")
            ("wrw" nil  nil   nil I   E   () nil nil "Gugu Warra")
            ("wrx" nil  nil   nil I   L   () nil nil "Wae Rana")
            ("wry" nil  nil   nil I   L   () nil nil "Merwari")
            ("wrz" nil  nil   nil I   E   () nil nil "Waray (Australia)")
            ("wsa" nil  nil   nil I   L   () nil nil "Warembori")
            ("wsg" nil  nil   nil I   L   () "Dravidian" nil "Adilabad Gondi")
            ("wsi" nil  nil   nil I   L   () nil nil "Wusi")
            ("wsk" nil  nil   nil I   L   () nil nil "Waskia")
            ("wsr" nil  nil   nil I   L   () nil nil "Owenia")
            ("wss" nil  nil   nil I   L   () nil nil "Wasa")
            ("wsu" nil  nil   nil I   E   () nil nil "Wasu")
            ("wsv" nil  nil   nil I   E   () nil nil "Wotapuri-Katarqalai")
            ("wtb" nil  nil   nil I   L   () "Niger–Congo" nil "Matambwe")
            ("wtf" nil  nil   nil I   L   () nil nil "Dumpu")
            ("wth" nil  nil   nil I   E   () nil nil "Wathawurrung")
            ("wti" nil  nil   nil I   L   () nil nil "Berta")
            ("wtk" nil  nil   nil I   L   () nil nil "Watakataui")
            ("wtm" nil  nil   nil I   L   () nil nil "Mewati")
            ("wtw" nil  nil   nil I   L   () nil nil "Wotu")
            ("wua" nil  nil   nil I   L   () nil nil "Wikngenchera")
            ("wub" nil  nil   nil I   L   () nil nil "Wunambal")
            ("wud" nil  nil   nil I   L   () nil nil "Wudu")
            ("wuh" nil  nil   nil I   L   () nil nil "Wutunhua")
            ("wul" nil  nil   nil I   L   () nil nil "Silimo")
            ("wum" nil  nil   nil I   L   () nil nil "Wumbvu")
            ("wun" nil  nil   nil I   L   () nil nil "Bungu")
            ("wur" nil  nil   nil I   E   () nil nil "Wurrugu")
            ("wut" nil  nil   nil I   L   () nil nil "Wutung")
            ("wuu" nil  nil   nil I   L   () "Chinese" "吴语" "Wu Chinese")
            ("wuv" nil  nil   nil I   L   () nil nil "Wuvulu-Aua")
            ("wux" nil  nil   nil I   L   () nil nil "Wulna")
            ("wuy" nil  nil   nil I   L   () nil nil "Wauyai")
            ("wwa" nil  nil   nil I   L   () nil nil "Waama")
            ("wwb" nil  nil   nil I   E   () nil nil "Wakabunga")
            ("wwo" nil  nil   nil I   L   () nil nil "Wetamut")
            ("wwr" nil  nil   nil I   E   () nil nil "Warrwa")
            ("www" nil  nil   nil I   L   () nil nil "Wawa")
            ("wxa" nil  nil   nil I   L   () nil nil "Waxianghua")
            ("wxw" nil  nil   nil I   E   () nil nil "Wardandi")
            ("wya" nil  nil   t   I   L   () nil nil "Wyandot")
            ("wyb" nil  nil   nil I   L   () nil nil "Wangaaybuwan-Ngiyambaa")
            ("wyi" nil  nil   nil I   E   () nil nil "Woiwurrung")
            ("wym" nil  nil   nil I   L   () nil "Wymysiöeryś" "Wymysorys")
            ("wyn" nil  nil   nil I   L   () "Iroquoian" nil "Wyandot")
            ("wyr" nil  nil   nil I   L   () nil nil "Wayoró")
            ("wyy" nil  nil   nil I   L   () nil nil "Fijian, Western")
            ;; X
            ("xaa" nil  nil   nil I   H   () nil nil "Andalusian Arabic")
            ("xab" nil  nil   nil I   L   () "Indo-European" nil "Sambe")
            ("xac" nil  nil   nil I   L   () nil nil "Kachari")
            ("xad" nil  nil   nil I   E   () nil nil "Adai")
            ("xae" nil  nil   nil I   A   () nil nil "Aequian")
            ("xag" nil  nil   nil I   A   () nil nil "Aghwan")
            ("xah" nil  nil   t   nil nil () nil nil "Kahayan")
            ("xai" nil  nil   nil I   E   () nil nil "Kaimbé")
            ("xaj" nil  nil   nil I   E   () "Tupian" nil "Ararandewára")
            ("xak" nil  nil   nil I   E   () "isolate" nil "Máku")
            ("xal" nil  "xal" nil I   L   () nil "хальмг" "Kalmyk; Oirat")
            ("xam" nil  nil   nil I   E   () nil nil "/Xam")
            ("xan" nil  nil   nil I   L   () nil nil "Xamtanga")
            ("xao" nil  nil   nil I   L   () nil nil "Khao")
            ("xap" nil  nil   nil I   E   () nil nil "Apalachee")
            ("xaq" nil  nil   nil I   A   () nil nil "Aquitanian")
            ("xar" nil  nil   nil I   E   () nil nil "Karami")
            ("xas" nil  nil   nil I   E   () nil nil "Kamas")
            ("xat" nil  nil   nil I   L   () nil nil "Katawixi")
            ("xau" nil  nil   nil I   L   () nil nil "Kauwera")
            ("xav" nil  nil   nil I   L   () nil nil "Xavánte")
            ("xaw" nil  nil   nil I   L   () nil nil "Kawaiisu")
            ("xay" nil  nil   nil I   L   () nil nil "Kayan Mahakam")
            ("xba" nil  nil   t   I   E   () nil nil "Kamba (Brazil)")
            ("xbb" nil  nil   nil I   E   () nil nil "Lower Burdekin")
            ("xbc" nil  nil   nil I   A   () nil nil "Bactrian")
            ("xbd" nil  nil   nil I   E   () nil nil "Bindal")
            ("xbe" nil  nil   nil I   E   () nil nil "Bigambal")
            ("xbg" nil  nil   nil I   E   () nil nil "Bunganditj")
            ("xbi" nil  nil   nil I   L   () nil nil "Kombio")
            ("xbj" nil  nil   nil I   E   () nil nil "Birrpayi")
            ("xbm" nil  nil   nil I   H   () nil nil "Breton, Middle")
            ("xbn" nil  nil   nil I   E   () nil nil "Kenaboi")
            ("xbo" nil  nil   nil I   H   () nil nil "Bolgarian")
            ("xbp" nil  nil   nil I   E   () nil nil "Bibbulman")
            ("xbr" nil  nil   nil I   L   () nil nil "Kambera")
            ("xbw" nil  nil   nil I   E   () nil nil "Kambiwá")
            ("xbx" nil  nil   t   I   E   () nil nil "Kabixí")
            ("xby" nil  nil   nil I   L   () nil nil "Batyala")
            ("xcb" nil  nil   nil I   H   () nil nil "Cumbric")
            ("xcc" nil  nil   nil I   A   () nil nil "Camunic")
            ("xce" nil  nil   nil I   A   () nil nil "Celtiberian")
            ("xcg" nil  nil   nil I   A   () nil nil "Gaulish, Cisalpine")
            ("xch" nil  nil   nil I   E   () nil nil "Chemakum")
            ("xcl" nil  nil   nil I   H   () nil nil "Armenian, Classical")
            ("xcm" nil  nil   nil I   E   () nil nil "Comecrudo")
            ("xcn" nil  nil   nil I   E   () nil nil "Cotoname")
            ("xco" nil  nil   nil I   A   () nil nil "Chorasmian")
            ("xcr" nil  nil   nil I   A   () nil nil "Carian")
            ("xct" nil  nil   nil I   H   () nil nil "Tibetan, Classical")
            ("xcu" nil  nil   nil I   H   () nil nil "Curonian")
            ("xcv" nil  nil   nil I   E   () nil nil "Chuvantsy")
            ("xcw" nil  nil   nil I   E   () nil nil "Coahuilteco")
            ("xcy" nil  nil   nil I   E   () nil nil "Cayuse")
            ("xda" nil  nil   nil I   L   () nil nil "Darkinyung")
            ("xdc" nil  nil   nil I   A   () nil nil "Dacian")
            ("xdk" nil  nil   nil I   E   () nil nil "Dharuk")
            ("xdm" nil  nil   nil I   A   () nil nil "Edomite")
            ("xdo" nil  nil   nil I   L   () "Niger–Congo" nil "Kwandu")
            ("xdq" nil  nil   nil I   L   () "Northeast Caucasian" nil "Kaitag")
            ("xdy" nil  nil   nil I   L   () nil nil "Malayic Dayak")
            ("xeb" nil  nil   nil I   A   () nil nil "Eblan")
            ("xed" nil  nil   nil I   L   () nil nil "Hdi")
            ("xeg" nil  nil   nil I   E   () nil nil "//Xegwi")
            ("xel" nil  nil   nil I   L   () nil nil "Kelo")
            ("xem" nil  nil   nil I   L   () nil nil "Kembayan")
            ("xep" nil  nil   nil I   A   () nil nil "Epi-Olmec")
            ("xer" nil  nil   nil I   L   () nil nil "Xerénte")
            ("xes" nil  nil   nil I   L   () nil nil "Kesawai")
            ("xet" nil  nil   nil I   L   () nil nil "Xetá")
            ("xeu" nil  nil   nil I   L   () nil nil "Keuru")
            ("xfa" nil  nil   nil I   A   () nil nil "Faliscan")
            ("xga" nil  nil   nil I   A   () nil nil "Galatian")
            ("xgb" nil  nil   nil I   E   () nil nil "Gbin")
            ("xgd" nil  nil   nil I   E   () nil nil "Gudang")
            ("xgf" nil  nil   nil I   E   () nil nil "Gabrielino-Fernandeño")
            ("xgg" nil  nil   nil I   E   () nil nil "Goreng")
            ("xgi" nil  nil   nil I   E   () nil nil "Garingbal")
            ("xgl" nil  nil   nil I   H   () nil nil "Galindan")
            ("xgm" nil  nil   nil I   E   () nil nil "Guwinmal")
            ("xgr" nil  nil   nil I   E   () nil nil "Garza")
            ("xgu" nil  nil   nil I   L   () nil nil "Unggumi")
            ("xgw" nil  nil   nil I   E   () nil nil "Guwa")
            ("xha" nil  nil   nil I   A   () nil nil "Harami")
            ("xhc" nil  nil   nil I   A   () nil nil "Hunnic")
            ("xhd" nil  nil   nil I   A   () nil nil "Hadrami")
            ("xhe" nil  nil   nil I   L   () nil nil "Khetrani")
            ("xhm" nil  nil   nil I   H   () "Austroasiatic" nil "Khmer, Middle (1400 to 1850 CE)")
            ("xho" "xh" "xho" nil I   L   () "Niger–Congo" "isiXhosa" "Xhosa")
            ("xhr" nil  nil   nil I   A   () nil nil "Hernican")
            ("xht" nil  nil   nil I   A   () nil nil "Hattic")
            ("xhu" nil  nil   nil I   A   () nil nil "Hurrian")
            ("xhv" nil  nil   nil I   L   () nil nil "Khua")
            ("xia" nil  nil   t   I   L   () nil nil "Xiandao")
            ("xib" nil  nil   nil I   A   () nil nil "Iberian")
            ("xii" nil  nil   nil I   L   () nil nil "Xiri")
            ("xil" nil  nil   nil I   A   () nil nil "Illyrian")
            ("xin" nil  nil   nil I   E   () nil nil "Xinca")
            ("xip" nil  nil   t   I   E   () nil nil "Xipináwa")
            ("xir" nil  nil   nil I   E   () nil nil "Xiriâna")
            ("xis" nil  nil   nil I   L   () "Indo-European" nil "Kisan")
            ("xiv" nil  nil   nil I   A   () nil nil "Indus Valley Language")
            ("xiy" nil  nil   nil I   L   () nil nil "Xipaya")
            ("xjb" nil  nil   nil I   E   () nil nil "Minjungbal")
            ("xjt" nil  nil   nil I   E   () nil nil "Jaitmatang")
            ("xka" nil  nil   nil I   L   () nil nil "Kalkoti")
            ("xkb" nil  nil   nil I   L   () nil nil "Ede Nago, Manigri-Kambolé")
            ("xkc" nil  nil   nil I   L   () nil nil "Kho'ini")
            ("xkd" nil  nil   nil I   L   () nil nil "Kayan, Mendalam")
            ("xke" nil  nil   nil I   L   () nil nil "Kereho-Uheng")
            ("xkf" nil  nil   nil I   L   () nil nil "Khengkha")
            ("xkg" nil  nil   nil I   L   () nil nil "Kagoro")
            ("xkh" nil  nil   t   I   L   () nil nil "Karahawyana")
            ("xki" nil  nil   nil I   L   () nil nil "Kenyan Sign Language")
            ("xkj" nil  nil   nil I   L   () nil nil "Kajali")
            ("xkk" nil  nil   nil I   L   () nil nil "Kaco'")
            ("xkl" nil  nil   nil I   L   () nil nil "Kenyah, Kelinyau")
            ("xkm" nil  nil   t   nil nil () nil nil "Mahakam Kenyah")
            ("xkn" nil  nil   nil I   L   () nil nil "Kayan, Kayan River")
            ("xko" nil  nil   nil I   L   () nil nil "Kiorr")
            ("xkp" nil  nil   nil I   L   () nil nil "Kabatei")
            ("xkq" nil  nil   nil I   L   () nil nil "Koroni")
            ("xkr" nil  nil   nil I   E   () nil nil "Xakriabá")
            ("xks" nil  nil   nil I   L   () nil nil "Kumbewaha")
            ("xkt" nil  nil   nil I   L   () nil nil "Kantosi")
            ("xku" nil  nil   nil I   L   () nil nil "Kaamba")
            ("xkv" nil  nil   nil I   L   () nil nil "Kgalagadi")
            ("xkw" nil  nil   nil I   L   () nil nil "Kembra")
            ("xkx" nil  nil   nil I   L   () nil nil "Karore")
            ("xky" nil  nil   nil I   L   () nil nil "Kenyah, Western")
            ("xkz" nil  nil   nil I   L   () nil nil "Kurtokha")
            ("xla" nil  nil   nil I   L   () nil nil "Kamula")
            ("xlb" nil  nil   nil I   E   () nil nil "Loup B")
            ("xlc" nil  nil   nil I   A   () nil nil "Lycian")
            ("xld" nil  nil   nil I   A   () nil nil "Lydian")
            ("xle" nil  nil   nil I   A   () nil nil "Lemnian")
            ("xlg" nil  nil   nil I   A   () nil nil "Ligurian (Ancient)")
            ("xli" nil  nil   nil I   A   () nil nil "Liburnian")
            ("xln" nil  nil   nil I   A   () nil nil "Alanic")
            ("xlo" nil  nil   nil I   E   () nil nil "Loup A")
            ("xlp" nil  nil   nil I   A   () nil nil "Lepontic")
            ("xls" nil  nil   nil I   A   () nil nil "Lusitanian")
            ("xlu" nil  nil   nil I   A   () nil nil "Luwian, Cuneiform")
            ("xly" nil  nil   nil I   A   () nil nil "Elymian")
            ("xma" nil  nil   nil I   L   () nil nil "Mushungulu")
            ("xmb" nil  nil   nil I   L   () nil nil "Mbonga")
            ("xmc" nil  nil   nil I   L   () nil nil "Makhuwa-Marrevone")
            ("xmd" nil  nil   nil I   L   () nil nil "Mbedam")
            ("xme" nil  nil   nil I   A   () nil nil "Median")
            ("xmf" nil  nil   nil I   L   () nil "მარგალური" "Mingrelian")
            ("xmg" nil  nil   nil I   L   () nil nil "Mengaka")
            ("xmh" nil  nil   nil I   L   () nil nil "Kuku-Muminh")
            ("xmi" nil  nil   t   nil nil () nil nil "Miarrã")
            ("xmj" nil  nil   nil I   L   () nil nil "Majera")
            ("xmk" nil  nil   nil I   A   () nil nil "Macedonian, Ancient")
            ("xml" nil  nil   nil I   L   () nil nil "Malaysian Sign Language")
            ("xmm" nil  nil   nil I   L   () nil nil "Malay, Manado")
            ("xmn" nil  nil   nil I   H   () nil nil "Manichaean Middle Persian")
            ("xmo" nil  nil   nil I   L   () nil nil "Morerebi")
            ("xmp" nil  nil   nil I   E   () nil nil "Kuku-Mu'inh")
            ("xmq" nil  nil   nil I   E   () nil nil "Kuku-Mangk")
            ("xmr" nil  nil   nil I   A   () nil nil "Meroitic")
            ("xms" nil  nil   nil I   L   () nil nil "Moroccan Sign Language")
            ("xmt" nil  nil   nil I   L   () nil nil "Matbat")
            ("xmu" nil  nil   nil I   E   () nil nil "Kamu")
            ("xmv" nil  nil   nil I   L   () nil nil "Malagasy, Antankarana")
            ("xmw" nil  nil   nil I   L   () nil nil "Malagasy, Tsimihety")
            ("xmx" nil  nil   nil I   L   () nil nil "Maden")
            ("xmy" nil  nil   nil I   L   () nil nil "Mayaguduna")
            ("xmz" nil  nil   nil I   L   () nil nil "Mori Bawah")
            ("xna" nil  nil   nil I   A   () nil nil "North Arabian, Ancient")
            ("xnb" nil  nil   nil I   L   () nil nil "Kanakanabu")
            ("xng" nil  nil   nil I   H   () nil nil "Mongolian, Middle")
            ("xnh" nil  nil   nil I   L   () nil nil "Kuanhua")
            ("xni" nil  nil   nil I   E   () nil nil "Ngarigu")
            ("xnj" nil  nil   nil I   L   () "Niger–Congo" nil "Ngoni (Tanzania)")
            ("xnk" nil  nil   nil I   E   () nil nil "Nganakarti")
            ("xnm" nil  nil   nil I   E   () nil nil "Ngumbarl")
            ("xnn" nil  nil   nil I   L   () nil nil "Kankanay, Northern")
            ("xno" nil  nil   nil I   H   () nil nil "Anglo-Norman")
            ("xnq" nil  nil   nil I   L   () "Niger–Congo" nil "Ngoni (Mozambique)")
            ("xnr" nil  nil   nil I   L   () nil nil "Kangri")
            ("xns" nil  nil   nil I   L   () nil nil "Kanashi")
            ("xnt" nil  nil   nil I   E   () nil nil "Narragansett")
            ("xnu" nil  nil   nil I   E   () nil nil "Nukunul")
            ("xny" nil  nil   nil I   L   () nil nil "Nyiyaparli")
            ("xnz" nil  nil   nil I   L   () nil nil "Kenzi")
            ("xoc" nil  nil   nil I   E   () nil nil "O'chi'chi'")
            ("xod" nil  nil   nil I   L   () nil nil "Kokoda")
            ("xog" nil  nil   nil I   L   () nil nil "Soga")
            ("xoi" nil  nil   nil I   L   () nil nil "Kominimung")
            ("xok" nil  nil   nil I   L   () nil nil "Xokleng")
            ("xom" nil  nil   nil I   L   () nil nil "Komo (Sudan)")
            ("xon" nil  nil   nil I   L   () nil nil "Konkomba")
            ("xoo" nil  nil   nil I   E   () nil nil "Xukurú")
            ("xop" nil  nil   nil I   L   () nil nil "Kopar")
            ("xor" nil  nil   nil I   L   () nil nil "Korubo")
            ("xow" nil  nil   nil I   L   () nil nil "Kowaki")
            ("xpa" nil  nil   nil I   E   () nil nil "Pirriya")
            ("xpb" nil  nil   nil I   E   () nil nil "Northeastern Tasmanian, Pyemmairrener")
            ("xpc" nil  nil   nil I   H   () nil nil "Pecheneg")
            ("xpd" nil  nil   nil I   E   () nil nil "Oyster Bay Tasmanian")
            ("xpe" nil  nil   nil I   L   () nil nil "Kpelle, Liberia")
            ("xpf" nil  nil   nil I   E   () nil nil "Nuenonne, Southeast Tasmanian")
            ("xpg" nil  nil   nil I   A   () nil nil "Phrygian")
            ("xph" nil  nil   nil I   E   () nil nil "North Midlands Tasmanian, Tyerrenoterpanner")
            ("xpi" nil  nil   nil I   H   () nil nil "Pictish")
            ("xpj" nil  nil   nil I   E   () nil nil "Mpalitjanh")
            ("xpk" nil  nil   nil I   L   () nil nil "Kulina Pano")
            ("xpl" nil  nil   nil I   E   () nil nil "Port Sorell Tasmanian")
            ("xpm" nil  nil   nil I   E   () nil nil "Pumpokol")
            ("xpn" nil  nil   nil I   E   () nil nil "Kapinawá")
            ("xpo" nil  nil   nil I   E   () nil nil "Pochutec")
            ("xpp" nil  nil   nil I   A   () nil nil "Puyo-Paekche")
            ("xpq" nil  nil   nil I   E   () nil nil "Mohegan-Pequot")
            ("xpr" nil  nil   nil I   A   () nil nil "Parthian")
            ("xps" nil  nil   nil I   A   () nil nil "Pisidian")
            ("xpt" nil  nil   nil I   E   () nil nil "Punthamara")
            ("xpu" nil  nil   nil I   A   () nil nil "Punic")
            ("xpv" nil  nil   nil I   E   () nil nil "Northern Tasmanian, Tommeginne")
            ("xpw" nil  nil   nil I   E   () nil nil "Northwestern Tasmanian, Peerapper")
            ("xpx" nil  nil   nil I   E   () nil nil "Southwestern Tasmanian, Toogee")
            ("xpy" nil  nil   nil I   A   () nil nil "Puyo")
            ("xpz" nil  nil   nil I   E   () nil nil "Bruny Island Tasmanian")
            ("xqa" nil  nil   nil I   H   () nil nil "Karakhanid")
            ("xqt" nil  nil   nil I   A   () nil nil "Qatabanian")
            ("xra" nil  nil   nil I   L   () nil nil "Krahô")
            ("xrb" nil  nil   nil I   L   () nil nil "Karaboro, Eastern")
            ("xrd" nil  nil   nil I   E   () nil nil "Gundungurra")
            ("xre" nil  nil   nil I   L   () nil nil "Kreye")
            ("xrg" nil  nil   nil I   E   () nil nil "Minang")
            ("xri" nil  nil   nil I   L   () nil nil "Krikati-Timbira")
            ("xrm" nil  nil   nil I   A   () nil nil "Armazic")
            ("xrn" nil  nil   nil I   E   () nil nil "Arin")
            ("xrq" nil  nil   t   I   E   () nil nil "Karranga")
            ("xrr" nil  nil   nil I   A   () nil nil "Raetic")
            ("xrt" nil  nil   nil I   E   () nil nil "Aranama-Tamique")
            ("xru" nil  nil   nil I   L   () nil nil "Marriammu")
            ("xrw" nil  nil   nil I   L   () nil nil "Karawa")
            ("xsa" nil  nil   nil I   A   () nil nil "Sabaean")
            ("xsb" nil  nil   nil I   L   () nil nil "Sambal, Tinà")
            ("xsc" nil  nil   nil I   A   () nil nil "Scythian")
            ("xsd" nil  nil   nil I   A   () nil nil "Sidetic")
            ("xse" nil  nil   nil I   L   () nil nil "Sempan")
            ("xsh" nil  nil   nil I   L   () nil nil "Shamang")
            ("xsi" nil  nil   nil I   L   () nil nil "Sio")
            ("xsj" nil  nil   nil I   L   () nil nil "Subi")
            ("xsk" nil  nil   t   nil nil () nil nil "Sakan")
            ("xsl" nil  nil   nil I   L   () nil "ᑌᓀᒐ" "Slavey, South")
            ("xsm" nil  nil   nil I   L   () nil nil "Kasem")
            ("xsn" nil  nil   nil I   L   () nil nil "Sanga (Nigeria)")
            ("xso" nil  nil   nil I   E   () nil nil "Solano")
            ("xsp" nil  nil   nil I   L   () nil nil "Silopi")
            ("xsq" nil  nil   nil I   L   () nil nil "Makhuwa-Saka")
            ("xsr" nil  nil   nil I   L   () nil nil "Sherpa")
            ("xss" nil  nil   t   I   E   () nil nil "Assan")
            ("xst" nil  nil   t   nil nil () nil nil "Silt'e")
            ("xsu" nil  nil   nil I   L   () nil nil "Sanumá")
            ("xsv" nil  nil   nil I   E   () nil "Sūdaviskai" "Sudovian")
            ("xsy" nil  nil   nil I   L   () nil nil "Saisiyat")
            ("xta" nil  nil   nil I   L   () nil nil "Mixtec, Alcozauca")
            ("xtb" nil  nil   nil I   L   () nil nil "Mixtec, Chazumba")
            ("xtc" nil  nil   nil I   L   () nil nil "Katcha-Kadugli-Miri")
            ("xtd" nil  nil   nil I   L   () nil nil "Mixtec, Diuxi-Tilantongo")
            ("xte" nil  nil   nil I   L   () nil nil "Ketengban")
            ("xtg" nil  nil   nil I   A   () nil nil "Gaulish, Transalpine")
            ("xth" nil  nil   nil I   E   () nil nil "Yitha Yitha")
            ("xti" nil  nil   nil I   L   () nil nil "Mixtec, Sinicahua")
            ("xtj" nil  nil   nil I   L   () nil nil "Mixtec, San Juan Teita")
            ("xtl" nil  nil   nil I   L   () nil nil "Mixtec, Tijaltepec")
            ("xtm" nil  nil   nil I   L   () nil nil "Mixtec, Magdalena Peñasco")
            ("xtn" nil  nil   nil I   L   () nil nil "Mixtec, Northern Tlaxiaco")
            ("xto" nil  nil   nil I   A   () nil nil "Tokharian A")
            ("xtp" nil  nil   nil I   L   () nil nil "Mixtec, San Miguel Piedras")
            ("xtq" nil  nil   nil I   H   () nil nil "Tumshuqese")
            ("xtr" nil  nil   nil I   A   () nil nil "Tripuri, Early")
            ("xts" nil  nil   nil I   L   () nil nil "Mixtec, Sindihui")
            ("xtt" nil  nil   nil I   L   () nil nil "Mixtec, Tacahua")
            ("xtu" nil  nil   nil I   L   () nil nil "Mixtec, Cuyamecalco")
            ("xtv" nil  nil   nil I   E   () nil nil "Thawa")
            ("xtw" nil  nil   nil I   L   () nil nil "Tawandê")
            ("xty" nil  nil   nil I   L   () nil nil "Mixtec, Yoloxochitl")
            ("xtz" nil  nil   t   I   E   () nil nil "Tasmanian")
            ("xua" nil  nil   nil I   L   () nil nil "Kurumba, Alu")
            ("xub" nil  nil   nil I   L   () nil nil "Kurumba, Betta")
            ("xud" nil  nil   nil I   E   () nil nil "Umiida")
            ("xuf" nil  nil   t   nil nil () nil nil "Kunfal")
            ("xug" nil  nil   nil I   L   () nil nil "Kunigami")
            ("xuj" nil  nil   nil I   L   () nil nil "Kurumba, Jennu")
            ("xul" nil  nil   nil I   E   () nil nil "Ngunawal")
            ("xum" nil  nil   nil I   A   () nil nil "Umbrian")
            ("xun" nil  nil   nil I   E   () nil nil "Unggarranggu")
            ("xuo" nil  nil   nil I   L   () nil nil "Kuo")
            ("xup" nil  nil   nil I   E   () nil nil "Umpqua, Upper")
            ("xur" nil  nil   nil I   A   () nil nil "Urartian")
            ("xut" nil  nil   nil I   E   () nil nil "Kuthant")
            ("xuu" nil  nil   nil I   L   () nil nil "Kxoe")
            ("xve" nil  nil   nil I   A   () nil nil "Venetic")
            ("xvi" nil  nil   nil I   L   () nil nil "Kamviri")
            ("xvn" nil  nil   nil I   A   () nil nil "Vandalic")
            ("xvo" nil  nil   nil I   A   () nil nil "Volscian")
            ("xvs" nil  nil   nil I   A   () nil nil "Vestinian")
            ("xwa" nil  nil   nil I   L   () nil nil "Kwaza")
            ("xwc" nil  nil   nil I   E   () nil nil "Woccon")
            ("xwd" nil  nil   nil I   E   () nil nil "Wadi Wadi")
            ("xwe" nil  nil   nil I   L   () nil nil "Gbe, Xwela")
            ("xwg" nil  nil   nil I   L   () nil nil "Kwegu")
            ("xwj" nil  nil   nil I   E   () nil nil "Wajuk")
            ("xwk" nil  nil   nil I   E   () nil nil "Wangkumara")
            ("xwl" nil  nil   nil I   L   () nil nil "Gbe, Western Xwla")
            ("xwo" nil  nil   nil I   E   () nil nil "Oirat, Written")
            ("xwr" nil  nil   nil I   L   () nil nil "Kwerba Mamberamo")
            ("xwt" nil  nil   nil I   E   () nil nil "Wotjobaluk")
            ("xww" nil  nil   nil I   E   () nil nil "Wemba Wemba")
            ("xxb" nil  nil   nil I   E   () nil nil "Boro (Ghana)")
            ("xxk" nil  nil   nil I   L   () nil nil "Ke'o")
            ("xxm" nil  nil   nil I   E   () nil nil "Minkin")
            ("xxr" nil  nil   nil I   E   () nil nil "Koropó")
            ("xxt" nil  nil   nil I   E   () nil nil "Tambora")
            ("xya" nil  nil   nil I   E   () nil nil "Yaygir")
            ("xyb" nil  nil   nil I   E   () nil nil "Yandjibara")
            ("xyj" nil  nil   nil I   E   () nil nil "Mayi-Yapi")
            ("xyk" nil  nil   nil I   E   () nil nil "Mayi-Kulan")
            ("xyl" nil  nil   nil I   E   () nil nil "Yalakalore")
            ("xyt" nil  nil   nil I   E   () nil nil "Mayi-Thakurti")
            ("xyy" nil  nil   nil I   L   () nil nil "Yorta Yorta")
            ("xzh" nil  nil   nil I   A   () nil nil "Zhang-Zhung")
            ("xzm" nil  nil   nil I   E   () nil nil "Zemgalian")
            ("xzp" nil  nil   nil I   H   () nil nil "Ancient Zapotec")
            ;; Y
            ("yaa" nil  nil   nil I   L   () nil nil "Yaminahua")
            ("yab" nil  nil   nil I   L   () nil nil "Yuhup")
            ("yac" nil  nil   nil I   L   () nil nil "Yali, Pass Valley")
            ("yad" nil  nil   nil I   L   () nil nil "Yagua")
            ("yae" nil  nil   nil I   L   () nil nil "Pumé")
            ("yaf" nil  nil   nil I   L   () nil nil "Yaka (Democratic Republic of Congo)")
            ("yag" nil  nil   nil I   L   () nil nil "Yámana")
            ("yah" nil  nil   nil I   L   () nil nil "Yazgulyam")
            ("yai" nil  nil   nil I   L   () nil nil "Yagnobi")
            ("yaj" nil  nil   nil I   L   () nil nil "Banda-Yangere")
            ("yak" nil  nil   nil I   L   () nil nil "Yakima")
            ("yal" nil  nil   nil I   L   () nil nil "Yalunka")
            ("yam" nil  nil   nil I   L   () nil nil "Yamba")
            ("yan" nil  nil   nil I   L   () nil nil "Mayangna")
            ("yao" nil  "yao" nil I   L   () nil nil "Yao")
            ("yap" nil  "yap" nil I   L   () nil nil "Yapese")
            ("yaq" nil  nil   nil I   L   () nil nil "Yaqui")
            ("yar" nil  nil   nil I   L   () nil nil "Yabarana")
            ("yas" nil  nil   nil I   L   () nil nil "Nugunu (Cameroon)")
            ("yat" nil  nil   nil I   L   () nil nil "Yambeta")
            ("yau" nil  nil   nil I   L   () nil nil "Yuwana")
            ("yav" nil  nil   nil I   L   () nil nil "Yangben")
            ("yaw" nil  nil   nil I   L   () nil nil "Yawalapití")
            ("yax" nil  nil   nil I   L   () nil nil "Yauma")
            ("yay" nil  nil   nil I   L   () nil nil "Agwagwune")
            ("yaz" nil  nil   nil I   L   () nil nil "Lokaa")
            ("yba" nil  nil   nil I   L   () nil nil "Yala")
            ("ybb" nil  nil   nil I   L   () nil nil "Yemba")
            ("ybd" nil  nil   t   I   L   () nil nil "Yangbye")
            ("ybe" nil  nil   nil I   L   () nil nil "Yugur, West")
            ("ybh" nil  nil   nil I   L   () nil nil "Yakha")
            ("ybi" nil  nil   nil I   L   () nil nil "Yamphu")
            ("ybj" nil  nil   nil I   L   () nil nil "Hasha")
            ("ybk" nil  nil   nil I   L   () nil nil "Bokha")
            ("ybl" nil  nil   nil I   L   () nil nil "Yukuben")
            ("ybm" nil  nil   nil I   L   () nil nil "Yaben")
            ("ybn" nil  nil   nil I   E   () nil nil "Yabaâna")
            ("ybo" nil  nil   nil I   L   () nil nil "Yabong")
            ("ybx" nil  nil   nil I   L   () nil nil "Yawiyo")
            ("yby" nil  nil   nil I   L   () nil nil "Yaweyuha")
            ("ych" nil  nil   nil I   L   () nil nil "Chesu")
            ("ycl" nil  nil   nil I   L   () nil nil "Yi, Central (Lolopo)")
            ("ycn" nil  nil   nil I   L   () nil nil "Yucuna")
            ("ycp" nil  nil   nil I   L   () nil nil "Chepya")
            ("ycr" nil  nil   nil I   L   () nil nil "Yilan Creole")
            ("yda" nil  nil   nil I   E   () nil nil "Yanda")
            ("ydd" nil  nil   nil I   L   () nil nil "Yiddish, Eastern")
            ("yde" nil  nil   nil I   L   () nil nil "Yangum Dey")
            ("ydg" nil  nil   nil I   L   () nil nil "Yidgha")
            ("ydk" nil  nil   nil I   L   () nil nil "Yoidik")
            ("yds" nil  nil   t   I   L   () nil nil "Yiddish Sign Language")
            ("yea" nil  nil   nil I   L   () nil nil "Ravula")
            ("yec" nil  nil   nil I   L   () nil nil "Yeniche")
            ("yee" nil  nil   nil I   L   () nil nil "Yimas")
            ("yei" nil  nil   nil I   E   () nil nil "Yeni")
            ("yej" nil  nil   nil I   L   () nil nil "Yevanic")
            ("yel" nil  nil   nil I   L   () nil nil "Yela")
            ("yen" nil  nil   t   I   L   () nil nil "Yendang")
            ("yer" nil  nil   nil I   L   () nil nil "Tarok")
            ("yes" nil  nil   nil I   L   () nil nil "Yeskwa")
            ("yet" nil  nil   nil I   L   () nil nil "Yetfa")
            ("yeu" nil  nil   nil I   L   () nil nil "Yerukula")
            ("yev" nil  nil   nil I   L   () nil nil "Yapunda")
            ("yey" nil  nil   nil I   L   () nil nil "Yeyi")
            ("yga" nil  nil   nil I   E   () nil nil "Malyangapa")
            ("ygi" nil  nil   nil I   E   () nil nil "Yiningayi")
            ("ygl" nil  nil   nil I   L   () nil nil "Yangum Gel")
            ("ygm" nil  nil   nil I   L   () nil nil "Yagomi")
            ("ygp" nil  nil   nil I   L   () nil nil "Gepo")
            ("ygr" nil  nil   nil I   L   () nil nil "Yagaria")
            ("ygs" nil  nil   nil I   L   () nil nil "Yolŋu Sign Language")
            ("ygu" nil  nil   nil I   L   () nil nil "Yugul")
            ("ygw" nil  nil   nil I   L   () nil nil "Yagwoia")
            ("yha" nil  nil   nil I   L   () nil nil "Baha Buyang")
            ("yhd" nil  nil   nil I   L   () "Arabic" nil "Arabic, Judeo-Iraqi")
            ("yhl" nil  nil   nil I   L   () nil nil "Hlepho Phowa")
            ("yhs" nil  nil   nil I   L   () nil nil "Yan-nhaŋu Sign Language")
            ("yia" nil  nil   nil I   L   () nil nil "Yinggarda")
            ("yib" nil  nil   t   nil nil () nil nil "Yinglish")
            ("yid" "yi" "yid" nil M   L   ("ydd" "yih") "Indo-European" "אידיש" "Yiddish")
            ("yif" nil  nil   nil I   L   () nil nil "Yi, Ache")
            ("yig" nil  nil   nil I   L   () nil nil "Yi, Wusa (Wusa Nasu)")
            ("yih" nil  nil   nil I   E   () nil nil "Yiddish, Western")
            ("yii" nil  nil   nil I   L   () nil nil "Yidiny")
            ("yij" nil  nil   nil I   L   () nil nil "Yindjibarndi")
            ("yik" nil  nil   nil I   L   () nil nil "Yi, Dongshanba Lalo")
            ("yil" nil  nil   nil I   E   () nil nil "Yindjilandji")
            ("yim" nil  nil   nil I   L   () nil nil "Naga, Yimchungru")
            ("yin" nil  nil   nil I   L   () nil nil "Yinchia")
            ("yio" nil  nil   t   nil nil () nil nil "Dayao Yi")
            ("yip" nil  nil   nil I   L   () nil nil "Yi, Poluo (Pholo)")
            ("yiq" nil  nil   nil I   L   () nil nil "Yi, Miqie")
            ("yir" nil  nil   nil I   L   () nil nil "Awyu, North")
            ("yis" nil  nil   nil I   L   () nil nil "Yis")
            ("yit" nil  nil   nil I   L   () nil nil "Yi, Eastern Lalu")
            ("yiu" nil  nil   nil I   L   () nil nil "Yi, Awu")
            ("yiv" nil  nil   nil I   L   () nil nil "Yi, Eshan-Xinping (Northern Nisu)")
            ("yix" nil  nil   nil I   L   () nil nil "Yi, Axi")
            ("yiy" nil  nil   t   I   L   () nil nil "Yir Yoront")
            ("yiz" nil  nil   nil I   L   () nil nil "Yi, Azhe")
            ("yka" nil  nil   nil I   L   () nil nil "Yakan")
            ("ykg" nil  nil   nil I   L   () nil nil "Yukaghir, Northern")
            ("ykh" nil  nil   nil I   L   () "Mongolic" nil "Khamnigan Mongol")
            ("yki" nil  nil   nil I   L   () nil nil "Yoke")
            ("ykk" nil  nil   nil I   L   () nil nil "Yakaikeke")
            ("ykl" nil  nil   nil I   L   () nil nil "Khlula")
            ("ykm" nil  nil   nil I   L   () nil nil "Yakamul")
            ("ykn" nil  nil   nil I   L   () nil nil "Kua-nsi")
            ("yko" nil  nil   nil I   L   () nil nil "Yasa")
            ("ykr" nil  nil   nil I   L   () nil nil "Yekora")
            ("ykt" nil  nil   nil I   L   () nil nil "Kathu")
            ("yku" nil  nil   nil I   L   () nil nil "Kuamasi")
            ("yky" nil  nil   nil I   L   () nil nil "Yakoma")
            ("yla" nil  nil   nil I   L   () nil nil "Yaul")
            ("ylb" nil  nil   nil I   L   () nil nil "Yaleba")
            ("yle" nil  nil   nil I   L   () nil nil "Yele")
            ("ylg" nil  nil   nil I   L   () nil nil "Yelogu")
            ("yli" nil  nil   nil I   L   () nil nil "Yali, Angguruk")
            ("yll" nil  nil   nil I   L   () nil nil "Yil")
            ("ylm" nil  nil   nil I   L   () nil nil "Yi, Limi")
            ("yln" nil  nil   nil I   L   () nil nil "Langnian Buyang")
            ("ylo" nil  nil   nil I   L   () nil nil "Yi, Naluo")
            ("ylr" nil  nil   nil I   E   () nil nil "Yalarnnga")
            ("ylu" nil  nil   nil I   L   () nil nil "Aribwaung")
            ("yly" nil  nil   nil I   L   () nil nil "Nyâlayu")
            ("yma" nil  nil   t   I   L   () nil nil "Yamphe")
            ("ymb" nil  nil   nil I   L   () nil nil "Yambes")
            ("ymc" nil  nil   nil I   L   () nil nil "Southern Muji")
            ("ymd" nil  nil   nil I   L   () nil nil "Muda")
            ("yme" nil  nil   nil I   E   () nil nil "Yameo")
            ("ymg" nil  nil   nil I   L   () nil nil "Yamongeri")
            ("ymh" nil  nil   nil I   L   () nil nil "Yi, Mili")
            ("ymi" nil  nil   nil I   L   () nil nil "Moji")
            ("ymj" nil  nil   t   nil nil () nil nil "Muji Yi")
            ("ymk" nil  nil   nil I   L   () nil nil "Makwe")
            ("yml" nil  nil   nil I   L   () nil nil "Iamalele")
            ("ymm" nil  nil   nil I   L   () nil nil "Maay")
            ("ymn" nil  nil   nil I   L   () nil nil "Yamna")
            ("ymo" nil  nil   nil I   L   () nil nil "Yangum Mon")
            ("ymp" nil  nil   nil I   L   () nil nil "Yamap")
            ("ymq" nil  nil   nil I   L   () nil nil "Qila Muji")
            ("ymr" nil  nil   nil I   L   () nil nil "Malasar")
            ("yms" nil  nil   nil I   A   () nil nil "Mysian")
            ("ymt" nil  nil   t   I   E   () nil nil "Mator-Taygi-Karagas")
            ("ymx" nil  nil   nil I   L   () nil nil "Northern Muji")
            ("ymz" nil  nil   nil I   L   () nil nil "Muzi")
            ("yna" nil  nil   nil I   L   () nil nil "Aluo")
            ("ynd" nil  nil   nil I   E   () nil nil "Yandruwandha")
            ("yne" nil  nil   nil I   L   () nil nil "Lang'e")
            ("yng" nil  nil   nil I   L   () nil nil "Yango")
            ("ynh" nil  nil   t   I   L   () nil nil "Yangho")
            ("ynk" nil  nil   nil I   L   () nil nil "Yupik, Naukan")
            ("ynl" nil  nil   nil I   L   () nil nil "Yangulam")
            ("ynn" nil  nil   nil I   E   () nil nil "Yana")
            ("yno" nil  nil   nil I   L   () nil nil "Yong")
            ("ynq" nil  nil   nil I   L   () nil nil "Yendang")
            ("yns" nil  nil   nil I   L   () nil nil "Yansi")
            ("ynu" nil  nil   nil I   E   () nil nil "Yahuna")
            ("yob" nil  nil   nil I   E   () nil nil "Yoba")
            ("yog" nil  nil   nil I   L   () nil nil "Yogad")
            ("yoi" nil  nil   nil I   L   () nil nil "Yonaguni")
            ("yok" nil  nil   nil I   L   () nil nil "Yokuts")
            ("yol" nil  nil   nil I   E   () nil nil "Yola")
            ("yom" nil  nil   nil I   L   () nil nil "Yombe")
            ("yon" nil  nil   nil I   L   () nil nil "Yonggom")
            ("yor" "yo" "yor" nil I   L   () "Niger–Congo" nil "Yoruba")
            ("yos" nil  nil   t   I   L   () nil nil "Yos")
            ("yot" nil  nil   nil I   L   () nil nil "Yotti")
            ("yox" nil  nil   nil I   L   () nil nil "Yoron")
            ("yoy" nil  nil   nil I   L   () nil nil "Yoy")
            ("ypa" nil  nil   nil I   L   () nil nil "Phala")
            ("ypb" nil  nil   nil I   L   () nil nil "Labo Phowa")
            ("ypg" nil  nil   nil I   L   () nil nil "Phola")
            ("yph" nil  nil   nil I   L   () nil nil "Phupha")
            (nil   nil  "ypk" nil C   nil () nil nil "Yupik languages")
            ("ypl" nil  nil   t   nil nil () nil nil "Pula Yi")
            ("ypm" nil  nil   nil I   L   () nil nil "Phuma")
            ("ypn" nil  nil   nil I   L   () nil nil "Ani Phowa")
            ("ypo" nil  nil   nil I   L   () nil nil "Alo Phola")
            ("ypp" nil  nil   nil I   L   () nil nil "Phupa")
            ("ypw" nil  nil   t   nil nil () nil nil "Puwa Yi")
            ("ypz" nil  nil   nil I   L   () nil nil "Phuza")
            ("yra" nil  nil   nil I   L   () nil nil "Yerakai")
            ("yrb" nil  nil   nil I   L   () nil nil "Yareba")
            ("yre" nil  nil   nil I   L   () nil nil "Yaouré")
            ("yri" nil  nil   t   I   L   () nil nil "Yarí")
            ("yrk" nil  nil   nil I   L   () nil nil "Nenets")
            ("yrl" nil  nil   nil I   L   () nil nil "Nhengatu")
            ("yrm" nil  nil   nil I   L   () nil nil "Yirrk-Mel")
            ("yrn" nil  nil   nil I   L   () nil nil "Yerong")
            ("yro" nil  nil   nil I   L   () "Yanomam" nil "Yaroamë")
            ("yrs" nil  nil   nil I   L   () nil nil "Yarsun")
            ("yrw" nil  nil   nil I   L   () nil nil "Yarawata")
            ("yry" nil  nil   nil I   L   () nil nil "Yarluyandi")
            ("ysc" nil  nil   nil I   E   () nil nil "Yassic")
            ("ysd" nil  nil   nil I   L   () nil nil "Samatao")
            ("ysg" nil  nil   nil I   L   () nil nil "Sonaga")
            ("ysl" nil  nil   nil I   L   () nil nil "Yugoslavian Sign Language")
            ("ysm" nil  nil   nil I   L   () nil nil "Myanmar Sign Language")
            ("ysn" nil  nil   nil I   L   () nil nil "Yi, Sani")
            ("yso" nil  nil   nil I   L   () nil nil "Yi, Southeastern Lolo (Nisi)")
            ("ysp" nil  nil   nil I   L   () nil nil "Yi, Southern Lolopho")
            ("ysr" nil  nil   nil I   E   () nil nil "Yupik, Sirenik")
            ("yss" nil  nil   nil I   L   () nil nil "Yessan-Mayo")
            ("ysy" nil  nil   nil I   L   () nil nil "Sanie")
            ("yta" nil  nil   nil I   L   () nil nil "Talu")
            ("ytl" nil  nil   nil I   L   () nil nil "Tanglang")
            ("ytp" nil  nil   nil I   L   () nil nil "Thopho")
            ("ytw" nil  nil   nil I   L   () nil nil "Yout Wam")
            ("yty" nil  nil   nil I   E   () nil nil "Yatay")
            ("yua" nil  nil   nil I   L   () nil nil "Maya, Yucatán")
            ("yub" nil  nil   nil I   E   () nil nil "Yugambal")
            ("yuc" nil  nil   nil I   L   () nil nil "Yuchi")
            ("yud" nil  nil   nil I   L   () nil nil "Arabic, Judeo-Tripolitanian")
            ("yue" nil  nil   nil I   L   () "Chinese" nil "Yue Chinese (Cantonese)")
            ("yuf" nil  nil   nil I   L   () nil nil "Havasupai-Walapai-Yavapai")
            ("yug" nil  nil   nil I   E   () nil nil "Yug")
            ("yui" nil  nil   nil I   L   () nil nil "Yurutí")
            ("yuj" nil  nil   nil I   L   () nil nil "Karkar-Yuri")
            ("yuk" nil  nil   nil I   E   () nil nil "Yuki")
            ("yul" nil  nil   nil I   L   () nil nil "Yulu")
            ("yum" nil  nil   nil I   L   () nil nil "Quechan")
            ("yun" nil  nil   nil I   L   () nil nil "Bena (Nigeria)")
            ("yup" nil  nil   nil I   L   () nil nil "Yukpa")
            ("yuq" nil  nil   nil I   L   () nil nil "Yuqui")
            ("yur" nil  nil   nil I   E   () nil nil "Yurok")
            ("yus" nil  nil   t   I   L   () nil nil "Maya, Chan Santa Cruz")
            ("yut" nil  nil   nil I   L   () nil nil "Yopno")
            ("yuu" nil  nil   t   I   L   () nil nil "Yugh")
            ("yuw" nil  nil   nil I   L   () nil nil "Yau (Morobe Province)")
            ("yux" nil  nil   nil I   L   () nil nil "Yukaghir, Southern")
            ("yuy" nil  nil   nil I   L   () nil nil "Yugur, East")
            ("yuz" nil  nil   nil I   L   () nil nil "Yuracare")
            ("yva" nil  nil   nil I   L   () nil nil "Yawa")
            ("yvt" nil  nil   nil I   E   () nil nil "Yavitero")
            ("ywa" nil  nil   nil I   L   () nil nil "Kalou")
            ("ywg" nil  nil   nil I   L   () nil nil "Yinhawangka")
            ("ywl" nil  nil   nil I   L   () nil nil "Yi, Western Lalu")
            ("ywm" nil  nil   t   nil nil () nil nil "Wumeng Yi")
            ("ywn" nil  nil   nil I   L   () nil nil "Yawanawa")
            ("ywq" nil  nil   nil I   L   () nil nil "Yi, Wuding-Luquan")
            ("ywr" nil  nil   nil I   L   () nil nil "Yawuru")
            ("ywt" nil  nil   nil I   L   () nil nil "Yi, Western (Xishanba Lalo)")
            ("ywu" nil  nil   nil I   L   () nil nil "Yi, Wusa (Wumeng Nasu)")
            ("yww" nil  nil   nil I   E   () nil nil "Yawarawarga")
            ("yxa" nil  nil   nil I   E   () nil nil "Mayawali")
            ("yxg" nil  nil   nil I   E   () nil nil "Yagara")
            ("yxl" nil  nil   nil I   E   () nil nil "Yardliyawarra")
            ("yxm" nil  nil   nil I   E   () nil nil "Yinwum")
            ("yxu" nil  nil   nil I   E   () nil nil "Yuyu")
            ("yxy" nil  nil   nil I   E   () nil nil "Yabula Yabula")
            ("yym" nil  nil   t   nil nil () nil nil "Yuanjiang-Mojiang Yi")
            ("yyr" nil  nil   nil I   E   () nil nil "Yir Yoront")
            ("yyu" nil  nil   nil I   L   () nil nil "Yau (Sandaun Province)")
            ("yyz" nil  nil   nil I   L   () nil nil "Ayizi")
            ("yzg" nil  nil   nil I   L   () nil nil "E'ma Buyang")
            ("yzk" nil  nil   nil I   L   () nil nil "Zokhuo")
            ;; Z
            ("zaa" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Sierra de Juárez")
            ("zab" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, San Juan Guelavía")
            ("zac" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Ocotlán")
            ("zad" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Cajonos")
            ("zae" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Yareni")
            ("zaf" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Ayoquesco")
            ("zag" nil  nil   nil I   L   () "Nilo-Saharan" nil "Zaghawa")
            ("zah" nil  nil   nil I   L   () "Niger–Congo" nil "Zangwal")
            ("zai" nil  nil   nil I   L   () "Zapotec" "Diidxazá'" "Zapotec, Isthmus")
            ("zaj" nil  nil   nil I   L   () "Niger–Congo" nil "Zaramo")
            ("zak" nil  nil   nil I   L   () "Niger–Congo" nil "Zanaki")
            ("zal" nil  nil   nil I   L   () "Sino-Tibetan" nil "Zauzou")
            ("zam" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Miahuatlán")
            ("zao" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Ozolotepec")
            ("zap" nil  "zap" nil M   L   ("zch" "zeh" "zgb"
                                           "zgm" "zgn" "zhd"
                                           "zhn" "zlj" "zln"
                                           "zlq" "zqe" "zyb"
                                           "zyg" "zyj" "zyn"
                                           "zzj" "ccx" "ccy") "Zapotec" nil "Zapotec")
            ("zaq" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Aloápam")
            ("zar" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Rincón")
            ("zas" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Santo Domingo Albarradas")
            ("zat" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Tabaa")
            ("zau" nil  nil   nil I   L   () "Sino-Tibetan" nil "Zangskari")
            ("zav" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Yatzachi")
            ("zaw" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Mitla")
            ("zax" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Xadani")
            ("zay" nil  nil   nil I   L   () "Afro-Asiatic" nil "Zayse-Zergulla")
            ("zaz" nil  nil   nil I   L   () "Afro-Asiatic" nil "Zari")
            ("zba" nil  nil   nil I   C   () "constructed" nil "Balaibalan")
            ("zbc" nil  nil   nil I   L   () "Austronesian" nil "Central Berawan")
            ("zbe" nil  nil   nil I   L   () "Austronesian" nil "East Berawan")
            ("zbl" nil  "zbl" nil I   C   () "constructed" nil "Blissymbols")
            ("zbt" nil  nil   nil I   L   () "Austronesian" nil "Batui")
            ("zbu" nil  nil   nil I   L   () "Afro-Asiatic" nil "Bu (Bauchi State)")
            ("zbw" nil  nil   nil I   L   () "Austronesian" nil "West Berawan")
            ("zca" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Coatecas Altas")
            ("zcd" nil  nil   nil I   L   () "Oto-Manguean" nil "Las Delicias Zapotec")
            ("zch" nil  nil   nil I   L   () "Tai–Kadai" nil "Central Hongshuihe Zhuang")
            ("zdj" nil  nil   nil I   L   () "Niger–Congo" nil "Comorian, Ngazidja")
            ("zea" nil  nil   nil I   L   () "Indo-European" nil "Zeeuws")
            ("zeg" nil  nil   nil I   L   () "Austronesian" nil "Zenag")
            ("zeh" nil  nil   nil I   L   () "Tai–Kadai" nil "Eastern Hongshuihe Zhuang")
            ("zem" nil  nil   nil I   L   () "Afro-Asiatic" nil "Zeem")
            ("zen" nil  "zen" nil I   L   () "Afro-Asiatic" nil "Zenaga")
            ("zga" nil  nil   nil I   L   () "Niger–Congo" nil "Kinga")
            ("zgb" nil  nil   nil I   L   () "Tai–Kadai" nil "Guibei Zhuang")
            ("zgh" nil  "zgh" nil I   L   () "Afro-Asiatic" nil "Standard Moroccan Tamazight")
            ("zgm" nil  nil   nil I   L   () "Tai–Kadai" nil "Minz Zhuang")
            ("zgn" nil  nil   nil I   L   () "Tai–Kadai" nil "Guibian Zhuang")
            ("zgr" nil  nil   nil I   L   () "Austronesian" nil "Magori")
            ("zha" "za" "zha" nil M   L   () "Tai–Kadai" nil "Zhuang")
            ("zhb" nil  nil   nil I   L   () "Sino-Tibetan" nil "Zhaba")
            ("zhd" nil  nil   nil I   L   () "Tai–Kadai" nil "Dai Zhuang")
            ("zhi" nil  nil   nil I   L   () "Niger–Congo" nil "Zhire")
            ("zhn" nil  nil   nil I   L   () "Tai–Kadai" nil "Nong Zhuang")
            ("zho" "zh" "chi" nil M   L   ("cdo"
                                           "cjy" "cmn" "cnp"
                                           "cpx" "csp" "czh"
                                           "czo" "gan" "hak"
                                           "hsn" "lzh" "mnp"
                                           "nan" "wuu" "yue") "Chinese" "中文; 汉语; 华语" "Chinese")
            ("zhw" nil  nil   nil I   L   () "Niger–Congo" nil "Zhoa")
            ("zia" nil  nil   nil I   L   () "Trans–New Guinea" nil "Zia")
            ("zib" nil  nil   nil I   L   () "isolate" nil "Zimbabwe Sign Language")
            ("zik" nil  nil   nil I   L   () "Trans–New Guinea" nil "Zimakani")
            ("zil" nil  nil   nil I   L   () "Niger–Congo" nil "Zialo")
            ("zim" nil  nil   nil I   L   () "Afro-Asiatic" nil "Mesme")
            ("zin" nil  nil   nil I   L   () "Niger–Congo" nil "Zinza")
            ("zir" nil  nil   t   I   E   () "Niger–Congo" nil "Ziriya")
            ("ziw" nil  nil   nil I   L   () "Niger–Congo" nil "Zigula")
            ("ziz" nil  nil   nil I   L   () "Afro-Asiatic" nil "Zizilivakan")
            ("zka" nil  nil   nil I   L   () "Austronesian" nil "Kaimbulawa")
            ("zkb" nil  nil   t   I   E   () "Uralic" nil "Koibal")
            ("zkd" nil  nil   nil I   L   () "Sino-Tibetan" nil "Kadu")
            ("zkg" nil  nil   nil I   A   () "Buyeo" nil "Koguryo")
            ("zkh" nil  nil   nil I   H   () "Turkic" nil "Khorezmian")
            ("zkk" nil  nil   nil I   E   () "unclassified" nil "Karankawa")
            ("zkn" nil  nil   nil I   L   () "Sino-Tibetan" nil "Kanan")
            ("zko" nil  nil   nil I   E   () "Dené–Yeniseian" nil "Kott")
            ("zkp" nil  nil   nil I   E   () "Macro-Jê" nil "Kaingáng, São Paulo")
            ("zkr" nil  nil   nil I   L   () "Sino-Tibetan" nil "Zakhring")
            ("zkt" nil  nil   nil I   H   () "Mongolic" nil "Khitan, Kitan")
            ("zku" nil  nil   nil I   L   () "Pama–Nyungan" nil "Kaurna")
            ("zkv" nil  nil   nil I   E   () "Uralic" nil "Krevinian")
            ("zkz" nil  nil   nil I   H   () "Turkic" nil "Khazar")
            ("zla" nil  nil   nil I   L   () "Niger–Congo" nil "Zula")
            ("zlj" nil  nil   nil I   L   () "Tai–Kadai" nil "Liujiang Zhuang")
            ("zlm" nil  nil   nil I   L   () "Austronesian" nil "Malay (individual language)")
            ("zln" nil  nil   nil I   L   () "Tai–Kadai" nil "Lianshan Zhuang")
            ("zlq" nil  nil   nil I   L   () "Tai–Kadai" nil "Liuqian Zhuang")
            ("zlu" nil  nil   nil I   L   () "Afro-Asiatic" nil "Zul")
            ("zma" nil  nil   nil I   L   () "Western Daly" nil "Manda (Australia)")
            ("zmb" nil  nil   nil I   L   () "Niger–Congo" nil "Zimba")
            ("zmc" nil  nil   nil I   E   () "Pama–Nyungan" nil "Margany")
            ("zmd" nil  nil   nil I   L   () "Western Daly" nil "Maridan")
            ("zme" nil  nil   nil I   E   () "Arnhem Land" nil "Mangerr")
            ("zmf" nil  nil   nil I   L   () "Niger–Congo" nil "Mfinu")
            ("zmg" nil  nil   nil I   L   () "Western Daly" nil "Marti Ke")
            ("zmh" nil  nil   nil I   E   () "Baining" nil "Makolkol")
            ("zmi" nil  nil   nil I   L   () "Austronesian" nil "Negeri Sembilan Malay")
            ("zmj" nil  nil   nil I   L   () "Western Daly" nil "Maridjabin")
            ("zmk" nil  nil   nil I   E   () "Pama–Nyungan" nil "Mandandanyi")
            ("zml" nil  nil   nil I   E   () "Eastern Daly" nil "Madngele")
            ("zmm" nil  nil   nil I   L   () "Western Daly" nil "Marimanindji")
            ("zmn" nil  nil   nil I   L   () "Niger–Congo" nil "Mbangwe")
            ("zmo" nil  nil   nil I   L   () "Nilo-Saharan" nil "Molo")
            ("zmp" nil  nil   nil I   L   () "Niger–Congo" nil "Mpuono")
            ("zmq" nil  nil   nil I   L   () "Niger–Congo" nil "Mituku")
            ("zmr" nil  nil   nil I   L   () "Western Daly" nil "Maranunggu")
            ("zms" nil  nil   nil I   L   () "Niger–Congo" nil "Mbesa")
            ("zmt" nil  nil   nil I   L   () "Western Daly" nil "Maringarr")
            ("zmu" nil  nil   nil I   E   () "Pama–Nyungan" nil "Muruwari")
            ("zmv" nil  nil   nil I   E   () "Pama–Nyungan" nil "Mbariman-Gudhinma")
            ("zmw" nil  nil   nil I   L   () "Niger–Congo" nil "Mbo (Democratic Republic of Congo)")
            ("zmx" nil  nil   nil I   L   () "Niger–Congo" nil "Bomitaba")
            ("zmy" nil  nil   nil I   L   () "Western Daly" nil "Mariyedi")
            ("zmz" nil  nil   nil I   L   () "Ubangian" nil "Mbandja")
            ("zna" nil  nil   nil I   L   () "Niger–Congo" nil "Zan Gula")
            (nil   nil  "znd" nil C   nil () nil nil "Zande languages")
            ("zne" nil  nil   nil I   L   () "Niger–Congo" nil "Zande (specific)")
            ("zng" nil  nil   nil I   L   () "Austroasiatic" nil "Mang")
            ("znk" nil  nil   nil I   E   () "Iwaidjan" nil "Manangkari")
            ("zns" nil  nil   nil I   L   () "Afro-Asiatic" nil "Mangas")
            ("zoc" nil  nil   nil I   L   () "Mixe-Zoquean" nil "Zoque, Copainalá")
            ("zoh" nil  nil   nil I   L   () "Mixe-Zoquean" nil "Zoque, Chimalapa")
            ("zom" nil  nil   nil I   L   () "Sino-Tibetan" nil "Zou")
            ("zoo" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Asunción Mixtepec")
            ("zoq" nil  nil   nil I   L   () "Mixe-Zoquean" nil "Zoque, Tabasco")
            ("zor" nil  nil   nil I   L   () "Mixe-Zoquean" nil "Zoque, Rayón")
            ("zos" nil  nil   nil I   L   () "Mixe-Zoquean" nil "Zoque, Francisco León")
            ("zpa" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Lachiguiri")
            ("zpb" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Yautepec")
            ("zpc" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Choapan")
            ("zpd" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Southeastern Ixtlán")
            ("zpe" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Petapa")
            ("zpf" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, San Pedro Quiatoni")
            ("zpg" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Guevea De Humboldt")
            ("zph" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Totomachapan")
            ("zpi" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Santa María Quiegolani")
            ("zpj" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Quiavicuzas")
            ("zpk" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Tlacolulita")
            ("zpl" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Lachixío")
            ("zpm" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Mixtepec")
            ("zpn" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Santa Inés Yatzechi")
            ("zpo" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Amatlán")
            ("zpp" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, El Alto")
            ("zpq" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Zoogocho")
            ("zpr" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Santiago Xanica")
            ("zps" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Coatlán")
            ("zpt" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, San Vicente Coatlán")
            ("zpu" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Yalálag")
            ("zpv" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Chichicapan")
            ("zpw" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Zaniza")
            ("zpx" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, San Baltazar Loxicha")
            ("zpy" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Mazaltepec")
            ("zpz" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Texmelucan")
            ("zqe" nil  nil   nil I   L   () "Tai–Kadai" nil "Qiubei Zhuang")
            ("zra" nil  nil   nil I   A   () "unclassified" nil "Kara (Korea)")
            ("zrg" nil  nil   nil I   L   () "Indo-European" nil "Mirgan")
            ("zrn" nil  nil   nil I   L   () "Afro-Asiatic" nil "Zirenkel")
            ("zro" nil  nil   nil I   L   () "Zaparoan" nil "Záparo")
            ("zrp" nil  nil   nil I   E   () "Indo-European" nil "Zarphatic")
            ("zrs" nil  nil   nil I   L   () "Mairasi" nil "Mairasi")
            ("zsa" nil  nil   nil I   L   () "Austronesian" nil "Sarasira")
            ("zsk" nil  nil   nil I   A   () "unclassified" nil "Kaskean")
            ("zsl" nil  nil   nil I   L   () "unclassified" nil "Zambian Sign Language")
            ("zsm" nil  nil   nil I   L   () "Austronesian" nil "Standard Malay")
            ("zsr" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Southern Rincon")
            ("zsu" nil  nil   nil I   L   () "Austronesian" nil "Sukurum")
            ("ztc" nil  nil   t   nil nil () "Zapotec" nil "Lachirioag Zapotec")
            ("zte" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Elotepec")
            ("ztg" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Xanaguía")
            ("ztl" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Santiago Lapaguía")
            ("ztm" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, San Agustín Mixtepec")
            ("ztn" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Santa Catarina Albarradas")
            ("ztp" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Loxicha")
            ("ztq" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Quioquitani-Quierí")
            ("zts" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Tilquiapan")
            ("ztt" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Tejalapan")
            ("ztu" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Güilá")
            ("ztx" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Zaachila")
            ("zty" nil  nil   nil I   L   () "Zapotec" nil "Zapotec, Yatee")
            ("zua" nil  nil   t   I   L   () "Afro-Asiatic" nil "Zeem")
            ("zuh" nil  nil   nil I   L   () "Trans–New Guinea" nil "Tokano")
            ("zul" "zu" "zul" nil I   L   () "Niger–Congo" "isiZulu" "Zulu")
            ("zum" nil  nil   nil I   L   () "Indo-European" nil "Kumzari")
            ("zun" nil  "zun" nil I   L   () "isolate" nil "Zuni")
            ("zuy" nil  nil   nil I   L   () "Afro-Asiatic" nil "Zumaya")
            ("zwa" nil  nil   nil I   L   () "Afro-Asiatic" nil "Zay")
            ("zxx" nil  "zxx" nil S   S   () nil nil "No linguistic content; Not applicable")
            ("zyb" nil  nil   nil I   L   () "Tai–Kadai" nil "Yongbei Zhuang")
            ("zyg" nil  nil   nil I   L   () "Tai–Kadai" nil "Yang Zhuang")
            ("zyj" nil  nil   nil I   L   () "Tai–Kadai" nil "Youjiang Zhuang")
            ("zyn" nil  nil   nil I   L   () "Tai–Kadai" nil "Yongnan Zhuang")
            ("zyp" nil  nil   nil I   L   () "Sino-Tibetan" nil "Zyphe")
            ("zza" nil  "zza" nil M   L   ("diq" "kiu") "Indo-European" nil "Zaza; Dimili")
            ("zzj" nil  nil   nil I   L   () "Tai–Kadai" nil "Zuojiang Zhuang"))
  "An list of all ISO 639-1, -2, and -3 codes along with info regarding them.

Each element is a list belonging to an individual language as designated by the
standardized nomenclatures.

The first element of each element is the ISO 639-3 code for the language and, in
cases where the language was – also – defined in ISO 639-2, the terminological
code for ISO 639-2.

The second element of any sublist is the ISO 639-1 code for a language, if
designated/defined by the standardized nomenclature.

The third element is the bibliographical code for ISO 639-2, if
designated/defined by the standardized nomenclature.

The fourth element is whether the ISO 639-3 code has been retired or not (a
boolean).

The fifth element is the Scope of the language, as a symbol of a single letter;
this gets expanded into something human-readable in
`iso-639--create-lang-container'.

The sixth element is the Type of the language, as a symbol of a single letter;
this gets expanded into something human-readable in
`iso-639--create-lang-container'.

The seventh element is a list of tags which are mapped to from the language
\(which is a macrolanguage, if the list is not empty).

The eighth element is the linguistic family that the language belongs to.

The ninth element is the name of the language in its native language.

The tenth element is the name of the language in English.")

(defalias 'iso-639--code-1      'cadr)
(defalias 'iso-639--code-2      'caddr)
(defalias 'iso-639--code-3      'car)
(defalias 'iso-639--retired     'cadddr)
(defun     iso-639--scope       (lang)
  "Helper function to get the scope from the language structure, LANG."

        (car    (cddddr lang)))
(defun     iso-639--type        (lang)
  "Helper function to get the type from the language structure, LANG."

        (cadr   (cddddr lang)))
(defun     iso-639--mappings    (lang)
  "Helper function to get the mappings, if a macrolanguage, from a LANG structure."

        (caddr  (cddddr lang)))
(defun     iso-639--family      (lang)
  "Helper function to get the language family from the language structure, LANG."

        (cadddr (cddddr lang)))
(defun     iso-639--name-native (lang)
  "Helper function to get the name, in the native spelling, from a LANG structure."

  (car  (cddddr (cddddr lang))))
(defun     iso-639--name        (lang)
  "Helper function to get the name from the language structure, LANG."

  (cadr (cddddr (cddddr lang))))

(defun iso-639--create-lang-container (lang &optional skipmacrop)
  "Form a recognizable data structure from LANG, an entry of `iso-639---languages'.

If SKIPMACROP is \\='t\\=', the function does not try to lookup any mappings to
other langauges – if LANG is a macrolanguage – or, alteratively, does not try to
see if LANG is a mapping to a macrolanguage."

  (delq nil
        `((code-3      . ,(iso-639--code-3 lang))
          (code-2      . ,(when (iso-639--code-2 lang)
                            `((biblio  . ,(iso-639--code-2 lang))
                              (termino . ,(or (iso-639--code-3 lang)
                                              (iso-639--code-2 lang))))))
          (code-1      . ,(iso-639--code-1 lang))
          (belongs-to  . ,(delq nil `(,(and (iso-639--code-1 lang) 'iso-639-1)
                                      ,(and (iso-639--code-2 lang) 'iso-639-2)
                                      ,(and (iso-639--code-3 lang) 'iso-639-3))))
          (retired     . ,(iso-639--retired lang))
          (scope       . ,(let ((s (iso-639--scope lang)))
                            (cond
                             ((eq s 'C) (concat "Collections of languages "
                                                "connected, for example "
                                                "genetically or by region"))
                             ((eq s 'I) "Individual language")
                             ((eq s 'M) "Macrolanguage")
                             ((eq s 'S) "Special code"))))
          (type        . ,(let ((p (iso-639--type lang)))
                            (cond
                             ((eq p 'A) "Ancient (extinct since ancient times)")
                             ((eq p 'C) "Constructed")
                             ((eq p 'E) "Extinct (in recent times)")
                             ((eq p 'H) "Historical (distinct from its modern form)")
                             ((eq p 'L) "Living")
                             ((eq p 'S) "Special code"))))
          ,(unless skipmacrop
             (let ((mappings (iso-639--mappings lang)))
               (cons (if mappings 'mappings 'macrolang)
                     (if mappings
                         (mapcar (lambda (mapping)
                                   (iso-639-find-by-code mapping t))
                                 mappings)
                       (when-let ((l (seq-find (lambda (ml)
                                                 (member (iso-639--code-3 lang)
                                                         (iso-639--mappings ml)))
                                               iso-639---languages)))
                         (iso-639--create-lang-container l t))))))
          (family      . ,(iso-639--family      lang))
          (name-native . ,(iso-639--name-native lang))
          (name        . ,(iso-639--name        lang)))))

(defun iso-639-find-by-code (code &optional skipmacrop)
  "Provided a two- or three-letter CODE, try to find a corresponding language.

All three – ISO 639-1, ISO 639-2, and ISO 639-3 – are searched to find a match.

If SKIPMACROP is \\='t\\=', the function does not try to lookup any mappings to
other langauges – if CODE is for a macrolanguage – or, alteratively, does not
try to see if CODE has a mapping to a macrolanguage."

  (if (or (not (stringp code)) (not (< 1 (length code) 4)))
      (error "Argument is not an ISO 639-1–3 code: %s" code)
    (when-let ((l (seq-find (lambda (lang)
                              (member (downcase code) (seq-subseq lang 0 3)))
                            iso-639---languages)))
      (iso-639--create-lang-container l skipmacrop))))
(defun iso-639-1-find-by-code (code &optional skipmacrop)
  "Provided a two-letter CODE, try to find a corresponding language.

Only ISO 639-1 codes are checked against.

If SKIPMACROP is \\='t\\=', the function does not try to lookup any mappings to
other langauges – if CODE is for a macrolanguage – or, alteratively, does not
try to see if CODE has a mapping to a macrolanguage."

  (if (or (not (stringp code)) (/= (length code) 2))
      (error "Argument is not an ISO 639-1 code: %s" code)
    (when-let ((l (seq-find (lambda (lang)
                              (string= (downcase code) (iso-639--code-1 lang)))
                            iso-639---languages)))
      (iso-639--create-lang-container l skipmacrop))))
(defun iso-639-2-find-by-code (code &optional skipmacrop)
  "Provided a three-letter CODE, try to find a corresponding language.

Only ISO 639-2 codes are checked against.

If SKIPMACROP is \\='t\\=', the function does not try to lookup any mappings to
other langauges – if CODE is for a macrolanguage – or, alteratively, does not
try to see if CODE has a mapping to a macrolanguage."

  (if (or (not (stringp code)) (/= (length code) 3))
      (error "Argument is not an ISO 639-2 code: %s" code)
    (when-let ((l (seq-find (lambda (lang)
                              (and (iso-639--code-2 lang)
                                   (member (downcase code)
                                           (list (iso-639--code-2 lang)
                                                 (iso-639--code-3 lang)))))
                            iso-639---languages)))
      (iso-639--create-lang-container l skipmacrop))))
(defun iso-639-3-find-by-code (code &optional skipmacrop)
  "Provided a three-letter CODE, try to find a corresponding language.

Only ISO 639-3 codes are checked against.

If SKIPMACROP is \\='t\\=', the function does not try to lookup any mappings to
other langauges – if CODE is for a macrolanguage – or, alteratively, does not
try to see if CODE has a mapping to a macrolanguage."

  (if (or (not (stringp code)) (/= (length code) 3))
      (error "Argument is not an ISO 639-3 code: %s" code)
    (when-let ((l (seq-find (lambda (lang)
                              (string= (downcase code) (iso-639--code-3 lang)))
                            iso-639---languages)))
      (iso-639--create-lang-container l skipmacrop))))

(defun iso-639-find-by-name (name)
  "Try to find a language which has a name which is closest to NAME.

This can never be precise; it is offered in case imprecise convenience is
desired or when a (possibly) \"good enough\" result may return."

  (let ((winner   nil)
        (distance  -1))
    (mapc (lambda (lang)
            (let ((d (string-distance (downcase name)
                                      (downcase (iso-639--name lang)))))
              (when (or (= distance -1) (< d distance))
                (setq distance d)
                (setq winner   lang))))
          iso-639---languages)

    (iso-639--create-lang-container winner)))



(defun iso-639-get-code-of-1 (lang)
  "Given a language structure of LANG, return its ISO 639-1 code (if it has one)."

  (alist-get 'code-1 lang))
(defun iso-639-get-code-of-2 (lang)
  "Given a language structure of LANG, return its ISO 639-2 codes (if it has any)."

  (alist-get 'code-2 lang))
(defun iso-639-get-code-of-2-bibliographical (lang)
  "Given a language structure of LANG, return its bibliographical ISO 639-2 code."

  (alist-get 'biblio (alist-get 'code-2 lang)))
(defun iso-639-get-code-of-2-terminological (lang)
  "Given a language structure of LANG, return its terminological ISO 639-2 code."

  (alist-get 'termino (alist-get 'code-2 lang)))
(defun iso-639-get-code-of-3 (lang)
  "Given a language structure of LANG, return its ISO 639-3 code (if it has one)."

  (alist-get 'code-3 lang))

(defun iso-639-1-p (lang)
  "Given a language structure of LANG, return \\='t\\=' if it has an ISO 639-1 code."

  (member 'iso-639-1 (alist-get 'belongs-to lang)))
(defun iso-639-2-p (lang)
  "Given a language structure of LANG, return \\='t\\=' if it has an ISO 639-2 code."

  (member 'iso-639-2 (alist-get 'belongs-to lang)))
(defun iso-639-3-p (lang)
  "Given a language structure of LANG, return \\='t\\=' if it has an ISO 639-3 code."

  (member 'iso-639-3 (alist-get 'belongs-to lang)))

(defun iso-639-retired-p (lang)
  "Given a language structure of LANG, return \\='t\\=' if the code's been retired.

The only codes for which retiring has occured have been ISO 639-3; nonetheless,
– if the language's corresponding ISO 639-3 has been retired – this function
will tell you.

Retired ISO 639-3 codes generally do not have ISO 639-2 or ISO 639-1 codes."

  (alist-get 'retired lang))

(defun iso-639-get-scope (lang)
  "Given a language structure of LANG, return its scope (if it has one)."

  (alist-get 'scope lang))

(defun iso-639-get-type (lang)
  "Given a language structure of LANG, return its type (if it has one)."

  (alist-get 'type lang))

(defun iso-639-get-macrolanguage-mappings (lang)
  "Given a language structure of LANG, – if it's a macrolanguage – return mappings."

  (alist-get 'mappings lang))
(defun iso-639-get-macrolanguage-parent (lang)
  "Given a language structure of LANG, return any macrolanguages it maps to."

  (alist-get 'macrolang lang))

(defun iso-639-get-family (lang)
  "Given a language structure of LANG, return its language family (if it has one)."

  (alist-get 'family lang))

(defun iso-639-get-name-in-native-spelling (lang)
  "Given a language structure of LANG, return its name, spelled in the language."

  (alist-get 'native lang))

(defun iso-639-get-name (lang)
  "Given a language structure of LANG, return its name."

  (alist-get 'name lang))



(provide 'iso-639)

;; Local Variables:
;; indent-tabs-mode: nil
;; End:
;;; iso-639.el ends here
