;;; org-people.el --- Work with a contact-list in org-mode files -*- lexical-binding: t; -*-

;; Copyright (C) 2026  Steve Kemp

;; Author: Steve Kemp
;; Maintainer: Steve Kemp
;; Package-Version: 1.8.1
;; Package-Revision: 0423d6d0bfa7
;; Package-Requires: ((emacs "29.1") (org "9.0"))
;; Keywords: outlines, contacts, people
;; URL: https://github.com/skx/org-people

;; This file is not part of GNU Emacs.

;; This file is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This file is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this file.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:

;; A package to collect contacts from org-agenda files, and allow
;; operations to be carried out on them.
;;
;; By default all tree-items with a "contact" tag, which contain
;; properties, are considered contacts.
;;
;; An example pair of contacts would look like this inside an org-mode
;; file:
;;
;;   * People
;;     ** Alice                        :family:contact:
;;     :PROPERTIES
;;     :ADDRESS: 32 Something Street
;;     :EMAIL: alice@example.com
;;     :PHONE: +123 456 789
;;     :CHILDREN: Mallory
;;     :NICKNAME: Allu
;;     :END
;;     ** Bob                           :colleague:contact:
;;     :PROPERTIES
;;     :ADDRESS: 32 Something Lane
;;     :EMAIL: bob@example.com
;;     :PHONE: +123 456 987
;;     :END
;;
;; The specific properties used don't matter, although it seems natural
;; to use :ADDRESS, :EMAIL, and :PHONE.  A contact will be recorded
;; if there is at least one property present and the "contact" tag
;; being present.
;;
;; The name of the tag used to search for entries is specified in the
;; org-people-search-tag variable.
;;

;;; Basic operations

;; Once your agenda files have been loaded `org-people-summary' will
;; show a buffer of all known contacts, and from there you can copy
;; attributes, jump to their definitions, & etc.
;;
;; `org-people-insert' allows you to interactively insert data from
;; your contacts with helpful TAB-completion, on the attribute name
;; and person.

;;; org-table helpers

;; If you tag the contacts with more than just the `contacts` value
;; then you may use those tags to build simple tables of matching entries.
;; For example the following can auto-update:
;;
;;    #+NAME: get-family-contacts
;;    #+BEGIN_SRC elisp :results value table
;;    (org-people-tags-to-table "family")
;;    #+END_SRC
;;
;; If you prefer to include different columns in your generated table you
;; can specify them directly:
;;
;;   #+NAME: get-family-contacts
;;   #+BEGIN_SRC elisp :results value table
;;   (org-people-tags-to-table "family" '(:LINK :PHONE))
;;   #+END_SRC
;;
;; You may also create a table including all known data about a single named
;; individual:
;;
;;    #+NAME: steve-kemp
;;    #+BEGIN_SRC elisp :results value table :colnames '("Field" "Value")
;;    (org-people-person-to-table "Steve Kemp")
;;    #+END_SRC
;;
;; In this case properties listed in `org-people-ignored-properties` will
;; be ignored and excluded from the generated table.


;;; Version history (brief)

;;
;; 1.8 - Improvement: Filtering on :TAGS property in `M-x org-people-summary`
;;       uses sub-string matches of entries, rather than membership testing.
;;
;; 1.7 - BugFix: First column in org-people-summary-properties is used as the
;;       default sort key.  Now allows a width to be defined too.
;;
;; 1.6 - Open the property drawers when jumping to a contact, to allow
;;       viewing all appropriate details.
;;
;; 1.5 - All linting fixes implemented.
;;
;; 1.4 - `org-people-summary' now allows you to specify the fields
;;       which are displayed, via the new `org-people-summary-properties'
;;       configuration value.
;;
;; 1.3 - `org-people-summary' can now be filtered against all known properties.
;;       Not just the ones which are visible.  (i.e. Filter against ":ADDRESS")
;;
;; 1.2 - Rudimentary (single-contact-only) VCF export.
;;
;; 1.1 - Special support for nickname, and case insensitivity by default for
;;       completion.
;;       org-people-ignored-properties was introduced to ignore specific
;;       properties from completion and table-generation.
;;
;; 1.0 - Process all agenda-files by default, via a tag search for
;;       ":contact:" (by default).  This is more generally useful, and
;;       removes configuration and our ad-hoc caching implementation.
;;
;; 0.9 - org-people-person-to-table shows all the data about one individual
;;       as an `org-mode' table.
;;       Added test-cases in new file, org-people-test.el
;;
;; 0.8 - Provide "[[org-person:Name Here]]" support with completion,
;;       clicking, and export attributes.
;;       Make org-people-browse-name public and usefully available.
;;
;; 0.7 - Provide annotations for name-completion.
;;       Switch the org-people-summary to using tabulated-list-mode.
;;
;; 0.6 - The table-creating function has been renamed and updated.
;;       Now you can specify the fields to return.
;;
;; 0.5 - Drop simple functions.  They can be user-driver.
;;       Added filtering options and rewrote code to use them.
;;
;; 0.4 - Allow searching by property value.
;;       Added org-people-get-by-email, etc, using this new facility.
;;
;; 0.3 - :TAGS shows up as a comma-separated list in org-people-summary.
;;       org-people-summary is set to view-mode, so "q" buries the buffer.
;;
;; 0.2 - Added org-people-summary.
;;       Updated all contacts to have :TAGS and :NAME properties
;;       where appropriate.
;;
;; 0.1 - initial release
;;

;;; Code:

(require 'cl-lib)
(require 'seq)
(require 'org)


;; Avoid byte-compile warnings for org functions
(declare-function org-map-entries "org" (&rest args))
(declare-function org-get-tags "org" (&optional pom inherit))
(declare-function org-entry-properties "org" (&optional pom scope))
(declare-function org-heading-components "org" (&optional pom))
(declare-function org-complex-heading-regexp-format "org" ())

;;; Configuration:

(defvar org-people-search-tag "contact"
  "This is the tag-filter for finding contacts.")

(defvar org-people-search-type 'agenda
  "The filter which is used for finding entries, via `org-map-entries'.

By default this is configured to allow all of your agenda files to
be processed.  You might consider replacing this with a list of
file-paths, in which case only those specific files will be read
for contacts.")

(defvar org-people-summary-buffer-name
  "*Contacts*"
  "The name of the buffer to create when `org-people-summary' is invoked.")

(defvar org-people-summary-properties
  '((:NAME 30)
    (:EMAIL 35)
    (:PHONE 15)
    (:TAGS  15))
  "List of properties to display in `org-people-summary'.")

(defvar org-people-ignored-properties
  (list :MARKER)
  "Properties which are generally ignored from contacts.

These are properties which are specifically excluded when creating
an `org-mode' table from a persons details, or when completion is
being invoked by `org-people-insert'.")

(defvar org-people-summary-mode-map
  (let ((map (make-sparse-keymap)))
    (define-key map (kbd "RET") #'org-people-summary--open)
    (define-key map (kbd "c") #'org-people-summary--copy-field)
    (define-key map (kbd "f") #'org-people-summary--filter-by-property)
    (define-key map (kbd "v") #'org-people-summary--vcard)
    map)
  "Keymap for `org-people-summary-mode'.")




;;
;; Core
;;

(defun org-people-parse ()
  "Return hash table of NAME -> PLIST from all agenda-files.

We only include data from headlines which have a tag matching
`org-people-search-tag', and at least one property.

It is assumed the `org-mode' caching and parsing layer is fast
enough that there won't be undue performance problems regardless
of the number of contacts you have."
  (let ((table (make-hash-table :test #'equal)))
    (org-map-entries
         (lambda ()
           (let ((name (nth 4 (org-heading-components)))
                 (plist nil)
                 ; remove "contacts" from the tag-list
                 (entry-tags (remove org-people-search-tag (org-get-tags))))
             ;; Get any associated properties
             (dolist (prop (org-entry-properties nil 'standard))
               (let ((key (intern (concat ":" (car prop))))
                     (val (cdr prop)))
                 (setq plist (plist-put plist key val))))
             ;; Save the details if we have more than one property present.
             (when (and plist (> (/ (length plist) 2) 1))
               (setq plist (plist-put plist :NAME name))
               (setq plist (plist-put plist :MARKER (point-marker)))
               (setq plist (plist-put plist :TAGS entry-tags))
               (puthash name plist table))
             table))
         (concat "+" org-people-search-tag)
         org-people-search-type)
    table))

;;;###autoload
(defun org-people-insert ()
  "Insert a specific piece of data from a contact.

This uses `org-people-select-interactively' to first prompt the user
for a contact, and then a second interactive selection of the specific
attribute value which should be inserted.

Properties which are included in `org-people-ignored-properties' are
excluded from the completion."
  (interactive)
  (let* ((person (org-people-select-interactively))
         (values (org-people-get-by-name person))
         (ignored org-people-ignored-properties)
         (completion-ignore-case t)
         (completion-styles '(basic substring partial-completion)))
    (if values
        ;; Collect keys and remove ignored ones
        (let* ((keys (cl-loop for (k ) on values by #'cddr
                              unless (memq k ignored)
                              collect k))
               ;; prompt
               (choice (intern (completing-read
                                "Attribute: "
                                (mapcar #'symbol-name keys)
                                nil t))))
          ;; insert
          (insert (or (plist-get values choice) "")))
      (if person
          (user-error "No properties present for contact %s" person)
        (user-error "No contact selected")))))





;;
;; Filtering and searching functions that build upon the data-structure
;; which org-people-parse returns.
;;

(defun org-people-names ()
  "Return all known contact names.

This uses `org-people-parse' to get the list of parsed/discovered contacts."
  (sort (hash-table-keys (org-people-parse)) #'string<))

(defun org-people--properties (table)
  "Return a sorted list of all distinct plist keys used in TABLE."
  (let (keys)
    (maphash
     (lambda (_name plist)
       (while plist
         (let ((key (car plist)))
           (unless (memq key org-people-ignored-properties)
             (push key keys)))
         (setq plist (cddr plist))))
     table)
    (sort (delete-dups keys)
          (lambda (a b)
            (string< (symbol-name a)
                     (symbol-name b))))))

(defun org-people--alias-table ()
  "Return a hash of completion-string -> canonical-name.

This is used to allow TAB-completion against :NICKNAME properties
which might be associated with contacts."
  (let ((alias-table (make-hash-table :test #'equal))
        (people (org-people-parse)))
    (maphash
     (lambda (name plist)
       ;; Always allow completion on canonical name
       (puthash name name alias-table)

       ;; Support single nickname
       (when-let ((nick (plist-get plist :NICKNAME)))
         (puthash nick name alias-table))

       ;; Optional: support multiple aliases separated by ;
       (when-let ((aliases (plist-get plist :ALIASES)))
         (dolist (alias (split-string aliases ";" t "[[:space:]]*"))
           (puthash alias name alias-table))))
     people)
    alias-table))





;;
;; Completion-related code
;;

(defun org-people--completion-annotation (name)
  "Return a annotation-string for the specific candidate.

NAME is the name of the contact."
  (let* ((plist (org-people-get-by-name name))
         (email (plist-get plist :EMAIL))
         (phone (plist-get plist :PHONE))
         (alias-table (org-people--alias-table))
         (canonical (gethash name alias-table)))
    (string-join
     (delq nil
           (list (when email (format " [%s]" email))
                 (when phone (format " ☎ %s" phone))
                 (when (not (string= name canonical))
                   (format " (%s)" canonical))))
     "  ")))

(defun org-people--completion-table (string pred action)
  "Return a completion-table for contact completion.

STRING PRED and ACTION are passed to `complete-with-action', and
not used directly."
  (let* ((alias-table (org-people--alias-table))
         (candidates (hash-table-keys alias-table)))
    (if (eq action 'metadata)
        '(metadata
          (annotation-function . org-people--completion-annotation)
          (category . org-people))
      (complete-with-action action candidates string pred))))

(defun org-people-select-interactively ()
  "Select a contact by name or nickname."
  (let* ((alias-table (org-people--alias-table))
         (completion-ignore-case t)
         (completion-styles '(basic substring partial-completion))
         (completion
          (completing-read
           "Contact name: "
           (lambda (string pred action)
             (org-people--completion-table string pred action))
           nil t)))
    ;; Return canonical name
    (gethash completion alias-table)))




;;
;; Selection code
;;

(defun org-people-get-by-name (name)
  "Return plist for NAME from the contact-file.

This is basically the way of getting all data known about a given person."
  (gethash name (org-people-parse)))

(defun org-people-get-by-property (property value &optional regexp)
  "Return contacts by searching the contents of a specific field.

PROPERTY is the name of the property associated with entries, and
VALUE is the string to match with.

By default string-equality is used for matching, however if REGEXP
is true then `string-match' is used instead."
  (org-people-filter (lambda(plist)
                       (let ((found (plist-get plist property)))
                         (if regexp
                             (if (string-match value (or found ""))
                                 t)
                           (if (string-equal value (or found ""))
                             t))))))

(defun org-people-filter (pred-p)
  "Filter all known contacts by the given predicate.

PRED-P should be a function which accepts the plist of properties associated
with a given contact, and returns t if they should be kept.

See `org-people-get-by-property' for an example use of this function."
  (cl-loop
   for plist being the hash-values of (org-people-parse)
   when (funcall pred-p plist)
   collect plist))




;;
;; Generate org-mode tables
;;

(defun org-people-person-to-table (name)
  "Return table-data about a named contact.

This function is designed to create an `org-mode' table, like so:

#+NAME: myself
#+BEGIN_SRC elisp :results value table :colnames \='(\"Field\" \"Value\")
\(org-people-person-to-table \"Steve Kemp\")
#+END_SRC

Properties listed in `org-people-ignored-properties' are excluded from
the generated table."
  (let* ((plist (org-people-get-by-name name))
         ;; Convert plist to list of (key . value) pairs
         (pairs (seq-partition plist 2))

         ;; Remove ignored keys - :MARKER, etc.
         (filtered (seq-remove (lambda (pair)
                                 (memq (car pair) org-people-ignored-properties))
                               pairs))

         ;; Sort pairs alphabetically by key name (without leading :)
         (sorted
          (sort filtered
                (lambda (a b)
                  (string<
                   (substring (symbol-name (car a)) 1)
                   (substring (symbol-name (car b)) 1)))))
         ;; Convert to rows
         (rows
          (cl-remove-if (lambda (s) (null s))
          (mapcar
           (lambda (pair)
             (if (cadr pair)
             (list
              (capitalize
               (substring (symbol-name (car pair)) 1))
              (cadr pair))))
           sorted))))
    rows))

(defun org-people-tags-to-table (tag &optional props)
  "Return a list of contacts filtered by TAG.

This function is designed to create an `org-mode' table, like so:

#+NAME: family-contacts
#+BEGIN_SRC elisp :results value table
\(org-people-tags-to-table \"family\" \='(:NAME :PHONE))
#+END_SRC

PROPS is a list of property symbols to include, is nil we
default to (:NAME :PHONE :EMAIL).

The special value :LINK: will expand to a clickable link,
using the org-people: handler."
  (let ((people (org-people-parse))
        (props (or props '(:LINK :PHONE :EMAIL)))
        (result))
    (push props result) ; header
    (maphash
     (lambda (name plist)
       (let ((tags (or (plist-get plist :TAGS) '())))
         (when (member tag tags)
           (push
            (mapcar
             (lambda (prop)
               (if (eq prop :NAME)
                   name
                 (if (eq prop :LINK)
                     (org-link-make-string (concat  "[[org-people:" name "][" name "]]"))
                   (or (plist-get plist prop) ""))))
             props)
            result))))
     people)
    (nreverse result)))





;;
;; summary-buffer code, and associated helpers
;;

(defun org-people-export-to-vcard (contact)
  "Pop up a new buffer containing CONTACT (a plist) formatted as a vCard 3.0."
  (interactive)
  (let* ((name    (plist-get contact :NAME))
         (safe-name (replace-regexp-in-string "[[:space:]]+" "_" name))
         (filename (concat safe-name ".vcf"))
         (email   (plist-get contact :EMAIL))
         (phone   (plist-get contact :PHONE))
         (address (plist-get contact :ADDRESS))
         (buf (find-file-noselect filename)))

    (with-current-buffer buf
      (insert "BEGIN:VCARD\n")
      (insert "VERSION:3.0\n")

      ;; Full name
      (when name
        (insert (format "FN:%s\n" name))
        ;; Structured name (basic split: last word = surname)
        (let* ((parts (split-string name " "))
               (first (string-join (butlast parts) " "))
               (last  (car (last parts))))
          (insert (format "N:%s;%s;;;\n" (or last "") (or first "")))))

      ;; Email
      (when email
        (insert (format "EMAIL;TYPE=INTERNET:%s\n" email)))

      ;; Phone
      (when phone
        (insert (format "TEL;TYPE=CELL:%s\n" phone)))

      ;; Single-line address
      ;; vCard ADR format is:
      ;; ADR;TYPE=HOME:POBOX;EXT;STREET;CITY;REGION;POSTCODE;COUNTRY
      ;; We'll put everything into STREET only.
      (when address
        (insert (format "ADR;TYPE=HOME:;;%s;;;;\n" address)))

      (insert "END:VCARD\n")
      (goto-char (point-min))

      ;; Enable vcard-mode if available
      (when (fboundp 'vcard-mode)
        (vcard-mode)))

    (pop-to-buffer buf)))


(defun org-people--open-properties ()
  "Open the property drawer beneath current headline."
  (save-excursion
    (org-back-to-heading t)
    (outline-show-subtree)
    (when (re-search-forward org-property-drawer-re
                             (save-excursion (org-end-of-subtree t t))
                             t)
      (org-fold-region (match-beginning 0)
                       (match-end 0)
                       nil))))

(defun org-people-browse-name (&optional name)
  "Open the Org entry for NAME.

If NAME is not set then prompt for it interactively.

This is used by our [[people:xxx]] handler."
  (interactive)
  (if (not name)
      (setq name (org-people-select-interactively)))
  (let ((marker (plist-get (org-people-get-by-name name) :MARKER)))
    (switch-to-buffer (marker-buffer marker))
    (goto-char marker)
    (org-reveal)
    (org-people--open-properties)))

(defun org-people--all-plists ()
  "Return a list of all contact plists."
  (cl-loop
   for plist being the hash-values of (org-people-parse)
   collect plist))

(defun org-people-summary--column-width (col-spec)
  "Return the width for COL-SPEC.

COL-SPEC can be a symbol (:NAME) or a list (:PROP WIDTH).
Defaults are used if WIDTH is not specified."
  (cond
   ;; (:PROP WIDTH) → use WIDTH
   ((and (listp col-spec) (symbolp (car col-spec)) (numberp (cadr col-spec)))
    (cadr col-spec))

   ;; just a symbol → default width
   ((symbolp col-spec)
    (pcase col-spec
      (:NAME 30)
      (:EMAIL 35)
      (:PHONE 15)
      (:TAGS  15)
      (_ 20)))  ; fallback for new properties

   ;; fallback in case of weird input
   (t 20)))

(defun org-people-summary--format ()
  "Build `tabulated-list-format' from `org-people-summary-properties'.

Supports `(:PROP WIDTH)` style for custom widths."
  (vconcat
   (mapcar
    (lambda (entry)
      (let ((prop (if (listp entry) (car entry) entry)))
        (list
         (capitalize (substring (symbol-name prop) 1))
         (org-people-summary--column-width entry)
         t)))
    org-people-summary-properties)))

(define-derived-mode org-people-summary-mode tabulated-list-mode "Org-People"
  "Major mode for listing Org People contacts."


  (setq tabulated-list-format
        (org-people-summary--format))

  (setq tabulated-list-padding 2)

  ;; Default sort = first column
  ;; We support both ":NAME" and ":NAME WIDTH" here, via the consp test.
  (setq tabulated-list-sort-key
        (let* ((first (car org-people-summary-properties))
               (name  (if (consp first)
                          (car first)
                        first)))
          (cons (capitalize
                 (substring
                  (symbol-name name) 1))
                nil)))

  (add-hook 'tabulated-list-revert-hook
            #'org-people-summary--refresh
            nil t)

  (tabulated-list-init-header))

(defun org-people-summary--entry (plist)
  "Convert PLIST to a `tabulated-list-mode' entry.

This formats using the value of `org-people-summary-properties' to
format the entry for display."
  (let* ((name  (or (plist-get plist :NAME) ""))
         (columns
          (mapcar
           (lambda (entry)
             (let ((prop (if (listp entry) (car entry) entry)))
               (cond
                ((eq prop :NAME) name)
                ((eq prop :TAGS)
                 (mapconcat #'identity
                            (or (plist-get plist :TAGS) '())
                            ","))
                (t (or (plist-get plist prop) "")))))
           org-people-summary-properties)))
    (list name (vconcat columns))))

(defun org-people-summary--refresh ()
  "Populate `tabulated-list-entries'."
  (setq tabulated-list-entries
        (mapcar #'org-people-summary--entry
                (org-people--all-plists))))

(defun org-people-summary--open ()
  "Open the Org entry for the contact at point."
  (interactive)
  (let* ((name (tabulated-list-get-id))
         (plist (org-people-get-by-name name)))
    (unless plist
      (user-error "No contact found: %s" name))
    (org-people-browse-name name)))

(defun org-people-summary--vcard ()
  "Create a vcard contact for the contact at point."
  (interactive)
  (let* ((name (tabulated-list-get-id))
         (plist (org-people-get-by-name name)))
    (if plist
        (org-people-export-to-vcard plist)
      (user-error "No contact found: %s" name))))

(defun org-people--export-person-link (path desc backend)
  "Export a person link for BACKEND.
PATH is the person name, DESC is the description.

We just make the name bold."
  (let ((name (or desc path)))
    (cond
     ((eq backend 'html)
      (format "<strong>%s</strong>" name))
     (t
      name))))

(defun org-people-summary--copy-field ()
  "Copy the value of the field under point to the clipboard."
 (interactive)
  (let* ((entry (tabulated-list-get-entry))
         (columns tabulated-list-format)
         (start 0)
         col)
    ;; Determine which column the point is in
    (catch 'found
      (dotimes (i (length columns))
        (let* ((col-info (if (vectorp columns) (aref columns i) (nth i columns)))
               (width (if (vectorp col-info) (aref col-info 1) (nth 1 col-info))))
          (when (<= start (current-column) (+ start width))
            (setq col i)
            (throw 'found t))
          (setq start (+ start width)))))
    ;; Copy value if found
    (if (and entry col)
        (let ((value (aref entry col)))  ;; entry is always a vector
          (kill-new value)
          (message "Copied: %s" value))
      (message "Could not determine field under point"))))

(defun org-people-summary--filter-by-property ()
  "Filter contacts interactively by a property value."
  (interactive)
  (let* ((completion-ignore-case t)
         (completion-styles '(basic substring partial-completion))
         (prop-str (completing-read
                    "Property to filter against: "
                    (org-people--properties (org-people-parse))
                    nil t))
         (prop (intern prop-str))
         (value (read-string (format "Value to match for %s: " prop-str))))
    ;; Filter list
    (let ((filtered
           (org-people-filter
             (lambda (plist)
               (let ((v (plist-get plist prop)))
                 (cond
                  ((listp v)
                   (seq-some (lambda (item)
                               (and (stringp item)
                                    (string-match-p value item)))
                             v))
                  ((stringp v)
                   (string-match-p value v))
                  (t nil)))))))
      ;; Refresh buffer
      (with-current-buffer org-people-summary-buffer-name
        (let ((inhibit-read-only t))
          (erase-buffer)
          (setq tabulated-list-entries
                (mapcar #'org-people-summary--entry filtered))
          (tabulated-list-print t))))))



;;;###autoload
(defun org-people-summary ()
  "Display contacts using `tabulated-list-mode'.

This allows sorting by each column, etc.

Filtering can be applied (using a regexp), and fields copied."
  (interactive)

  (let ((buf (get-buffer-create org-people-summary-buffer-name)))
    (with-current-buffer buf
      (org-people-summary-mode)
      (org-people-summary--refresh)
      (tabulated-list-print t))
    (pop-to-buffer buf)))





;;
;; Define a handler for a link of the form "org-person:XXX"
;;

(org-link-set-parameters
 "org-people"
 :complete #'org-people-select-interactively
 :export   #'org-people--export-person-link
 :follow   #'org-people-browse-name
 :help-echo "Open the contacts-file at the position of the named person, via org-people")




;;
;; Utility functions for users
;;

;;;###autoload
(defun org-people-add-descriptions ()
  "Populate descriptions to all [[org-people:XXX]] links in the current buffer.

This ensures that all links have a description which matches the name of the
contact, descriptions are only added if they are missing."
  (interactive)
  (save-excursion
    (goto-char (point-min))
    ;; Search for org-people links
    (while (re-search-forward "\\[\\[org-people:\\([^]]+\\)\\]\\]" nil t)
      (let ((target (match-string 1)))
        ;; Replace with [[org-people:XXX][XXX]]
        (replace-match (format "[[org-people:%s][%s]]" target target) t t)))))


(provide 'org-people)
;;; org-people.el ends here
