;;; selected-window-contrast.el --- Highlight by brightness of text and background   -*- lexical-binding: t -*-

;; Copyright (c) 2025 github.com/Anoncheg1,codeberg.org/Anoncheg
;; Author: <github.com/Anoncheg1,codeberg.org/Anoncheg>
;; Keywords:  color, windows, faces, buffer, background
;; URL: https://codeberg.org/Anoncheg/selected-window-contrast
;; Package-Version: 0.2
;; Package-Revision: 72ff01857119
;; Created: 11 dec 2024
;; Package-Requires: ((emacs "29.4"))
;; SPDX-License-Identifier: AGPL-3.0-or-later

;;; License
;; This file is not part of GNU Emacs.

;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU Affero General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU Affero General Public License for more details.

;; You should have received a copy of the GNU Affero General Public License
;; along with this program.  If not,
;; see <https://www.gnu.org/licenses/agpl-3.0.en.html>.

;;; Commentary:
;; Highlight selected window by adjusting contrast of text ;;
;;  "foreground" and background.
;; Working good if you switch temes frequently, contrast will be kept.
;; Also this works for modeline.
;;  We also highligh cursor position, this may be disabled with
;;  M-x customize-variable RET selected-window-contrast-text-switch-mode

;; Usage:

;; (add-to-list 'load-path "path_to/selected-window-contrast") ; optional
;; (when (require 'selected-window-contrast nil 'noerror)
;;   (setopt selected-window-contrast-bg-selected 0.95)
;;   (setopt selected-window-contrast-bg-others 0.75)
;;   (setopt selected-window-contrast-text-selected 0.9)
;;   (setopt selected-window-contrast-text-others 0.6)
;;   (add-hook 'buffer-list-update-hook
;;             #'selected-window-contrast-highlight-selected-window))


;; To disable highlighting window with rectangle around pointer use:
;; (setopt selected-window-contrast-mode 1)


;; How this works:
;;  1) We get color with `face-attribute' `selected-frame' for
;;  foreground and backgraound.
;;  2) Convert color to HSL
;;  3) adjust brightness in direction of foreground-background average
;;  4) convert color to RGB, then to HEX
;;  5) apply color

;; Customize: M-x customize-group RET selected-window-contrast

;; Donate:
;; - BTC (Bitcoin) address: 1CcDWSQ2vgqv5LxZuWaHGW52B9fkT5io25
;; - USDT (Tether) address: TVoXfYMkVYLnQZV3mGZ6GvmumuBfGsZzsN
;; - TON (Telegram) address: UQC8rjJFCHQkfdp7KmCkTZCb5dGzLFYe2TzsiZpfsnyTFt9D

;; Other packages:
;; - Modern navigation in major modes https://github.com/Anoncheg1/firstly-search
;; - Search with Chinese	https://github.com/Anoncheg1/pinyin-isearch
;; - Ediff no 3-th window	https://github.com/Anoncheg1/ediffnw
;; - Dired history		https://github.com/Anoncheg1/dired-hist
;; - Copy link to clipboard	https://github.com/Anoncheg1/emacs-org-links
;; - Solution for "callback hell"	https://github.com/Anoncheg1/emacs-async1
;; - Restore buffer state	https://github.com/Anoncheg1/emacs-unmodified-buffer1
;; - outline.el usage		https://github.com/Anoncheg1/emacs-outline-it
;; - Call LLMs & AIfrom Org-mode block.  https://github.com/Anoncheg1/emacs-oai

;;; Code:
;; Touch: Global variables bound deep is not good, it is a type of the inversion of control.
;; I am the best that is why I am the winner.
(require 'color)
(require 'rect)

;; - configurable:
(defcustom selected-window-contrast-bg-selected nil
  "Non-nil used to set selected window background contrast in [0-1] range.
Higher value increase contrast between text and background.
This value change contrast of text regarding to background."
  :group 'selected-window-contrast
  :type '(choice (number :tag "contrast in [0-1] range")
                 (const :tag "Don't change default contrast of theme." nil)))

(defcustom selected-window-contrast-bg-others 0.8
  "Non-nil used to set not selected windows background contrast.
in [0-1] range."
  :group 'selected-window-contrast
  :type '(choice (number :tag "contrast [0-1].")
                 (const :tag "Don't change default contrast of theme." nil)))

(defcustom selected-window-contrast-text-selected nil
  "Non-nil used to set not selected windows text contrast in [0-1] range."
  :group 'selected-window-contrast
  :type '(choice (number :tag "Text contrast [0-1].")
                 (const :tag "Don't change default contrast of theme." nil)))

(defcustom selected-window-contrast-text-others nil
  "Non-nil used to set not selected windows text contrast in [0-1] range."
  :group 'selected-window-contrast
  :type '(choice (number :tag "Text contrast [0-1].")
                 (const :tag "Don't change default contrast of theme." nil)))

(defcustom selected-window-contrast-mode 3
  "Highlight window by 3 contrast and region, 1 by contrast, 2 by region."
  :group 'selected-window-contrast
  :type 'number)

(defcustom selected-window-contrast-region-timeout 0.5
  "Hightlight cursor position: Second for which to show rectangle around."
  :group 'selected-window-contrast
  :type 'float)

(defun selected-window-contrast--get-current-colors ()
  "Get current text and background color of default face.
Returns list: (foreground background), both strings."
  (list (face-attribute 'default :foreground (selected-frame))
        (face-attribute 'default :background (selected-frame))))

(defun selected-window-contrast--parse-color (color)
  "Return normalized RGB list for COLOR (hex or name).
COLOR: string, hex like '#rrggbb' or '#rrrrggggbbbb', or color name."
  (cond
   ((and (stringp color) (string-prefix-p "#" color))
    (let* ((hex (substring color 1))
           (len (length hex))
           (digits (/ len 3))
           (maxval (if (= digits 2) 255.0 65535.0)))
      (mapcar
       (lambda (i)
         (/ (string-to-number (substring hex (* i digits) (* (+ i 1) digits)) 16)
            maxval))
       '(0 1 2))))
   ((and (stringp color) (color-name-to-rgb color))
    (color-name-to-rgb color))
   (t (error "Invalid color: %s" color))))

(defun selected-window-contrast--color-to-hsl (color)
  "Convert COLOR (hex or name) to a normalized HSL list.
COLOR: string, hex or color name."
  (apply #'color-rgb-to-hsl (selected-window-contrast--parse-color color)))

(defun selected-window-contrast-adjust-contrast (text-color background-color bg-mag text-mag)
  "Maximize visual contrast between TEXT-COLOR and BACKGROUND-COLOR.
This function remaps the lightness component of both input colors so that:
- Colors above the midpoint (0.5) are pushed towards full white.
- Colors below the midpoint are pushed towards full black.
Arguments:
 TEXT-COLOR        String, name or hex.
 BACKGROUND-COLOR  String, name or hex.
 BG-MAG (float in [0,1]) controls stretching of contrast for background.
 TEXT-MAG (float in [0,1], optional): stretching of contrast for text.
Returns:
 List: (NEW-TEXT-RGB NEW-BACKGROUND-RGB), each as (R G B) floats in [0,1]."
  (let ((text-hsl (selected-window-contrast--color-to-hsl text-color))
        (bg-hsl   (selected-window-contrast--color-to-hsl background-color))
        (mid 0.5))
    (let ((res-text-rgb
           (if (not text-mag)
               (apply #'color-hsl-to-rgb text-hsl)
             ;; else
             (let* ((t-l (nth 2 text-hsl))
                    (new-t-l (if (> t-l mid)
                                 (+ mid (* text-mag (- 1.0 mid)))
                               (- mid (* text-mag (- mid 0))))))
               (apply #'color-hsl-to-rgb (list (nth 0 text-hsl) (nth 1 text-hsl) new-t-l)))))
          (res-bg-rgb (if (not bg-mag)
                          (apply #'color-hsl-to-rgb bg-hsl)
                        ;; else
                        (let* ((b-l (nth 2 bg-hsl))
                               (new-b-l (if (> b-l mid)
                                            (+ mid (* bg-mag (- 1.0 mid)))
                                          (- mid (* bg-mag (- mid 0))))))
                          (apply #'color-hsl-to-rgb (list (nth 0 bg-hsl) (nth 1 bg-hsl) new-b-l))))))
      (list res-text-rgb res-bg-rgb))))

(defun selected-window-contrast--rgb-to-hex (rgb &optional digits)
  "Convert normalized RGB list to hex string.
RGB: list of 3 floats in [0,1].  DIGITS: 2 or 4 digits/component."
  (apply #'color-rgb-to-hex (append rgb (list (or digits 2)))))

(defun selected-window-contrast-change-window (contrast-background contrast-text)
  "Increase contrast between text and background in buffer.
CONTRAST-BACKGROUND, CONTRAST-TEXT: float in [0,1]; of contrast 1 - is
full contrast, 0 - no contrast.
Works on both dark (light text/dark bg) and light (dark text/light bg) themes."
  (unless (or (when (and contrast-background
                         (or (not (numberp contrast-background))
                             (not (<= 0 contrast-background 1))))
                (message "Contrast-background must be floats in [0,1]"))
              (when (and contrast-text
                         (or (not (numberp contrast-text))
                             (not (<= 0 contrast-text 1))))
                (message "contrast-text must be floats in [0,1]")))
    (let* ((current-colors (selected-window-contrast--get-current-colors))
           (new-colors (selected-window-contrast-adjust-contrast (nth 0 current-colors)
                                                                 (nth 1 current-colors)
                                                                 contrast-background
                                                                 contrast-text)))
      (let ((background-rgb (nth 1 new-colors))
            (text-rgb (nth 0 new-colors)))
        (if (and contrast-background contrast-text)
            ;; :foreground (selected-window-contrast--rgb-to-hex text-rgb)
            (buffer-face-set (list :foreground (selected-window-contrast--rgb-to-hex text-rgb)
                                   :background (selected-window-contrast--rgb-to-hex background-rgb)))
          ;; else
          (when contrast-text
            (buffer-face-set (list :foreground (selected-window-contrast--rgb-to-hex text-rgb)
                                   :background (face-attribute 'default :background))))
          (when contrast-background
            (buffer-face-set (list :foreground (face-attribute 'default :foreground)
                                   :background (selected-window-contrast--rgb-to-hex background-rgb)))))))))

(defun selected-window-contrast-change-modeline (contrast-background contrast-text)
  "Adjust modeline brightness of text and background.
Arguments CONTRAST-BACKGROUND, CONTRAST-TEXT is float value to increase
or decrease contrast."
  (let* ((back (face-attribute 'mode-line-active :background))
         (fore (face-attribute 'mode-line-active :foreground)))
    (when (or (eq back 'unspecified) (eq back 'unspecified-bg))
      (setq back (face-attribute 'default :background)))
    (when (eq fore 'unspecified)
      (setq fore (face-attribute 'default :foreground)))

    (if (or (eq back 'unspecified)
            (eq back 'unspecified-bg)
            (eq fore 'unspecified))
        (message "backgound or foreground color is unspecified in active mode line.")
      ;; else
      (let* ((new-colors (selected-window-contrast-adjust-contrast fore
                                               back
                                               contrast-background
                                               contrast-text))
             (new-fore (apply #'selected-window-contrast--rgb-to-hex (nth 0 new-colors)))
             (new-back (apply #'selected-window-contrast--rgb-to-hex (nth 1 new-colors))))
        (set-face-attribute 'mode-line-active nil
                            :foreground new-fore
                            :background new-back)
        t))))

(defvar selected-window-contrast-prev-window nil)

(defun selected-window-contrast-mark-small-rectangle-temporary (window)
  "Mark a 2x2 rectangle around point for 1 sec, to hightlight WINDOW.
Use `rectangle-mark-mode'.  Deactivate rectangle after 1 second or less."
  (interactive)
  (when (and (eq window (selected-window))
             (not (window-minibuffer-p window)))
    ;; Enable rectangle selection.
    (progn (rectangle-mark-mode 1)
           (rectangle-next-line 2)
           (rectangle-forward-char 8)
           (rectangle-exchange-point-and-mark))
    ;; Start timer to deactivate mark and rectangle mode.
    (run-with-timer selected-window-contrast-region-timeout
                    nil (lambda (buf)
                          (with-current-buffer buf
                            (when (region-active-p)
                              ;; (exchange-point-and-mark)
                              (deactivate-mark))))
                    (current-buffer))))

(defun selected-window-contrast-highlight-selected-window-with-timeout ()
  "Highlight not selected windows with a different background color.
Timeout 0.1 sec.
For case of opening new frame with new buffer by call:
$ emacsclient -c ~/file"
  (run-with-idle-timer 0.4 nil #'selected-window-contrast-highlight-selected-window))

(defun selected-window-contrast-highlight-selected-window ()
  "Highlight not selected windows with a different background color."
  (let ((cbn (buffer-name (current-buffer)))
        (sw (selected-window)))
    (when (/= (aref cbn 0) ?\s) ; ignore system buffers
      ;; - not selected:
      (when (or (eq selected-window-contrast-mode 1)
                (eq selected-window-contrast-mode 3))
        (walk-windows (lambda (w)
                        (unless (or (eq sw w)
                                    (eq cbn (buffer-name (window-buffer w))))
                          (with-selected-window w
                            (buffer-face-set 'default)
                            (selected-window-contrast-change-window
                             selected-window-contrast-bg-others
                             selected-window-contrast-text-others))))
                      -1)) ; -1 means to not include minimuber

      ;; - selected:
      (when (or (eq selected-window-contrast-mode 1)
                (eq selected-window-contrast-mode 3))
        (selected-window-contrast-change-window selected-window-contrast-bg-selected selected-window-contrast-text-selected))
      (when (or (eq selected-window-contrast-mode 2)
                (eq selected-window-contrast-mode 3))
        (add-hook 'window-selection-change-functions #'selected-window-contrast-mark-small-rectangle-temporary nil t))
      )))

(provide 'selected-window-contrast)
;;; selected-window-contrast.el ends here
